"""
Model: L-Bracket with Mounting Holes
Description: A sturdy L-bracket for mounting, with reinforcement gusset
Author: Claude Code
Units: mm (millimeters)
"""

from build123d import *
from michelangelocc import MichelangeloModel, ModelMetadata


# === Parameters ===
# Main dimensions
arm_length = 60.0       # Length of each arm (mm)
arm_width = 40.0        # Width of the bracket (mm)
arm_thickness = 5.0     # Thickness of material (mm)

# Mounting holes
hole_diameter = 5.2     # M5 clearance hole (mm)
num_holes_per_arm = 2   # Number of holes per arm
hole_edge_distance = 12.0  # Distance from hole center to edge (mm)

# Reinforcement gusset
gusset_enabled = True   # Add triangular gusset for strength
gusset_thickness = 4.0  # Thickness of gusset (mm)
gusset_size = 25.0      # Size of gusset triangle (mm)

# Fillets
fillet_radius = 3.0     # Fillet on inside corner (mm)
edge_fillet = 1.0       # Small fillet on edges (mm)


# === Model Construction ===
# Create horizontal arm
horizontal_arm = Box(arm_length, arm_width, arm_thickness)
horizontal_arm = Pos(arm_length / 2, 0, arm_thickness / 2) * horizontal_arm

# Create vertical arm
vertical_arm = Box(arm_thickness, arm_width, arm_length)
vertical_arm = Pos(arm_thickness / 2, 0, arm_length / 2) * vertical_arm

# Combine arms
part = horizontal_arm + vertical_arm

# Add inside corner fillet
# The inside corner is at the junction of the two arms
inside_edges = part.edges().filter_by(
    lambda e: (
        abs(e.center().X - arm_thickness) < 2 and
        abs(e.center().Z - arm_thickness) < 2
    )
)
if len(inside_edges) > 0:
    part = fillet(inside_edges, fillet_radius)

# Add reinforcement gusset
if gusset_enabled:
    # Create triangular gusset using extrusion
    with BuildPart() as gusset_builder:
        with BuildSketch(Plane.XZ.offset(arm_width / 2 - gusset_thickness / 2)):
            with BuildLine():
                # Triangle from corner
                Polyline([
                    (arm_thickness, arm_thickness),
                    (arm_thickness + gusset_size, arm_thickness),
                    (arm_thickness, arm_thickness + gusset_size),
                    (arm_thickness, arm_thickness),
                ])
            make_face()
        extrude(amount=gusset_thickness)

    gusset = gusset_builder.part
    part += gusset

# Add mounting holes to horizontal arm
hole_spacing = (arm_length - 2 * hole_edge_distance) / max(1, num_holes_per_arm - 1)
for i in range(num_holes_per_arm):
    x_pos = hole_edge_distance + i * hole_spacing
    hole = Pos(x_pos, 0, 0) * Cylinder(hole_diameter / 2, arm_thickness * 2)
    part -= hole

# Add mounting holes to vertical arm
for i in range(num_holes_per_arm):
    z_pos = hole_edge_distance + i * hole_spacing
    hole = Pos(0, 0, z_pos) * Rot(0, 90, 0) * Cylinder(hole_diameter / 2, arm_thickness * 2)
    part -= hole

# Add small fillets to all outer edges for comfort and strength
try:
    outer_edges = part.edges().filter_by(
        lambda e: e.length < arm_length  # Exclude very long edges
    )
    part = fillet(outer_edges, edge_fillet)
except Exception:
    pass  # Skip if fillet fails on complex geometry


# === Export ===
model = MichelangeloModel(
    part=part,
    metadata=ModelMetadata(
        name="l_bracket",
        description=f"L-bracket {arm_length}x{arm_length}mm with M5 mounting holes",
        units="mm",
        tags=["bracket", "mechanical", "mounting", "structural"]
    )
)
