"""
Model: Box with Snap-Fit Lid
Category: mechanical
Patterns: mating parts, tolerances, snap-fits, shell operation, lip/groove
Description: A box with a removable lid using snap-fit connection.
             Demonstrates tolerance design for mating parts.
Author: Claude Code
Units: mm
"""

from build123d import *
from michelangelocc import MichelangeloModel, ModelMetadata

# === Parameters ===
# Overall dimensions (outer)
box_length = 60.0       # mm
box_width = 40.0        # mm
box_height = 30.0       # mm

# Wall and structural
wall_thickness = 2.5    # mm - sturdy walls
bottom_thickness = 2.0  # mm

# Lid parameters
lid_height = 8.0        # mm - total lid height
lid_lip_height = 4.0    # mm - lip that goes inside box
lid_lip_inset = 0.3     # mm - TOLERANCE: clearance for easy fit

# Corner treatment
corner_radius = 3.0     # mm

# === Construction ===

# PATTERN: Create hollow box using shell operation
# Start with solid, then hollow it out
with BuildPart() as box_builder:
    # Outer shell with rounded corners
    with BuildSketch():
        RectangleRounded(box_length, box_width, corner_radius)
    extrude(amount=box_height)

    # PATTERN: Shell operation using offset() to hollow out
    # Select top face as opening, offset inward (negative amount)
    top_face = box_builder.part.faces().sort_by(Axis.Z)[-1]
    offset(amount=-wall_thickness, openings=[top_face])

box = box_builder.part

# PATTERN: Create mating lid with lip
# Lid has outer cap + inner lip that fits inside box
with BuildPart() as lid_builder:
    # Outer cap (sits on top of box)
    with BuildSketch():
        RectangleRounded(box_length, box_width, corner_radius)
    extrude(amount=lid_height - lid_lip_height)

    # PATTERN: Inner lip with TOLERANCE for mating
    # Lip dimensions = box inner dimensions - clearance
    inner_length = box_length - 2 * wall_thickness - 2 * lid_lip_inset
    inner_width = box_width - 2 * wall_thickness - 2 * lid_lip_inset

    # Add lip extending downward
    with BuildSketch(Plane.XY):
        RectangleRounded(inner_length, inner_width, max(0.5, corner_radius - wall_thickness))
    extrude(amount=-lid_lip_height)

    # PATTERN: Hollow the lip to reduce material and improve fit
    lip_wall = 1.5  # mm - lip wall thickness
    hollow_length = inner_length - 2 * lip_wall
    hollow_width = inner_width - 2 * lip_wall

    if hollow_length > 2 and hollow_width > 2:
        with BuildSketch(Plane.XY):
            RectangleRounded(hollow_length, hollow_width, max(0.5, corner_radius - wall_thickness - lip_wall))
        extrude(amount=-lid_lip_height, mode=Mode.SUBTRACT)

lid = lid_builder.part

# === Export ===
# NOTE: For this example, we export just the box.
# The lid would be exported separately for actual printing.
# To export lid: create separate MichelangeloModel with part=lid
model = MichelangeloModel(
    part=box,  # Export box only for valid single-part model
    metadata=ModelMetadata(
        name="snap_fit_enclosure",
        description=f"Box {box_length}x{box_width}x{box_height}mm with snap-fit lid, {lid_lip_inset}mm tolerance",
        units="mm"
    )
)
