"""
Model: Parametric Spur Gear
Category: mechanical
Patterns: parametric design, derived parameters, polar patterns, boolean operations
Description: A parametric spur gear with configurable tooth count and dimensions.
             Demonstrates mathematical calculations for mechanical parts.
Author: Claude Code
Units: mm
"""

from build123d import *
from michelangelocc import MichelangeloModel, ModelMetadata
import math

# === Parameters ===
# Gear specifications
num_teeth = 20              # Number of teeth
module = 2.0                # Module (tooth size) in mm
pressure_angle = 20.0       # Pressure angle in degrees
gear_height = 10.0          # Thickness of gear in mm

# Shaft hole
shaft_diameter = 8.0        # Center hole diameter in mm
keyway_width = 3.0          # Keyway width (set to 0 to disable)
keyway_depth = 1.5          # Keyway depth

# Hub (optional raised center)
hub_diameter = 20.0         # Hub outer diameter (0 to disable)
hub_height = 5.0            # Hub height above gear face

# === Derived Parameters ===
# PATTERN: Calculate derived dimensions from base parameters
pitch_diameter = module * num_teeth
outer_diameter = pitch_diameter + 2 * module
root_diameter = pitch_diameter - 2.5 * module

# === Model Construction ===

# PATTERN: Create gear profile as single 2D sketch with teeth
with BuildPart() as gear_builder:
    with BuildSketch():
        # Start with outer circle
        Circle(outer_diameter / 2)

        # PATTERN: Subtract tooth gaps using polar pattern
        # Each gap is a wedge cut between teeth
        tooth_angle = 360 / num_teeth
        gap_angle = tooth_angle * 0.5  # Gap is ~50% of tooth pitch

        for i in range(num_teeth):
            angle = i * tooth_angle + tooth_angle / 2  # Center gap between teeth
            angle_rad = math.radians(angle)

            # Create wedge for tooth gap
            gap_outer = outer_diameter / 2 + 1  # Extend past outer
            gap_inner = root_diameter / 2

            # Calculate gap wedge points
            half_gap_rad = math.radians(gap_angle / 2)

            # Inner arc points
            x1_inner = gap_inner * math.cos(angle_rad - half_gap_rad)
            y1_inner = gap_inner * math.sin(angle_rad - half_gap_rad)
            x2_inner = gap_inner * math.cos(angle_rad + half_gap_rad)
            y2_inner = gap_inner * math.sin(angle_rad + half_gap_rad)

            # Outer arc points
            x1_outer = gap_outer * math.cos(angle_rad - half_gap_rad)
            y1_outer = gap_outer * math.sin(angle_rad - half_gap_rad)
            x2_outer = gap_outer * math.cos(angle_rad + half_gap_rad)
            y2_outer = gap_outer * math.sin(angle_rad + half_gap_rad)

            # Create tooth gap as polygon
            with BuildLine():
                Polyline([
                    (x1_inner, y1_inner),
                    (x1_outer, y1_outer),
                    (x2_outer, y2_outer),
                    (x2_inner, y2_inner),
                    (x1_inner, y1_inner),
                ])
            make_face(mode=Mode.SUBTRACT)

    # Extrude gear profile
    extrude(amount=gear_height)

    # PATTERN: Subtract center shaft hole
    with BuildSketch():
        Circle(shaft_diameter / 2)
    extrude(amount=gear_height, mode=Mode.SUBTRACT)

    # PATTERN: Conditional feature - keyway
    if keyway_width > 0:
        with BuildSketch():
            with Locations((shaft_diameter / 2 + keyway_depth / 2, 0)):
                Rectangle(keyway_depth + 0.1, keyway_width)
        extrude(amount=gear_height, mode=Mode.SUBTRACT)

part = gear_builder.part

# PATTERN: Conditional feature - hub
if hub_diameter > 0 and hub_height > 0:
    with BuildPart() as hub_builder:
        with BuildSketch(Plane.XY.offset(gear_height)):
            Circle(hub_diameter / 2)
        extrude(amount=hub_height)

        # Subtract shaft hole from hub
        with BuildSketch(Plane.XY.offset(gear_height)):
            Circle(shaft_diameter / 2)
        extrude(amount=hub_height, mode=Mode.SUBTRACT)

        # Keyway in hub
        if keyway_width > 0:
            with BuildSketch(Plane.XY.offset(gear_height)):
                with Locations((shaft_diameter / 2 + keyway_depth / 2, 0)):
                    Rectangle(keyway_depth + 0.1, keyway_width)
            extrude(amount=hub_height, mode=Mode.SUBTRACT)

    hub = hub_builder.part
    part = part + hub

# === Export ===
model = MichelangeloModel(
    part=part,
    metadata=ModelMetadata(
        name="parametric_gear",
        description=f"Spur gear with {num_teeth} teeth, module {module}mm, {outer_diameter:.1f}mm OD",
        units="mm"
    )
)
