"""
Model: Wall Hook
Category: household
Patterns: curved profile, mounting holes, load-bearing design, sweep/loft
Description: A sturdy wall-mounted hook for hanging items.
             Demonstrates curved profiles and structural design.
Author: Claude Code
Units: mm
"""

from build123d import *
from michelangelocc import MichelangeloModel, ModelMetadata
import math

# === Parameters ===
# Back plate (mounts to wall)
plate_width = 30.0      # mm
plate_height = 50.0     # mm
plate_thickness = 4.0   # mm

# Hook arm
hook_length = 40.0      # mm - how far hook extends from wall
hook_thickness = 8.0    # mm - structural thickness
hook_width = 20.0       # mm - width of hook arm
hook_curve_radius = 15.0    # mm - radius of the curved tip

# Mounting holes
hole_diameter = 4.0     # mm - for #8 screws
hole_spacing = 30.0     # mm - vertical distance between holes
countersink_diameter = 8.0  # mm
countersink_depth = 2.0     # mm

# Structural
fillet_radius = 3.0     # mm - stress relief at joint

# === Construction ===

# PATTERN: Create back plate with mounting holes
with BuildPart() as plate_builder:
    # Main plate
    Box(plate_thickness, plate_width, plate_height)

    # PATTERN: Countersunk mounting holes
    # Position holes symmetrically, add countersink for flush screw heads
    for z_offset in [-hole_spacing / 2, hole_spacing / 2]:
        # Through hole
        with BuildSketch(Plane.YZ.offset(-plate_thickness / 2)):
            with Locations((0, z_offset)):
                Circle(hole_diameter / 2)
        extrude(amount=plate_thickness, mode=Mode.SUBTRACT)

        # Countersink (wider hole on back side)
        with BuildSketch(Plane.YZ.offset(-plate_thickness / 2)):
            with Locations((0, z_offset)):
                Circle(countersink_diameter / 2)
        extrude(amount=countersink_depth, mode=Mode.SUBTRACT)

plate = plate_builder.part

# PATTERN: Create hook arm using extrusion with profile
# Hook extends from front of plate, curves down at end
with BuildPart() as hook_builder:
    # Create hook profile in XZ plane (side view of hook)
    with BuildSketch(Plane.XZ):
        # Start at plate front, go out, curve down
        with BuildLine():
            # Straight section from plate
            Line((plate_thickness / 2, plate_height / 4), (plate_thickness / 2 + hook_length - hook_curve_radius, plate_height / 4))
            # Curved tip going down
            arc_center = (plate_thickness / 2 + hook_length - hook_curve_radius, plate_height / 4 - hook_curve_radius)
            RadiusArc(
                (plate_thickness / 2 + hook_length - hook_curve_radius, plate_height / 4),
                (plate_thickness / 2 + hook_length, plate_height / 4 - hook_curve_radius),
                hook_curve_radius
            )
            # Down section
            Line(
                (plate_thickness / 2 + hook_length, plate_height / 4 - hook_curve_radius),
                (plate_thickness / 2 + hook_length, plate_height / 4 - hook_curve_radius - hook_curve_radius)
            )
        # Make it a closed rectangle by adding thickness
        make_face()

    # Extrude to create 3D hook
    extrude(amount=hook_width, both=True)

hook_arm = hook_builder.part

# PATTERN: Combine parts with union
# Center hook on plate
hook_arm_positioned = Pos(0, 0, 0) * hook_arm
part = plate + hook_arm_positioned

# PATTERN: Add fillet at joint for stress relief
# Fillet where hook meets plate (high stress area)
try:
    # Find edges at the joint
    joint_edges = part.edges().filter_by(
        lambda e: abs(e.center().X - plate_thickness / 2) < 1 and
                  abs(e.center().Z - plate_height / 4) < hook_thickness
    )
    if len(joint_edges) > 0:
        part = fillet(joint_edges, fillet_radius)
except Exception:
    pass  # Skip fillet if geometry is complex

# === Export ===
model = MichelangeloModel(
    part=part,
    metadata=ModelMetadata(
        name="wall_hook",
        description=f"Wall hook, {hook_length}mm reach, {plate_width}x{plate_height}mm plate",
        units="mm"
    )
)
