"""
Model: Custom Keychain
Category: functional
Patterns: text embossing, thin features, ring holes, rounded edges
Description: A personalized keychain with embossed text and a ring hole.
             Demonstrates text operations, thin geometry, and hole placement.
Author: Claude Code
Units: mm
"""

from build123d import *
from michelangelocc import MichelangeloModel, ModelMetadata

# === Parameters ===
# Main body dimensions
body_length = 50.0      # mm - length of keychain body
body_width = 20.0       # mm - width of keychain body
body_thickness = 4.0    # mm - thickness (min 3mm for strength)

# Ring hole parameters
ring_hole_diameter = 5.0    # mm - for standard keyring
ring_hole_offset = 5.0      # mm - distance from edge to hole center

# Text parameters
text_content = "HELLO"      # Text to emboss
text_size = 8.0             # mm - font size
text_depth = 1.0            # mm - emboss height (positive) or engrave depth (negative)
emboss = True               # True = raised text, False = engraved text

# Edge treatment
corner_radius = 3.0         # mm - rounded corners
edge_fillet = 0.5           # mm - edge softening

# === Construction ===

# PATTERN: Create base body with rounded corners
# Use RectangleRounded for smooth corners on thin parts
with BuildPart() as builder:
    with BuildSketch():
        RectangleRounded(body_length, body_width, corner_radius)
    extrude(amount=body_thickness)

    # PATTERN: Add ring hole for keyring attachment
    # Position hole with offset from edge, ensure it goes through
    hole_x = -body_length / 2 + ring_hole_offset
    with BuildSketch(Plane.XY.offset(body_thickness)):
        with Locations((hole_x, 0)):
            Circle(ring_hole_diameter / 2)
    extrude(amount=-body_thickness, mode=Mode.SUBTRACT)

    # PATTERN: Add text embossing/engraving
    # Place text on top face, centered on the right portion (away from hole)
    text_x = ring_hole_offset / 2  # Shift text right to avoid hole area
    top_face = builder.part.faces().sort_by(Axis.Z)[-1]

    with BuildSketch(top_face):
        with Locations((text_x, 0)):
            Text(text_content, font_size=text_size, align=(Align.CENTER, Align.CENTER))

    if emboss:
        extrude(amount=text_depth)  # Raised text
    else:
        extrude(amount=-text_depth, mode=Mode.SUBTRACT)  # Engraved text

    # PATTERN: Soften edges with small fillet for comfort
    # Use try/except as some edges may be too small to fillet
    try:
        fillet(builder.part.edges(), edge_fillet)
    except Exception:
        pass  # Skip fillet if geometry is too complex

part = builder.part

# === Export ===
model = MichelangeloModel(
    part=part,
    metadata=ModelMetadata(
        name="custom_keychain",
        description=f"Keychain with '{text_content}' text, {body_length}x{body_width}x{body_thickness}mm",
        units="mm"
    )
)
