"""
Model: Phone Stand
Category: household
Patterns: angled surfaces, cable routing, stability design, cutouts
Description: A simple phone stand with angled back support and cable slot.
             Demonstrates building connected geometry in single BuildPart.
Author: Claude Code
Units: mm
"""

from build123d import *
from michelangelocc import MichelangeloModel, ModelMetadata

# === Parameters ===
# Base dimensions
base_width = 80.0       # mm - width of stand
base_depth = 50.0       # mm - front to back
base_height = 6.0       # mm - thickness of base

# Back support (vertical wall to lean phone against)
back_height = 60.0      # mm - height of back support
back_thickness = 5.0    # mm

# Phone lip (front edge to hold phone)
lip_height = 10.0       # mm - how tall the front lip is
lip_depth = 8.0         # mm - how deep the lip is

# Cable routing
cable_slot_width = 15.0     # mm - for charging cable
cable_slot_depth = 10.0     # mm

# Structural
corner_radius = 3.0     # mm

# === Construction ===

# PATTERN: Build all connected parts in single BuildPart for solid model
with BuildPart() as builder:
    # Base plate
    with BuildSketch():
        RectangleRounded(base_width, base_depth, corner_radius)
    extrude(amount=base_height)

    # PATTERN: Back support wall - at back edge of base
    # Positioned to overlap with base for solid connection
    with BuildSketch(Plane.XY.offset(base_height)):
        with Locations((0, -base_depth / 2 + back_thickness / 2)):
            Rectangle(base_width - 2 * corner_radius, back_thickness)
    extrude(amount=back_height)

    # PATTERN: Front lip - at front edge of base
    with BuildSketch(Plane.XY.offset(base_height)):
        with Locations((0, base_depth / 2 - lip_depth / 2)):
            Rectangle(base_width - 2 * corner_radius, lip_depth)
    extrude(amount=lip_height)

    # PATTERN: Cable routing slot through the lip
    # Cut a slot for the charging cable
    with BuildSketch(Plane.XZ.offset(base_depth / 2)):
        with Locations((0, base_height + lip_height / 2)):
            RectangleRounded(cable_slot_width, lip_height + 2, 2)
    extrude(amount=-lip_depth - 2, mode=Mode.SUBTRACT)

    # Also cut through base for cable
    with BuildSketch(Plane.XZ.offset(base_depth / 2 - lip_depth)):
        with Locations((0, base_height / 2)):
            RectangleRounded(cable_slot_width, base_height + 2, 1)
    extrude(amount=-cable_slot_depth, mode=Mode.SUBTRACT)

part = builder.part

# PATTERN: Fillet edges for comfort
try:
    # Fillet top edges of back support
    top_edges = part.edges().filter_by(Axis.Z).sort_by(Axis.Z)[-4:]
    part = fillet(top_edges, 2.0)
except Exception:
    pass  # Skip if geometry doesn't support it

# === Export ===
model = MichelangeloModel(
    part=part,
    metadata=ModelMetadata(
        name="phone_stand",
        description=f"Phone stand with {back_height}mm back support, {base_width}x{base_depth}mm base",
        units="mm"
    )
)
