"""
Model: Twisted Organic Vase
Description: An artistic vase with smooth curves and a twist
Author: Claude Code
Units: mm (millimeters)
"""

from build123d import *
from michelangelocc import MichelangeloModel, ModelMetadata
import math


# === Parameters ===
# Overall dimensions
base_diameter = 60.0    # mm
top_diameter = 40.0     # mm
height = 120.0          # mm
wall_thickness = 3.0    # mm

# Shape parameters
num_sides = 8           # Number of sides (use high number for round)
twist_angle = 90.0      # Total twist from bottom to top (degrees)
bulge_amount = 0.3      # How much the middle bulges out (0-1)
bulge_position = 0.4    # Where the bulge is (0=bottom, 1=top)

# Quality
num_sections = 16       # Number of loft sections (higher = smoother)


# === Helper Functions ===
def ease_in_out(t: float) -> float:
    """Smooth interpolation function."""
    return t * t * (3 - 2 * t)


def calculate_radius(t: float, base_r: float, top_r: float, bulge: float, bulge_pos: float) -> float:
    """Calculate radius at height t (0 to 1) with optional bulge."""
    # Linear interpolation
    linear_r = base_r + (top_r - base_r) * t

    # Add bulge using gaussian-like curve
    bulge_factor = math.exp(-((t - bulge_pos) ** 2) / 0.1)
    bulge_r = linear_r * (1 + bulge * bulge_factor)

    return bulge_r


# === Model Construction ===
profiles = []

for i in range(num_sections):
    # Position along height (0 to 1)
    t = i / (num_sections - 1)
    z = t * height

    # Calculate radius at this height
    radius = calculate_radius(
        t,
        base_diameter / 2,
        top_diameter / 2,
        bulge_amount,
        bulge_position
    )

    # Calculate rotation for twist
    rotation = twist_angle * ease_in_out(t)

    # Create profile at this height
    with BuildSketch(Plane.XY.offset(z)) as profile:
        if num_sides >= 32:
            # Use circle for high side count
            Circle(radius)
        else:
            # Use polygon
            RegularPolygon(radius, num_sides, rotation=rotation)

    profiles.append(profile.sketch)

# Loft between all profiles to create outer shape
outer_part = loft(profiles)

# Create inner profiles for hollowing (thinner walls)
inner_profiles = []

for i in range(num_sections):
    t = i / (num_sections - 1)
    z = t * height

    radius = calculate_radius(
        t,
        base_diameter / 2,
        top_diameter / 2,
        bulge_amount,
        bulge_position
    )

    # Inner radius (accounting for wall thickness)
    inner_radius = max(radius - wall_thickness, 5.0)  # Minimum 5mm inner

    rotation = twist_angle * ease_in_out(t)

    with BuildSketch(Plane.XY.offset(z)) as profile:
        if num_sides >= 32:
            Circle(inner_radius)
        else:
            RegularPolygon(inner_radius, num_sides, rotation=rotation)

    inner_profiles.append(profile.sketch)

# Create inner void (skip bottom profile to create floor)
inner_part = loft(inner_profiles[1:])  # Start from second profile

# Subtract inner from outer
part = outer_part - Pos(0, 0, wall_thickness) * inner_part

# Optional: Add a decorative rim at the top
rim_height = 3.0
rim_thickness = 1.5

top_outer_radius = top_diameter / 2
top_inner_radius = top_outer_radius - wall_thickness

# Rim as a torus section
# rim = Pos(0, 0, height) * Torus(top_outer_radius - rim_thickness/2, rim_thickness/2)
# part += rim

# Add small fillet to top edge for comfort
# top_edges = part.edges().filter_by(lambda e: abs(e.center().Z - height) < 1)
# part = fillet(top_edges, 1.0)


# === Export ===
model = MichelangeloModel(
    part=part,
    metadata=ModelMetadata(
        name="twisted_vase",
        description=f"Organic vase with {twist_angle} degree twist, {num_sides} sides",
        units="mm",
        tags=["vase", "organic", "artistic", "decorative"]
    )
)
