---
name: agi-memory
description: Use when working with the AGI Memory System - creating memories, querying, testing, database operations, or understanding the memory architecture.
---

# AGI Memory System Skill

Use this skill when working with memory operations, database queries, tests, or the memory API.

## Quick Start - MCP Tools (Recommended)

Use the `adam-memory` MCP server tools for direct access:

```
# Search memories
mcp__adam-memory__memory_recall(query="user preferences", limit=10)

# Store semantic memory (facts, knowledge)
mcp__adam-memory__memory_store_semantic(
    content="User prefers yarn over npm",
    confidence=0.9,
    categories=["preference", "tools"],
    importance=0.8
)

# Store episodic memory (events)
mcp__adam-memory__memory_store_episodic(
    content="Fixed memory gateway parsing",
    action="bug_fix",
    context={"project": "adam"},
    result={"status": "success"}
)

# Working memory (temporary, auto-expires)
mcp__adam-memory__memory_working_add(content="Currently debugging hooks", duration_hours=2)

# Get packed context for prompts
mcp__adam-memory__memory_pack_context(query="react patterns", max_tokens=4000)

# System stats
mcp__adam-memory__memory_stats()
mcp__adam-memory__memory_recent(limit=10)
```

## CLI Helper (Alternative)

The `mem.sh` script in the project directory provides reliable database access:

```bash
cd ~/projects/agi-memory/agi-memory

# Search memories
./mem.sh recall "query here" 10

# Create semantic memory (facts, knowledge)
./mem.sh semantic "User prefers dark mode" 0.9 0.8

# Create episodic memory (events, conversations)
./mem.sh episodic "Completed project setup"

# Add to working memory (temporary, expires)
./mem.sh working "Currently working on memory integration" 2

# Check health
./mem.sh health

# Count memories
./mem.sh count

# Recent memories
./mem.sh recent 10

# Raw SQL
./mem.sh sql "SELECT * FROM memories LIMIT 5;"
```

## Memory Types

| Type | Command | Use For |
|------|---------|---------|
| Semantic | `./mem.sh semantic` | Facts, knowledge, user preferences |
| Episodic | `./mem.sh episodic` | Events, conversations, sessions |
| Procedural | `./mem.sh sql` | How-to steps (use SQL directly) |
| Strategic | `./mem.sh sql` | Patterns, strategies (use SQL directly) |
| Working | `./mem.sh working` | Temporary context (auto-expires) |

## When to Store Memories

**Semantic (facts):**
- User preferences discovered during conversation
- Important facts about projects or codebase
- Knowledge that should persist across sessions

**Episodic (events):**
- Work completed during sessions
- Conversations or interactions
- Debugging sessions and their outcomes

**Working (temporary):**
- Current task context
- Active conversation state
- Short-term reminders

## SQL Functions Reference

### Memory Creation

```sql
-- Semantic memory (facts)
SELECT create_semantic_memory(
    'content',           -- TEXT
    0.9,                 -- confidence (0-1)
    ARRAY['cat1'],       -- categories (optional)
    ARRAY['concept1'],   -- concepts (optional)
    '{"source": "x"}'::jsonb,  -- sources (optional)
    0.8                  -- importance (0-1)
);

-- Episodic memory (events)
SELECT create_episodic_memory(
    'content',
    '{"type": "action"}'::jsonb,    -- action
    '{"project": "x"}'::jsonb,      -- context
    '{"outcome": "success"}'::jsonb, -- result
    0.5,                 -- emotional valence (-1 to 1)
    NOW(),               -- event time
    0.7                  -- importance
);

-- Working memory (temporary)
SELECT add_to_working_memory('content', INTERVAL '1 hour');
```

### Memory Retrieval

```sql
-- Primary retrieval (use this!)
SELECT * FROM fast_recall('query', 10);

-- Search with filters
SELECT * FROM search_similar_memories(
    'query',
    10,                  -- limit
    ARRAY['semantic']::memory_type[],  -- types filter
    0.5                  -- min importance
);

-- Working memory
SELECT * FROM search_working_memory('query', 10);

-- Token-aware context packing (for LLM prompts)
SELECT * FROM pack_context(
    'query',             -- search query
    4000,                -- max tokens budget
    ARRAY['episodic', 'semantic']::memory_type[],  -- preferred types (optional)
    true,                -- include_skills (default true)
    true                 -- include_working_memory (default true)
);
-- Returns: packed_context (markdown), tokens_used, memory_count, skill_count, sections

-- Search learned skills
SELECT * FROM search_skills(
    'how to debug tests',  -- query
    5,                     -- limit
    0.3,                   -- min_confidence (default 0.3)
    0.0                    -- min_success_rate (default 0.0)
);
-- Returns: id, name, use_when, tool_sequence, confidence, success_rate
```

## Direct Docker Access

If mem.sh isn't available, use docker directly:

```bash
docker compose exec -T db psql -U postgres -d agi_memory -c "
    LOAD 'age';
    SET search_path = ag_catalog, public;
    SELECT * FROM fast_recall('query here', 10);
"
```

## System Health

```bash
./mem.sh health            # Memory statistics
./mem.sh count             # Count by type
./mem.sh recent 10         # Recent memories
```

Or via SQL:
```sql
SELECT * FROM memory_health;
SELECT * FROM stale_neighborhoods;
SELECT check_embedding_service_health();
```

## Common Commands

```bash
# Database management
./db-manage.sh start      # Start database
./db-manage.sh status     # Check health
./db-manage.sh shell      # Open psql
./db-manage.sh reset      # Reset (DESTRUCTIVE)

# Testing
pytest test.py -v
pytest test.py::test_name -v
```

## Troubleshooting

| Issue | Solution |
|-------|----------|
| `mem.sh` not found | `cd ~/projects/agi-memory/agi-memory` first |
| Container not running | `./db-manage.sh start` |
| AGE errors | Ensure using `mem.sh` or loading AGE manually |
| Embeddings fail | `docker compose logs embeddings` |
| No results | Check `./mem.sh count` for existing memories |
