# SQL Functions Quick Reference

## Memory Creation

### create_semantic_memory
```sql
SELECT create_semantic_memory(
    content TEXT,
    confidence FLOAT,           -- 0.0-1.0
    categories TEXT[],          -- optional
    concepts TEXT[],            -- optional
    sources JSONB,              -- optional
    importance FLOAT            -- 0.0-1.0
);
-- Returns: UUID
```

### create_episodic_memory
```sql
SELECT create_episodic_memory(
    content TEXT,
    action JSONB,               -- {"type": "...", ...}
    context JSONB,              -- {"key": "value", ...}
    result JSONB,               -- {"outcome": "...", ...}
    emotional_valence FLOAT,    -- -1.0 to 1.0
    event_time TIMESTAMPTZ,     -- usually NOW()
    importance FLOAT            -- 0.0-1.0
);
-- Returns: UUID
```

### create_procedural_memory
```sql
SELECT create_procedural_memory(
    content TEXT,
    steps JSONB,                -- ["step1", "step2", ...]
    prerequisites JSONB,        -- ["prereq1", ...] or NULL
    importance FLOAT            -- 0.0-1.0
);
-- Returns: UUID
```

### create_strategic_memory
```sql
SELECT create_strategic_memory(
    content TEXT,
    pattern TEXT,
    confidence FLOAT,           -- 0.0-1.0
    evidence JSONB,
    applicability JSONB,
    importance FLOAT            -- 0.0-1.0
);
-- Returns: UUID
```

### add_to_working_memory
```sql
SELECT add_to_working_memory(
    content TEXT,
    expiry INTERVAL             -- '1 hour', '30 minutes', etc.
);
-- Returns: UUID
```

## Memory Retrieval

### fast_recall (PRIMARY - use this!)
```sql
SELECT * FROM fast_recall(
    query_text TEXT,
    limit_count INTEGER
);
-- Returns: memory_id, content, memory_type, score, source
```

### search_similar_memories
```sql
SELECT * FROM search_similar_memories(
    query_text TEXT,
    limit_count INTEGER,
    memory_types memory_type[], -- ARRAY['semantic', 'episodic']::memory_type[]
    min_importance FLOAT        -- optional, e.g., 0.5
);
-- Returns: memory_id, content, type, similarity, importance
```

### search_working_memory
```sql
SELECT * FROM search_working_memory(
    query_text TEXT,
    limit_count INTEGER
);
-- Returns: memory_id, content, similarity, created_at
```

### pack_context (Token-aware retrieval for LLM prompts)
```sql
SELECT * FROM pack_context(
    query_text TEXT,
    max_tokens INTEGER,              -- default 4000
    prefer_types memory_type[],      -- optional, e.g., ARRAY['episodic']
    include_skills BOOLEAN,          -- default TRUE
    include_working_memory BOOLEAN   -- default TRUE
);
-- Returns: packed_context, tokens_used, memory_count, skill_count, sections
-- Budget allocation: 15% working memory, 20% skills, 65% memories
```

### search_skills
```sql
SELECT * FROM search_skills(
    query_text TEXT,
    limit_count INTEGER,             -- default 5
    min_confidence FLOAT,            -- default 0.3
    min_success_rate FLOAT           -- default 0.0
);
-- Returns: id, name, use_when, tool_sequence, confidence, success_rate
```

## Skills Management

### create_learned_skill
```sql
SELECT create_learned_skill(
    name TEXT,                       -- "Debug failing test"
    use_when TEXT,                   -- "when pytest shows failures"
    tool_sequence JSONB,             -- [{"tool": "grep", "args": {...}}, ...]
    description TEXT,                -- optional
    preferences JSONB,               -- optional user customizations
    prerequisites JSONB,             -- optional requirements
    source_memories UUID[],          -- episodic memories that taught this
    confidence FLOAT                 -- 0.0-1.0
);
-- Returns: UUID
```

### record_skill_application
```sql
SELECT record_skill_application(
    skill_id UUID,
    outcome TEXT,                    -- 'success', 'failure', 'partial', 'unknown'
    context JSONB,                   -- execution context
    outcome_details JSONB,           -- what happened
    duration_ms INTEGER              -- optional execution time
);
-- Returns: UUID (application record id)
```

## Relationships

### create_memory_relationship
```sql
SELECT create_memory_relationship(
    from_memory_id UUID,
    to_memory_id UUID,
    relationship_type graph_edge_type,  -- See types below
    properties JSONB
);
```

**Edge Types:**
- `TEMPORAL_NEXT` - Sequence in time
- `CAUSES` - Causal relationship
- `DERIVED_FROM` - Episodic → semantic
- `CONTRADICTS` - Conflict
- `SUPPORTS` - Evidence
- `INSTANCE_OF` - Memory → concept
- `PARENT_OF` - Concept hierarchy
- `ASSOCIATED` - Co-activation

### link_memory_to_concept
```sql
SELECT link_memory_to_concept(
    memory_id UUID,
    concept_name TEXT,
    strength FLOAT              -- 0.0-1.0
);
```

## Clusters

### create_memory_cluster
```sql
SELECT create_memory_cluster(
    name TEXT,
    cluster_type cluster_type,  -- 'theme', 'temporal', etc.
    description TEXT,
    keywords TEXT[]
);
-- Returns: UUID
```

### assign_memory_to_clusters
```sql
SELECT assign_memory_to_clusters(
    memory_id UUID,
    max_clusters INTEGER        -- typically 3
);
```

### recalculate_cluster_centroid
```sql
SELECT recalculate_cluster_centroid(cluster_id UUID);
```

## Maintenance

### cleanup_working_memory
```sql
SELECT cleanup_working_memory();
-- Returns: count of deleted items
```

### cleanup_embedding_cache
```sql
SELECT cleanup_embedding_cache(older_than INTERVAL);
-- e.g., INTERVAL '1 day'
```

## System Views

```sql
-- Memory statistics by type
SELECT * FROM memory_health;

-- Cluster analysis
SELECT * FROM cluster_insights;

-- Episode summaries
SELECT * FROM episode_summary;

-- Neighborhoods needing refresh
SELECT * FROM stale_neighborhoods;

-- Embedding service status
SELECT check_embedding_service_health();
```

## Debugging Queries

```sql
-- Count memories by type
SELECT type, COUNT(*) FROM memories GROUP BY type;

-- Recent memories
SELECT id, content, type, created_at
FROM memories ORDER BY created_at DESC LIMIT 10;

-- Check if memory exists
SELECT * FROM memories WHERE id = 'uuid-here'::uuid;

-- Memory with full details
SELECT m.*, sm.confidence, sm.categories
FROM memories m
JOIN semantic_memories sm ON m.id = sm.memory_id
WHERE m.id = 'uuid-here'::uuid;

-- Stale neighborhood count
SELECT COUNT(*) FROM memory_neighborhoods WHERE is_stale = TRUE;

-- Active clusters
SELECT * FROM memory_clusters ORDER BY importance_score DESC LIMIT 5;
```
