"""
Test template for AGI Memory System

Copy this template when creating new tests.
"""
import pytest
import asyncio
import asyncpg
import json
import time
import uuid

# Required: session-scoped async
pytestmark = pytest.mark.asyncio(loop_scope="session")

# Test session ID for unique identifiers
TEST_SESSION_ID = str(uuid.uuid4())[:8]


def get_test_identifier(test_name: str) -> str:
    """Generate unique identifier for test data to avoid conflicts"""
    return f"{test_name}_{TEST_SESSION_ID}_{int(time.time() * 1000)}"


@pytest.fixture(scope="session")
async def db_pool():
    """Create connection pool for testing"""
    pool = await asyncpg.create_pool(
        "postgresql://postgres:password@localhost:5432/agi_memory",
        ssl=False,
        min_size=2,
        max_size=20,
        command_timeout=60.0
    )
    yield pool
    await pool.close()


@pytest.fixture(autouse=True)
async def setup_db(db_pool):
    """Setup AGE extension before each test"""
    async with db_pool.acquire() as conn:
        await conn.execute("LOAD 'age';")
        await conn.execute("SET search_path = ag_catalog, public;")
    yield


# =============================================================================
# Example Tests - Replace with your tests
# =============================================================================

async def test_example_memory_creation(db_pool):
    """Test creating a memory"""
    async with db_pool.acquire() as conn:
        await conn.execute("LOAD 'age';")
        await conn.execute("SET search_path = ag_catalog, public;")

        test_id = get_test_identifier("example")
        content = f"Test memory {test_id}"

        # Create memory
        memory_id = await conn.fetchval("""
            SELECT create_semantic_memory($1, $2, NULL, NULL, NULL, $3)
        """, content, 0.9, 0.8)

        assert memory_id is not None

        # Verify it exists
        result = await conn.fetchrow("""
            SELECT * FROM memories WHERE id = $1
        """, memory_id)

        assert result is not None
        assert result['content'] == content


async def test_example_fast_recall(db_pool):
    """Test fast_recall function"""
    async with db_pool.acquire() as conn:
        await conn.execute("LOAD 'age';")
        await conn.execute("SET search_path = ag_catalog, public;")

        test_id = get_test_identifier("recall")
        content = f"Unique test content {test_id}"

        # Create memory first
        await conn.fetchval("""
            SELECT create_semantic_memory($1, $2, NULL, NULL, NULL, $3)
        """, content, 0.9, 0.8)

        # Recall it
        results = await conn.fetch("""
            SELECT * FROM fast_recall($1, 10)
        """, content)

        assert len(results) > 0
        # First result should match our content
        assert test_id in results[0]['content']


async def test_example_working_memory(db_pool):
    """Test working memory operations"""
    async with db_pool.acquire() as conn:
        await conn.execute("LOAD 'age';")
        await conn.execute("SET search_path = ag_catalog, public;")

        test_id = get_test_identifier("working")
        content = f"Working memory test {test_id}"

        # Add to working memory
        wm_id = await conn.fetchval("""
            SELECT add_to_working_memory($1, $2)
        """, content, "1 hour")

        assert wm_id is not None

        # Search working memory
        results = await conn.fetch("""
            SELECT * FROM search_working_memory($1, 5)
        """, content)

        assert len(results) > 0
