#!/usr/bin/env python3
"""
Alert system for drug discovery intelligence.

Usage:
    python alert_system.py --create my-alert --target "EGFR"
    python alert_system.py --check my-alert
    python alert_system.py --list
"""

import argparse
import json
import os
import sys
from datetime import datetime, timedelta
from typing import Optional, Dict, Any, List

try:
    import requests
except ImportError:
    requests = None


class AlertSystem:
    """Monitor and alert on drug discovery events."""

    def __init__(self, config_dir: str = None):
        self.config_dir = config_dir or os.path.expanduser("~/.drug-discovery-alerts")
        os.makedirs(self.config_dir, exist_ok=True)

        self.alerts_file = os.path.join(self.config_dir, "alerts.json")
        self.history_file = os.path.join(self.config_dir, "history.json")

        self.alerts = self._load_alerts()
        self.history = self._load_history()

        # Mock data sources
        self.MOCK_EVENTS = {
            "clinical": [
                {
                    "date": "2024-12-15",
                    "event": "trial_start",
                    "nct_id": "NCT01234567",
                    "sponsor": "AstraZeneca",
                    "phase": "3",
                    "indication": "NSCLC",
                    "targets": ["EGFR"]
                },
                {
                    "date": "2024-12-12",
                    "event": "status_change",
                    "nct_id": "NCT07890123",
                    "previous_status": "Recruiting",
                    "new_status": "Active, not recruiting",
                    "targets": ["EGFR", "T790M"]
                }
            ],
            "regulatory": [
                {
                    "date": "2024-12-12",
                    "event": "approval",
                    "agency": "FDA",
                    "sponsor": "AstraZeneca",
                    "drug": "Osimertinib",
                    "indication": "Adjuvant NSCLC",
                    "targets": ["EGFR"]
                }
            ],
            "publications": [
                {
                    "date": "2024-12-08",
                    "event": "publication",
                    "journal": "Nature Cancer",
                    "title": "Fourth-generation EGFR inhibitors targeting C797S",
                    "authors": "Zhang et al.",
                    "targets": ["EGFR", "C797S"]
                },
                {
                    "date": "2024-12-05",
                    "event": "publication",
                    "journal": "Lancet Oncology",
                    "title": "Combination therapy: EGFR + MET inhibition",
                    "authors": "Smith et al.",
                    "targets": ["EGFR", "MET"]
                }
            ],
            "patents": [
                {
                    "date": "2024-12-10",
                    "event": "patent_granted",
                    "assignee": "Merck",
                    "title": "EGFR C797S inhibitors",
                    "patent_id": "US2024/XXX",
                    "targets": ["EGFR", "C797S"]
                }
            ]
        }

    def create_alert(self, name: str, targets: List[str],
                     events: List[str] = None, competitors: List[str] = None) -> Dict:
        """Create a new alert configuration."""
        try:
            alert = {
                "name": name,
                "created": datetime.now().isoformat(),
                "targets": targets,
                "events": events or ["clinical", "regulatory", "publications", "patents"],
                "competitors": competitors or [],
                "filters": {
                    "phase": [2, 3],
                    "countries": ["US", "EU", "CN", "JP"],
                    "minimum_significance": "medium"
                },
                "notifications": {
                    "enabled": True,
                    "frequency": "weekly"
                },
                "last_checked": None
            }

            self.alerts[name] = alert
            self._save_alerts()

            print(f"✓ Alert '{name}' created")
            return alert

        except Exception as e:
            print(f"✗ Create alert error: {e}")
            return {}

    def check_alert(self, name: str, days: int = 7) -> Dict:
        """Check for new events matching alert criteria."""
        try:
            if name not in self.alerts:
                print(f"✗ Alert '{name}' not found")
                return {}

            alert = self.alerts[name]
            alert["last_checked"] = datetime.now().isoformat()
            self._save_alerts()

            cutoff_date = datetime.now() - timedelta(days=days)

            # Find matching events
            matches = {
                "clinical": [],
                "regulatory": [],
                "publications": [],
                "patents": []
            }

            for event_type in alert["events"]:
                if event_type in self.MOCK_EVENTS:
                    for event in self.MOCK_EVENTS[event_type]:
                        event_date = datetime.strptime(event["date"], "%Y-%m-%d")
                        if event_date >= cutoff_date:
                            if self._matches_alert(event, alert):
                                matches[event_type].append(event)

            # Generate report
            report = self._generate_report(name, alert, matches, days)
            self._save_to_history(name, report)

            print(f"✓ Alert '{name}' checked: {sum(len(v) for v in matches.values())} new events")
            return report

        except Exception as e:
            print(f"✗ Check alert error: {e}")
            return {}

    def list_alerts(self) -> List[Dict]:
        """List all configured alerts."""
        alerts_list = []

        for name, alert in self.alerts.items():
            alerts_list.append({
                "name": name,
                "targets": alert.get("targets", []),
                "events": alert.get("events", []),
                "last_checked": alert.get("last_checked", "Never"),
                "enabled": alert.get("notifications", {}).get("enabled", True)
            })

        return alerts_list

    def delete_alert(self, name: str) -> bool:
        """Delete an alert."""
        if name in self.alerts:
            del self.alerts[name]
            self._save_alerts()
            print(f"✓ Alert '{name}' deleted")
            return True
        else:
            print(f"✗ Alert '{name}' not found")
            return False

    def _matches_alert(self, event: Dict, alert: Dict) -> bool:
        """Check if event matches alert criteria."""
        # Check targets
        if alert.get("targets"):
            event_targets = event.get("targets", [])
            if not any(t.upper() in [at.upper() for at in alert["targets"]]
                      for t in event_targets):
                return False

        # Check competitors
        if alert.get("competitors"):
            sponsor = event.get("sponsor", event.get("assignee", ""))
            if not any(c.lower() in sponsor.lower() for c in alert["competitors"]):
                # Not a competitor event, but might still match targets
                pass

        return True

    def _generate_report(self, name: str, alert: Dict,
                        matches: Dict, days: int) -> Dict:
        """Generate alert report."""
        report = {
            "alert_name": name,
            "period": f"Last {days} days",
            "generated": datetime.now().isoformat(),
            "summary": {
                k: len(v) for k, v in matches.items()
            },
            "events": matches,
            "recommendations": self._generate_recommendations(matches)
        }

        return report

    def _generate_recommendations(self, matches: Dict) -> List[str]:
        """Generate actionable recommendations."""
        recommendations = []

        if matches.get("regulatory"):
            recommendations.append("Review regulatory updates for competitive impact")

        if matches.get("clinical"):
            recommendations.append("Assess new clinical trials for competitive threat")

        if matches.get("patents"):
            recommendations.append("Review new patents for FTO implications")

        if matches.get("publications"):
            recommendations.append("Review key publications for scientific updates")

        return recommendations

    def _load_alerts(self) -> Dict:
        """Load alerts from file."""
        if os.path.exists(self.alerts_file):
            try:
                with open(self.alerts_file) as f:
                    return json.load(f)
            except:
                return {}
        return {}

    def _save_alerts(self):
        """Save alerts to file."""
        with open(self.alerts_file, 'w') as f:
            json.dump(self.alerts, f, indent=2)

    def _load_history(self) -> Dict:
        """Load alert history from file."""
        if os.path.exists(self.history_file):
            try:
                with open(self.history_file) as f:
                    return json.load(f)
            except:
                return {}
        return {}

    def _save_to_history(self, alert_name: str, report: Dict):
        """Save report to history."""
        if alert_name not in self.history:
            self.history[alert_name] = []

        self.history[alert_name].append({
            "timestamp": datetime.now().isoformat(),
            "summary": report["summary"]
        })

        # Keep only last 100 reports
        if len(self.history[alert_name]) > 100:
            self.history[alert_name] = self.history[alert_name][-100:]

        with open(self.history_file, 'w') as f:
            json.dump(self.history, f, indent=2)


def main():
    parser = argparse.ArgumentParser(description="Alert system for drug discovery")
    parser.add_argument("--create", help="Create new alert (name)")
    parser.add_argument("--target", action="append", help="Target to monitor (can specify multiple)")
    parser.add_argument("--events", help="Event types (comma-separated)")
    parser.add_argument("--competitor", action="append", help="Competitors to monitor")
    parser.add_argument("--check", help="Check alert for updates")
    parser.add_argument("--list", action="store_true", help="List all alerts")
    parser.add_argument("--delete", help="Delete an alert")
    parser.add_argument("--days", type=int, default=7, help="Days to check back")
    parser.add_argument("--export", help="Export alert history")
    parser.add_argument("--format", choices=["json", "summary"],
                        default="summary", help="Output format")

    args = parser.parse_args()

    system = AlertSystem()

    if args.create:
        # Create new alert
        targets = args.target or []
        events = args.events.split(",") if args.events else None
        competitors = args.competitor or []

        alert = system.create_alert(args.create, targets, events, competitors)

        if args.format == "summary":
            print_alert_summary(alert)

    elif args.check:
        # Check alert
        report = system.check_alert(args.check, days=args.days)

        if args.format == "summary":
            print_alert_report(report)
        else:
            print(json.dumps(report, indent=2))

    elif args.list:
        # List alerts
        alerts = system.list_alerts()

        if args.format == "summary":
            print_alerts_list(alerts)
        else:
            print(json.dumps(alerts, indent=2))

    elif args.delete:
        # Delete alert
        system.delete_alert(args.delete)
    else:
        parser.print_help()


def print_alert_summary(alert: Dict):
    """Print alert configuration summary."""
    if not alert:
        print("No alert created")
        return

    print("\n" + "="*60)
    print(f"ALERT CREATED: {alert.get('name')}")
    print("="*60 + "\n")

    print(f"**Targets**: {', '.join(alert.get('targets', []))}")
    print(f"**Events**: {', '.join(alert.get('events', []))}")
    print(f"**Competitors**: {', '.join(alert.get('competitors', []))}")

    filters = alert.get('filters', {})
    print(f"\n**Filters**:")
    print(f"  Phase: {filters.get('phase', [])}")
    print(f"  Countries: {filters.get('countries', [])}")

    print("\n" + "="*60 + "\n")


def print_alert_report(report: Dict):
    """Print alert check report."""
    if not report:
        print("No report generated")
        return

    print("\n" + "="*70)
    print(f"ALERT REPORT: {report.get('alert_name')} ({report.get('period')})")
    print("="*70 + "\n")

    summary = report.get("summary", {})
    total = sum(summary.values())

    print(f"**Summary**: {total} new events\n")

    for event_type, count in summary.items():
        if count > 0:
            print(f"  {event_type.capitalize()}: {count}")

    events = report.get("events", {})

    if events.get("clinical"):
        print("\n**Clinical Updates**:")
        for event in events["clinical"][:5]:
            print(f"  [{event['date']}] {event.get('sponsor', 'Unknown')}: {event.get('event', 'Unknown')}")

    if events.get("regulatory"):
        print("\n**Regulatory Updates**:")
        for event in events["regulatory"]:
            print(f"  [{event['date']}] {event.get('agency', 'Unknown')}: {event.get('event', 'Unknown')} - {event.get('drug', 'Unknown')}")

    if events.get("publications"):
        print("\n**Key Publications**:")
        for event in events["publications"][:3]:
            print(f"  [{event['date']}] {event.get('title', 'Unknown')} ({event.get('journal', 'Unknown')})")

    if events.get("patents"):
        print("\n**Patent Filings**:")
        for event in events["patents"][:3]:
            print(f"  [{event['date']}] {event.get('assignee', 'Unknown')}: {event.get('title', 'Unknown')}")

    recommendations = report.get("recommendations", [])
    if recommendations:
        print("\n**Recommendations**:")
        for rec in recommendations:
            print(f"  • {rec}")

    print("\n" + "="*70 + "\n")


def print_alerts_list(alerts: List[Dict]):
    """Print list of alerts."""
    print("\n" + "="*60)
    print("CONFIGURED ALERTS")
    print("="*60 + "\n")

    if not alerts:
        print("No alerts configured")
        return

    for alert in alerts:
        print(f"**{alert['name']}**")
        print(f"  Targets: {', '.join(alert['targets'] or ['None'])}")
        print(f"  Events: {', '.join(alert['events'] or ['None'])}")
        print(f"  Last checked: {alert['last_checked']}")
        print(f"  Enabled: {alert['enabled']}")
        print()

    print("="*60 + "\n")


if __name__ == "__main__":
    main()
