# API Development Standards and Best Practices

This reference provides coding standards, best practices, and guidelines for creating professional APIs that the APIBuilder skill should follow.

## Code Structure and Organization

### File Organization
```
project/
├── app.py                 # Main application entry point
├── config.py             # Configuration settings
├── models/               # Database models
│   ├── __init__.py
│   └── user.py
├── routes/               # API routes
│   ├── __init__.py
│   └── user_routes.py
├── utils/                # Utility functions
│   ├── __init__.py
│   └── validators.py
├── middleware/           # Middleware functions
│   ├── __init__.py
│   └── auth.py
└── tests/                # Test files
    ├── __init__.py
    └── test_users.py
```

### Import Order (PEP 8)
1. Standard library imports
2. Related third-party imports
3. Local application/library specific imports
4. Separate each group with a blank line

```python
import os
import sys
from datetime import datetime

import requests
from flask import Flask, jsonify

from .models import User
from .utils import validate_email
```

## Error Handling Standards

### Try-Catch Best Practices

#### Basic Error Handling
```python
import logging
from flask import jsonify

def safe_api_call():
    try:
        # Business logic here
        result = perform_operation()
        return jsonify({'success': True, 'data': result}), 200
    except ValueError as e:
        logging.error(f"Value error in safe_api_call: {str(e)}")
        return jsonify({'success': False, 'error': 'Invalid input'}), 400
    except DatabaseError as e:
        logging.error(f"Database error in safe_api_call: {str(e)}")
        return jsonify({'success': False, 'error': 'Database error'}), 500
    except Exception as e:
        logging.error(f"Unexpected error in safe_api_call: {str(e)}")
        return jsonify({'success': False, 'error': 'Internal server error'}), 500
```

#### Specific Exception Handling
```python
def get_user(user_id):
    try:
        user = User.query.get(user_id)
        if not user:
            return jsonify({'error': 'User not found'}), 404
        return jsonify(user.to_dict()), 200
    except ValueError:
        # Handle invalid user_id format
        return jsonify({'error': 'Invalid user ID format'}), 400
    except Exception as e:
        # Log the error and return generic response
        app.logger.error(f"Error retrieving user {user_id}: {str(e)}")
        return jsonify({'error': 'Internal server error'}), 500
```

### Logging Standards
```python
import logging

# Configure logger
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

# Use appropriate log levels
logger.debug("Detailed debug information")
logger.info("General information")
logger.warning("Warning message")
logger.error("Error message")
logger.critical("Critical error message")
```

## API Design Standards

### HTTP Status Codes
- **200**: Success for GET, PUT, PATCH
- **201**: Created for POST
- **204**: No Content for DELETE
- **400**: Bad Request - client error
- **401**: Unauthorized - authentication required
- **403**: Forbidden - insufficient permissions
- **404**: Not Found
- **422**: Unprocessable Entity - validation error
- **500**: Internal Server Error

### Response Format
```python
# Success response
{
    "success": true,
    "data": {...},
    "message": "Operation successful"
}

# Error response
{
    "success": false,
    "error": "Error message",
    "details": "Additional details if needed"
}
```

## Input Validation

### Request Validation
```python
from flask import request
from marshmallow import Schema, fields, ValidationError

class UserSchema(Schema):
    name = fields.Str(required=True, validate=lambda x: len(x) > 0)
    email = fields.Email(required=True)
    age = fields.Int(validate=lambda x: 0 <= x <= 120)

def create_user():
    try:
        schema = UserSchema()
        user_data = schema.load(request.json)
        # Process valid data
        return jsonify({'success': True, 'data': user_data}), 201
    except ValidationError as err:
        return jsonify({
            'success': False,
            'error': 'Validation failed',
            'details': err.messages
        }), 400
```

## Security Best Practices

### Authentication and Authorization
```python
from functools import wraps
from flask import request, jsonify

def require_auth(f):
    @wraps(f)
    def decorated_function(*args, **kwargs):
        token = request.headers.get('Authorization')
        if not token:
            return jsonify({'error': 'Authorization token required'}), 401

        try:
            # Verify token (implementation depends on auth method)
            user = verify_token(token)
            request.current_user = user
        except Exception:
            return jsonify({'error': 'Invalid token'}), 401

        return f(*args, **kwargs)
    return decorated_function
```

### Input Sanitization
```python
import re
from html import escape

def sanitize_input(data):
    """Sanitize input to prevent injection attacks"""
    if isinstance(data, str):
        # Escape HTML characters
        data = escape(data)
        # Remove potentially dangerous characters
        data = re.sub(r'[<>"\']', '', data)
    elif isinstance(data, dict):
        return {k: sanitize_input(v) for k, v in data.items()}
    elif isinstance(data, list):
        return [sanitize_input(item) for item in data]
    return data
```

## Code Quality Standards

### Function Documentation
```python
def calculate_user_score(user_id: int, activity_data: dict) -> float:
    """
    Calculate user engagement score based on activity data.

    Args:
        user_id (int): Unique identifier for the user
        activity_data (dict): Dictionary containing user activity metrics

    Returns:
        float: Calculated score between 0.0 and 100.0

    Raises:
        ValueError: If user_id is invalid or activity_data is malformed
        DatabaseError: If unable to retrieve user data from database
    """
    # Implementation here
    pass
```

### Naming Conventions
- Use descriptive variable names
- Follow snake_case for variables and functions
- Use PascalCase for classes
- Use UPPER_CASE for constants

```python
# Good
user_account_balance = 1000.0
MAX_LOGIN_ATTEMPTS = 3
USER_STATUS_ACTIVE = 'active'

class UserAccountManager:
    def calculate_monthly_fees(self):
        pass
```

## Performance Considerations

### Database Query Optimization
```python
# Use specific queries instead of fetching all data
# Bad
users = User.query.all()
active_users = [u for u in users if u.is_active]

# Good
active_users = User.query.filter_by(is_active=True).all()

# Use pagination for large datasets
users = User.query.paginate(
    page=request.args.get('page', 1, type=int),
    per_page=request.args.get('per_page', 10, type=int)
)
```

### Caching
```python
from flask_caching import Cache

cache = Cache(app, config={'CACHE_TYPE': 'simple'})

@cache.cached(timeout=300, key_prefix='user_data')
def get_user_data(user_id):
    # Expensive operation
    return expensive_database_query(user_id)
```

## Testing Standards

### Unit Test Structure
```python
import unittest
from app import app
import json

class TestUserAPI(unittest.TestCase):
    def setUp(self):
        self.app = app.test_client()

    def test_create_user_success(self):
        # Arrange
        payload = {
            'name': 'John Doe',
            'email': 'john@example.com'
        }

        # Act
        response = self.app.post('/api/users',
                                data=json.dumps(payload),
                                content_type='application/json')

        # Assert
        self.assertEqual(response.status_code, 201)
        self.assertTrue(response.json['success'])

    def test_create_user_invalid_email(self):
        # Test error case
        payload = {
            'name': 'John Doe',
            'email': 'invalid-email'
        }

        response = self.app.post('/api/users',
                                data=json.dumps(payload),
                                content_type='application/json')

        self.assertEqual(response.status_code, 400)
```

## Documentation Standards

### API Documentation
```python
from flask import Flask, request
from flask_restx import Api, Resource, fields

app = Flask(__name__)
api = Api(app, doc='/docs')

user_model = api.model('User', {
    'name': fields.String(required=True, description='User name'),
    'email': fields.String(required=True, description='User email')
})

@api.route('/users')
class UserResource(Resource):
    @api.doc('create_user')
    @api.expect(user_model)
    @api.response(201, 'User created successfully')
    @api.response(400, 'Invalid input')
    def post(self):
        """Create a new user"""
        # Implementation here
        pass
```

Following these standards ensures that APIs created by the APIBuilder skill are:
- Secure and robust
- Well-documented and maintainable
- Following industry best practices
- Consistent in structure and error handling