# API Testing Framework

This reference provides detailed guidance on creating comprehensive test suites for APIs generated with the APIBuilder skill.

## Test Client Architecture

### Base API Client Class
```python
import requests
import json
from typing import Dict, List, Optional

class BaseAPIClient:
    def __init__(self, base_url: str, auth_token: Optional[str] = None):
        self.base_url = base_url.rstrip('/')
        self.auth_token = auth_token
        self.headers = {
            'Content-Type': 'application/json'
        }
        if auth_token:
            self.headers['Authorization'] = f'Bearer {auth_token}'

    def _make_request(self, method: str, endpoint: str, data: Optional[Dict] = None) -> Dict:
        url = f"{self.base_url}/{endpoint.lstrip('/')}"
        try:
            if method.upper() == 'GET':
                response = requests.get(url, headers=self.headers)
            elif method.upper() == 'POST':
                response = requests.post(url, headers=self.headers, json=data)
            elif method.upper() == 'PUT':
                response = requests.put(url, headers=self.headers, json=data)
            elif method.upper() == 'DELETE':
                response = requests.delete(url, headers=self.headers)
            else:
                raise ValueError(f"Unsupported HTTP method: {method}")

            return {
                'status_code': response.status_code,
                'success': response.ok,
                'data': response.json() if response.content else None,
                'url': url,
                'method': method
            }
        except Exception as e:
            return {
                'status_code': None,
                'success': False,
                'error': str(e),
                'url': url,
                'method': method
            }
```

## Testing Patterns

### CRUD Operation Tests
For each resource, implement comprehensive tests:

#### 1. Create Operations
- Test successful creation with valid data
- Test error handling with invalid data
- Test duplicate creation prevention
- Test required field validation

#### 2. Read Operations
- Test successful retrieval of single item
- Test successful retrieval of multiple items
- Test error handling for non-existent items
- Test pagination (if applicable)

#### 3. Update Operations
- Test successful update with valid data
- Test partial updates
- Test error handling with invalid data
- Test update of non-existent items

#### 4. Delete Operations
- Test successful deletion
- Test error handling for non-existent items
- Test idempotent deletion (deleting twice)

### Authentication & Authorization Tests
- Test access without authentication
- Test access with invalid authentication
- Test access with valid authentication
- Test access control for different user roles

### Error Condition Tests
- Test malformed JSON requests
- Test missing required fields
- Test invalid data types
- Test rate limiting (if implemented)
- Test server error scenarios

## Test Data Management

### Mock Data Generation
```python
def generate_test_data(resource_type: str) -> Dict:
    """Generate valid test data for different resource types"""
    import uuid
    from datetime import datetime

    if resource_type == 'user':
        return {
            'username': f'testuser_{uuid.uuid4().hex[:8]}',
            'email': f'test_{uuid.uuid4().hex[:8]}@example.com',
            'name': 'Test User'
        }
    elif resource_type == 'medication':
        return {
            'name': 'Test Medication',
            'dosage': '100mg',
            'frequency': 'Once daily',
            'time': 'Morning',
            'instructions': 'Take with food'
        }
    # Add more resource types as needed
    return {}
```

## Test Execution Framework

### Test Runner
```python
def run_api_tests(client_class, base_url: str, user_id: str):
    """Execute comprehensive tests for an API"""
    client = client_class(base_url, user_id)

    test_results = {
        'passed': 0,
        'failed': 0,
        'tests': []
    }

    # Execute all test methods in the client
    for attr_name in dir(client):
        if attr_name.startswith('test_'):
            test_method = getattr(client, attr_name)
            try:
                result = test_method()
                if result.get('success'):
                    test_results['passed'] += 1
                else:
                    test_results['failed'] += 1
                test_results['tests'].append({
                    'test': attr_name,
                    'result': result
                })
            except Exception as e:
                test_results['failed'] += 1
                test_results['tests'].append({
                    'test': attr_name,
                    'error': str(e)
                })

    return test_results
```

## Integration with CI/CD

### Environment Configuration
```bash
# Environment variables for testing
TEST_BASE_URL="https://staging-api.example.com"
TEST_USER_ID="test-user-123"
TEST_AUTH_TOKEN="test-token-456"
TEST_ENVIRONMENT="staging"
```

### Test Reporting
- Generate detailed test reports
- Track API response times
- Monitor error rates
- Alert on breaking changes

## Performance Testing

### Load Testing
- Test concurrent requests
- Measure response times under load
- Identify performance bottlenecks
- Test resource limits

### Stress Testing
- Test maximum capacity
- Verify graceful degradation
- Test recovery from failures

## Security Testing

### Input Validation
- Test SQL injection attempts
- Test XSS attempts
- Test malformed input
- Test boundary conditions

### Authentication Testing
- Test token expiration
- Test token replay attacks
- Test privilege escalation
- Test session management

This framework ensures comprehensive testing of all API endpoints and functionality, providing confidence in the API's reliability and security.