---
name: artemis-debug-secure
description: Database investigation skill for Jira tickets with secure credential handling. Multi-Agent Swarm for 3x faster parallel execution. Auto-learns from investigations, searches similar tickets, integrates with Jira, and detects anomalies.
---

# Artemis Database Debug Skill (SWARM Edition)

## Performance

| Metric | Classic | Swarm | Improvement |
|--------|---------|-------|-------------|
| Investigation time | 45-60s | 15-20s | **3x faster** |
| Browser logins | Every time | Pooled (3) | **No login wait** |
| Query execution | Sequential | Parallel | **2-3x faster** |
| Screenshots | One-by-one | Batched | **50% faster** |

## Features

| Feature | Description |
|---------|-------------|
| Multi-Agent Swarm | 5 specialized agents working in parallel |
| Connection Pooling | 3 pre-logged browser sessions, reused |
| Auto-Learn | Learns patterns from resolved tickets |
| Similar Search | Finds past tickets with same issue |
| Jira Integration | Auto-fetch ticket, auto-comment results |
| Error Detection | Flags anomalies in query results |
| Multi-User | Per-laptop credentials, shareable |

---

## Multi-Agent Swarm Architecture

```
                         ┌────────────────────────┐
                         │   COORDINATOR AGENT    │
                         │   (Orchestrates all)   │
                         └───────────┬────────────┘
                                     │
          ┌──────────────┬───────────┼───────────┬──────────────┐
          ▼              ▼           ▼           ▼              ▼
    ┌──────────┐  ┌──────────┐  ┌──────────┐  ┌──────────┐
    │ DB AGENT │  │JIRA AGENT│  │ ANALYSIS │  │ LEARNING │
    │          │  │          │  │  AGENT   │  │  AGENT   │
    │ -Queries │  │ -Fetch   │  │ -Errors  │  │ -Brain   │
    │ -Pool    │  │ -Similar │  │ -RCA     │  │ -Patterns│
    │ -Screen  │  │ -Comment │  │ -Summary │  │ -Learn   │
    └──────────┘  └──────────┘  └──────────┘  └──────────┘
          │              │           │              │
          └──────────────┴───────────┴──────────────┘
                                │
                   ┌────────────┴────────────┐
                   │    SHARED RESOURCES     │
                   │  • Message Bus (async)  │
                   │  • Connection Pool (3)  │
                   │  • Result Store         │
                   └─────────────────────────┘
```

### 4-Phase Parallel Workflow

| Phase | Tasks | Execution |
|-------|-------|-----------|
| 1. Init | Fetch ticket, Load brain, Get customer_id | **PARALLEL** |
| 2. Data | Execute queries, Search similar, Screenshots | **PARALLEL** |
| 3. Analysis | Detect errors, Determine RCA, Find solutions | **PARALLEL** |
| 4. Output | Generate summary, Format To CAS, Learn | Sequential |

---

## Claude Workflow (AUTO)

### Step 1: Check Credentials

```bash
SKILL_DIR=~/.claude/skills/artemis-debug-secure
USER=$(whoami)
if [ -f "$SKILL_DIR/users/$USER/.credentials" ]; then
  echo "READY"
else
  echo "NEED_SETUP"
fi
```

### Step 2: If NEED_SETUP -> Ask User

Use `AskUserQuestion` tool:
```
Questions:
1. "Artemis username?"
2. "Artemis password?"
```

Then save credentials:
```bash
SKILL_DIR=~/.claude/skills/artemis-debug-secure
USER=$(whoami)
mkdir -p "$SKILL_DIR/users/$USER"
cat > "$SKILL_DIR/users/$USER/.credentials" << 'EOF'
ARTEMIS_URL=https://artemis.568winex.com
ARTEMIS_USER={username_from_user}
ARTEMIS_PASS={password_from_user}
EOF
chmod 600 "$SKILL_DIR/users/$USER/.credentials"
```

### Step 3: Fetch Jira Ticket (if ticket key provided)

```
Use mcp__jira__jira_get_issue(ticketKey)
Parse description to extract: username, webId, date, type
```

### Step 4: Search Similar Tickets

```
Use mcp__jira__jira_search_issues with JQL:
  project = TCP AND summary ~ "{ticket_type}" AND status = Done ORDER BY created DESC
Review past solutions before investigating
```

### Step 5: Run Investigation

```bash
cd ~/.claude/skills/artemis-debug-secure
python3 scripts/investigate.py -t {type} -w {webId} -u "{username}" -k {ticket_key} --learn
```

### Step 6: Add Jira Comment

```
Use mcp__jira__jira_add_comment(ticketKey, body)
Include: findings, root cause, conclusion, screenshots
```

### Step 7: Output To CAS (500 chars max)

```
To CAS:
{Root cause}. {Conclusion}. {Details}.
```

---

## Auto-Detect Type

| Keywords in Ticket | Type |
|-------------------|------|
| promotion, bonus, reject, FP, lucky wheel | `promotion` |
| deposit, withdrawal, pending, payment | `payment` |
| vip, upgrade, level, benefits | `vip` |
| bet, settlement, winning, void | `betting` |
| login, locked, 2fa, suspended | `login` |

## Auto-Detect WebId

| Site Mention | WebId |
|--------------|-------|
| Saffaluck, SFL | 20154 |
| NocmakatiInc, NMI | 20107 |
| Bet25 | 20120 |
| GBW, gbw777 | 20109 |
| Lucky7 | 20132 |

See `docs/webids.md` for full list.

---

## SQL Rules (MUST FOLLOW)

| Rule | Correct | Wrong |
|------|---------|-------|
| Columns | `SELECT [Id], [Name]` | `SELECT *` |
| Count | `COUNT(1)` | `COUNT(*)` |
| Hint | `WITH(NOLOCK)` | None |
| Limit | `TOP 100` | None |
| Sort | `ORDER BY DESC` | None |

---

## Error Detection (AUTO)

The skill automatically detects these issues:

| Type | Detection |
|------|-----------|
| Customer | Suspended, Closed, Deleted, Negative balance |
| Promotion | Multiple rejections, Same FP/IP conflicts |
| Transaction | Rejected, Balance mismatch, Large amounts |
| VIP | Downgrades, Rejected bonuses |
| Betting | Voided bets, Resettlements, Large stakes |

Detected issues are flagged as ALERTS (critical) or WARNINGS (potential).

---

## Auto-Learn System

After each investigation, the skill learns:
- **Pattern**: ticket type + root cause + solution
- **Indicators**: key fields from results (RejectSetting, Status, etc.)
- **Frequency**: how often each cause appears

Use learned data:
```bash
# Show common causes for a type
python3 scripts/investigate.py --show-common promotion

# Output: [5x] SameFP: Fingerprint conflict...
#         [3x] TurnoverNotMet: Wagering requirement...
```

---

## File Reference

| Need | File |
|------|------|
| Query templates | `docs/queries.md` |
| Status codes | `docs/status-codes.md` |
| WebId mapping | `docs/webids.md` |
| Playbooks | `playbooks/{type}.md` |
| Brain patterns | `memory/brain.json` |
| Learnings | `memory/learnings.json` |
| Rules | `memory/rules.md` |

## Script Reference

### Swarm (Recommended - 3x faster)

| Script | Purpose |
|--------|---------|
| `scripts/swarm_investigate.py` | **Main entry point** - Multi-agent parallel |
| `scripts/agents/coordinator.py` | Orchestrates 4-phase workflow |
| `scripts/agents/db_agent.py` | Database operations with pooling |
| `scripts/agents/jira_agent.py` | Ticket parsing & formatting |
| `scripts/agents/analysis_agent.py` | Error detection & RCA |
| `scripts/agents/learning_agent.py` | Pattern learning & matching |
| `scripts/swarm/pool.py` | Connection pool (3 browsers) |
| `scripts/swarm/bus.py` | Async message bus |
| `scripts/swarm/context.py` | Shared investigation context |
| `scripts/swarm/store.py` | Result aggregation |

### Classic (Legacy)

| Script | Purpose |
|--------|---------|
| `scripts/investigate.py` | Classic sequential investigation |
| `scripts/core.py` | Core classes (Artemis, UserConfig) |
| `scripts/brain.py` | Auto-learning module |
| `scripts/jira_integration.py` | Jira parsing & formatting |
| `scripts/error_detector.py` | Anomaly detection |
| `scripts/queries.json` | Query definitions |

---

## CLI Options

### Swarm (Recommended)

```bash
python3 scripts/swarm_investigate.py [options]

Required (one of):
  --ticket          Jira ticket key (e.g., TCP-12345)
  OR
  -t, --type        Type: promotion, payment, vip, betting, login
  -w, --webid       WebId number
  -u, --username    Player username

Optional:
  --customer-id     CustomerId (auto-fetched if not provided)
  --headless        Run browsers in headless mode
  --pool-size       Connection pool size (default: 3)
  --benchmark       Run performance benchmark (3 iterations)
  --json            Output results as JSON
```

### Classic (Legacy)

```bash
python3 scripts/investigate.py [options]

Required:
  -t, --type        Type: promotion, payment, vip, betting, login
  -w, --webid       WebId number
  -u, --username    Player username

Optional:
  -k, --ticket      Jira ticket key (e.g., TCP-12345)
  -c, --customerid  CustomerId (auto-fetched if not provided)
  --headless        Run without browser window
  --learn           Save learning from this investigation
  --show-common     Show common causes for a type
  -o, --output      Save results to JSON file
```

---

## Multi-User Support

The `users/` folder is **gitignored**. Each laptop creates its own:
- `users/{system_username}/.credentials`

When skill is shared, new user's credentials are auto-created on first run.

---

## Output Format

### To CAS (500 chars max)
```
{Root cause}. {Conclusion}. {Details}.
```

### Jira Comment
```
*DATABASE INVESTIGATION RESULTS*
*Ticket:* TCP-12345
*Player:* username
*WebId:* 20154

*Queries Executed:*
- Customer Info
- Rejection Records
- Balance History

*Key Findings:*
- Rejected: Same Fingerprint
- FP used by: other_player

*ROOT CAUSE:* Fingerprint conflict
*CONCLUSION:* Player's device was used by another account
```

---

## Example Usage

### With Jira Ticket
```
User: Investigate TCP-92018

Claude:
1. Fetch TCP-92018 via mcp__jira__jira_get_issue
2. Parse: username=player123, webId=20154 (Saffaluck), type=promotion
3. Search similar: mcp__jira__jira_search_issues(JQL)
4. Found 3 similar tickets - TCP-91234 had same root cause
5. Check credentials
6. Run: python3 investigate.py -t promotion -w 20154 -u player123 -k TCP-92018 --learn
7. Auto-detect: Fingerprint conflict
8. Comment results to Jira
9. Output To CAS response
```

### Manual Mode
```
User: Check promotion for player123 on Saffaluck

Claude:
1. Infer: webId=20154, type=promotion
2. Check credentials
3. Run: python3 investigate.py -t promotion -w 20154 -u player123
4. Output To CAS response
```
