# AAA Database Schema

## Overview
The AAA database handles authentication, authorization, and accounting functionalities, including user groups, permissions, login history, and resource access control.

## Tables

### Activity
Tracks user actions for auditing and monitoring.

| Column      | Type            | Description                                  |
|-------------|-----------------|----------------------------------------------|
| Id          | INT, PK         | Unique identifier for the activity log       |
| CustomerId  | INT             | Customer/user identifier                     |
| Username    | NVARCHAR(100)   | Username of the customer                     |
| Action      | NVARCHAR(100)   | Action performed (e.g., login, update)       |
| Description | NVARCHAR(MAX)   | Details about the action                     |
| CreatedOn   | DATETIME        | When the action was performed                |
| CreatedBy   | NVARCHAR(100)   | Who performed the action                     |
| WebId       | INT             | Website/application identifier               |
| Ip          | NVARCHAR(50)    | IP address of the user                      |
| Device      | NVARCHAR(MAX)   | Device information                          |
| Agent       | NVARCHAR(50)    | User agent string                           |

```sql
CREATE TABLE [Activity] (
    [Id] INT IDENTITY(1,1) PRIMARY KEY,
    [CustomerId] INT NOT NULL,
    [Username] NVARCHAR(100) NOT NULL,
    [Action] NVARCHAR(100) NOT NULL,
    [Description] NVARCHAR(MAX),
    [CreatedOn] DATETIME NOT NULL,
    [CreatedBy] NVARCHAR(100) NOT NULL,
    [WebId] INT NOT NULL,
    [Ip] NVARCHAR(50),
    [Device] NVARCHAR(MAX),
    [Agent] NVARCHAR(50)
)
```

### CustomerLoginHistory
Records login and logout events for customers.

| Column      | Type            | Description                                  |
|-------------|-----------------|----------------------------------------------|
| Id          | INT, PK         | Unique identifier for the login record       |
| CustomerId  | INT             | Customer/user identifier                     |
| Username    | NVARCHAR(100)   | Username of the customer                     |
| LoginTime   | DATETIME        | Login timestamp                              |
| LogoutTime  | DATETIME        | Logout timestamp (nullable)                  |
| WebId       | INT             | Website/application identifier               |
| Ip          | NVARCHAR(50)    | IP address used for login                    |
| Device      | NVARCHAR(MAX)   | Device information                          |
| Agent       | NVARCHAR(50)    | User agent string                           |
| Status      | NVARCHAR(50)    | Login status (e.g., Success, Failed)         |

```sql
CREATE TABLE [CustomerLoginHistory] (
    [Id] INT IDENTITY(1,1) PRIMARY KEY,
    [CustomerId] INT NOT NULL,
    [Username] NVARCHAR(100) NOT NULL,
    [LoginTime] DATETIME NOT NULL,
    [LogoutTime] DATETIME,
    [WebId] INT NOT NULL,
    [Ip] NVARCHAR(50),
    [Device] NVARCHAR(MAX),
    [Agent] NVARCHAR(50),
    [Status] NVARCHAR(50)
)
```

### CustomerLoginToken
Stores active login tokens for session management.

| Column      | Type            | Description                                  |
|-------------|-----------------|----------------------------------------------|
| Id          | INT, PK         | Unique identifier for the token              |
| CustomerId  | INT             | Customer/user identifier                     |
| Username    | NVARCHAR(100)   | Username of the customer                     |
| Token       | NVARCHAR(MAX)   | Authentication token                         |
| ExpiredOn   | DATETIME        | Token expiration timestamp                   |
| CreatedOn   | DATETIME        | When the token was created                   |
| WebId       | INT             | Website/application identifier               |
| Ip          | NVARCHAR(50)    | IP address used for login                    |
| Device      | NVARCHAR(MAX)   | Device information                          |
| Agent       | NVARCHAR(50)    | User agent string                           |

```sql
CREATE TABLE [CustomerLoginToken] (
    [Id] INT IDENTITY(1,1) PRIMARY KEY,
    [CustomerId] INT NOT NULL,
    [Username] NVARCHAR(100) NOT NULL,
    [Token] NVARCHAR(MAX) NOT NULL,
    [ExpiredOn] DATETIME NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [WebId] INT NOT NULL,
    [Ip] NVARCHAR(50),
    [Device] NVARCHAR(MAX),
    [Agent] NVARCHAR(50)
)
```

### IpWhiteList
Defines allowed IPs for access control by group and website.

| Column      | Type            | Description                                  |
|-------------|-----------------|----------------------------------------------|
| Id          | INT, PK         | Unique identifier for the whitelist entry    |
| WebId       | INT             | Website/application identifier               |
| GroupId     | INT             | Group identifier                            |
| ModifiedBy  | NVARCHAR(50)    | Who last modified the entry                  |
| ModifiedOn  | DATETIME        | When the entry was last modified             |
| CreatedBy   | NVARCHAR(50)    | Who created the entry                        |
| CreatedOn   | DATETIME        | When the entry was created                   |
| IsEnabled   | BIT             | Whether the whitelist entry is active        |
| IpWhiteList | NVARCHAR(MAX)   | Whitelisted IP addresses (comma-separated)   |

```sql
CREATE TABLE [IpWhiteList] (
    [Id] INT IDENTITY(1,1) PRIMARY KEY,
    [WebId] INT NOT NULL,
    [GroupId] INT NOT NULL,
    [ModifiedBy] NVARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [IsEnabled] BIT NOT NULL,
    [IpWhiteList] NVARCHAR(MAX) NOT NULL
)
```

### EncryptMemberInformation
Controls encryption permissions for member information by group and website.

| Column      | Type            | Description                                  |
|-------------|-----------------|----------------------------------------------|
| WebId       | INT             | Website/application identifier               |
| GroupId     | INT, PK         | Group identifier                            |
| Permission  | INT             | Permission level (bitmask or enum)           |
| ModifiedBy  | NVARCHAR(50)    | Who last modified the entry                  |
| ModifiedOn  | DATETIME        | When the entry was last modified             |
| CreatedBy   | NVARCHAR(50)    | Who created the entry                        |
| CreatedOn   | DATETIME        | When the entry was created                   |

```sql
CREATE TABLE [EncryptMemberInformation] (
    [WebId] INT NOT NULL,
    [GroupId] INT NOT NULL,
    [Permission] INT NOT NULL,
    [ModifiedBy] NVARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL
)
```

### Resource
Defines resources (features, pages, APIs) for access control.

| Column           | Type             | Description                                 |
|------------------|------------------|---------------------------------------------|
| Id               | INT, PK          | Unique identifier for the resource          |
| Name             | NVARCHAR(255)    | Resource name                              |
| Description      | NVARCHAR(50)     | Short description of the resource           |
| Category         | NVARCHAR(50)     | Resource category (e.g., API, Page)         |
| IsActive         | BIT              | Whether the resource is active              |
| IsRootOnly       | BIT              | If true, only root users can access         |
| ModifiedBy       | NVARCHAR(50)     | Who last modified the resource              |
| ModifiedOn       | DATETIME         | When the resource was last modified         |
| CreatedBy        | NVARCHAR(50)     | Who created the resource                    |
| CreatedOn        | DATETIME         | When the resource was created               |
| NewCategory      | NVARCHAR(100)    | New/extended category (optional)            |
| NewCategoryOrder | NVARCHAR(10)     | Order for new category (optional)           |

```sql
CREATE TABLE [Resource] (
    [Id] INT IDENTITY(1,1) PRIMARY KEY,
    [Name] NVARCHAR(255) NOT NULL,
    [Description] NVARCHAR(50),
    [Category] NVARCHAR(50),
    [IsActive] BIT NOT NULL,
    [IsRootOnly] BIT NOT NULL,
    [ModifiedBy] NVARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [NewCategory] NVARCHAR(100),
    [NewCategoryOrder] NVARCHAR(10)
)
```

### ResourcePermission
Maps resources to owners (users or groups) and defines permissions.

| Column      | Type            | Description                                  |
|-------------|-----------------|----------------------------------------------|
| WebId       | INT, PK         | Website/application identifier               |
| OwnerId     | INT, PK         | User or group identifier                     |
| ResourceId  | INT, PK         | Resource identifier                          |
| IsGroup     | BIT, PK         | True if OwnerId is a group, false if user    |
| Permission  | INT             | Permission level (bitmask or enum)           |
| ModifiedBy  | NVARCHAR(50)    | Who last modified the entry                  |
| ModifiedOn  | DATETIME        | When the entry was last modified             |
| CreatedBy   | NVARCHAR(50)    | Who created the entry                        |
| CreatedOn   | DATETIME        | When the entry was created                   |

```sql
CREATE TABLE [ResourcePermission] (
    [WebId] INT NOT NULL,
    [OwnerId] INT NOT NULL,
    [ResourceId] INT NOT NULL,
    [IsGroup] BIT NOT NULL,
    [Permission] INT NOT NULL,
    [ModifiedBy] NVARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL
)
```

### AccountGroup
Links accounts to groups for group-based access control.

| Column      | Type            | Description                                  |
|-------------|-----------------|----------------------------------------------|
| AccountId   | INT, PK         | Account/user identifier                      |
| GroupId     | INT             | Group identifier                            |
| ModifiedBy  | NVARCHAR(50)    | Who last modified the entry                  |
| ModifiedOn  | DATETIME        | When the entry was last modified             |
| WebId       | INT, PK         | Website/application identifier               |

```sql
CREATE TABLE [AccountGroup] (
    [AccountId] INT NOT NULL,
    [GroupId] INT NOT NULL,
    [ModifiedBy] NVARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL,
    [WebId] INT NOT NULL,
    PRIMARY KEY ([AccountId], [GroupId], [WebId])
)
```

### Group
Defines user groups for organizing accounts and permissions.

| Column      | Type            | Description                                  |
|-------------|-----------------|----------------------------------------------|
| Id          | INT, PK         | Unique identifier for the group              |
| Name        | NVARCHAR(50)    | Group name                                  |
| Remark      | NVARCHAR(500)   | Description or notes about the group         |
| IsDefault   | BIT             | Whether this is the default group            |
| OperatorId  | INT             | Operator/creator identifier                  |
| AccountType | INT             | Type of account (enum or bitmask)            |
| WebId       | INT             | Website/application identifier               |
| ModifiedBy  | NVARCHAR(50)    | Who last modified the group                  |
| ModifiedOn  | DATETIME        | When the group was last modified             |
| CreatedBy   | NVARCHAR(50)    | Who created the group                        |
| CreatedOn   | DATETIME        | When the group was created                   |

```sql
CREATE TABLE [Group] (
    [Id] INT IDENTITY(1,1) PRIMARY KEY,
    [Name] NVARCHAR(50) NOT NULL,
    [Remark] NVARCHAR(500),
    [IsDefault] BIT NOT NULL,
    [OperatorId] INT NOT NULL,
    [AccountType] INT NOT NULL,
    [WebId] INT NOT NULL,
    [ModifiedBy] NVARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL
)
```

## Indexes

### Activity
- `IX_Activity_CustomerId` on `CustomerId`
- `IX_Activity_Username` on `Username`
- `IX_Activity_WebId` on `WebId`

### CustomerLoginHistory
- `IX_CustomerLoginHistory_CustomerId` on `CustomerId`
- `IX_CustomerLoginHistory_Username` on `Username`
- `IX_CustomerLoginHistory_WebId` on `WebId`

### CustomerLoginToken
- `IX_CustomerLoginToken_CustomerId` on `CustomerId`
- `IX_CustomerLoginToken_Username` on `Username`
- `IX_CustomerLoginToken_WebId` on `WebId`
- `IX_CustomerLoginToken_Token` on `Token`

### IpWhiteList
- `IX_IpWhiteList_WebId` on `WebId`

### EncryptMemberInformation
- `IX_EncryptMemberInformation_WebId` on `WebId`

### Resource
- `IX_Resource_Name` on `Name`

### ResourcePermission
- `IX_ResourcePermission_WebId` on `WebId`

### AccountGroup
- `IX_AccountGroup_AccountId` on `AccountId`
- `IX_AccountGroup_GroupId` on `GroupId`
- `IX_AccountGroup_WebId` on `WebId`

### Group
- `IX_Group_Name` on `Name`

## Relationships
- All tables have a relationship with the Customer table in the Main database through the `CustomerId` field
- All tables are linked to website configuration through the `WebId` field
- `CustomerId` and `AccountId` link to the Customer/Account tables in the Main database
- `GroupId` and `OwnerId` are used for group-based permissions and access control
- `ResourcePermission` and `AccountGroup` enable flexible, role-based access management
