# AI Database Schema

## Overview
The AI database manages AI-related functionalities in the Coloris Framework, including conversation management, prompt handling, and embeddings storage for AI models.

## Core Tables

### Conversation
Manages conversation sessions between users and AI assistants within the Coloris Framework.

```sql
CREATE TABLE [dbo].[Conversation] (
  [Id] INT NOT NULL PRIMARY KEY IDENTITY(1,1),
  [WebId] INT NOT NULL,
  [CustomerId] INT NOT NULL,
  [CreatedOn] DATETIME NOT NULL,
  [CreatedBy] NVARCHAR(100)
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Primary key, auto-incrementing unique identifier for each conversation |
| WebId | INT | Foreign key reference to the website/application where the conversation takes place |
| CustomerId | INT | Foreign key reference to the customer/user participating in the conversation |
| CreatedOn | DATETIME | Timestamp when the conversation was initiated |
| CreatedBy | NVARCHAR(100) | Optional field to track who/what created the conversation record |

### Prompt
Stores individual messages and interactions within conversations, supporting various types of AI communication.

```sql
CREATE TABLE [dbo].[Prompt] (
  [Id] INT NOT NULL PRIMARY KEY IDENTITY(1,1),
  [ConversationId] INT NOT NULL,
  [WebId] INT NOT NULL,
  [PromptType] NVARCHAR(100),
  [Content] NVARCHAR(MAX),
  [CustomerId] INT NOT NULL,
  [CreatedOn] DATETIME NOT NULL,
  [CreatedBy] NVARCHAR(100),
  CONSTRAINT CK_Prompt_PromptType CHECK ([PromptType] IN ('Systam', 'User', 'Assistant', 'Tool'))
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Primary key, auto-incrementing unique identifier for each prompt |
| ConversationId | INT | Foreign key linking to the parent conversation |
| WebId | INT | Foreign key reference to the website/application context |
| PromptType | NVARCHAR(100) | Categorizes the type of prompt with constrained values: 'Systam', 'User', 'Assistant', 'Tool' |
| Content | NVARCHAR(MAX) | The actual text content of the prompt/message (unlimited length) |
| CustomerId | INT | Foreign key reference to the customer/user |
| CreatedOn | DATETIME | Timestamp when the prompt was created |
| CreatedBy | NVARCHAR(100) | Optional field to track the creator of the prompt |

### Embedding
Stores text content along with their corresponding vector embeddings for semantic search and AI operations.

```sql
CREATE TABLE [dbo].[Embedding] (
  [Id] INT NOT NULL PRIMARY KEY IDENTITY(1,1),
  [Content] NVARCHAR(MAX) NOT NULL,
  [Embedding] NVARCHAR(MAX) NOT NULL,
  [CreatedOn] DATETIME
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Primary key, auto-incrementing unique identifier for each embedding record |
| Content | NVARCHAR(MAX) | The original text content that was converted to an embedding vector |
| Embedding | NVARCHAR(MAX) | The vector representation of the content, typically stored as a serialized array or JSON string |
| CreatedOn | DATETIME | Optional timestamp indicating when the embedding was generated |

## Key Features

1. **Conversation Management**
   - Tracks conversations between users and AI
   - Associates conversations with specific websites and customers
   - Maintains creation timestamps and audit trails

2. **Prompt Handling**
   - Supports multiple prompt types:
     - System prompts ('Systam')
     - User inputs ('User')
     - Assistant responses ('Assistant')
     - Tool interactions ('Tool')
   - Links prompts to conversations
   - Maintains context through conversation IDs

3. **Embedding Storage**
   - Stores text content and their vector embeddings
   - Enables semantic search capabilities
   - Tracks creation timestamps for embeddings

## Core Stored Procedures

1. **Conversation Management**
   - `Chatbodia_GetNewConversationId`: Creates a new conversation and returns its ID
   - `Chatbodia_AddPromptToConversation`: Adds a new prompt to an existing conversation

2. **Embedding Management**
   - `Chatbodia_AddEmbedding`: Stores new content with its embedding
   - `Chatbodia_GetAllEmbeddings`: Retrieves all stored embeddings

## Security

1. **Role-Based Access**
   - Custom role `RoleChatBodia` for AI operations
   - Dedicated login `N_ChatBodia` for AI service access

2. **Permissions**
   - Execute permissions on all stored procedures
   - Select, Update, Insert, and Delete permissions on all tables
   - Schema-level permissions for the `dbo` schema

## Relationships
- `Prompt` references `Conversation` through `ConversationId`
- All tables maintain audit trails with creation timestamps
- Tables are linked to website configuration through `WebId`
- Customer tracking through `CustomerId` in relevant tables

## Best Practices
1. Always use stored procedures for data operations
2. Maintain proper conversation context through `ConversationId`
3. Ensure proper cleanup of old conversations and embeddings
4. Follow the established prompt type constraints
5. Implement proper error handling in AI operations
6. Regular maintenance of embedding data 