# Account Database Schema

## Overview
The Account database manages all financial and transactional aspects of the platform, including player accounts, banking information, transactions, statements, and financial reports.

## Core Tables

### Statement
Stores all financial statements and transactions for customers across different products and currencies.

```sql
CREATE TABLE [dbo].[Statement] (
    [StatementId] INT IDENTITY(1,1),
    [RefNo] NVARCHAR(50) NOT NULL,
    [StatementDate] DATETIME NOT NULL,
    [StatementType] INT NOT NULL,
    [TransDate] DATETIME NOT NULL,
    [CustomerId] INT NOT NULL,
    [UserName] NVARCHAR(50) NOT NULL,
    [CustomerStatus] INT NOT NULL,
    [CashIn] DECIMAL(19,6) NOT NULL,
    [CashOut] DECIMAL(19,6) NOT NULL,
    [Currency] CHAR(3) NOT NULL,
    [ExchangeRate] DECIMAL(12,8) NOT NULL,
    [WebId] INT NOT NULL,
    [WithdrawalLimit] DECIMAL(19,6) NOT NULL DEFAULT(0)
)
```

| Column | Type | Description |
|--------|------|-------------|
| StatementId | INT, PK | Unique identifier for the statement record |
| RefNo | NVARCHAR(50) | Reference number for the transaction |
| StatementDate | DATETIME | Date when the statement was generated |
| StatementType | INT | Type of statement (0=bet, 1=deposit, 2=withdrawal, etc.) |
| TransDate | DATETIME | Actual transaction date |
| CustomerId | INT | Customer/user identifier |
| UserName | NVARCHAR(50) | Username of the customer |
| CustomerStatus | INT | Status of the customer account |
| CashIn | DECIMAL(19,6) | Amount credited to the account |
| CashOut | DECIMAL(19,6) | Amount debited from the account |
| Currency | CHAR(3) | Currency code (USD, EUR, etc.) |
| ExchangeRate | DECIMAL(12,8) | Exchange rate used for currency conversion |
| WebId | INT | Website/application identifier for multi-tenancy |
| WithdrawalLimit | DECIMAL(19,6) | Withdrawal limit associated with this transaction |

### TransactionRequest
Records all transaction requests including deposits, withdrawals, and transfers between customers.

```sql
CREATE TABLE [dbo].[TransactionRequest] (
    [ID] INT IDENTITY(1,1) NOT NULL,
    [TransactionType] INT NOT NULL,
    [ParentId] INT NOT NULL,
    [CustomerId] INT NULL,
    [Username] NVARCHAR(50) NULL,
    [Amount] DECIMAL(19,6) NOT NULL,
    [Currency] CHAR(3) NULL,
    [ExchangeRate] DECIMAL(12,8) NULL,
    [TransactionStatus] INT NOT NULL,
    [BeforeBalance] DECIMAL(19,6) NULL,
    [AfterBalance] DECIMAL(19,6) NULL,
    [WebId] INT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| ID | INT, PK | Unique identifier for the transaction request |
| TransactionType | INT | Type of transaction (1=deposit, 2=withdrawal, 3=transfer, etc.) |
| ParentId | INT | Parent transaction or related entity ID |
| CustomerId | INT | Customer/user identifier |
| Username | NVARCHAR(50) | Username of the customer |
| Amount | DECIMAL(19,6) | Transaction amount |
| Currency | CHAR(3) | Currency code for the transaction |
| ExchangeRate | DECIMAL(12,8) | Exchange rate applied to the transaction |
| TransactionStatus | INT | Status of the transaction (0=pending, 1=approved, 2=rejected, etc.) |
| BeforeBalance | DECIMAL(19,6) | Account balance before the transaction |
| AfterBalance | DECIMAL(19,6) | Account balance after the transaction |
| WebId | INT | Website/application identifier |

### CompanyBanks
Manages company bank accounts used for customer deposits and withdrawals.

```sql
CREATE TABLE [dbo].[CompanyBanks] (
    [ID] INT IDENTITY(1,1) NOT NULL,
    [AdminId] INT NOT NULL,
    [DepositMin] DECIMAL(19,6) NULL,
    [DepositMax] DECIMAL(19,6) NULL,
    [WithdrawalMin] DECIMAL(19,6) NULL,
    [WithdrawalMax] DECIMAL(19,6) NULL,
    [Status] INT NULL,
    [Currency] CHAR(3) NULL,
    [BankAccountNumber] VARCHAR(50) NULL,
    [BankAccountName] NVARCHAR(100) NULL,
    [BankName] NVARCHAR(100) NULL,
    [WebID] INT NULL,
    [BankBalance] DECIMAL(19,6) NOT NULL DEFAULT(0),
    [IsAutoDeposit] BIT NOT NULL DEFAULT(0)
)
```

| Column | Type | Description |
|--------|------|-------------|
| ID | INT, PK | Unique identifier for the company bank account |
| AdminId | INT | Administrator who manages this bank account |
| DepositMin | DECIMAL(19,6) | Minimum deposit amount allowed |
| DepositMax | DECIMAL(19,6) | Maximum deposit amount allowed |
| WithdrawalMin | DECIMAL(19,6) | Minimum withdrawal amount allowed |
| WithdrawalMax | DECIMAL(19,6) | Maximum withdrawal amount allowed |
| Status | INT | Status of the bank account (1=active, 0=inactive) |
| Currency | CHAR(3) | Currency supported by this bank account |
| BankAccountNumber | VARCHAR(50) | Bank account number |
| BankAccountName | NVARCHAR(100) | Name on the bank account |
| BankName | NVARCHAR(100) | Name of the bank |
| WebID | INT | Website identifier this bank account belongs to |
| BankBalance | DECIMAL(19,6) | Current balance in the bank account |
| IsAutoDeposit | BIT | Whether automatic deposit processing is enabled |

### DailyTransactionSummary
Summarizes daily transaction activity for each customer by currency and transaction type.

```sql
CREATE TABLE [dbo].[DailyTransactionSummary] (
    [Id] INT PRIMARY KEY IDENTITY(1,1),
    [Date] DATE NOT NULL,
    [WebId] INT NOT NULL,
    [Currency] NVARCHAR(3) NOT NULL,
    [CustomerId] INT NOT NULL,
    [Username] NVARCHAR(50),
    [ParentId] INT NOT NULL,
    [DepositCount] INT DEFAULT(0),
    [DepositAmount] DECIMAL(19,6) DEFAULT(0),
    [WithdrawalCount] INT DEFAULT(0),
    [WithdrawalAmount] DECIMAL(19,6) DEFAULT(0)
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the daily summary record |
| Date | DATE | Date of the transaction summary |
| WebId | INT | Website/application identifier |
| Currency | NVARCHAR(3) | Currency code for the summary |
| CustomerId | INT | Customer identifier |
| Username | NVARCHAR(50) | Username of the customer |
| ParentId | INT | Parent entity identifier in the hierarchy |
| DepositCount | INT | Number of deposit transactions |
| DepositAmount | DECIMAL(19,6) | Total deposit amount for the day |
| WithdrawalCount | INT | Number of withdrawal transactions |
| WithdrawalAmount | DECIMAL(19,6) | Total withdrawal amount for the day |

## Banking Tables

### PlayerPaymentWithdrawalBankInfo
Stores player bank information for withdrawal processing.

```sql
CREATE TABLE [dbo].[PlayerPaymentWithdrawalBankInfo] (
    [WebId] INT NOT NULL,
    [CustomerId] INT NOT NULL,
    [Username] VARCHAR(50) NOT NULL,
    [Currency] CHAR(3) NOT NULL,
    [BankName] NVARCHAR(100) NOT NULL,
    [BankAccountName] NVARCHAR(MAX) NOT NULL,
    [BankAccountNumber] VARCHAR(50) NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| WebId | INT | Website/application identifier |
| CustomerId | INT | Customer identifier |
| Username | VARCHAR(50) | Username of the customer |
| Currency | CHAR(3) | Currency for withdrawal |
| BankName | NVARCHAR(100) | Name of the player's bank |
| BankAccountName | NVARCHAR(MAX) | Name on the player's bank account |
| BankAccountNumber | VARCHAR(50) | Player's bank account number |

### PlayerBankListAfterFirstTimeDeposit
Tracks player bank information after their first successful deposit.

```sql
CREATE TABLE [dbo].[PlayerBankListAfterFirstTimeDeposit] (
    [Id] BIGINT IDENTITY(1,1) NOT NULL,
    [CustomerId] INT NOT NULL,
    [PlayerBankId] INT NOT NULL,
    [PlayerBankAccountName] NVARCHAR(MAX) NOT NULL,
    [PlayerBankAccountNumber] VARCHAR(50) NOT NULL,
    [Currency] CHAR(4) NOT NULL,
    [WebId] INT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | BIGINT, PK | Unique identifier for the record |
| CustomerId | INT | Customer identifier |
| PlayerBankId | INT | Bank identifier from bank list |
| PlayerBankAccountName | NVARCHAR(MAX) | Name on the player's bank account |
| PlayerBankAccountNumber | VARCHAR(50) | Player's bank account number |
| Currency | CHAR(4) | Currency code |
| WebId | INT | Website identifier |

## Financial Reporting

### FinancialReport
Stores monthly financial reports aggregated by customer, product, and currency.

```sql
CREATE TABLE [dbo].[FinancialReport] (
    [YearMonth] CHAR(6) NOT NULL,
    [Customer] NVARCHAR(50) NOT NULL,
    [ProductId] INT NOT NULL,
    [Product] NVARCHAR(100) NOT NULL,
    [AccountId] NVARCHAR(50) NOT NULL,
    [DataType] VARCHAR(20) NOT NULL,
    [Currency] CHAR(3) NOT NULL,
    [BetCount] INT NOT NULL,
    [ActiveUser] INT NOT NULL,
    [Turnover] DECIMAL(19,6) NOT NULL,
    [MemberGrWinLoss] DECIMAL(19,6) NOT NULL,
    [CustomerWinLoss] DECIMAL(19,6) NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| YearMonth | CHAR(6), PK | Year and month in YYYYMM format |
| Customer | NVARCHAR(50), PK | Customer identifier |
| ProductId | INT, PK | Product identifier |
| Product | NVARCHAR(100), PK | Product name |
| AccountId | NVARCHAR(50), PK | Account identifier |
| DataType | VARCHAR(20), PK | Type of data (summary, detail, etc.) |
| Currency | CHAR(3), PK | Currency code |
| BetCount | INT | Total number of bets placed |
| ActiveUser | INT | Number of active users |
| Turnover | DECIMAL(19,6) | Total betting turnover amount |
| MemberGrWinLoss | DECIMAL(19,6) | Member gross win/loss amount |
| CustomerWinLoss | DECIMAL(19,6) | Customer win/loss amount |

## Payment Gateway Tables

### PaymentGatewayTransaction
Records all transactions processed through payment gateways for deposits and withdrawals.

```sql
CREATE TABLE [dbo].[PaymentGatewayTransaction] (
    [ID] INT IDENTITY(1,1) NOT NULL,
    [TransactionType] INT NOT NULL,
    [TransactionTypeStr] NVARCHAR(50) NOT NULL,
    [ParentId] INT NOT NULL,
    [CustomerId] INT NULL,
    [Username] NVARCHAR(50) NULL,
    [Amount] DECIMAL(19,6) NOT NULL,
    [UsdAmount] DECIMAL(19,6) NULL,
    [BeforeBalance] DECIMAL(19,6) NULL,
    [AfterBalance] DECIMAL(19,6) NULL,
    [Currency] CHAR(3) NULL,
    [ExchangeRate] DECIMAL(12,8) NULL,
    [UsdExchangeRate] DECIMAL(12,8) NULL,
    [TransactionStatus] INT NOT NULL,
    [TransactionStatusStr] NVARCHAR(100) NOT NULL,
    [PaymentProviderErrorCode] NVARCHAR(50) DEFAULT ('') NULL,
    [PaymentProviderErrorMessage] NVARCHAR(250) DEFAULT ('') NULL,
    [IsShowTransactionErrorDetail] BIT,
    [IP] NVARCHAR(50) NULL,
    [ModifiedBy] VARCHAR(100) NULL,
    [ModifiedOn] DATETIME NULL,
    [CreatedBy] VARCHAR(100) NULL,
    [CreatedOn] DATETIME NULL,
    [Description] NVARCHAR(250) NULL,
    [Remark] NVARCHAR(250) NULL,
    [PlayerRemark] NVARCHAR(250) NULL,
    [WebId] INT NOT NULL,
    [PaymentMethod] NVARCHAR(50) NOT NULL,
    [PaymentProviderId] INT NOT NULL,
    [PaymentProvider] NVARCHAR(50) NOT NULL,
    [BankCode] NVARCHAR(50) NULL,
    [BankName] NVARCHAR(100) NULL,
    [TransactionToken] NVARCHAR(50) NULL,
    [ExternalRefNo] NCHAR(100) NULL,
    [PercentageTakesByPaymentProvider] DECIMAL(5,4) DEFAULT 0,
    [PercentageWeSellToCustomer] DECIMAL(5,4) DEFAULT 0,
    [MinFeeTakeByPaymentProvider] DECIMAL(10,4) DEFAULT 0,
    [TransactionFeeWeGet] DECIMAL(10,4) DEFAULT 0,
    [PlayerBankAccountName] NVARCHAR(100) DEFAULT '',
    [PlayerBankAccountNumber] NVARCHAR(50) DEFAULT '',
    [WalletAddress] NVARCHAR(100) DEFAULT '',
    [CryptoCurrencyNetwork] NVARCHAR(50) DEFAULT '',
    [GasFeeRate] DECIMAL(5,4) DEFAULT 1,
    [IsArbitragePlayer] BIT DEFAULT 0 NOT NULL,
    [ExtraParameter] NVARCHAR(MAX) DEFAULT '' NOT NULL,
    [AvailableWithdrawAmount] DECIMAL(19,6) NOT NULL DEFAULT 0,
    [ExtraInternalParameter] NVARCHAR(MAX)
)
```

| Column | Type | Description |
|--------|------|-------------|
| ID | INT, PK | Unique identifier for the payment gateway transaction |
| TransactionType | INT | Type of transaction (1=deposit, 2=withdrawal) |
| TransactionTypeStr | NVARCHAR(50) | String representation of transaction type |
| ParentId | INT | Parent entity or transaction identifier |
| CustomerId | INT | Customer identifier |
| Username | NVARCHAR(50) | Customer username |
| Amount | DECIMAL(19,6) | Transaction amount in original currency |
| UsdAmount | DECIMAL(19,6) | Transaction amount in USD |
| BeforeBalance | DECIMAL(19,6) | Account balance before transaction |
| AfterBalance | DECIMAL(19,6) | Account balance after transaction |
| Currency | CHAR(3) | Transaction currency code |
| ExchangeRate | DECIMAL(12,8) | Exchange rate for currency conversion |
| UsdExchangeRate | DECIMAL(12,8) | USD exchange rate |
| TransactionStatus | INT | Status code (0=pending, 1=success, 2=failed, etc.) |
| TransactionStatusStr | NVARCHAR(100) | String representation of status |
| PaymentProviderErrorCode | NVARCHAR(50) | Error code from payment provider |
| PaymentProviderErrorMessage | NVARCHAR(250) | Error message from payment provider |
| IsShowTransactionErrorDetail | BIT | Whether to show error details to user |
| IP | NVARCHAR(50) | IP address of the transaction request |
| ModifiedBy | VARCHAR(100) | User who last modified the record |
| ModifiedOn | DATETIME | Timestamp of last modification |
| CreatedBy | VARCHAR(100) | User who created the record |
| CreatedOn | DATETIME | Timestamp of record creation |
| Description | NVARCHAR(250) | Transaction description |
| Remark | NVARCHAR(250) | Administrative remarks |
| PlayerRemark | NVARCHAR(250) | Player's remarks or notes |
| WebId | INT | Website identifier |
| PaymentMethod | NVARCHAR(50) | Payment method used (bank_transfer, e_wallet, etc.) |
| PaymentProviderId | INT | Payment provider identifier |
| PaymentProvider | NVARCHAR(50) | Payment provider name |
| BankCode | NVARCHAR(50) | Bank code for bank transfers |
| BankName | NVARCHAR(100) | Bank name |
| TransactionToken | NVARCHAR(50) | Unique token for the transaction |
| ExternalRefNo | NCHAR(100) | External reference number from payment provider |
| PercentageTakesByPaymentProvider | DECIMAL(5,4) | Fee percentage taken by payment provider |
| PercentageWeSellToCustomer | DECIMAL(5,4) | Fee percentage charged to customer |
| MinFeeTakeByPaymentProvider | DECIMAL(10,4) | Minimum fee taken by payment provider |
| TransactionFeeWeGet | DECIMAL(10,4) | Fee amount received |
| PlayerBankAccountName | NVARCHAR(100) | Player's bank account name |
| PlayerBankAccountNumber | NVARCHAR(50) | Player's bank account number |
| WalletAddress | NVARCHAR(100) | Cryptocurrency wallet address |
| CryptoCurrencyNetwork | NVARCHAR(50) | Cryptocurrency network (Bitcoin, Ethereum, etc.) |
| GasFeeRate | DECIMAL(5,4) | Gas fee rate for cryptocurrency transactions |
| IsArbitragePlayer | BIT | Flag indicating if player is flagged for arbitrage |
| ExtraParameter | NVARCHAR(MAX) | Additional parameters in JSON format |
| AvailableWithdrawAmount | DECIMAL(19,6) | Amount available for withdrawal after this transaction |
| ExtraInternalParameter | NVARCHAR(MAX) | Internal parameters for system use |

### PaymentGatewayBanks
Configuration of banks available through each payment gateway provider.

```sql
CREATE TABLE [dbo].[PaymentGatewayBanks] (
    [ID] INT PRIMARY KEY IDENTITY(1,1),
    [BankCode] NVARCHAR(100) NOT NULL,
    [BankName] NVARCHAR(100) NOT NULL,
    [PaymentProviderId] INT NOT NULL,
    [PaymentMethod] NVARCHAR(200) DEFAULT ('InternetBanking') NOT NULL,
    [Currency] NVARCHAR(5) NOT NULL,
    [CountryCode] NVARCHAR(5) NOT NULL,
    [IsEnabled] BIT NOT NULL,
    [ModifiedOn] DATETIME NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| ID | INT, PK | Unique identifier for the payment gateway bank |
| BankCode | NVARCHAR(100) | Unique bank code used by payment provider |
| BankName | NVARCHAR(100) | Display name of the bank |
| PaymentProviderId | INT | Payment provider that supports this bank |
| PaymentMethod | NVARCHAR(200) | Payment method type (InternetBanking, ATM, etc.) |
| Currency | NVARCHAR(5) | Currency supported by this bank |
| CountryCode | NVARCHAR(5) | Country code where bank operates |
| IsEnabled | BIT | Whether this bank is currently enabled |
| ModifiedOn | DATETIME | Last modification timestamp |

### PaymentGatewayCompanyBankSettings
Company-specific settings for payment gateway banks.

```sql
CREATE TABLE [dbo].[PaymentGatewayCompanyBankSettings] (
    [ID] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [PaymentProviderId] INT NOT NULL,
    [BankCode] NVARCHAR(100) NOT NULL,
    [IsEnabled] BIT NOT NULL,
    [ModifiedBy] NVARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| ID | INT, PK | Unique identifier for the setting |
| WebId | INT | Website identifier |
| PaymentProviderId | INT | Payment provider identifier |
| BankCode | NVARCHAR(100) | Bank code |
| IsEnabled | BIT | Whether this bank is enabled for this company |
| ModifiedBy | NVARCHAR(50) | User who last modified the setting |
| ModifiedOn | DATETIME | Last modification timestamp |
| CreatedBy | NVARCHAR(50) | User who created the setting |
| CreatedOn | DATETIME | Creation timestamp |

### PaymentProviders
Master list of payment gateway providers available in the system.

```sql
CREATE TABLE [dbo].[PaymentProviders] (
    [ID] INT IDENTITY(1,1) NOT NULL,
    [Name] NVARCHAR(100) NOT NULL,
    [Code] NVARCHAR(50) NOT NULL,
    [IsEnabled] BIT NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [ModifiedBy] NVARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| ID | INT, PK | Unique identifier for the payment provider |
| Name | NVARCHAR(100) | Display name of the payment provider |
| Code | NVARCHAR(50) | Unique code for the payment provider |
| IsEnabled | BIT | Whether this provider is currently enabled |
| CreatedBy | NVARCHAR(50) | User who created the provider record |
| CreatedOn | DATETIME | Creation timestamp |
| ModifiedBy | NVARCHAR(50) | User who last modified the provider |
| ModifiedOn | DATETIME | Last modification timestamp |

### PaymentMethodDisplayOrder
Controls the display order of payment methods in the user interface.

```sql
CREATE TABLE [dbo].[PaymentMethodDisplayOrder] (
    [ID] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [PaymentMethod] NVARCHAR(100) NOT NULL,
    [DisplayOrder] INT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| ID | INT, PK | Unique identifier for the display order setting |
| WebId | INT | Website identifier |
| PaymentMethod | NVARCHAR(100) | Name of the payment method |
| DisplayOrder | INT | Order number for display (lower numbers appear first) |

### PaymentPercentagesDealWithCustomer
Defines fee structures and percentages for payment transactions.

```sql
CREATE TABLE [dbo].[PaymentPercentagesDealWithCustomer] (
    [ID] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [PaymentProviderId] INT NOT NULL,
    [Currency] NVARCHAR(5) NOT NULL,
    [PercentageTakesByPaymentProvider] DECIMAL(5,4) NOT NULL,
    [PercentageWeSellToCustomer] DECIMAL(5,4) NOT NULL,
    [MinFeeTakeByPaymentProvider] DECIMAL(10,4) NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [ModifiedBy] NVARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| ID | INT, PK | Unique identifier for the fee structure |
| WebId | INT | Website identifier |
| PaymentProviderId | INT | Payment provider identifier |
| Currency | NVARCHAR(5) | Currency this fee structure applies to |
| PercentageTakesByPaymentProvider | DECIMAL(5,4) | Percentage fee taken by payment provider |
| PercentageWeSellToCustomer | DECIMAL(5,4) | Percentage fee charged to customer |
| MinFeeTakeByPaymentProvider | DECIMAL(10,4) | Minimum fee amount taken by provider |
| CreatedBy | NVARCHAR(50) | User who created the fee structure |
| CreatedOn | DATETIME | Creation timestamp |
| ModifiedBy | NVARCHAR(50) | User who last modified the structure |
| ModifiedOn | DATETIME | Last modification timestamp |

## Game Provider & Transfer Tables

### TransferGameProviderDailyStatement
Daily statements for game provider transfers and betting activity.

```sql
CREATE TABLE [dbo].[TransferGameProviderDailyStatement] (
    [Transid] BIGINT IDENTITY(1,1) NOT NULL,
    [CustomerId] INT NOT NULL,
    [UserName] NVARCHAR(50) NOT NULL,
    [WinLostDate] SMALLDATETIME NOT NULL,
    [AgtID] INT DEFAULT(0) NOT NULL,
    [MaID] INT DEFAULT(0) NOT NULL,
    [SmaID] INT DEFAULT(0) NOT NULL,
    [SSmaID] INT DEFAULT(0) NOT NULL,
    [SHID] INT DEFAULT(0) NOT NULL,
    [StatementType] TINYINT DEFAULT(0) NOT NULL,
    [Currency] CHAR(3) NULL,
    [CustomerStatus] INT DEFAULT(0) NOT NULL,
    [ProductType] INT DEFAULT(0) NOT NULL,
    [StatementStatus] INT DEFAULT(0) NULL,
    [CashIn] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [CashOut] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [CommIn] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [CommOut] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [DiscountIn] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [DiscountOut] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [ActualRate] DECIMAL(12,8) NULL,
    [TotalCashIn] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [TotalCashOut] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [TransDate] DATETIME DEFAULT(getdate()) NOT NULL,
    [CasinoTotalCashIn] DECIMAL(19,6) DEFAULT(0) NULL,
    [CasinoTotalCashOut] DECIMAL(19,6) DEFAULT(0) NULL,
    [NetTurnover] DECIMAL(19,6) DEFAULT(0) NULL,
    [tstamp] ROWVERSION NULL,
    [WebId] INT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Transid | BIGINT, PK | Unique transaction identifier |
| CustomerId | INT | Customer identifier |
| UserName | NVARCHAR(50) | Customer username |
| WinLostDate | SMALLDATETIME | Date of win/loss calculation |
| AgtID | INT | Agent identifier in hierarchy |
| MaID | INT | Master Agent identifier |
| SmaID | INT | Super Master Agent identifier |
| SSmaID | INT | Super Super Master Agent identifier |
| SHID | INT | Super Head identifier |
| StatementType | TINYINT | Type of statement (0=betting, 1=commission, etc.) |
| Currency | CHAR(3) | Currency code |
| CustomerStatus | INT | Customer account status |
| ProductType | INT | Game product type identifier |
| StatementStatus | INT | Status of the statement |
| CashIn | DECIMAL(19,6) | Amount won by customer |
| CashOut | DECIMAL(19,6) | Amount lost by customer |
| CommIn | DECIMAL(19,6) | Commission earned |
| CommOut | DECIMAL(19,6) | Commission paid |
| DiscountIn | DECIMAL(19,6) | Discount received |
| DiscountOut | DECIMAL(19,6) | Discount given |
| ActualRate | DECIMAL(12,8) | Actual exchange rate applied |
| TotalCashIn | DECIMAL(19,6) | Total cash in amount |
| TotalCashOut | DECIMAL(19,6) | Total cash out amount |
| TransDate | DATETIME | Transaction processing date |
| CasinoTotalCashIn | DECIMAL(19,6) | Casino-specific cash in amount |
| CasinoTotalCashOut | DECIMAL(19,6) | Casino-specific cash out amount |
| NetTurnover | DECIMAL(19,6) | Net turnover amount |
| tstamp | ROWVERSION | Row version for concurrency control |
| WebId | INT | Website identifier |

### DailyStatement
Daily aggregated statements for all customer betting activity.

```sql
CREATE TABLE [dbo].[DailyStatement] (
    [Transid] BIGINT IDENTITY(1,1) NOT NULL,
    [CustID] INT NOT NULL,
    [UserName] NVARCHAR(50) NOT NULL,
    [WinLostDate] SMALLDATETIME NOT NULL,
    [AgtID] INT DEFAULT(0) NOT NULL,
    [MaID] INT DEFAULT(0) NOT NULL,
    [SmaID] INT DEFAULT(0) NOT NULL,
    [SSmaID] INT DEFAULT(0) NOT NULL,
    [SHID] INT DEFAULT(0) NOT NULL,
    [StatementType] TINYINT DEFAULT(0) NOT NULL,
    [Currency] CHAR(3) NULL,
    [CustomerStatus] INT DEFAULT(0) NOT NULL,
    [ProductType] INT DEFAULT(0) NOT NULL,
    [StatementStatus] INT DEFAULT(0) NULL,
    [CashIn] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [CashOut] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [CommIn] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [CommOut] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [DiscountIn] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [DiscountOut] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [ActualRate] DECIMAL(12,8) NULL,
    [TotalCashIn] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [TotalCashOut] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [TransDate] DATETIME DEFAULT(getdate()) NOT NULL,
    [CasinoTotalCashIn] DECIMAL(19,6) DEFAULT(0) NULL,
    [CasinoTotalCashOut] DECIMAL(19,6) DEFAULT(0) NULL,
    [NetTurnover] DECIMAL(19,6) DEFAULT(0) NULL,
    [tstamp] ROWVERSION NULL,
    [WebId] INT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Transid | BIGINT, PK | Unique transaction identifier |
| CustID | INT | Customer identifier |
| UserName | NVARCHAR(50) | Customer username |
| WinLostDate | SMALLDATETIME | Date of win/loss calculation |
| AgtID | INT | Agent identifier in hierarchy |
| MaID | INT | Master Agent identifier |
| SmaID | INT | Super Master Agent identifier |
| SSmaID | INT | Super Super Master Agent identifier |
| SHID | INT | Super Head identifier |
| StatementType | TINYINT | Type of statement |
| Currency | CHAR(3) | Currency code |
| CustomerStatus | INT | Customer account status |
| ProductType | INT | Game product type identifier |
| StatementStatus | INT | Status of the statement |
| CashIn | DECIMAL(19,6) | Amount won by customer |
| CashOut | DECIMAL(19,6) | Amount lost by customer |
| CommIn | DECIMAL(19,6) | Commission earned |
| CommOut | DECIMAL(19,6) | Commission paid |
| DiscountIn | DECIMAL(19,6) | Discount received |
| DiscountOut | DECIMAL(19,6) | Discount given |
| ActualRate | DECIMAL(12,8) | Actual exchange rate applied |
| TotalCashIn | DECIMAL(19,6) | Total cash in amount |
| TotalCashOut | DECIMAL(19,6) | Total cash out amount |
| TransDate | DATETIME | Transaction processing date |
| CasinoTotalCashIn | DECIMAL(19,6) | Casino-specific cash in amount |
| CasinoTotalCashOut | DECIMAL(19,6) | Casino-specific cash out amount |
| NetTurnover | DECIMAL(19,6) | Net turnover amount |
| tstamp | ROWVERSION | Row version for concurrency control |
| WebId | INT | Website identifier |

### AutoTransferTransaction
Records automatic transfers between different game products or wallets.

```sql
CREATE TABLE [dbo].[AutoTransferTransaction] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [CustomerId] INT NOT NULL,
    [Username] NVARCHAR(50) NOT NULL,
    [FromProduct] INT NOT NULL,
    [ToProduct] INT NOT NULL,
    [Amount] DECIMAL(19,6) NOT NULL,
    [Currency] CHAR(3) NOT NULL,
    [Status] INT NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [ModifiedOn] DATETIME NOT NULL,
    [WebId] INT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the auto transfer |
| CustomerId | INT | Customer identifier |
| Username | NVARCHAR(50) | Customer username |
| FromProduct | INT | Source product/wallet identifier |
| ToProduct | INT | Destination product/wallet identifier |
| Amount | DECIMAL(19,6) | Transfer amount |
| Currency | CHAR(3) | Currency code |
| Status | INT | Transfer status (0=pending, 1=completed, 2=failed) |
| CreatedOn | DATETIME | Creation timestamp |
| ModifiedOn | DATETIME | Last modification timestamp |
| WebId | INT | Website identifier |

## Currency Management Tables

### Currency
Manages currency settings, exchange rates, and limits for each website.

```sql
CREATE TABLE [dbo].[Currency] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [Currency] CHAR(3) NOT NULL,
    [ExchangeRate] DECIMAL(10,6) NULL,
    [IsEnabled] BIT NULL,
    [IsDecimalAllow] BIT DEFAULT(1),
    [CashAgentId] INT NOT NULL,
    [DepositMin] INT NULL,
    [DepositMax] INT NULL,
    [WithdrawalMin] INT NULL,
    [WithdrawalMax] INT NULL,
    [ModifiedBy] NVARCHAR(50) NULL,
    [ModifiedOn] DATETIME NULL,
    [MaxManualWithdrawal] DECIMAL(19,6) NOT NULL DEFAULT 0,
    [ManualDepositLimitation] DECIMAL(19,6) NOT NULL DEFAULT 0,
    [TotalManualDeposit] DECIMAL(19,6) NOT NULL DEFAULT 0,
    [IsLimitManualDeposit] BIT NOT NULL DEFAULT 0,
    [IsLimitMaxManualWithdrawal] BIT NOT NULL DEFAULT 0
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the currency record |
| WebId | INT | Website identifier |
| Currency | CHAR(3) | Currency code (USD, EUR, THB, etc.) |
| ExchangeRate | DECIMAL(10,6) | Exchange rate to base currency |
| IsEnabled | BIT | Whether this currency is enabled for the website |
| IsDecimalAllow | BIT | Whether decimal amounts are allowed |
| CashAgentId | INT | Cash agent responsible for this currency |
| DepositMin | INT | Minimum deposit amount |
| DepositMax | INT | Maximum deposit amount |
| WithdrawalMin | INT | Minimum withdrawal amount |
| WithdrawalMax | INT | Maximum withdrawal amount |
| ModifiedBy | NVARCHAR(50) | User who last modified the record |
| ModifiedOn | DATETIME | Last modification timestamp |
| MaxManualWithdrawal | DECIMAL(19,6) | Maximum manual withdrawal limit |
| ManualDepositLimitation | DECIMAL(19,6) | Manual deposit limitation amount |
| TotalManualDeposit | DECIMAL(19,6) | Total manual deposits processed |
| IsLimitManualDeposit | BIT | Whether manual deposit is limited |
| IsLimitMaxManualWithdrawal | BIT | Whether max manual withdrawal is enforced |

### CurrencyEUR
Stores EUR exchange rates for different currencies.

```sql
CREATE TABLE [dbo].[CurrencyEUR] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [Currency] CHAR(3) NOT NULL,
    [ExchangeRate] DECIMAL(10,6) NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL,
    [ModifiedBy] NVARCHAR(50) NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the EUR exchange rate record |
| Currency | CHAR(3) | Currency code |
| ExchangeRate | DECIMAL(10,6) | Exchange rate to EUR |
| CreatedOn | DATETIME | Creation timestamp |
| CreatedBy | NVARCHAR(50) | User who created the record |
| ModifiedOn | DATETIME | Last modification timestamp |
| ModifiedBy | NVARCHAR(50) | User who last modified the record |

### CurrencyUSD
Stores USD exchange rates for different currencies.

```sql
CREATE TABLE [dbo].[CurrencyUSD] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [Currency] CHAR(3) NOT NULL,
    [ExchangeRate] DECIMAL(10,6) NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL,
    [ModifiedBy] NVARCHAR(50) NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the USD exchange rate record |
| Currency | CHAR(3) | Currency code |
| ExchangeRate | DECIMAL(10,6) | Exchange rate to USD |
| CreatedOn | DATETIME | Creation timestamp |
| CreatedBy | NVARCHAR(50) | User who created the record |
| ModifiedOn | DATETIME | Last modification timestamp |
| ModifiedBy | NVARCHAR(50) | User who last modified the record |

### CurrencyNew
Stores new currency exchange rates.

```sql
CREATE TABLE [dbo].[CurrencyNew] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [Currency] CHAR(3) NOT NULL,
    [ExchangeRate] DECIMAL(10,6) NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the new currency record |
| Currency | CHAR(3) | Currency code |
| ExchangeRate | DECIMAL(10,6) | Exchange rate |
| CreatedOn | DATETIME | Creation timestamp |
| CreatedBy | NVARCHAR(50) | User who created the record |

### CurrencyWhiteList
Manages which currencies are allowed for each website.

```sql
CREATE TABLE [dbo].[CurrencyWhiteList] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [Currency] CHAR(3) NOT NULL,
    [IsEnabled] BIT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the whitelist entry |
| WebId | INT | Website identifier |
| Currency | CHAR(3) | Currency code |
| IsEnabled | BIT | Whether this currency is enabled for the website |

## Company & Organization Tables

### Company
Stores company information for each website.

```sql
CREATE TABLE [dbo].[Company] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [CompanyName] NVARCHAR(100) NOT NULL,
    [CompanyCode] NVARCHAR(50) NOT NULL,
    [IsEnabled] BIT NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [ModifiedBy] NVARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the company |
| WebId | INT | Website identifier |
| CompanyName | NVARCHAR(100) | Official company name |
| CompanyCode | NVARCHAR(50) | Unique company code |
| IsEnabled | BIT | Whether the company is currently active |
| CreatedBy | NVARCHAR(50) | User who created the company record |
| CreatedOn | DATETIME | Creation timestamp |
| ModifiedBy | NVARCHAR(50) | User who last modified the record |
| ModifiedOn | DATETIME | Last modification timestamp |

### CompanyPaymentProviders
Links companies with their available payment providers.

```sql
CREATE TABLE [dbo].[CompanyPaymentProviders] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [PaymentProviderId] INT NOT NULL,
    [IsEnabled] BIT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the company payment provider link |
| WebId | INT | Website identifier |
| PaymentProviderId | INT | Payment provider identifier |
| IsEnabled | BIT | Whether this provider is enabled for the company |

### CompanyCap
Manages company financial caps and alert settings.

```sql
CREATE TABLE [dbo].[CompanyCap] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [Prepayment] DECIMAL(19,6) NOT NULL,
    [StartDate] DATETIME NOT NULL,
    [Multiple] INT NOT NULL,
    [CurrentWinlose] DECIMAL(19,6) NOT NULL,
    [IsSentAnnouncement] BIT NOT NULL,
    [Email1] NVARCHAR(2000) NULL,
    [Email2] NVARCHAR(2000) NULL,
    [Email3] NVARCHAR(2000) NULL,
    [Email4] NVARCHAR(2000) NULL,
    [AlertPercentage1] DECIMAL(5,4) NULL,
    [AlertPercentage2] DECIMAL(5,4) NULL,
    [AlertPercentage3] DECIMAL(5,4) NULL,
    [AlertPercentage4] DECIMAL(5,4) NULL,
    [IsEmail1Sent] BIT NOT NULL,
    [IsEmail2Sent] BIT NOT NULL,
    [IsEmail3Sent] BIT NOT NULL,
    [IsEmail4Sent] BIT NOT NULL,
    [IsEnabled] BIT NOT NULL,
    [IsPrepaymentProgram] BIT NOT NULL,
    [ModifiedOn] DATETIME NOT NULL,
    [ModifiedBy] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the company cap |
| WebId | INT | Website identifier |
| Prepayment | DECIMAL(19,6) | Prepayment amount |
| StartDate | DATETIME | Start date for the cap period |
| Multiple | INT | Multiplier for cap calculation |
| CurrentWinlose | DECIMAL(19,6) | Current win/loss amount |
| IsSentAnnouncement | BIT | Whether announcement has been sent |
| Email1 | NVARCHAR(2000) | First alert email address |
| Email2 | NVARCHAR(2000) | Second alert email address |
| Email3 | NVARCHAR(2000) | Third alert email address |
| Email4 | NVARCHAR(2000) | Fourth alert email address |
| AlertPercentage1 | DECIMAL(5,4) | First alert threshold percentage |
| AlertPercentage2 | DECIMAL(5,4) | Second alert threshold percentage |
| AlertPercentage3 | DECIMAL(5,4) | Third alert threshold percentage |
| AlertPercentage4 | DECIMAL(5,4) | Fourth alert threshold percentage |
| IsEmail1Sent | BIT | Whether first alert email was sent |
| IsEmail2Sent | BIT | Whether second alert email was sent |
| IsEmail3Sent | BIT | Whether third alert email was sent |
| IsEmail4Sent | BIT | Whether fourth alert email was sent |
| IsEnabled | BIT | Whether the cap is enabled |
| IsPrepaymentProgram | BIT | Whether prepayment program is active |
| ModifiedOn | DATETIME | Last modification timestamp |
| ModifiedBy | NVARCHAR(50) | User who last modified the record |
| CreatedOn | DATETIME | Creation timestamp |
| CreatedBy | NVARCHAR(50) | User who created the record |

### AllDomain
Stores all domains associated with websites.

```sql
CREATE TABLE [dbo].[AllDomain] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [Domain] NVARCHAR(200) NOT NULL,
    [IsEnabled] BIT NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [ModifiedBy] NVARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL,
    [IsAgentExclusive] BIT NOT NULL DEFAULT 0
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the domain |
| WebId | INT | Website identifier |
| Domain | NVARCHAR(200) | Domain name |
| IsEnabled | BIT | Whether the domain is active |
| CreatedBy | NVARCHAR(50) | User who created the domain record |
| CreatedOn | DATETIME | Creation timestamp |
| ModifiedBy | NVARCHAR(50) | User who last modified the record |
| ModifiedOn | DATETIME | Last modification timestamp |
| IsAgentExclusive | BIT | Whether domain is exclusive to agents |

### DownlineRelationship
Tracks hierarchical relationships between entities.

```sql
CREATE TABLE [dbo].[DownlineRelationship] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [ParentId] INT NOT NULL,
    [ChildId] INT NOT NULL,
    [Depth] INT NOT NULL,
    [CreatedOn] DATETIME NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the relationship |
| WebId | INT | Website identifier |
| ParentId | INT | Parent entity identifier |
| ChildId | INT | Child entity identifier |
| Depth | INT | Depth level in the hierarchy |
| CreatedOn | DATETIME | Creation timestamp |

## OTP & Authentication Tables

### OtpVerification
Stores OTP verification records for various authentication processes.

```sql
CREATE TABLE [dbo].[OtpVerification] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [Phone] VARCHAR(15) NOT NULL DEFAULT (''),
    [Email] NVARCHAR(250) NOT NULL DEFAULT (''),
    [CountryCode] NVARCHAR(2) NOT NULL,
    [SmsUuid] NVARCHAR(MAX) NOT NULL,
    [Captcha] NVARCHAR(8) NOT NULL,
    [CreatedOn] DATETIME DEFAULT(getdate()) NOT NULL,
    [ModifiedOn] DATETIME DEFAULT(getdate()) NOT NULL,
    [Remark] NVARCHAR(250),
    [IsVerify] BIT,
    [OtpType] NVARCHAR(50),
    [IsVerifyByWithdrawal] BIT,
    [Username] NVARCHAR(100),
    [Ip] NVARCHAR(50)
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the OTP verification |
| WebId | INT | Website identifier |
| Phone | VARCHAR(15) | Phone number for SMS OTP |
| Email | NVARCHAR(250) | Email address for email OTP |
| CountryCode | NVARCHAR(2) | Country code for phone number |
| SmsUuid | NVARCHAR(MAX) | UUID for SMS tracking |
| Captcha | NVARCHAR(8) | OTP code sent |
| CreatedOn | DATETIME | Creation timestamp |
| ModifiedOn | DATETIME | Last modification timestamp |
| Remark | NVARCHAR(250) | Additional remarks |
| IsVerify | BIT | Whether OTP was verified |
| OtpType | NVARCHAR(50) | Type of OTP (SMS, Email, etc.) |
| IsVerifyByWithdrawal | BIT | Whether verified for withdrawal |
| Username | NVARCHAR(100) | Username requesting OTP |
| Ip | NVARCHAR(50) | IP address of the request |

### OperatorOtpAccount
Stores OTP provider account information for operators.

```sql
CREATE TABLE [dbo].[OperatorOtpAccount] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [OtpProviderId] INT NOT NULL,
    [ApiKey] NVARCHAR(500) NOT NULL,
    [SecretKey] NVARCHAR(500) NOT NULL,
    [IsEnabled] BIT NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the operator OTP account |
| WebId | INT | Website identifier |
| OtpProviderId | INT | OTP provider identifier |
| ApiKey | NVARCHAR(500) | API key for OTP provider |
| SecretKey | NVARCHAR(500) | Secret key for OTP provider |
| IsEnabled | BIT | Whether the account is enabled |
| CreatedBy | NVARCHAR(50) | User who created the account |
| CreatedOn | DATETIME | Creation timestamp |

### OtpProviders
Master list of OTP service providers.

```sql
CREATE TABLE [dbo].[OtpProviders] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [Name] NVARCHAR(100) NOT NULL,
    [Code] NVARCHAR(50) NOT NULL,
    [IsEnabled] BIT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the OTP provider |
| Name | NVARCHAR(100) | Name of the OTP provider |
| Code | NVARCHAR(50) | Unique code for the provider |
| IsEnabled | BIT | Whether the provider is enabled |

## Security & Access Control Tables

### IpWhiteList
Manages IP addresses that are allowed access to the system.

```sql
CREATE TABLE [dbo].[IpWhiteList] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [IpAddress] NVARCHAR(50) NOT NULL,
    [Description] NVARCHAR(200) NULL,
    [IsEnabled] BIT NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [ModifiedBy] NVARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the whitelist entry |
| WebId | INT | Website identifier |
| IpAddress | NVARCHAR(50) | IP address or CIDR range |
| Description | NVARCHAR(200) | Description of the IP entry |
| IsEnabled | BIT | Whether the whitelist entry is active |
| CreatedBy | NVARCHAR(50) | User who created the entry |
| CreatedOn | DATETIME | Creation timestamp |
| ModifiedBy | NVARCHAR(50) | User who last modified the entry |
| ModifiedOn | DATETIME | Last modification timestamp |

### IpBlacklist
Manages IP addresses that are blocked from accessing the system.

```sql
CREATE TABLE [dbo].[IpBlacklist] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [IpAddress] NVARCHAR(50) NOT NULL,
    [Description] NVARCHAR(200) NULL,
    [IsEnabled] BIT NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [ModifiedBy] NVARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the blacklist entry |
| WebId | INT | Website identifier |
| IpAddress | NVARCHAR(50) | IP address or CIDR range to block |
| Description | NVARCHAR(200) | Reason for blocking this IP |
| IsEnabled | BIT | Whether the blacklist entry is active |
| CreatedBy | NVARCHAR(50) | User who created the entry |
| CreatedOn | DATETIME | Creation timestamp |
| ModifiedBy | NVARCHAR(50) | User who last modified the entry |
| ModifiedOn | DATETIME | Last modification timestamp |

## Player Banking Tables

### PlayerBankOptions
Manages available bank options for players in different countries and currencies.

```sql
CREATE TABLE [dbo].[PlayerBankOptions] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [BankName] NVARCHAR(100) NOT NULL,
    [BankCode] NVARCHAR(50) NOT NULL,
    [CountryCode] NVARCHAR(5) NOT NULL,
    [Currency] CHAR(3) NOT NULL,
    [IsEnabled] BIT NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [ModifiedBy] NVARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the player bank option |
| WebId | INT | Website identifier |
| BankName | NVARCHAR(100) | Name of the bank |
| BankCode | NVARCHAR(50) | Unique bank code |
| CountryCode | NVARCHAR(5) | Country code where bank operates |
| Currency | CHAR(3) | Currency supported by the bank |
| IsEnabled | BIT | Whether this bank option is enabled |
| CreatedBy | NVARCHAR(50) | User who created the bank option |
| CreatedOn | DATETIME | Creation timestamp |
| ModifiedBy | NVARCHAR(50) | User who last modified the option |
| ModifiedOn | DATETIME | Last modification timestamp |

### PlayerBankGroupSettings
Manages bank group settings for different currencies and websites.

```sql
CREATE TABLE [dbo].[PlayerBankGroupSettings] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [BankGroup] NVARCHAR(100) NOT NULL,
    [Currency] CHAR(3) NOT NULL,
    [IsEnabled] BIT NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [ModifiedBy] NVARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the bank group setting |
| WebId | INT | Website identifier |
| BankGroup | NVARCHAR(100) | Name of the bank group |
| Currency | CHAR(3) | Currency for this bank group |
| IsEnabled | BIT | Whether this bank group is enabled |
| CreatedBy | NVARCHAR(50) | User who created the setting |
| CreatedOn | DATETIME | Creation timestamp |
| ModifiedBy | NVARCHAR(50) | User who last modified the setting |
| ModifiedOn | DATETIME | Last modification timestamp |

### PlayerFirstTimeDeposit
Tracks whether a player has made their first deposit.

```sql
CREATE TABLE [dbo].[PlayerFirstTimeDeposit] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [CustomerId] INT NOT NULL,
    [WebId] INT NOT NULL,
    [IsFirstTimeDeposit] BIT NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [ModifiedOn] DATETIME NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the first deposit record |
| CustomerId | INT | Customer identifier |
| WebId | INT | Website identifier |
| IsFirstTimeDeposit | BIT | Whether customer has made first deposit |
| CreatedOn | DATETIME | Creation timestamp |
| ModifiedOn | DATETIME | Last modification timestamp |

## Promotion & Bonus Tables

### PromotionRequest
Records player requests for promotions and bonuses.

```sql
CREATE TABLE [dbo].[PromotionRequest] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [CustomerId] INT NOT NULL,
    [Username] NVARCHAR(50) NOT NULL,
    [PromotionId] INT NOT NULL,
    [Amount] DECIMAL(19,6) NOT NULL,
    [Currency] CHAR(3) NOT NULL,
    [Status] INT NOT NULL,
    [RequestDate] DATETIME NOT NULL,
    [ApprovedDate] DATETIME NULL,
    [ApprovedBy] NVARCHAR(50) NULL,
    [Remark] NVARCHAR(500) NULL,
    [WebId] INT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the promotion request |
| CustomerId | INT | Customer identifier |
| Username | NVARCHAR(50) | Customer username |
| PromotionId | INT | Promotion identifier |
| Amount | DECIMAL(19,6) | Requested promotion amount |
| Currency | CHAR(3) | Currency code |
| Status | INT | Status of the request (0=pending, 1=approved, 2=rejected) |
| RequestDate | DATETIME | Date when request was made |
| ApprovedDate | DATETIME | Date when request was approved |
| ApprovedBy | NVARCHAR(50) | User who approved the request |
| Remark | NVARCHAR(500) | Additional remarks |
| WebId | INT | Website identifier |

### PromotionApply
Records applied promotions with their requirements and progress.

```sql
CREATE TABLE [dbo].[PromotionApply] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [CustomerId] INT NOT NULL,
    [Username] NVARCHAR(50) NOT NULL,
    [PromotionId] INT NOT NULL,
    [Amount] DECIMAL(19,6) NOT NULL,
    [Currency] CHAR(3) NOT NULL,
    [Status] INT NOT NULL,
    [AppliedDate] DATETIME NOT NULL,
    [CompletedDate] DATETIME NULL,
    [Remark] NVARCHAR(500) NULL,
    [WebId] INT NOT NULL,
    [BonusAmount] DECIMAL(19,6) NOT NULL DEFAULT 0,
    [TurnoverRequirement] DECIMAL(19,6) NOT NULL DEFAULT 0,
    [CurrentTurnover] DECIMAL(19,6) NOT NULL DEFAULT 0
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the applied promotion |
| CustomerId | INT | Customer identifier |
| Username | NVARCHAR(50) | Customer username |
| PromotionId | INT | Promotion identifier |
| Amount | DECIMAL(19,6) | Applied promotion amount |
| Currency | CHAR(3) | Currency code |
| Status | INT | Status of the applied promotion |
| AppliedDate | DATETIME | Date when promotion was applied |
| CompletedDate | DATETIME | Date when promotion was completed |
| Remark | NVARCHAR(500) | Additional remarks |
| WebId | INT | Website identifier |
| BonusAmount | DECIMAL(19,6) | Bonus amount received |
| TurnoverRequirement | DECIMAL(19,6) | Required turnover to complete promotion |
| CurrentTurnover | DECIMAL(19,6) | Current turnover progress |

### PromotionDailyCommissions
Tracks daily commissions from promotions.

```sql
CREATE TABLE [dbo].[PromotionDailyCommissions] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [Date] DATE NOT NULL,
    [CustomerId] INT NOT NULL,
    [Username] NVARCHAR(50) NOT NULL,
    [CommissionAmount] DECIMAL(19,6) NOT NULL,
    [Currency] CHAR(3) NOT NULL,
    [Status] INT NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [WebId] INT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the daily commission |
| Date | DATE | Date of the commission |
| CustomerId | INT | Customer identifier |
| Username | NVARCHAR(50) | Customer username |
| CommissionAmount | DECIMAL(19,6) | Commission amount earned |
| Currency | CHAR(3) | Currency code |
| Status | INT | Status of the commission |
| CreatedOn | DATETIME | Creation timestamp |
| WebId | INT | Website identifier |

### PromotionDetailForDeposit
Links deposit transactions with promotion details.

```sql
CREATE TABLE [dbo].[PromotionDetailForDeposit] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [CustomerId] INT NOT NULL,
    [TransactionRequestId] INT NOT NULL,
    [PromotionId] INT NOT NULL,
    [BonusAmount] DECIMAL(19,6) NOT NULL,
    [TurnoverRequirement] DECIMAL(19,6) NOT NULL,
    [Status] INT NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [WebId] INT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the promotion detail |
| CustomerId | INT | Customer identifier |
| TransactionRequestId | INT | Related transaction request ID |
| PromotionId | INT | Promotion identifier |
| BonusAmount | DECIMAL(19,6) | Bonus amount for this deposit |
| TurnoverRequirement | DECIMAL(19,6) | Turnover requirement for this bonus |
| Status | INT | Status of the promotion detail |
| CreatedOn | DATETIME | Creation timestamp |
| WebId | INT | Website identifier |

### PromotionWalletAutoTransfer
Manages automatic wallet transfers for promotions.

```sql
CREATE TABLE [dbo].[PromotionWalletAutoTransfer] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [CustomerId] INT NOT NULL,
    [IsEnabled] BIT NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [ModifiedOn] DATETIME NOT NULL,
    [WebId] INT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the auto transfer setting |
| CustomerId | INT | Customer identifier |
| IsEnabled | BIT | Whether auto transfer is enabled |
| CreatedOn | DATETIME | Creation timestamp |
| ModifiedOn | DATETIME | Last modification timestamp |
| WebId | INT | Website identifier |

### PlayerInjectBonus
Manages scheduled bonus injections for players.

```sql
CREATE TABLE [dbo].[PlayerInjectBonus] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [CustomerId] INT NOT NULL,
    [Amount] DECIMAL(19,6) NOT NULL,
    [Status] INT NOT NULL,
    [ScheduledDate] DATETIME NOT NULL,
    [WebId] INT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the inject bonus |
| CustomerId | INT | Customer identifier |
| Amount | DECIMAL(19,6) | Bonus amount to inject |
| Status | INT | Status of the bonus injection |
| ScheduledDate | DATETIME | Scheduled date for bonus injection |
| WebId | INT | Website identifier |

### ScheduledInjectBonusSet
Manages sets of scheduled bonus injections.

```sql
CREATE TABLE [dbo].[ScheduledInjectBonusSet] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [Name] NVARCHAR(100) NOT NULL,
    [Amount] DECIMAL(19,6) NOT NULL,
    [ScheduledDate] DATETIME NOT NULL,
    [Status] INT NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [WebId] INT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the bonus set |
| Name | NVARCHAR(100) | Name of the bonus set |
| Amount | DECIMAL(19,6) | Total bonus amount in the set |
| ScheduledDate | DATETIME | Scheduled execution date |
| Status | INT | Status of the bonus set |
| CreatedBy | NVARCHAR(50) | User who created the bonus set |
| CreatedOn | DATETIME | Creation timestamp |
| WebId | INT | Website identifier |

## Referral System Tables

### ReferralDetail
Tracks individual referral transactions and commissions.

```sql
CREATE TABLE [dbo].[ReferralDetail] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [ReferrerId] INT NOT NULL,
    [ReferredId] INT NOT NULL,
    [CommissionAmount] DECIMAL(19,6) NOT NULL,
    [Currency] CHAR(3) NOT NULL,
    [Date] DATE NOT NULL,
    [Status] INT NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [WebId] INT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the referral detail |
| ReferrerId | INT | Customer ID of the referrer |
| ReferredId | INT | Customer ID of the referred player |
| CommissionAmount | DECIMAL(19,6) | Commission amount earned |
| Currency | CHAR(3) | Currency code |
| Date | DATE | Date of the referral transaction |
| Status | INT | Status of the referral commission |
| CreatedOn | DATETIME | Creation timestamp |
| WebId | INT | Website identifier |

### ReferralSummary
Summarizes total referral activity for each customer.

```sql
CREATE TABLE [dbo].[ReferralSummary] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [CustomerId] INT NOT NULL,
    [TotalReferrals] INT NOT NULL,
    [TotalCommission] DECIMAL(19,6) NOT NULL,
    [Currency] CHAR(3) NOT NULL,
    [LastUpdated] DATETIME NOT NULL,
    [WebId] INT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the referral summary |
| CustomerId | INT | Customer identifier |
| TotalReferrals | INT | Total number of successful referrals |
| TotalCommission | DECIMAL(19,6) | Total commission earned from referrals |
| Currency | CHAR(3) | Currency code |
| LastUpdated | DATETIME | Last update timestamp |
| WebId | INT | Website identifier |

### RedeemableReferralAmount
Tracks redeemable amounts from referral commissions.

```sql
CREATE TABLE [dbo].[RedeemableReferralAmount] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [CustomerId] INT NOT NULL,
    [Amount] DECIMAL(19,6) NOT NULL,
    [Currency] CHAR(3) NOT NULL,
    [LastUpdated] DATETIME NOT NULL,
    [WebId] INT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the redeemable amount |
| CustomerId | INT | Customer identifier |
| Amount | DECIMAL(19,6) | Amount available for redemption |
| Currency | CHAR(3) | Currency code |
| LastUpdated | DATETIME | Last update timestamp |
| WebId | INT | Website identifier |

### StockReferralExpense
Tracks expenses related to stock referral programs.

```sql
CREATE TABLE [dbo].[StockReferralExpense] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [Date] DATE NOT NULL,
    [CustomerId] INT NOT NULL,
    [Amount] DECIMAL(19,6) NOT NULL,
    [Currency] CHAR(3) NOT NULL,
    [Status] INT NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [WebId] INT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the stock referral expense |
| Date | DATE | Date of the expense |
| CustomerId | INT | Customer identifier |
| Amount | DECIMAL(19,6) | Expense amount |
| Currency | CHAR(3) | Currency code |
| Status | INT | Status of the expense |
| CreatedOn | DATETIME | Creation timestamp |
| WebId | INT | Website identifier |

## Revenue & Stock Tables

### DailyStockRevenue
Tracks daily stock-based revenue for customers.

```sql
CREATE TABLE [dbo].[DailyStockRevenue] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [Date] DATE NOT NULL,
    [WebId] INT NOT NULL,
    [CustomerId] INT NOT NULL,
    [Revenue] DECIMAL(19,6) NOT NULL,
    [Currency] CHAR(3) NOT NULL,
    [Status] INT NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [ModifiedOn] DATETIME NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the daily stock revenue |
| Date | DATE | Date of the revenue |
| WebId | INT | Website identifier |
| CustomerId | INT | Customer identifier |
| Revenue | DECIMAL(19,6) | Revenue amount generated |
| Currency | CHAR(3) | Currency code |
| Status | INT | Status of the revenue record |
| CreatedOn | DATETIME | Creation timestamp |
| ModifiedOn | DATETIME | Last modification timestamp |

## Product Configuration Tables

### ProductSetting
Manages product settings and availability for each website.

```sql
CREATE TABLE [dbo].[ProductSetting] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [ProductId] INT NOT NULL,
    [ProductName] NVARCHAR(100) NOT NULL,
    [IsEnabled] BIT NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [ModifiedBy] NVARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the product setting |
| WebId | INT | Website identifier |
| ProductId | INT | Product identifier |
| ProductName | NVARCHAR(100) | Name of the product |
| IsEnabled | BIT | Whether the product is enabled |
| CreatedBy | NVARCHAR(50) | User who created the setting |
| CreatedOn | DATETIME | Creation timestamp |
| ModifiedBy | NVARCHAR(50) | User who last modified the setting |
| ModifiedOn | DATETIME | Last modification timestamp |

### ProductPercentagesDealWithPlatformCustomer
Manages percentage deals between products and platform customers.

```sql
CREATE TABLE [dbo].[ProductPercentagesDealWithPlatformCustomer] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [ProductId] INT NOT NULL,
    [PercentageRate] DECIMAL(5,4) NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the percentage deal |
| WebId | INT | Website identifier |
| ProductId | INT | Product identifier |
| PercentageRate | DECIMAL(5,4) | Percentage rate for the deal |
| CreatedBy | NVARCHAR(50) | User who created the deal |
| CreatedOn | DATETIME | Creation timestamp |

### MaxPTConfig
Configures maximum PT (Point/Token) limits for different roles.

```sql
CREATE TABLE [dbo].[MaxPTConfig] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [RoleId] INT NOT NULL,
    [MaxPT] DECIMAL(19,6) NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [ModifiedBy] NVARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the max PT config |
| WebId | INT | Website identifier |
| RoleId | INT | Role identifier |
| MaxPT | DECIMAL(19,6) | Maximum PT amount allowed |
| CreatedBy | NVARCHAR(50) | User who created the config |
| CreatedOn | DATETIME | Creation timestamp |
| ModifiedBy | NVARCHAR(50) | User who last modified the config |
| ModifiedOn | DATETIME | Last modification timestamp |

## Seamless Wallet Tables

### SeamlessWalletUrl
Stores URLs for seamless wallet integration with different products.

```sql
CREATE TABLE [dbo].[SeamlessWalletUrl] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [ProductId] INT NOT NULL,
    [Url] NVARCHAR(500) NOT NULL,
    [IsEnabled] BIT NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [ModifiedOn] DATETIME NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the seamless wallet URL |
| WebId | INT | Website identifier |
| ProductId | INT | Product identifier |
| Url | NVARCHAR(500) | URL for seamless wallet API |
| IsEnabled | BIT | Whether this URL is enabled |
| CreatedOn | DATETIME | Creation timestamp |
| ModifiedOn | DATETIME | Last modification timestamp |

### SeamlessWalletNoResendErrorCode
Defines error codes that should not trigger resend attempts in seamless wallet operations.

```sql
CREATE TABLE [dbo].[SeamlessWalletNoResendErrorCode] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [ProductId] INT NOT NULL,
    [ErrorCode] NVARCHAR(50) NOT NULL,
    [Description] NVARCHAR(200) NULL,
    [IsEnabled] BIT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the error code rule |
| ProductId | INT | Product identifier |
| ErrorCode | NVARCHAR(50) | Error code that should not trigger resend |
| Description | NVARCHAR(200) | Description of the error code |
| IsEnabled | BIT | Whether this rule is enabled |

## Statement & Betting Tables

### StatementGivenBetCount
Tracks betting activity and awards based on bet count thresholds.

```sql
CREATE TABLE [dbo].[StatementGivenBetCount] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [CustomerId] INT NOT NULL,
    [ProductId] INT NOT NULL,
    [BetCount] INT NOT NULL,
    [Amount] DECIMAL(19,6) NOT NULL,
    [Currency] CHAR(3) NOT NULL,
    [Date] DATE NOT NULL,
    [Status] INT NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [WebId] INT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the bet count award |
| CustomerId | INT | Customer identifier |
| ProductId | INT | Product identifier |
| BetCount | INT | Number of bets placed |
| Amount | DECIMAL(19,6) | Award amount given |
| Currency | CHAR(3) | Currency code |
| Date | DATE | Date of the award |
| Status | INT | Status of the award |
| CreatedOn | DATETIME | Creation timestamp |
| WebId | INT | Website identifier |

## Withdrawal Management Tables

### WithdrawalLimit
Manages withdrawal limits and requirements for customers.

```sql
CREATE TABLE [dbo].[WithdrawalLimit] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [CustomerId] INT NOT NULL,
    [WithdrawalLimit] DECIMAL(19,6) NOT NULL,
    [AvailableWithdrawAmount] DECIMAL(19,6) NOT NULL,
    [Winlost] DECIMAL(19,6) NOT NULL,
    [Status] NVARCHAR(100) NOT NULL,
    [ValidTurnover] DECIMAL(19,6) NOT NULL,
    [Type] NVARCHAR(100) NOT NULL,
    [EventId] NVARCHAR(100) NULL,
    [WithdrawalLimitRequirement] DECIMAL(19,6) NOT NULL,
    [TransactionAmount] DECIMAL(19,6) NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL,
    [ModifiedBy] NVARCHAR(50) NOT NULL,
    [Remark] NVARCHAR(255) NULL,
    [IsStatusCompleteByLosingBefore] BIT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the withdrawal limit |
| WebId | INT | Website identifier |
| CustomerId | INT | Customer identifier |
| WithdrawalLimit | DECIMAL(19,6) | Maximum withdrawal limit |
| AvailableWithdrawAmount | DECIMAL(19,6) | Currently available withdrawal amount |
| Winlost | DECIMAL(19,6) | Win/loss amount affecting withdrawal |
| Status | NVARCHAR(100) | Status of the withdrawal limit |
| ValidTurnover | DECIMAL(19,6) | Valid turnover for withdrawal eligibility |
| Type | NVARCHAR(100) | Type of withdrawal limit |
| EventId | NVARCHAR(100) | Related event identifier |
| WithdrawalLimitRequirement | DECIMAL(19,6) | Required amount for withdrawal eligibility |
| TransactionAmount | DECIMAL(19,6) | Transaction amount related to this limit |
| CreatedOn | DATETIME | Creation timestamp |
| CreatedBy | NVARCHAR(50) | User who created the limit |
| ModifiedOn | DATETIME | Last modification timestamp |
| ModifiedBy | NVARCHAR(50) | User who last modified the limit |
| Remark | NVARCHAR(255) | Additional remarks |
| IsStatusCompleteByLosingBefore | BIT | Whether status was completed by losing |

## Device & System Tables

### TvBoxDeviceInfo
Manages TV box device information and configurations.

```sql
CREATE TABLE [dbo].[TvBoxDeviceInfo] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [DeviceId] NVARCHAR(200) NOT NULL,
    [WebId] INT NOT NULL,
    [TargetWebId] INT NOT NULL,
    [TargetApk] NVARCHAR(1000) NOT NULL,
    [CheckDate] DATETIME NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL,
    [ModifiedBy] NVARCHAR(50) NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the TV box device |
| DeviceId | NVARCHAR(200) | Unique device identifier |
| WebId | INT | Website identifier |
| TargetWebId | INT | Target website for the device |
| TargetApk | NVARCHAR(1000) | Target APK information |
| CheckDate | DATETIME | Last check date |
| CreatedOn | DATETIME | Creation timestamp |
| CreatedBy | NVARCHAR(50) | User who created the device record |
| ModifiedOn | DATETIME | Last modification timestamp |
| ModifiedBy | NVARCHAR(50) | User who last modified the record |

### AppReleaseInfo
Manages application release information and update details.

```sql
CREATE TABLE [dbo].[AppReleaseInfo] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [Version] NVARCHAR(50) NOT NULL,
    [ReleaseNotes] NVARCHAR(MAX) NULL,
    [DownloadUrl] NVARCHAR(500) NOT NULL,
    [IsForceUpdate] BIT NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the app release |
| Version | NVARCHAR(50) | Version number of the release |
| ReleaseNotes | NVARCHAR(MAX) | Release notes and changelog |
| DownloadUrl | NVARCHAR(500) | URL for downloading the release |
| IsForceUpdate | BIT | Whether this is a forced update |
| CreatedOn | DATETIME | Creation timestamp |
| CreatedBy | NVARCHAR(50) | User who created the release record |

### SystemTimeZone
Manages time zone settings for different websites.

```sql
CREATE TABLE [dbo].[SystemTimeZone] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [TimeZone] NVARCHAR(100) NOT NULL,
    [UtcOffset] INT NOT NULL,
    [IsEnabled] BIT NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [ModifiedOn] DATETIME NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the time zone setting |
| WebId | INT | Website identifier |
| TimeZone | NVARCHAR(100) | Time zone identifier |
| UtcOffset | INT | UTC offset in hours |
| IsEnabled | BIT | Whether this time zone is enabled |
| CreatedOn | DATETIME | Creation timestamp |
| ModifiedOn | DATETIME | Last modification timestamp |

## Notification & Alert Tables

### TelegramFailAlert
Records failed Telegram alert attempts for monitoring.

```sql
CREATE TABLE [dbo].[TelegramFailAlert] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [Message] NVARCHAR(MAX) NOT NULL,
    [ErrorMessage] NVARCHAR(MAX) NULL,
    [CreatedOn] DATETIME NOT NULL,
    [Status] INT NOT NULL,
    [WebId] INT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the failed alert |
| Message | NVARCHAR(MAX) | Original message that failed to send |
| ErrorMessage | NVARCHAR(MAX) | Error message from Telegram API |
| CreatedOn | DATETIME | Creation timestamp |
| Status | INT | Status of the alert |
| WebId | INT | Website identifier |

### CasherNotificationToken
Stores notification tokens for casher applications.

```sql
CREATE TABLE [dbo].[CasherNotificationToken] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [Token] NVARCHAR(500) NOT NULL,
    [IsEnabled] BIT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the notification token |
| WebId | INT | Website identifier |
| Token | NVARCHAR(500) | Notification token |
| IsEnabled | BIT | Whether this token is enabled |

## Operator & Account Management Tables

### OperatorPaymentAccount
Manages payment account information for operators.

```sql
CREATE TABLE [dbo].[OperatorPaymentAccount] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [PaymentProviderId] INT NOT NULL,
    [AccountName] NVARCHAR(100) NOT NULL,
    [AccountNumber] NVARCHAR(100) NOT NULL,
    [ApiKey] NVARCHAR(500) NOT NULL,
    [SecretKey] NVARCHAR(500) NOT NULL,
    [IsEnabled] BIT NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [ModifiedBy] NVARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the operator payment account |
| WebId | INT | Website identifier |
| PaymentProviderId | INT | Payment provider identifier |
| AccountName | NVARCHAR(100) | Name of the payment account |
| AccountNumber | NVARCHAR(100) | Account number |
| ApiKey | NVARCHAR(500) | API key for payment provider |
| SecretKey | NVARCHAR(500) | Secret key for payment provider |
| IsEnabled | BIT | Whether the account is enabled |
| CreatedBy | NVARCHAR(50) | User who created the account |
| CreatedOn | DATETIME | Creation timestamp |
| ModifiedBy | NVARCHAR(50) | User who last modified the account |
| ModifiedOn | DATETIME | Last modification timestamp |

### ApiCustomerAccountInfo
Stores API access information for customer accounts.

```sql
CREATE TABLE [dbo].[ApiCustomerAccountInfo] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [CustomerId] INT NOT NULL,
    [Username] NVARCHAR(50) NOT NULL,
    [ApiKey] NVARCHAR(500) NOT NULL,
    [SecretKey] NVARCHAR(500) NOT NULL,
    [IsEnabled] BIT NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [ModifiedBy] NVARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL,
    [WebId] INT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the API customer account |
| CustomerId | INT | Customer identifier |
| Username | NVARCHAR(50) | Customer username |
| ApiKey | NVARCHAR(500) | API key for customer |
| SecretKey | NVARCHAR(500) | Secret key for customer |
| IsEnabled | BIT | Whether API access is enabled |
| CreatedBy | NVARCHAR(50) | User who created the API access |
| CreatedOn | DATETIME | Creation timestamp |
| ModifiedBy | NVARCHAR(50) | User who last modified the access |
| ModifiedOn | DATETIME | Last modification timestamp |
| WebId | INT | Website identifier |

### AffiliateOperatorAccount
Links affiliate operators with their accounts.

```sql
CREATE TABLE [dbo].[AffiliateOperatorAccount] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [OperatorId] INT NOT NULL,
    [IsEnabled] BIT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the affiliate operator account |
| WebId | INT | Website identifier |
| OperatorId | INT | Operator identifier |
| IsEnabled | BIT | Whether the affiliate account is enabled |

### AffiliateProvider
Master list of affiliate providers.

```sql
CREATE TABLE [dbo].[AffiliateProvider] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [Name] NVARCHAR(100) NOT NULL,
    [Code] NVARCHAR(50) NOT NULL,
    [IsEnabled] BIT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the affiliate provider |
| Name | NVARCHAR(100) | Name of the affiliate provider |
| Code | NVARCHAR(50) | Unique code for the provider |
| IsEnabled | BIT | Whether the provider is enabled |

### CompanyAffiliateProvider
Links companies with their affiliate providers.

```sql
CREATE TABLE [dbo].[CompanyAffiliateProvider] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [AffiliateProviderId] INT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the company affiliate provider link |
| WebId | INT | Website identifier |
| AffiliateProviderId | INT | Affiliate provider identifier |

## Banking Sync Tables

### BankSyncTransactionRecord
Records bank synchronization transactions for reconciliation.

```sql
CREATE TABLE [dbo].[BankSyncTransactionRecord] (
    [Id] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [BankAccountNumber] NVARCHAR(50) NOT NULL,
    [TransactionDate] DATETIME NOT NULL,
    [Amount] DECIMAL(19,6) NOT NULL,
    [Currency] CHAR(3) NOT NULL,
    [Description] NVARCHAR(500) NOT NULL,
    [Reference] NVARCHAR(100) NOT NULL,
    [Status] INT NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [ModifiedOn] DATETIME NOT NULL,
    [MatchedTransactionId] INT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the bank sync record |
| WebId | INT | Website identifier |
| BankAccountNumber | NVARCHAR(50) | Bank account number |
| TransactionDate | DATETIME | Date of the bank transaction |
| Amount | DECIMAL(19,6) | Transaction amount |
| Currency | CHAR(3) | Currency code |
| Description | NVARCHAR(500) | Transaction description from bank |
| Reference | NVARCHAR(100) | Bank reference number |
| Status | INT | Synchronization status |
| CreatedOn | DATETIME | Creation timestamp |
| ModifiedOn | DATETIME | Last modification timestamp |
| MatchedTransactionId | INT | ID of matched internal transaction |

## Customer Account Management Tables

### CustomerAccount
Manages customer account balances and status for each currency.

```sql
CREATE TABLE [dbo].[CustomerAccount] (
    [Id] INT IDENTITY(1,1) PRIMARY KEY,
    [CustomerId] INT NOT NULL,
    [Username] NVARCHAR(100) NOT NULL,
    [Currency] NVARCHAR(3) NOT NULL,
    [Balance] DECIMAL(19,6) NOT NULL DEFAULT 0,
    [FreezeBalance] DECIMAL(19,6) NOT NULL DEFAULT 0,
    [Status] NVARCHAR(50) NOT NULL DEFAULT 'Active',
    [CreatedOn] DATETIME NOT NULL,
    [CreatedBy] NVARCHAR(100) NOT NULL,
    [ModifiedOn] DATETIME,
    [ModifiedBy] NVARCHAR(100),
    [WebId] INT NOT NULL,
    CONSTRAINT CK_CustomerAccount_Status CHECK ([Status] IN ('Active', 'Suspended', 'Closed')),
    CONSTRAINT UQ_CustomerAccount_CustomerId_Currency UNIQUE ([CustomerId], [Currency])
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the customer account |
| CustomerId | INT | Customer identifier |
| Username | NVARCHAR(100) | Customer username |
| Currency | NVARCHAR(3) | Currency code for this account |
| Balance | DECIMAL(19,6) | Current available balance |
| FreezeBalance | DECIMAL(19,6) | Amount currently frozen/held |
| Status | NVARCHAR(50) | Account status (Active, Suspended, Closed) |
| CreatedOn | DATETIME | Account creation timestamp |
| CreatedBy | NVARCHAR(100) | User who created the account |
| ModifiedOn | DATETIME | Last modification timestamp |
| ModifiedBy | NVARCHAR(100) | User who last modified the account |
| WebId | INT | Website identifier |

### CustomerAccountTransaction
Records all transactions affecting customer account balances.

```sql
CREATE TABLE [dbo].[CustomerAccountTransaction] (
    [Id] INT IDENTITY(1,1) PRIMARY KEY,
    [CustomerId] INT NOT NULL,
    [Username] NVARCHAR(100) NOT NULL,
    [Currency] NVARCHAR(3) NOT NULL,
    [TransactionType] INT NOT NULL,
    [Amount] DECIMAL(19,6) NOT NULL,
    [Balance] DECIMAL(19,6) NOT NULL,
    [Reference] NVARCHAR(100),
    [Description] NVARCHAR(MAX),
    [CreatedOn] DATETIME NOT NULL,
    [CreatedBy] NVARCHAR(100) NOT NULL,
    [WebId] INT NOT NULL,
    CONSTRAINT CK_CustomerAccountTransaction_Amount CHECK ([Amount] != 0)
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the transaction |
| CustomerId | INT | Customer identifier |
| Username | NVARCHAR(100) | Customer username |
| Currency | NVARCHAR(3) | Currency code for the transaction |
| TransactionType | INT | Type of transaction (1=deposit, 2=withdrawal, 3=bet, 4=win, etc.) |
| Amount | DECIMAL(19,6) | Transaction amount (positive for credits, negative for debits) |
| Balance | DECIMAL(19,6) | Account balance after this transaction |
| Reference | NVARCHAR(100) | Reference number or external ID |
| Description | NVARCHAR(MAX) | Detailed description of the transaction |
| CreatedOn | DATETIME | Transaction timestamp |
| CreatedBy | NVARCHAR(100) | User or system that created the transaction |
| WebId | INT | Website identifier |

### CustomerAccountFreeze
Manages temporary holds or freezes on customer account funds.

```sql
CREATE TABLE [dbo].[CustomerAccountFreeze] (
    [Id] INT IDENTITY(1,1) PRIMARY KEY,
    [CustomerId] INT NOT NULL,
    [Username] NVARCHAR(100) NOT NULL,
    [Currency] NVARCHAR(3) NOT NULL,
    [Amount] DECIMAL(19,6) NOT NULL,
    [Reason] NVARCHAR(MAX),
    [Status] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [CreatedBy] NVARCHAR(100) NOT NULL,
    [ModifiedOn] DATETIME,
    [ModifiedBy] NVARCHAR(100),
    [WebId] INT NOT NULL,
    CONSTRAINT CK_CustomerAccountFreeze_Status CHECK ([Status] IN ('Active', 'Released', 'Cancelled'))
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the freeze record |
| CustomerId | INT | Customer identifier |
| Username | NVARCHAR(100) | Customer username |
| Currency | NVARCHAR(3) | Currency code for the frozen amount |
| Amount | DECIMAL(19,6) | Amount that is frozen |
| Reason | NVARCHAR(MAX) | Reason for freezing the funds |
| Status | NVARCHAR(50) | Status of the freeze (Active, Released, Cancelled) |
| CreatedOn | DATETIME | Freeze creation timestamp |
| CreatedBy | NVARCHAR(100) | User who created the freeze |
| ModifiedOn | DATETIME | Last modification timestamp |
| ModifiedBy | NVARCHAR(100) | User who last modified the freeze |
| WebId | INT | Website identifier |

## Extended Transaction Request Fields

### TransactionRequest Extended
Complete structure with all fields for the TransactionRequest table.

| Column | Type | Description |
|--------|------|-------------|
| ID | INT, PK | Unique identifier for the transaction request |
| TransactionType | INT | Type of transaction (1=deposit, 2=withdrawal, 3=transfer, etc.) |
| ParentId | INT | Parent transaction or related entity ID |
| CustomerId | INT | Customer/user identifier |
| Username | NVARCHAR(50) | Username of the customer |
| Amount | DECIMAL(19,6) | Transaction amount |
| Currency | CHAR(3) | Currency code for the transaction |
| ExchangeRate | DECIMAL(12,8) | Exchange rate applied to the transaction |
| USDTExchangeRate | DECIMAL(19,6) | USDT exchange rate (if applicable) |
| CompanyBankID | INT | Related company bank account ID |
| PlayerBankName | NVARCHAR(100) | Player's bank name |
| PlayerBankAccountNumber | VARCHAR(50) | Player's bank account number |
| PlayerBankAccountName | NVARCHAR(MAX) | Player's bank account name |
| TransactionStatus | INT | Status of the transaction (0=pending, 1=approved, 2=rejected, etc.) |
| BeforeBalance | DECIMAL(19,6) | Account balance before the transaction |
| AfterBalance | DECIMAL(19,6) | Account balance after the transaction |
| VerifiedAmount | DECIMAL(19,6) | Verified transaction amount |
| BankReference | VARCHAR(50) | Bank reference number |
| IP | VARCHAR(50) | IP address of the request |
| ModifiedBy | VARCHAR(100) | User who last modified the record |
| ModifiedOn | DATETIME | Last modification timestamp |
| CreatedBy | VARCHAR(100) | User who created the record |
| CreatedOn | DATETIME | Creation timestamp |
| Description | NVARCHAR(250) | Transaction description |
| Remark | NVARCHAR(250) | Administrative remarks |
| PlayerRemark | NVARCHAR(250) | Player's remarks |
| WebId | INT | Website identifier |
| DepositTime | DATETIME | Deposit timestamp |
| SlipImage | NVARCHAR(MAX) | Image of the deposit slip |
| BranchName | NVARCHAR(100) | Bank branch name |
| CustomizeBankName | NVARCHAR(100) | Custom bank name |
| CPFNumber | VARCHAR(20) | CPF number (for Brazil, etc.) |
| BSB | NVARCHAR(100) | Bank State Branch code (Australia) |
| PayId | NVARCHAR(100) | PayID (Australia) |
| IsArbitragePlayer | BIT | Whether the player is flagged for arbitrage |
| RFC | NVARCHAR(100) | RFC (Mexico tax ID) |
| CURP | NVARCHAR(100) | CURP (Mexico personal ID) |
| AvailableWithdrawAmount | DECIMAL(19,6) | Amount available for withdrawal |
| ExtraInternalParameter | NVARCHAR(MAX) | Extra internal parameters |

## Extended PlayerBankListAfterFirstTimeDeposit Fields

### PlayerBankListAfterFirstTimeDeposit Extended
Complete structure with all fields for the PlayerBankListAfterFirstTimeDeposit table.

| Column | Type | Description |
|--------|------|-------------|
| Id | BIGINT, PK | Unique identifier for the record |
| CustomerId | INT | Customer/user identifier |
| PlayerBankId | INT | Player's bank ID |
| PlayerBankAccountName | NVARCHAR(MAX) | Player's bank account name |
| PlayerBankAccountNumber | VARCHAR(50) | Player's bank account number |
| ModifiedBy | VARCHAR(50) | User who last modified the record |
| ModifiedOn | DATETIME | Last modification timestamp |
| Currency | CHAR(4) | Currency code |
| WebId | INT | Website/application identifier |
| ExtraCompanyFlowField | NVARCHAR(MAX) | Extra company-specific field |
| BranchName | NVARCHAR(100) | Bank branch name |

## Extended CompanyBanks Fields

### CompanyBanks Extended
Complete structure with all fields for the CompanyBanks table.

| Column | Type | Description |
|--------|------|-------------|
| ID | INT, PK | Unique identifier for the company bank account |
| AdminId | INT | Administrator who manages this bank account |
| DepositMin | DECIMAL(19,6) | Minimum deposit amount allowed |
| DepositMax | DECIMAL(19,6) | Maximum deposit amount allowed |
| WithdrawalMin | DECIMAL(19,6) | Minimum withdrawal amount allowed |
| WithdrawalMax | DECIMAL(19,6) | Maximum withdrawal amount allowed |
| Status | INT | Status of the bank account (1=active, 0=inactive) |
| CreatedBy | VARCHAR(50) | User who created the record |
| CreatedOn | DATETIME | Creation timestamp |
| ModifiedBy | VARCHAR(50) | User who last modified the record |
| ModifiedOn | DATETIME | Last modification timestamp |
| Currency | CHAR(3) | Currency supported by this bank account |
| Group | INT | Group identifier (default 1) |
| BankAccountNumber | VARCHAR(50) | Bank account number |
| BankAccountName | NVARCHAR(100) | Name on the bank account |
| BankName | NVARCHAR(100) | Name of the bank |
| WebID | INT | Website identifier this bank account belongs to |
| BankBalance | DECIMAL(19,6) | Current balance in the bank account |
| IsAutoDeposit | BIT | Whether automatic deposit processing is enabled |
| Identifier | NVARCHAR(100) | Unique identifier for the bank account |
| BankLogo | NVARCHAR(255) | Bank logo image URL |
| BankQrCode | NVARCHAR(255) | Bank QR code image URL |
| IsCryptoCurrencyTransfer | BIT | Whether the account is for crypto transfers |

## Extended DailyTransactionSummary Fields

### DailyTransactionSummary Extended
Complete structure with all fields for the DailyTransactionSummary table.

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the daily summary record |
| Date | DATE | Date of the transaction summary |
| WebId | INT | Website/application identifier |
| Currency | NVARCHAR(3) | Currency code for the summary |
| CustomerId | INT | Customer identifier |
| Username | NVARCHAR(50) | Username of the customer |
| ParentId | INT | Parent entity identifier in the hierarchy |
| DepositCount | INT | Number of deposit transactions |
| DepositAmount | DECIMAL(19,6) | Total deposit amount for the day |
| WithdrawalCount | INT | Number of withdrawal transactions |
| WithdrawalAmount | DECIMAL(19,6) | Total withdrawal amount for the day |
| PaymentDeposit | DECIMAL(19,6) | Total payment deposit amount |
| PaymentWithdrawal | DECIMAL(19,6) | Total payment withdrawal amount |
| BankTransferDeposit | DECIMAL(19,6) | Total bank transfer deposit amount |
| BankTransferWithdrawal | DECIMAL(19,6) | Total bank transfer withdrawal amount |
| ManualDeposit | DECIMAL(19,6) | Total manual deposit amount |
| ManualWithdrawal | DECIMAL(19,6) | Total manual withdrawal amount |
| ManualBonus | DECIMAL(19,6) | Total manual bonus amount |
| ManualRebate | DECIMAL(19,6) | Total manual rebate amount |
| InjectBonus | DECIMAL(19,6) | Total injected bonus amount |
| CreatedOn | DATETIME | Creation timestamp |
| CreatedBy | NVARCHAR(50) | User who created the record |
| ModifiedOn | DATETIME | Last modification timestamp |
| ModifiedBy | NVARCHAR(50) | User who last modified the record |

## Extended PlayerPaymentWithdrawalBankInfo Fields

### PlayerPaymentWithdrawalBankInfo Extended
Complete structure with all fields for the PlayerPaymentWithdrawalBankInfo table.

| Column | Type | Description |
|--------|------|-------------|
| WebId | INT | Website/application identifier |
| CustomerId | INT | Customer identifier |
| Username | VARCHAR(50) | Username of the customer |
| Currency | CHAR(3) | Currency for withdrawal |
| BankName | NVARCHAR(100) | Name of the player's bank |
| BankAccountName | NVARCHAR(MAX) | Name on the player's bank account |
| BankAccountNumber | VARCHAR(50) | Player's bank account number |
| ModifiedBy | VARCHAR(50) | User who last modified the record |
| ModifiedOn | DATETIME | Last modification timestamp |

## Extended FinancialReport Fields

### FinancialReport Extended
Complete structure with all fields for the FinancialReport table.

| Column | Type | Description |
|--------|------|-------------|
| YearMonth | CHAR(6), PK | Year and month in YYYYMM format |
| Customer | NVARCHAR(50), PK | Customer identifier |
| ProductId | INT, PK | Product identifier |
| Product | NVARCHAR(100), PK | Product name |
| AccountId | NVARCHAR(50), PK | Account identifier |
| DataType | VARCHAR(20), PK | Type of data (summary, detail, etc.) |
| Currency | CHAR(3), PK | Currency code |
| BetCount | INT | Total number of bets placed |
| ActiveUser | INT | Number of active users |
| Turnover | DECIMAL(19,6) | Total betting turnover amount |
| MemberGrWinLoss | DECIMAL(19,6) | Member gross win/loss amount |
| CustomerWinLoss | DECIMAL(19,6) | Customer win/loss amount |
| CustomerWinLossEuro | DECIMAL(19,6) | Customer win/loss in EUR |
| Rate | DECIMAL(19,6) | Exchange rate |
| MinRate | DECIMAL(19,6) | Minimum exchange rate (nullable) |
| CreatedOn | DATETIME | Creation timestamp |
| CreatedBy | NVARCHAR(50) | User who created the record |
| ModifiedOn | DATETIME | Last modification timestamp |
| ModifiedBy | NVARCHAR(50) | User who last modified the record |

## Key Features Summary

1. **Complete Transaction Management**
   - Comprehensive transaction lifecycle tracking across all tables
   - Multiple transaction types with detailed status management
   - Balance tracking with before/after amounts
   - Currency and exchange rate handling for multi-currency support

2. **Advanced Payment Gateway Integration**
   - Support for multiple payment providers with individual configurations
   - Comprehensive fee structure management
   - Bank integration with real-time balance monitoring
   - Cryptocurrency support with gas fee tracking
   - Error handling and retry logic management

3. **Sophisticated Banking Operations**
   - Company bank account management with automated processes
   - Player bank information tracking with first-time deposit handling
   - Auto-deposit capabilities with balance synchronization
   - Bank reconciliation through sync transaction records
   - Comprehensive deposit/withdrawal limits and controls

4. **Comprehensive Financial Reporting**
   - Daily transaction summaries with detailed breakdowns
   - Monthly financial reports by customer, product, and currency
   - Real-time win/loss tracking across all products
   - Stock revenue tracking and management
   - Multi-currency reporting with exchange rate handling

5. **Advanced Customer Account Management**
   - Multi-currency account support per customer
   - Balance and freeze balance tracking
   - Comprehensive transaction history
   - Account status management with business rules
   - Withdrawal limit management with turnover requirements

6. **Sophisticated Promotion & Bonus System**
   - Promotion request and approval workflows
   - Bonus injection management with scheduling
   - Commission tracking for various promotion types
   - Turnover requirement tracking and validation
   - Auto-transfer capabilities between wallets

7. **Comprehensive Security & Access Control**
   - IP whitelist/blacklist management by website
   - Multi-provider OTP verification system
   - API key management for customers and operators
   - Comprehensive audit trails with user tracking

8. **Advanced Game Provider Integration**
   - Transfer game provider daily statements
   - Product-wise statement aggregation
   - Seamless wallet integration with error handling
   - Statement reconciliation across multiple products
   - Commission and discount tracking

9. **Sophisticated Referral System**
   - Multi-level referral tracking
   - Commission calculation and distribution
   - Redeemable amount management
   - Stock referral expense tracking
   - Comprehensive referral reporting

10. **Multi-tenancy & Organization Support**
    - Website-based data segregation
    - Company and domain management
    - Hierarchical relationship tracking
    - Operator and affiliate management
    - Time zone support per website

## Database Relationships & Constraints

### Primary Relationships
- All tables maintain `WebId` for multi-tenant isolation
- Customer tracking through `CustomerId` across all customer-related tables
- Currency-based operations with proper exchange rate handling
- Parent-child relationships through `ParentId` fields
- Product-based segregation for different game types

### Constraints & Business Rules
- Account status constraints ensure valid states
- Transaction amount constraints prevent zero-value transactions
- Currency whitelist ensures only approved currencies are used
- Withdrawal limits enforce business rules and regulations
- Freeze balance constraints maintain account integrity

### Audit & Compliance
- All tables include creation and modification timestamps
- User tracking for all create/update operations
- Comprehensive transaction trails for financial auditing
- Status tracking for regulatory compliance
- Document and reference number tracking

### Performance Considerations
- Proper indexing on frequently queried columns
- Optimized table structures for reporting queries
- Partitioning considerations for large transaction tables
- Archival strategies for historical data
- Real-time balance calculation optimization

## Best Practices for Implementation

1. **Transaction Management**
   - Always use database transactions for financial operations
   - Implement proper locking mechanisms for balance updates
   - Maintain transaction atomicity across related tables
   - Regular reconciliation processes

2. **Security Implementation**
   - Encrypt sensitive data (API keys, account numbers)
   - Implement proper access controls
   - Regular security audits and monitoring
   - Secure communication channels

3. **Performance Optimization**
   - Regular index maintenance and optimization
   - Query performance monitoring
   - Proper connection pooling
   - Caching strategies for frequently accessed data

4. **Data Integrity**
   - Regular data validation and reconciliation
   - Backup and recovery procedures
   - Data archival and retention policies
   - Cross-table consistency checks

5. **Monitoring & Alerting**
   - Real-time transaction monitoring
   - Alert systems for failed transactions
   - Performance metrics tracking
   - Regulatory compliance monitoring
