# ApplicationSetting Database Schema

## Overview
The ApplicationSetting database manages application-wide configuration settings, scheduler configurations, customer settings, and web cluster management for the Coloris Framework. It provides centralized configuration management across different websites and environments with comprehensive override capabilities, language support, message queue configuration, security restrictions, company flow customization, geographic IP management, database configuration, and service monitoring.

## Core Tables

### SimpleSettings
Manages basic application configuration settings per website and environment with versioning support.

```sql
CREATE TABLE [dbo].[SimpleSettings] (
    [Website]        NVARCHAR (50)   NOT NULL,
    [IsUAT]          BIT             NOT NULL,
    [Type]           NVARCHAR (50)   NOT NULL,
    [Id]             NVARCHAR (100)  NOT NULL,
    [Value]          NVARCHAR (1000) NOT NULL,
    [Decription]     NVARCHAR (1000) CONSTRAINT [DF_SimpleSettings_Decription] DEFAULT ('') NOT NULL,
    [PossibleValues] NVARCHAR (1000) CONSTRAINT [DF_SimpleSettings_PossibleValues] DEFAULT ('') NOT NULL,
    [Responsible]    NVARCHAR (50)   NOT NULL,
    [ModifiedBy]     NVARCHAR (50)   NOT NULL,
    [ModifiedOn]     DATETIME        CONSTRAINT [DF_SimpleSettings_ModifiedOn] DEFAULT (getdate()) NOT NULL,
    [IsEnabled]      BIT             CONSTRAINT [DF_SimpleSettings_IsEnabled] DEFAULT ((1)) NOT NULL,
    CONSTRAINT [PK_SimpleSettings] PRIMARY KEY CLUSTERED ([Website] ASC, [IsUAT] ASC, [Type] ASC, [Id] ASC)
)
```

| Column | Type | Description |
|--------|------|-------------|
| Website | NVARCHAR(50), PK | Website identifier for multi-tenant configuration |
| IsUAT | BIT, PK | Flag indicating UAT environment (part of composite key) |
| Type | NVARCHAR(50), PK | Setting category or type classification |
| Id | NVARCHAR(100), PK | Unique setting identifier within the type |
| Value | NVARCHAR(1000) | The actual configuration value |
| Decription | NVARCHAR(1000) | Description of the setting's purpose and usage |
| PossibleValues | NVARCHAR(1000) | Comma-separated list of valid values for the setting |
| Responsible | NVARCHAR(50) | Person or team responsible for this setting |
| ModifiedBy | NVARCHAR(50) | User who last modified the setting |
| ModifiedOn | DATETIME | Timestamp of last modification (defaults to current time) |
| IsEnabled | BIT | Flag indicating if the setting is currently active |

### SimpleSettingsOverride
Provides override capabilities for SimpleSettings to customize specific configurations without modifying base settings.

```sql
CREATE TABLE [dbo].[SimpleSettingsOverride] (
    [Website]        VARCHAR (50)   NOT NULL,
    [IsUAT]          BIT             NOT NULL,
    [Type]           VARCHAR (50)   NOT NULL,
    [Id]             VARCHAR (100)  NOT NULL,
    [Value]          VARCHAR (1000) NOT NULL,
    [Decription]     VARCHAR (1000) CONSTRAINT [DF_SimpleSettingsOverride_Decription] DEFAULT ('') NOT NULL,
    [PossibleValues] VARCHAR (1000) CONSTRAINT [DF_SimpleSettingsOverride_PossibleValues] DEFAULT ('') NOT NULL,
    [Responsible]    VARCHAR (50)   NOT NULL,
    [ModifiedBy]     VARCHAR (50)   NOT NULL,
    [ModifiedOn]     DATETIME        CONSTRAINT [DF_SimpleSettingsOverride_ModifiedOn] DEFAULT (getdate()) NOT NULL,
    [IsEnabled]      BIT             CONSTRAINT [DF_SimpleSettingsOverride_IsEnabled] DEFAULT ((1)) NOT NULL,
    CONSTRAINT [PK_SimpleSettingsOverride] PRIMARY KEY CLUSTERED ([Website] ASC, [IsUAT] ASC, [Type] ASC, [Id] ASC)
)
```

| Column | Type | Description |
|--------|------|-------------|
| Website | VARCHAR(50), PK | Website identifier for the override |
| IsUAT | BIT, PK | Environment flag (UAT vs Production) |
| Type | VARCHAR(50), PK | Setting type matching the base SimpleSettings |
| Id | VARCHAR(100), PK | Setting identifier matching the base SimpleSettings |
| Value | VARCHAR(1000) | Override value that takes precedence over base setting |
| Decription | VARCHAR(1000) | Description of the override purpose |
| PossibleValues | VARCHAR(1000) | Valid values for the override |
| Responsible | VARCHAR(50) | Person responsible for the override |
| ModifiedBy | VARCHAR(50) | User who last modified the override |
| ModifiedOn | DATETIME | Timestamp of last modification |
| IsEnabled | BIT | Flag indicating if the override is active |

### SimpleSettingTypes
Defines available setting type categories for use in SimpleSettings and related tables.

```sql
CREATE TABLE [dbo].[SimpleSettingTypes] (
    [Type] NVARCHAR (50) NOT NULL,
    CONSTRAINT [PK_SimpleSettingGroupNames] PRIMARY KEY CLUSTERED ([Type] ASC)
)
```

| Column | Type | Description |
|--------|------|-------------|
| Type | NVARCHAR(50), PK | Setting type category identifier (e.g., 'Database', 'Email', 'Security') |

### CustomerSettings
Manages customer-specific configuration settings that can be modified by customers themselves.

```sql
CREATE TABLE [dbo].[CustomerSettings] (
    [Id]             NVARCHAR (100)  NOT NULL,
    [Value]          NVARCHAR (MAX) NOT NULL,
    [Description]     NVARCHAR (1000) CONSTRAINT [DF_CustomerSettings_Description] DEFAULT ('') NOT NULL,
    [PossibleValues] NVARCHAR (1000) CONSTRAINT [DF_CustomerSettings_PossibleValues] DEFAULT ('') NOT NULL,
    [Responsible]    NVARCHAR (50)   NOT NULL,
    [ModifiedBy]     NVARCHAR (50)   NOT NULL,
    [ModifiedOn]     DATETIME        CONSTRAINT [DF_CustomerSettings_ModifiedOn] DEFAULT (getdate()) NOT NULL,
    [IsEnabled]      BIT             CONSTRAINT [DF_CustomerSettings_IsEnabled] DEFAULT ((1)) NOT NULL,
    [IsAbleToChangeByCustomer] BIT   CONSTRAINT [DF_CustomerSettings_IsAbleToChangeByCustomer] DEFAULT ((0)) NOT NULL,
    CONSTRAINT [PK_CustomerSettings] PRIMARY KEY CLUSTERED ([Id] ASC)
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | NVARCHAR(100), PK | Unique identifier for the customer setting |
| Value | NVARCHAR(MAX) | The setting value (supports large text values) |
| Description | NVARCHAR(1000) | Description of the setting's purpose |
| PossibleValues | NVARCHAR(1000) | Valid values or format description |
| Responsible | NVARCHAR(50) | Team or person responsible for this setting |
| ModifiedBy | NVARCHAR(50) | User who last modified the setting |
| ModifiedOn | DATETIME | Timestamp of last modification |
| IsEnabled | BIT | Flag indicating if the setting is active |
| IsAbleToChangeByCustomer | BIT | Flag allowing customers to modify this setting themselves |

### CustomerSettingsOverride
Provides website-specific overrides for customer settings across different clients.

```sql
CREATE TABLE [dbo].[CustomerSettingsOverride] (
    [WebId]          INT			 NOT NULL,
    [Id]             NVARCHAR (100)  NOT NULL,
    [Value]          NVARCHAR (MAX) NOT NULL,
    [Description]     NVARCHAR (1000) CONSTRAINT [DF_CustomerSettingsOverride_Description] DEFAULT ('') NOT NULL,
    [PossibleValues] NVARCHAR (1000) CONSTRAINT [DF_CustomerSettingsOverride_PossibleValues] DEFAULT ('') NOT NULL,
    [Responsible]    NVARCHAR (50)   NOT NULL,
    [ModifiedBy]     NVARCHAR (50)   NOT NULL,
    [ModifiedOn]     DATETIME        CONSTRAINT [DF_CustomerSettingsOverride_ModifiedOn] DEFAULT (getdate()) NOT NULL,
    [IsEnabled]      BIT             CONSTRAINT [DF_CustomerSettingsOverride_IsEnabled] DEFAULT ((1)) NOT NULL,
    [IsAbleToChangeByCustomer] BIT   CONSTRAINT [DF_CustomerSettingsOverride_IsAbleToChangeByCustomer] DEFAULT ((0)) NOT NULL
    CONSTRAINT [PK_CustomerSettingsOverride] PRIMARY KEY CLUSTERED ([WebId] ASC,[Id] ASC)
)
```

| Column | Type | Description |
|--------|------|-------------|
| WebId | INT, PK | Website identifier for customer-specific override |
| Id | NVARCHAR(100), PK | Customer setting identifier to override |
| Value | NVARCHAR(MAX) | Override value for the specific website |
| Description | NVARCHAR(1000) | Description of the override purpose |
| PossibleValues | NVARCHAR(1000) | Valid values for the override |
| Responsible | NVARCHAR(50) | Person responsible for the override |
| ModifiedBy | NVARCHAR(50) | User who last modified the override |
| ModifiedOn | DATETIME | Timestamp of last modification |
| IsEnabled | BIT | Flag indicating if the override is active |
| IsAbleToChangeByCustomer | BIT | Flag allowing customers to modify this override |

### SchedulerSettings
Configures scheduled job execution parameters for automated system tasks.

```sql
CREATE TABLE [dbo].[SchedulerSettings] (
    [Website]          NVARCHAR (50)   NOT NULL,
    [IsUAT]            BIT             NOT NULL,
    [Id]               NVARCHAR (100)  NOT NULL,
    [ClassName]        NVARCHAR (500)  NOT NULL,
    [Interval]         INT             NOT NULL,
    [Args]             NVARCHAR (500)  CONSTRAINT [DF_SchedulerSettings_Args] DEFAULT ('') NOT NULL,
    [DuplicationLevel] INT             NOT NULL,
    [Dependencies]     NVARCHAR (2000) CONSTRAINT [DF_SchedulerSettings_Dependencies] DEFAULT ('') NOT NULL,
    [IsLazy]           BIT             CONSTRAINT [DF_SchedulerSettings_IsLazy] DEFAULT ((0)) NOT NULL,
    [Description]      NVARCHAR (1000) NOT NULL,
    [ModifiedBy]       NVARCHAR (50)   NOT NULL,
    [ModifiedOn]       DATETIME        CONSTRAINT [DF_SchedulerSettings_ModifiedOn] DEFAULT (getdate()) NOT NULL,
    [IsEnabled]        BIT             CONSTRAINT [DF_SchedulerSettings_IsEnabled] DEFAULT ((1)) NOT NULL,
    [IsUnique]         INT             DEFAULT ((0)) NOT NULL,
    [IsDailyJob]       BIT             CONSTRAINT [DF_SchedulerSettings_IsDailyJob] DEFAULT ((0)) NOT NULL,
    [DailyJobTime]     TIME            CONSTRAINT [DF_SchedulerSettings_DailyJobTime] DEFAULT ('00:00:00') NOT NULL,
    CONSTRAINT [PK_SchedulerSettings] PRIMARY KEY CLUSTERED ([Website] ASC, [IsUAT] ASC, [Id] ASC)
)
```

| Column | Type | Description |
|--------|------|-------------|
| Website | NVARCHAR(50), PK | Website identifier for the scheduled job |
| IsUAT | BIT, PK | Environment flag (UAT vs Production) |
| Id | NVARCHAR(100), PK | Unique identifier for the scheduled job |
| ClassName | NVARCHAR(500) | Fully qualified class name of the job to execute |
| Interval | INT | Execution interval in seconds |
| Args | NVARCHAR(500) | Arguments to pass to the job execution |
| DuplicationLevel | INT | Number of concurrent instances allowed |
| Dependencies | NVARCHAR(2000) | Comma-separated list of job dependencies |
| IsLazy | BIT | Flag indicating if job should wait for optimal execution time |
| Description | NVARCHAR(1000) | Human-readable description of the job's purpose |
| ModifiedBy | NVARCHAR(50) | User who last modified the job configuration |
| ModifiedOn | DATETIME | Timestamp of last modification |
| IsEnabled | BIT | Flag indicating if the job is currently active |
| IsUnique | INT | Flag ensuring only one instance can run at a time |
| IsDailyJob | BIT | Flag indicating if this is a daily recurring job |
| DailyJobTime | TIME | Specific time for daily job execution |

### SchedulerSettingsOverride
Provides environment-specific overrides for scheduler configurations without modifying base settings.

```sql
CREATE TABLE [dbo].[SchedulerSettingsOverride] (
    [Website]          NVARCHAR (50)   NOT NULL,
    [IsUAT]            BIT             NOT NULL,
    [Id]               NVARCHAR (100)  NOT NULL,
    [ClassName]        NVARCHAR (500)  NOT NULL,
    [Interval]         INT             NOT NULL,
    [Args]             NVARCHAR (500)  CONSTRAINT [DF_SchedulerSettingsOverride_Args] DEFAULT ('') NOT NULL,
    [DuplicationLevel] INT             NOT NULL,
    [Dependencies]     NVARCHAR (2000) CONSTRAINT [DF_SchedulerSettingsOverride_Dependencies] DEFAULT ('') NOT NULL,
    [IsLazy]           BIT             CONSTRAINT [DF_SchedulerSettingsOverride_IsLazy] DEFAULT ((0)) NOT NULL,
    [Description]      NVARCHAR (1000) NOT NULL,
    [ModifiedBy]       NVARCHAR (50)   NOT NULL,
    [ModifiedOn]       DATETIME        CONSTRAINT [DF_SchedulerSettingsOverride_ModifiedOn] DEFAULT (getdate()) NOT NULL,
    [IsEnabled]        BIT             CONSTRAINT [DF_SchedulerSettingsOverride_IsEnabled] DEFAULT ((1)) NOT NULL,
    [IsUnique]         INT             DEFAULT ((0)) NOT NULL,
    [IsDailyJob]     BIT             CONSTRAINT [DF_SchedulerSettingsOverride_IsDailyJob] DEFAULT ((0)) NOT NULL,
    [DailyJobTime]   TIME            CONSTRAINT [DF_SchedulerSettingsOverride_DailyJobTime] DEFAULT ('00:00:00') NOT NULL,
    CONSTRAINT [PK_SchedulerSettingsOverride] PRIMARY KEY CLUSTERED ([Website] ASC, [IsUAT] ASC, [Id] ASC)
)
```

| Column | Type | Description |
|--------|------|-------------|
| Website | NVARCHAR(50), PK | Website identifier for the scheduler override |
| IsUAT | BIT, PK | Environment flag (UAT vs Production) |
| Id | NVARCHAR(100), PK | Scheduler job identifier to override |
| ClassName | NVARCHAR(500) | Override class name for job execution |
| Interval | INT | Override execution interval in seconds |
| Args | NVARCHAR(500) | Override arguments for job execution |
| DuplicationLevel | INT | Override concurrent instance limit |
| Dependencies | NVARCHAR(2000) | Override job dependency list |
| IsLazy | BIT | Override lazy execution flag |
| Description | NVARCHAR(1000) | Description of the override purpose |
| ModifiedBy | NVARCHAR(50) | User who last modified the override |
| ModifiedOn | DATETIME | Timestamp of last modification |
| IsEnabled | BIT | Flag indicating if the override is active |
| IsUnique | INT | Override unique execution flag |
| IsDailyJob | BIT | Override daily job flag |
| DailyJobTime | TIME | Override daily job execution time |

## Web Cluster Management

### WebClusters
Manages load balancing and cluster configuration for web applications across different environments.

```sql
CREATE TABLE [dbo].[WebClusters] (
    [Website]           NVARCHAR (50)   NOT NULL,
    [IsUAT]             BIT             NOT NULL,
    [SubdomainSelected] INT             NOT NULL,
    [MinimumLoad]       INT             NOT NULL,
    [BalanceRange]      INT             NOT NULL,
    [Description]       NVARCHAR (1000) CONSTRAINT [DF_WebClusters_Description] DEFAULT ('') NOT NULL,
    [ModifiedBy]        NVARCHAR (50)   NOT NULL,
    [ModifiedOn]        DATETIME        CONSTRAINT [DF_WebClusters_ModifiedOn] DEFAULT (getdate()) NOT NULL,
    [IsEnabled]         BIT             CONSTRAINT [DF_WebClusterSettings_IsEnabled] DEFAULT ((1)) NOT NULL,
    CONSTRAINT [PK_WebClusterSettings] PRIMARY KEY CLUSTERED ([Website] ASC, [IsUAT] ASC)
)
```

| Column | Type | Description |
|--------|------|-------------|
| Website | NVARCHAR(50), PK | Website identifier for cluster configuration |
| IsUAT | BIT, PK | Environment flag (UAT vs Production) |
| SubdomainSelected | INT | Selected subdomain configuration ID |
| MinimumLoad | INT | Minimum load threshold for load balancing |
| BalanceRange | INT | Load balancing range configuration |
| Description | NVARCHAR(1000) | Description of the cluster configuration |
| ModifiedBy | NVARCHAR(50) | User who last modified the cluster settings |
| ModifiedOn | DATETIME | Timestamp of last modification |
| IsEnabled | BIT | Flag indicating if the cluster is currently active |

### WebClusterNodes
Manages individual server nodes within web clusters with subdomain routing capabilities.

```sql
CREATE TABLE [dbo].[WebClusterNodes] (
    [Website]     NVARCHAR (50)   NOT NULL,
    [IsUAT]       BIT             NOT NULL,
    [HostName]    NVARCHAR (50)   NOT NULL,
    [ServerMode]  CHAR (1)        NOT NULL,
    [ServerGroup] VARCHAR (20)    CONSTRAINT [DF_WebClusterNodes_ServerGroup] DEFAULT ('') NOT NULL,
    [Subdomain0]  NVARCHAR (50)   CONSTRAINT [DF__WebCluste__Subdo__22751F6C] DEFAULT ('') NOT NULL,
    [Subdomain1]  NVARCHAR (50)   NULL,
    [Subdomain2]  NVARCHAR (50)   NULL,
    [Subdomain3]  NVARCHAR (50)   NULL,
    [Subdomain4]  NVARCHAR (50)   NULL,
    [Subdomain5]  NVARCHAR (50)   NULL,
    [Subdomain6]  NVARCHAR (50)   NULL,
    [Subdomain7]  NVARCHAR (50)   NULL,
    [Subdomain8]  NVARCHAR (50)   NULL,
    [Subdomain9]  NVARCHAR (50)   NULL,
    [Subdomain10] NVARCHAR (50)   NULL,
    [Subdomain11] NVARCHAR (50)   NULL,
    [Subdomain12] NVARCHAR (50)   NULL,
    [Subdomain13] NVARCHAR (50)   NULL,
    [Subdomain14] NVARCHAR (50)   NULL,
    [Description] NVARCHAR (1000) CONSTRAINT [DF_WebClusterNodes_Description] DEFAULT ('') NOT NULL,
    [ModifiedBy]  NVARCHAR (50)   NOT NULL,
    [ModifiedOn]  DATETIME        CONSTRAINT [DF_WebClusterNodes_ModifiedOn] DEFAULT (getdate()) NOT NULL,
    [IsEnabled]   BIT             CONSTRAINT [DF__WebCluste__IsEna__2DE6D218] DEFAULT ((1)) NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Website | NVARCHAR(50), PK | Website identifier for the cluster node |
| IsUAT | BIT, PK | Environment flag (UAT vs Production) |
| HostName | NVARCHAR(50), PK | Server hostname or IP address |
| ServerMode | CHAR(1) | Server operation mode (e.g., 'A' for Active, 'S' for Standby) |
| ServerGroup | VARCHAR(20) | Logical grouping of servers for management |
| Subdomain0-14 | NVARCHAR(50) | Up to 15 subdomain routing configurations per node |
| Description | NVARCHAR(1000) | Description of the server node |
| ModifiedBy | NVARCHAR(50) | User who last modified the node configuration |
| ModifiedOn | DATETIME | Timestamp of last modification |
| IsEnabled | BIT | Flag indicating if the node is currently active |

### WebClusterModes
Defines available server operation modes for web cluster nodes.

```sql
CREATE TABLE [dbo].[WebClusterModes] (
    [Mode]       CHAR (1)       NOT NULL,
    [Desription] NVARCHAR (100) NOT NULL,
    CONSTRAINT [PK_WebClusterModes] PRIMARY KEY CLUSTERED ([Mode] ASC)
)
```

| Column | Type | Description |
|--------|------|-------------|
| Mode | CHAR(1), PK | Single character mode identifier (e.g., 'A' for Active, 'S' for Standby) |
| Desription | NVARCHAR(100) | Human-readable description of the server mode |

## Messaging & Communication

### MsmqSettings
Configures Microsoft Message Queuing (MSMQ) settings for inter-service communication.

```sql
CREATE TABLE [dbo].[MsmqSettings] (
    [Website]    NVARCHAR (50)   NOT NULL,
    [IsUAT]      BIT             NOT NULL,
    [Key]        NVARCHAR (20)   NOT NULL,
    [QueueName]  NVARCHAR (50)   NOT NULL,
    [DNS]        NVARCHAR (50)   NOT NULL,
    [IsPrivate]  BIT             NOT NULL,
    [Type]       NVARCHAR (20)   NOT NULL,
    [Interval]   INT             NOT NULL,
    [Aliases]    NVARCHAR (20)   NOT NULL,
    [Decription] NVARCHAR (1000) NOT NULL,
    [ModifiedBy] NVARCHAR (50)   NOT NULL,
    [ModifiedOn] DATETIME        CONSTRAINT [DF_MsmqSettings_ModifiedOn] DEFAULT (getdate()) NOT NULL,
    [IsEnabled]  BIT             CONSTRAINT [DF_MsmqSettings_IsEnabled] DEFAULT ((1)) NOT NULL,
    CONSTRAINT [PK_MsmqSettings] PRIMARY KEY CLUSTERED ([Website] ASC, [IsUAT] ASC, [Key] ASC)
)
```

| Column | Type | Description |
|--------|------|-------------|
| Website | NVARCHAR(50), PK | Website identifier for the queue configuration |
| IsUAT | BIT, PK | Environment flag (UAT vs Production) |
| Key | NVARCHAR(20), PK | Unique key identifier for the queue setting |
| QueueName | NVARCHAR(50) | Name of the queue in the MSMQ server |
| DNS | NVARCHAR(50) | DNS address of the MSMQ server |
| IsPrivate | BIT | Flag indicating if this is a private queue (Y) or public queue (N) |
| Type | NVARCHAR(20) | Queue type: local, workgroup, or remote |
| Interval | INT | Interval for scheduler processing this queue |
| Aliases | NVARCHAR(20) | Alternative names or aliases for the queue |
| Decription | NVARCHAR(1000) | Description of the queue's purpose |
| ModifiedBy | NVARCHAR(50) | User who last modified the queue settings |
| ModifiedOn | DATETIME | Timestamp of last modification |
| IsEnabled | BIT | Flag indicating if the queue is currently active |

## Internationalization

### Language
Manages supported languages and localization settings for the application.

```sql
CREATE TABLE [dbo].[Language] (
    [Id]               int Identity(1,1),
    [LanguageName]     Nvarchar(100),
    [LanguageLocalName]     Nvarchar(100),
    [LanguageCode]     Nvarchar(100),
    [IsEnable]         BIT,
    [Description]      NVARCHAR(MAX),
    [ModifiedBy]       NVARCHAR (50)   NOT NULL,
    [ModifiedOn]       DATETIME        CONSTRAINT [DF_Language_ModifiedOn] DEFAULT (GETDATE()) NOT NULL,
    CONSTRAINT [PK_Language] PRIMARY KEY CLUSTERED ([Id] ASC)
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Primary key, auto-incrementing language identifier |
| LanguageName | NVARCHAR(100) | English name of the language |
| LanguageLocalName | NVARCHAR(100) | Native name of the language |
| LanguageCode | NVARCHAR(100) | ISO language code (e.g., 'en-US', 'zh-CN') |
| IsEnable | BIT | Flag indicating if the language is currently enabled |
| Description | NVARCHAR(MAX) | Additional description or notes about the language |
| ModifiedBy | NVARCHAR(50) | User who last modified the language settings |
| ModifiedOn | DATETIME | Timestamp of last modification |

### LanguageOverride
Provides website-specific language configuration overrides for multi-tenant environments.

```sql
CREATE TABLE [dbo].[LanguageOverride] (
    [Id]               int Identity(1,1),
    [WebId]            int NOT NULL,
    [LanguageName]     Nvarchar(100),
    [LanguageLocalName]     Nvarchar(100),
    [LanguageCode]     Nvarchar(100),
    [IsEnable]         BIT,
    [Description]      NVARCHAR(MAX),
    [ModifiedBy]       NVARCHAR (50)   NOT NULL,
    [ModifiedOn]       DATETIME        CONSTRAINT [DF_LanguageOverride_ModifiedOn] DEFAULT (GETDATE()) NOT NULL,
    CONSTRAINT [PK_LanguageOverride] PRIMARY KEY CLUSTERED ([Id] ASC)
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Primary key, auto-incrementing language override identifier |
| WebId | INT | Website identifier for website-specific language configuration |
| LanguageName | NVARCHAR(100) | English name of the language for this website |
| LanguageLocalName | NVARCHAR(100) | Native name of the language |
| LanguageCode | NVARCHAR(100) | ISO language code override for the website |
| IsEnable | BIT | Flag indicating if this language override is enabled |
| Description | NVARCHAR(MAX) | Additional description or notes about the language override |
| ModifiedBy | NVARCHAR(50) | User who last modified the language override |
| ModifiedOn | DATETIME | Timestamp of last modification |

## Security Management

### AdminIPGroup
Manages IP address groups for centralized IP-based access control configuration.

```sql
CREATE TABLE [dbo].[AdminIPGroup] (
    [IPGroupId]  INT            IDENTITY (1, 1) NOT NULL,
    [GroupName]  NVARCHAR (50)  NOT NULL,
    [IPList]     VARCHAR (500)  NOT NULL,
    [Remark]     NVARCHAR (128) NULL,
    [CreatedBy]  VARCHAR (50)   NOT NULL,
    [CreatedOn]  DATETIME       NULL,
    [ModifiedBy] VARCHAR (50)   NOT NULL,
    [ModifiedOn] DATETIME       NULL,
    CONSTRAINT [PK_AdminIPGroup] PRIMARY KEY CLUSTERED ([IPGroupId] ASC)
)
```

| Column | Type | Description |
|--------|------|-------------|
| IPGroupId | INT, PK | Primary key, auto-incrementing IP group identifier |
| GroupName | NVARCHAR(50) | Name of the IP group for identification |
| IPList | VARCHAR(500) | Comma-separated list of IP addresses or IP ranges |
| Remark | NVARCHAR(128) | Optional remarks or description for the IP group |
| CreatedBy | VARCHAR(50) | User who created the IP group |
| CreatedOn | DATETIME | Timestamp when the IP group was created |
| ModifiedBy | VARCHAR(50) | User who last modified the IP group |
| ModifiedOn | DATETIME | Timestamp of last modification |

### AdminIPRestriction
Manages IP-based access restrictions for administrative pages and functions.

```sql
CREATE TABLE [dbo].[AdminIPRestriction] (
    [ID]            INT          IDENTITY (1, 1) NOT NULL,
    [SiteKey]       VARCHAR (50) NOT NULL,
    [PageKey]       VARCHAR (50) NOT NULL,
    [IPGroupID]     INT          NOT NULL,
    [IsProdEnabled] BIT          NOT NULL,
    [IsUATEnabled]  BIT          NOT NULL,
    [CreatedBy]     VARCHAR (50) NOT NULL,
    [CreatedOn]     DATETIME     NULL,
    [ModifiedBy]    VARCHAR (50) NOT NULL,
    [ModifiedOn]    DATETIME     NULL,
    CONSTRAINT [PK_AdminIPRestriction] PRIMARY KEY CLUSTERED ([ID] ASC)
)
```

| Column | Type | Description |
|--------|------|-------------|
| ID | INT, PK | Primary key, auto-incrementing restriction identifier |
| SiteKey | VARCHAR(50) | Website or site identifier for the restriction |
| PageKey | VARCHAR(50) | Specific page or function key being restricted |
| IPGroupID | INT | Reference to IP group containing allowed IP addresses |
| IsProdEnabled | BIT | Flag indicating if restriction is active in production |
| IsUATEnabled | BIT | Flag indicating if restriction is active in UAT |
| CreatedBy | VARCHAR(50) | User who created the restriction |
| CreatedOn | DATETIME | Timestamp when the restriction was created |
| ModifiedBy | VARCHAR(50) | User who last modified the restriction |
| ModifiedOn | DATETIME | Timestamp of last modification |

## Service Management

### ServerUM
Manages server uptime monitoring and service provider status tracking.

```sql
CREATE TABLE [dbo].[ServerUM] (
    [ID]                INT           IDENTITY (1, 1) NOT NULL,
    [ServiceProviderID] INT           NOT NULL,
    [ServiceType]       INT           NOT NULL,
    [Status]            INT           CONSTRAINT [DF__ServerUM__Status__0E6E26BF] DEFAULT ((0)) NOT NULL,
    [StartTime]         DATETIME      NOT NULL,
    [EndTime]           DATETIME      NOT NULL,
    [ModifiedBy]        NVARCHAR (50) NOT NULL,
    [ModifiedOn]        DATETIME      NOT NULL,
    [isUAT]             BIT           CONSTRAINT [DF_ServerUM_isUAT] DEFAULT ((0)) NOT NULL,
    CONSTRAINT [PK_ServerUM] PRIMARY KEY CLUSTERED ([ID] ASC)
)
```

| Column | Type | Description |
|--------|------|-------------|
| ID | INT, PK | Primary key, auto-incrementing server monitoring record identifier |
| ServiceProviderID | INT | Identifier of the service provider being monitored |
| ServiceType | INT | Type of service being monitored (enumerated value) |
| Status | INT | Current status of the service (0=default, other values indicate specific states) |
| StartTime | DATETIME | Timestamp when the monitoring period or service started |
| EndTime | DATETIME | Timestamp when the monitoring period or service ended |
| ModifiedBy | NVARCHAR(50) | User who last modified the monitoring record |
| ModifiedOn | DATETIME | Timestamp of last modification |
| isUAT | BIT | Environment flag indicating UAT (1) vs Production (0) |

## Media Services

### VideoServerSetting
Configures video streaming servers with quality settings and performance thresholds.

```sql
CREATE TABLE [dbo].[VideoServerSetting] (
    [ID]            INT            IDENTITY (1, 1) NOT NULL,
    [GPID]          INT            NOT NULL,
    [ServerName]    NVARCHAR (30)  NOT NULL,
    [VideoURL]      NVARCHAR (200) NOT NULL,
    [TestSpeedFile] NVARCHAR (200) NOT NULL,
    [CreatedTime]   DATETIME       CONSTRAINT [DF_VideoServerSetting_CreateTime1] DEFAULT (getdate()) NOT NULL,
    [CreatedBy]     NVARCHAR (50)  CONSTRAINT [DF_VideoServerSetting_CreateTime] DEFAULT ((0)) NOT NULL,
    [ModifiedTime]  DATETIME       CONSTRAINT [DF_VideoServerSetting_ModifiedTime1] DEFAULT (getdate()) NOT NULL,
    [ModifiedBy]    NVARCHAR (50)  CONSTRAINT [DF_VideoServerSetting_ModifiedTime] DEFAULT ((0)) NOT NULL,
    [MediumKbps]    INT            NOT NULL,
    [HighKbps]      INT            NOT NULL,
    [IsUAT]         BIT            NOT NULL,
    [Type]          INT            NULL,
    [format]        VARCHAR (50)   NULL,
    CONSTRAINT [PK_VideoServerSetting] PRIMARY KEY CLUSTERED ([ID] ASC)
)
```

| Column | Type | Description |
|--------|------|-------------|
| ID | INT, PK | Primary key, auto-incrementing video server configuration identifier |
| GPID | INT | Game Provider ID associated with this video server |
| ServerName | NVARCHAR(30) | Name or identifier of the video server |
| VideoURL | NVARCHAR(200) | Base URL for video streaming from this server |
| TestSpeedFile | NVARCHAR(200) | File path used for speed testing/bandwidth measurement |
| CreatedTime | DATETIME | Timestamp when the server configuration was created |
| CreatedBy | NVARCHAR(50) | User who created the server configuration |
| ModifiedTime | DATETIME | Timestamp of last modification |
| ModifiedBy | NVARCHAR(50) | User who last modified the configuration |
| MediumKbps | INT | Bandwidth threshold for medium quality (if speed > this, use medium; else use low) |
| HighKbps | INT | Bandwidth threshold for high quality (if speed > this, use high quality) |
| IsUAT | BIT | Environment flag (Dev, Staging, UAT, Production) |
| Type | INT | Video server type or category (nullable) |
| format | VARCHAR(50) | Video format or codec configuration (nullable) |

## Service Management (Extended)

### VMServiceStatus
Tracks virtual machine service status for infrastructure monitoring.

```sql
CREATE TABLE [dbo].[VMServiceStatus] (
    [VMID]                  INT           IDENTITY (1, 1)   NOT NULL,
    [InternalDomain]        NVARCHAR(50)                    NOT NULL,
    [HostName]              NVARCHAR(50)                    NOT NULL,
    [Status]                INT                             NOT NULL,
    [CreatedBy]             NVARCHAR (50)                   NOT NULL,
    [CreatedOn]             DATETIME                        NOT NULL,
    [ModifiedBy]            NVARCHAR (50)                   NOT NULL,
    [ModifiedOn]            DATETIME                        NOT NULL,
    CONSTRAINT [PK_VMServiceStatus] PRIMARY KEY CLUSTERED ([VMID] ASC)
)
```

| Column | Type | Description |
|--------|------|-------------|
| VMID | INT, PK | Primary key, auto-incrementing virtual machine identifier |
| InternalDomain | NVARCHAR(50) | Internal domain name for the virtual machine |
| HostName | NVARCHAR(50) | Hostname or identifier of the virtual machine |
| Status | INT | Current status code of the VM service |
| CreatedBy | NVARCHAR(50) | User who created the VM status record |
| CreatedOn | DATETIME | Timestamp when the VM was registered |
| ModifiedBy | NVARCHAR(50) | User who last modified the VM status |
| ModifiedOn | DATETIME | Timestamp of last status modification |

### ProjectServiceStatus
Monitors the operational status of various project services.

```sql
CREATE TABLE [dbo].[ProjectServiceStatus] (
    [ID]            INT          IDENTITY (1, 1) NOT NULL,
    [ServiceName]   VARCHAR (50) NOT NULL,
    [MachineName]   VARCHAR (50) NOT NULL,
    [Status]        INT          NOT NULL,
    [CreatedBy]     VARCHAR (50) NOT NULL,
    [CreatedOn]     DATETIME     NOT NULL,
    [ModifiedBy]    VARCHAR (50) NOT NULL,
    [ModifiedOn]    DATETIME     NOT NULL,
    CONSTRAINT [PK_ProjectServiceStatus] PRIMARY KEY CLUSTERED ([ID] ASC)
)
```

| Column | Type | Description |
|--------|------|-------------|
| ID | INT, PK | Primary key, auto-incrementing service status identifier |
| ServiceName | VARCHAR(50) | Name of the service being monitored |
| MachineName | VARCHAR(50) | Machine or server name where the service runs |
| Status | INT | Current status code of the service |
| CreatedBy | VARCHAR(50) | User who created the service status record |
| CreatedOn | DATETIME | Timestamp when the service was registered |
| ModifiedBy | VARCHAR(50) | User who last modified the service status |
| ModifiedOn | DATETIME | Timestamp of last status modification |

## Company Flow Management

### CompanyFlowDefaultSetting
Defines default configuration for company workflow forms and field validation rules.

```sql
CREATE TABLE [dbo].[CompanyFlowDefaultSetting] (
    [Type]                  NVARCHAR (50)  NOT NULL,
    [PropertyName]			NVARCHAR(100)  NOT NULL,
    [Status]			    NVARCHAR(100)  NOT NULL,
    [Remark]                NVARCHAR(500)  NULL,
    [CreatedBy]             NVARCHAR (50)  NOT NULL,
    [CreatedOn]             DATETIME       CONSTRAINT [DF_CompanyFlowDefaultSetting_CreatedOn] DEFAULT (getdate()) NOT NULL,
    [ModifiedBy]            NVARCHAR (50)  NOT NULL,
    [ModifiedOn]            DATETIME       CONSTRAINT [DF_CompanyFlowDefaultSetting_ModifiedOn] DEFAULT (getdate()) NOT NULL,
    [DisplayOrder]          INT            NULL,
    [BODisplayOrder]        INT            NULL,
    [FieldType]             NVARCHAR(100)   NULL CONSTRAINT CK_CompanyFlowDefaultSetting_FieldType CHECK (FieldType IN ('number', 'date', 'text','symbol','letter','fileImage')),
    [LengthLimit]           NVARCHAR(50)   NULL Default(''),
    [FieldValidation]       NVARCHAR(100) NULL,
    [DisplayName]           NVARCHAR(100) NOT NULL Default(''),
    [OptionalValidation]    NVARCHAR(100) NOT NULL Default(''),
    [IsModify]              BIT NULL Default(1),
    CONSTRAINT [PK_CompanyFlowDefaultSetting] PRIMARY KEY CLUSTERED ([Type],[PropertyName]ASC) WITH (FILLFACTOR = 85)
)
```

| Column | Type | Description |
|--------|------|-------------|
| Type | NVARCHAR(50), PK | Form type or category for the company flow |
| PropertyName | NVARCHAR(100), PK | Name of the form field or property |
| Status | NVARCHAR(100) | Status or state of the field configuration |
| Remark | NVARCHAR(500) | Optional remarks or notes about the field |
| CreatedBy | NVARCHAR(50) | User who created the field configuration |
| CreatedOn | DATETIME | Timestamp when the configuration was created |
| ModifiedBy | NVARCHAR(50) | User who last modified the configuration |
| ModifiedOn | DATETIME | Timestamp of last modification |
| DisplayOrder | INT | Order for displaying this field in forms |
| BODisplayOrder | INT | Display order for back-office interfaces |
| FieldType | NVARCHAR(100) | Type of field (number, date, text, symbol, letter, fileImage) |
| LengthLimit | NVARCHAR(50) | Character or value length limitations |
| FieldValidation | NVARCHAR(100) | Validation rules for the field |
| DisplayName | NVARCHAR(100) | Human-readable name for the field |
| OptionalValidation | NVARCHAR(100) | Additional optional validation rules |
| IsModify | BIT | Flag indicating if the field can be modified |

### CompanyFlowCustomizeSetting
Provides website-specific customizations for company workflow configurations.

```sql
CREATE TABLE [dbo].[CompanyFlowCustomizeSetting] (
    [WebId]					INT             NOT NULL,
    [DomainId]	            INT             NOT NULL CONSTRAINT [DF_CompanyFlowCustomizeSetting_DomainId] DEFAULT ((0)),
    [Type]			        NVARCHAR(50)    NOT NULL,
    [PropertyName]			NVARCHAR(100)   NOT NULL,
    [Language]              NVARCHAR (50)   NOT NULL CONSTRAINT [DF_CompanyFlowCustomizeSetting_Language] DEFAULT 'en',
    [Status]			    INT             NOT NULL,
    [CreatedBy]             NVARCHAR (50)   NOT NULL,
    [CreatedOn]             DATETIME        CONSTRAINT [DF_CompanyFlowCustomizeValue_CreatedOn] DEFAULT (getdate()) NOT NULL,
    [ModifiedBy]            NVARCHAR (50)   NOT NULL,
    [ModifiedOn]            DATETIME        CONSTRAINT [DF_CompanyFlowCustomizeValue_ModifiedOn] DEFAULT (getdate()) NOT NULL,
    [DisplayOrder]          INT             NULL,
    [isSettingForApp]       BIT             CONSTRAINT [DF_CompanyFlowCustomizeValue_IsSettingForApp] DEFAULT (0) NOT NULL,
    [DisplayName]           NVARCHAR(100)   NULL,
    CONSTRAINT [PK_CompanyFlowCustomizeValue] PRIMARY KEY CLUSTERED ([Webid],[DomainId],[Type],[PropertyName],[Language] ASC) WITH (FILLFACTOR = 85)
)
```

| Column | Type | Description |
|--------|------|-------------|
| WebId | INT, PK | Website identifier for the customization |
| DomainId | INT, PK | Domain identifier for sub-domain specific settings |
| Type | NVARCHAR(50), PK | Form type matching the default setting |
| PropertyName | NVARCHAR(100), PK | Property name matching the default setting |
| Language | NVARCHAR(50), PK | Language code for localized customizations |
| Status | INT | Custom status value for this website/domain |
| CreatedBy | NVARCHAR(50) | User who created the customization |
| CreatedOn | DATETIME | Timestamp when the customization was created |
| ModifiedBy | NVARCHAR(50) | User who last modified the customization |
| ModifiedOn | DATETIME | Timestamp of last modification |
| DisplayOrder | INT | Custom display order for this website |
| isSettingForApp | BIT | Flag indicating if this setting is for mobile app |
| DisplayName | NVARCHAR(100) | Custom display name for this website/language |

### CompanyFlowCustomizeSettingDetail
Stores detailed customization values for company flow properties with extended configuration options.

```sql
CREATE TABLE CompanyFlowCustomizeSettingDetail (
  [Id] INT            IDENTITY (1, 1) NOT NULL,
  [WebId] INT,
  [DomainId] INT,
  [Language] NVARCHAR(50),
  [PropertyName] NVARCHAR(100),
  [Name] NVARCHAR(250),
  [Value] NVARCHAR(100),
  [CreatedOn] DATETIME,
  [CreatedBy] NVARCHAR(100),
  [ModifiedOn] DATETIME,
  [ModifiedBy] NVARCHAR(100)
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Primary key, auto-incrementing detail identifier |
| WebId | INT | Website identifier for the detailed customization |
| DomainId | INT | Domain identifier for sub-domain specific details |
| Language | NVARCHAR(50) | Language code for localized detail values |
| PropertyName | NVARCHAR(100) | Property name this detail belongs to |
| Name | NVARCHAR(250) | Name or label for this detail item |
| Value | NVARCHAR(100) | Specific value for this detail configuration |
| CreatedOn | DATETIME | Timestamp when the detail was created |
| CreatedBy | NVARCHAR(100) | User who created the detail |
| ModifiedOn | DATETIME | Timestamp of last modification |
| ModifiedBy | NVARCHAR(100) | User who last modified the detail |

## Geographic IP Management

### GEOIP
Stores geographic IP location database files for IP-based location services.

```sql
CREATE TABLE [dbo].[GEOIP] (
    [FileID]           INT             IDENTITY (1, 1) NOT NULL,
    [LastModifiedDate] DATETIME        NOT NULL,
    [FileContent]      VARBINARY (MAX) NOT NULL,
    CONSTRAINT [PK_GEOIP] PRIMARY KEY CLUSTERED ([FileID] ASC)
)
```

| Column | Type | Description |
|--------|------|-------------|
| FileID | INT, PK | Primary key, auto-incrementing file identifier |
| LastModifiedDate | DATETIME | Timestamp when the GeoIP file was last updated |
| FileContent | VARBINARY(MAX) | Binary content of the GeoIP database file |

### GEOIPV6
Stores IPv6 geographic location database files for IPv6 address location services.

```sql
CREATE TABLE [dbo].[GEOIPV6] (
    [FileID]           INT             IDENTITY (1, 1) NOT NULL,
    [LastModifiedDate] DATETIME        NOT NULL,
    [FileContent]      VARBINARY (MAX) NOT NULL,
    CONSTRAINT [PK_GEOIPV6] PRIMARY KEY CLUSTERED ([FileID] ASC)
)
```

| Column | Type | Description |
|--------|------|-------------|
| FileID | INT, PK | Primary key, auto-incrementing IPv6 file identifier |
| LastModifiedDate | DATETIME | Timestamp when the GeoIP IPv6 file was last updated |
| FileContent | VARBINARY(MAX) | Binary content of the GeoIP IPv6 database file |

## Database Configuration

### DBConfigNew
Manages database connection configurations with failover support and connection pooling settings.

```sql
CREATE TABLE [dbo].[DBConfigNew] (
    [Project]          VARCHAR (20)  NOT NULL,
    [DBRole]           VARCHAR (50)  NOT NULL,
    [ConnectionString] VARCHAR (500) NOT NULL,
    [Enabled]          BIT           NULL,
    [Remark]           NCHAR (10)    NOT NULL,
    [CreatedOn]        DATETIME      NULL,
    [LastModifiedOn]   DATETIME      NULL,
    [CreatedBy]        VARCHAR (50)  NULL,
    [LastModifiedBy]   VARCHAR (50)  NULL,
    [FailOver]         VARCHAR (50)  NULL,
    [Server]           VARCHAR (50)  NULL,
    [MaxReuse]         INT           NULL,
    [DB]               VARCHAR (50)  NULL,
    [DefaultEnabled]   BIT           NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Project | VARCHAR(20) | Project identifier for the database configuration |
| DBRole | VARCHAR(50) | Database role or purpose (e.g., 'Primary', 'ReadOnly', 'Reporting') |
| ConnectionString | VARCHAR(500) | Complete database connection string |
| Enabled | BIT | Flag indicating if this configuration is currently enabled |
| Remark | NCHAR(10) | Short remarks or notes about the configuration |
| CreatedOn | DATETIME | Timestamp when the configuration was created |
| LastModifiedOn | DATETIME | Timestamp of last modification |
| CreatedBy | VARCHAR(50) | User who created the configuration |
| LastModifiedBy | VARCHAR(50) | User who last modified the configuration |
| FailOver | VARCHAR(50) | Failover server or configuration |
| Server | VARCHAR(50) | Database server name or address |
| MaxReuse | INT | Maximum connection reuse count for connection pooling |
| DB | VARCHAR(50) | Database name |
| DefaultEnabled | BIT | Flag indicating if this is the default enabled configuration |

### DBConfig
Legacy database configuration table with similar structure to DBConfigNew for backward compatibility.

```sql
CREATE TABLE [dbo].[DBConfig] (
    [Project]          VARCHAR (20)  NOT NULL,
    [DBRole]           VARCHAR (50)  NOT NULL,
    [ConnectionString] VARCHAR (500) NOT NULL,
    [Enabled]          BIT           NULL,
    [Remark]           NCHAR (10)    NOT NULL,
    [CreatedOn]        DATETIME      NULL,
    [LastModifiedOn]   DATETIME      NULL,
    [CreatedBy]        VARCHAR (50)  NULL,
    [LastModifiedBy]   VARCHAR (50)  NULL,
    [FailOver]         VARCHAR (50)  NULL,
    [Server]           VARCHAR (50)  NULL,
    [MaxReuse]         INT           NULL,
    [DB]               VARCHAR (50)  NULL,
    [DefaultEnabled]   BIT           NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Project | VARCHAR(20) | Project identifier for legacy database configuration |
| DBRole | VARCHAR(50) | Database role or purpose (legacy version) |
| ConnectionString | VARCHAR(500) | Complete database connection string |
| Enabled | BIT | Flag indicating if this legacy configuration is enabled |
| Remark | NCHAR(10) | Short remarks or notes |
| CreatedOn | DATETIME | Creation timestamp |
| LastModifiedOn | DATETIME | Last modification timestamp |
| CreatedBy | VARCHAR(50) | User who created the configuration |
| LastModifiedBy | VARCHAR(50) | User who last modified the configuration |
| FailOver | VARCHAR(50) | Failover server configuration |
| Server | VARCHAR(50) | Database server name |
| MaxReuse | INT | Connection reuse limit |
| DB | VARCHAR(50) | Database name |
| DefaultEnabled | BIT | Default enabled flag |

## Game Provider & Fund Management

### FundProvider
Manages financial service providers and their security configurations for payment processing.

```sql
CREATE TABLE [dbo].[FundProvider] (
    [ID]          INT           IDENTITY (1, 1) NOT NULL,
    [FPName]      VARCHAR (30)  NOT NULL,
    [CreatedBy]   VARCHAR (50)  NOT NULL,
    [CreatedOn]   DATETIME      NOT NULL,
    [ModifiedBy]  VARCHAR (50)  NOT NULL,
    [ModifiedOn]  DATETIME      NOT NULL,
    [SecurityKey] VARCHAR (64)  NOT NULL,
    [ServerURL]   VARCHAR (200) NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| ID | INT, PK | Primary key, auto-incrementing fund provider identifier |
| FPName | VARCHAR(30) | Name of the fund provider |
| CreatedBy | VARCHAR(50) | User who created the fund provider record |
| CreatedOn | DATETIME | Timestamp when the provider was registered |
| ModifiedBy | VARCHAR(50) | User who last modified the provider |
| ModifiedOn | DATETIME | Timestamp of last modification |
| SecurityKey | VARCHAR(64) | Security key for authentication with the provider |
| ServerURL | VARCHAR(200) | API endpoint URL for the fund provider |

### GPServicedFP
Maps relationships between game providers and fund providers for payment routing.

```sql
CREATE TABLE [dbo].[GPServicedFP] (
    [ID]         INT           IDENTITY (1, 1) NOT NULL,
    [GPID]       INT           NOT NULL,
    [FPID]       INT           NOT NULL,
    [Status]     INT           NOT NULL,
    [CreatedBy]  VARCHAR (50) NOT NULL,
    [CreatedOn]  DATETIME      NOT NULL,
    [ModifiedBy] VARCHAR (50) NOT NULL,
    [ModifiedOn] DATETIME      NOT NULL,
    CONSTRAINT [PK_GPServicedFP] PRIMARY KEY CLUSTERED ([ID] ASC)
)
```

| Column | Type | Description |
|--------|------|-------------|
| ID | INT, PK | Primary key, auto-incrementing relationship identifier |
| GPID | INT | Game Provider ID (foreign key reference) |
| FPID | INT | Fund Provider ID (foreign key reference) |
| Status | INT | Status of the relationship (active, inactive, etc.) |
| CreatedBy | VARCHAR(50) | User who created the relationship |
| CreatedOn | DATETIME | Timestamp when the relationship was established |
| ModifiedBy | VARCHAR(50) | User who last modified the relationship |
| ModifiedOn | DATETIME | Timestamp of last modification |

## Monitoring & Logging

### SchedulerMonitor
Tracks execution status and performance of scheduled jobs for monitoring purposes.

```sql
CREATE TABLE [dbo].[SchedulerMonitor] (
    [SchedulerName] VARCHAR (100)  NOT NULL,
    [WebSite]       VARCHAR (100)  NOT NULL,
    [TimeStamp]     VARCHAR (1000) NOT NULL,
    [ExecutedOn]    DATETIME       NOT NULL,
    CONSTRAINT [PK_ElipsSchedulerMonitor] PRIMARY KEY CLUSTERED ([SchedulerName] ASC)
)
```

| Column | Type | Description |
|--------|------|-------------|
| SchedulerName | VARCHAR(100), PK | Name of the scheduled job being monitored |
| WebSite | VARCHAR(100) | Website identifier where the scheduler executed |
| TimeStamp | VARCHAR(1000) | Detailed timestamp information for execution tracking |
| ExecutedOn | DATETIME | Actual execution timestamp |

### Schedule
Manages provider-specific schedule assignments and server allocations.

```sql
CREATE TABLE [dbo].[Schedule] (
    [ProviderUniqueName] VARCHAR (50) NOT NULL,
    [ScheduleId]         VARCHAR (50) NOT NULL,
    [ServerName]         VARCHAR (50) NOT NULL,
    [CreatedOn]          DATETIME     NOT NULL,
    [ModifiedOn]         DATETIME     NOT NULL,
    [CreatedBy]          VARCHAR (50) NOT NULL,
    CONSTRAINT [PK_Schedule_1] PRIMARY KEY CLUSTERED ([ProviderUniqueName] ASC, [ScheduleId] ASC)
)
```

| Column | Type | Description |
|--------|------|-------------|
| ProviderUniqueName | VARCHAR(50), PK | Unique name identifier for the provider |
| ScheduleId | VARCHAR(50), PK | Schedule identifier for the provider |
| ServerName | VARCHAR(50) | Server assigned to handle this schedule |
| CreatedOn | DATETIME | Timestamp when the schedule was created |
| ModifiedOn | DATETIME | Timestamp of last modification |
| CreatedBy | VARCHAR(50) | User who created the schedule |

### AppSettingLog
Audit log for tracking all changes made to application settings across different tables.

```sql
CREATE TABLE [dbo].[AppSettingLog] (
    [Id]         INT             IDENTITY (1, 1) NOT NULL,
    [WebSite]    NVARCHAR (50)   NOT NULL,
    [TableName]  NVARCHAR (50)   NOT NULL,
    [Action]     NVARCHAR (10)   NOT NULL,
    [ModefiedOn] DATETIME        NOT NULL,
    [ModifiedBy] NVARCHAR (50)   NOT NULL,
    [Data]       NVARCHAR (4000) NULL,
    PRIMARY KEY CLUSTERED ([Id] ASC)
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Primary key, auto-incrementing log entry identifier |
| WebSite | NVARCHAR(50) | Website identifier for the setting change |
| TableName | NVARCHAR(50) | Name of the table that was modified |
| Action | NVARCHAR(10) | Type of action performed (INSERT, UPDATE, DELETE) |
| ModefiedOn | DATETIME | Timestamp when the modification occurred |
| ModifiedBy | NVARCHAR(50) | User who made the modification |
| Data | NVARCHAR(4000) | Serialized data of the change (before/after values) |

## Key Features

1. **Multi-Environment Configuration**
   - Support for UAT and Production environments through IsUAT flag
   - Environment-specific setting isolation
   - Consistent configuration management across environments

2. **Hierarchical Settings Management**
   - SimpleSettings for basic application configuration with type categorization
   - Override capabilities through SimpleSettingsOverride
   - CustomerSettings for user-customizable options with self-service capabilities
   - Environment-specific and website-specific overrides

3. **Automated Job Scheduling**
   - Comprehensive scheduler configuration with dependency management
   - Support for daily recurring jobs and interval-based execution
   - Job duplication control and lazy execution options
   - Override capabilities for environment-specific scheduler configurations

4. **Load Balancing & Clustering**
   - Web cluster configuration for high availability
   - Individual node management with subdomain routing (up to 15 subdomains per node)
   - Load balancing parameters and server grouping
   - Configurable server modes (Active, Standby, etc.)

5. **Message Queue Management**
   - MSMQ configuration for inter-service communication
   - Support for private and public queues
   - Environment-specific queue settings

6. **Internationalization Support**
   - Multi-language configuration with ISO language codes
   - Website-specific language overrides for multi-tenant environments
   - Language code management and localization

7. **Security & Access Control**
   - IP-based restrictions for administrative functions with group management
   - Environment-specific security settings
   - Centralized IP group configuration for access control
   - Audit trails and modification tracking

8. **Service Monitoring & Management**
   - Server uptime monitoring through ServerUM
   - Virtual machine service status tracking
   - Project service status monitoring across multiple machines
   - Time-based monitoring periods with start/end tracking

9. **Media Streaming Services**
   - Video server configuration and management
   - Adaptive quality streaming based on bandwidth thresholds
   - Performance testing capabilities with speed file configuration
   - Multi-format video support with game provider integration

10. **Company Flow Management**
    - Configurable workflow forms with field validation
    - Website and domain-specific customizations
    - Multi-language support for form fields
    - Detailed customization options with extended configuration
    - Field type validation (number, date, text, symbol, letter, fileImage)

11. **Geographic IP Services**
    - GeoIP database file management for IPv4 and IPv6
    - Binary file storage for location services
    - Automatic file versioning with modification tracking

12. **Database Configuration Management**
    - Multiple database connection configurations with failover support
    - Connection pooling settings with reuse limits
    - Legacy and new configuration table support
    - Project-specific database role management

13. **Game Provider & Fund Management**
    - Financial service provider configuration
    - Security key management for payment processing
    - Game provider to fund provider relationship mapping
    - Status tracking for provider relationships

14. **Comprehensive Monitoring & Logging**
    - Scheduler execution monitoring and performance tracking
    - Provider-specific schedule assignments
    - Complete audit logging for all setting changes
    - Detailed change tracking with before/after data

## Security

1. **Role-Based Access**
   - Controlled access to configuration settings
   - Audit trails through ModifiedBy fields
   - Responsible party tracking for accountability

2. **Permissions**
   - Read/Write access control on configuration tables
   - Customer self-service capabilities where appropriate
   - Environment isolation for UAT vs Production

3. **IP Restrictions**
   - Administrative page access control
   - Environment-specific IP filtering
   - Centralized IP group management

## Relationships
- All configuration tables are linked by Website and IsUAT for environment consistency
- Override tables extend base settings for specific customizations
- Scheduler settings can reference dependencies creating job execution chains
- Customer settings integrate with application-wide simple settings
- WebClusterNodes relate to WebClusters for cluster management
- AdminIPRestriction references IP groups for access control

## Best Practices
1. Always maintain environment separation through IsUAT flag
2. Document configuration changes through Description fields
3. Use override tables rather than modifying base settings directly
4. Regularly audit configuration changes through ModifiedBy/ModifiedOn tracking
5. Test scheduler configurations in UAT before production deployment
6. Implement proper backup strategies for configuration data
7. Monitor cluster performance and adjust load balancing parameters accordingly
8. Regularly review and update IP restrictions for security
9. Use proper language codes for internationalization
10. Configure message queues with appropriate security settings 