# GameProvider Database Schema

## Overview
The GameProvider database manages gaming and betting operations in the Coloris Framework, including game provider management, betting records, player transactions, compliance monitoring, and daily summaries across multiple gaming platforms.

## Core Provider Management Tables

### AllGameProviders
Manages all game providers available in the system with their configurations and settings.

```sql
CREATE TABLE [dbo].[AllGameProviders] (
  [APIGameProviderId] INT NOT NULL DEFAULT (1),
  [GameProviderId] INT NOT NULL,
  [GameProviderName] VARCHAR(30) NOT NULL,
  [IsEnabled] BIT NOT NULL,
  [CreatedBy] VARCHAR(50) NOT NULL,
  [CreatedOn] DATETIME NOT NULL,
  [ModifiedBy] VARCHAR(50) NOT NULL,
  [ModifiedOn] DATETIME NOT NULL,
  [DisplayName] VARCHAR(100) NULL,
  [ShortName] VARCHAR(100) NULL,
  [GameProviderType] INT NOT NULL CONSTRAINT [DF_AllGameProviders_GameproviderType] DEFAULT(0),
  [IsReadyToOpen] BIT NULL DEFAULT (1),
  [IsAbleToSetBetSetting] BIT NULL DEFAULT (0),
  [MaxBetSettingCount] INT NOT NULL DEFAULT(0),
  [IsAbleToSetMultiBetSettings] BIT NOT NULL DEFAULT (0),
  [IsShowBetDetailButton] BIT NOT NULL DEFAULT (0),
  [IsTransferGameProvider] BIT NOT NULL DEFAULT (0),
  [IsMaintainByGameProvider] BIT NOT NULL DEFAULT (0),
  [IsSupportCommission] BIT NOT NULL DEFAULT (0),
  [IsHot] BIT NOT NULL DEFAULT (0),
  [IsNew] BIT NOT NULL DEFAULT (0),
  [IsSupportStreaming] BIT NOT NULL DEFAULT (0),
  [DisplayOrder] INT CONSTRAINT [DF_AllGameProviders_DisplayOrder] DEFAULT (dbo.GetMaxDisplayOrderForWebId()) NOT NULL,
  [IsSupportIframe] BIT NOT NULL DEFAULT (0),
  CONSTRAINT Unique_AllGameProviders UNIQUE(APIGameProviderId,GameProviderId)
)
```

| Column | Type | Description |
|--------|------|-------------|
| APIGameProviderId | INT | API game provider identifier |
| GameProviderId | INT | Unique identifier for the game provider |
| GameProviderName | VARCHAR(30) | Name of the game provider |
| IsEnabled | BIT | Whether the provider is currently enabled |
| CreatedBy | VARCHAR(50) | User who created the record |
| CreatedOn | DATETIME | Timestamp when the record was created |
| ModifiedBy | VARCHAR(50) | User who last modified the record |
| ModifiedOn | DATETIME | Timestamp when the record was last modified |
| DisplayName | VARCHAR(100) | Display name for the provider |
| ShortName | VARCHAR(100) | Short name for the provider |
| GameProviderType | INT | Type classification of the provider |
| IsReadyToOpen | BIT | Whether the provider is ready to accept bets |
| IsAbleToSetBetSetting | BIT | Whether betting settings can be configured |
| MaxBetSettingCount | INT | Maximum number of bet settings allowed |
| IsAbleToSetMultiBetSettings | BIT | Whether multiple bet settings are supported |
| IsShowBetDetailButton | BIT | Whether to show bet detail button |
| IsTransferGameProvider | BIT | Whether this is a transfer-based provider |
| IsMaintainByGameProvider | BIT | Whether maintenance is handled by the provider |
| IsSupportCommission | BIT | Whether commission is supported |
| IsHot | BIT | Whether this provider is marked as hot/popular |
| IsNew | BIT | Whether this provider is marked as new |
| IsSupportStreaming | BIT | Whether streaming is supported |
| DisplayOrder | INT | Order for display purposes |
| IsSupportIframe | BIT | Whether iframe integration is supported |

### CompanyGameProviders
Manages game providers specific to each company/website configuration.

```sql
CREATE TABLE [dbo].[CompanyGameProviders] (
  [WebId] INT NOT NULL,
  [APIGameProviderId] INT NOT NULL,
  [GameProviderId] INT NOT NULL,
  [GameProviderName] VARCHAR(30) NOT NULL,
  [IsEnabled] BIT NOT NULL,
  [CreatedBy] VARCHAR(50) NOT NULL,
  [CreatedOn] DATETIME NOT NULL,
  [ModifiedBy] VARCHAR(50) NOT NULL,
  [ModifiedOn] DATETIME NOT NULL,
  [DisplayOrder] INT NOT NULL,
  [IsAuthorizeByPlatformProvider] BIT DEFAULT (1) NOT NULL,
  [IsAuthorizeByPlatformProviderModifiedBy] NVARCHAR(50) DEFAULT ('System Init') NOT NULL,
  [IsAuthorizeByPlatformProviderModifiedOn] DATETIME DEFAULT GETDATE() NOT NULL,
  [IsHot] BIT NOT NULL DEFAULT (0),
  [IsNew] BIT NOT NULL DEFAULT (0)
)
```

| Column | Type | Description |
|--------|------|-------------|
| WebId | INT | Website/company identifier |
| APIGameProviderId | INT | API game provider identifier |
| GameProviderId | INT | Game provider identifier |
| GameProviderName | VARCHAR(30) | Name of the game provider for this company |
| IsEnabled | BIT | Whether the provider is enabled for this company |
| CreatedBy | VARCHAR(50) | User who created the record |
| CreatedOn | DATETIME | Creation timestamp |
| ModifiedBy | VARCHAR(50) | User who last modified the record |
| ModifiedOn | DATETIME | Last modification timestamp |
| DisplayOrder | INT | Display order for this company |
| IsAuthorizeByPlatformProvider | BIT | Whether authorized by platform provider |
| IsAuthorizeByPlatformProviderModifiedBy | NVARCHAR(50) | Who modified the authorization |
| IsAuthorizeByPlatformProviderModifiedOn | DATETIME | When authorization was modified |
| IsHot | BIT | Whether marked as hot for this company |
| IsNew | BIT | Whether marked as new for this company |

### AllAPIGameProviders
Master list of all API game providers.

```sql
CREATE TABLE [dbo].[AllAPIGameProviders] (
  [APIGameProviderId] INT NOT NULL PRIMARY KEY,
  [Name] NVARCHAR(100) NOT NULL,
  [DisplayName] NVARCHAR(100) NULL,
  [IsEnabled] BIT NOT NULL DEFAULT 1,
  [CreatedOn] DATETIME NOT NULL DEFAULT GETDATE(),
  [CreatedBy] NVARCHAR(50) NOT NULL,
  [ModifiedOn] DATETIME NOT NULL DEFAULT GETDATE(),
  [ModifiedBy] NVARCHAR(50) NOT NULL
)
```

### CompanyAPIGameProviders
Links API game providers to specific companies.

```sql
CREATE TABLE [dbo].[CompanyAPIGameProviders] (
  [WebId] INT NOT NULL,
  [APIGameProviderId] INT NOT NULL,
  [IsEnabled] BIT NOT NULL,
  [CreatedBy] VARCHAR(50) NOT NULL,
  [CreatedOn] DATETIME NOT NULL,
  [ModifiedBy] VARCHAR(50) NOT NULL,
  [ModifiedOn] DATETIME NOT NULL,
  [DisplayOrder] INT NOT NULL
)
```

## Game Management Tables

### AllGames
Stores information about all games available across all providers.

```sql
CREATE TABLE [dbo].[AllGames] (
  [GameId] INT NOT NULL IDENTITY(1,1),
  [GameIdAtGameProvider] INT NOT NULL,
  [APIGameProviderId] INT NOT NULL DEFAULT (1),
  [GameProviderId] INT NOT NULL DEFAULT (-1),
  [GameName] NVARCHAR(255) NOT NULL,
  [GameCode] NVARCHAR(255) NOT NULL,
  [GameCategory] INT NOT NULL,
  [GameCategoryAtGameProvider] INT NOT NULL,
  [GameType] INT CONSTRAINT [DF_AllGames_GameType] DEFAULT (0) NOT NULL,
  [PlayersiteTabType] INT NULL,
  [IsEnabled] BIT CONSTRAINT [DF_AllGames_IsEnabled] DEFAULT (0) NOT NULL,
  [BSIPriority] INT CONSTRAINT [DF_AllGames_BSIPriority] DEFAULT (0) NOT NULL,
  [ASIPriority] INT CONSTRAINT [DF_AllGames_ASIPriority] DEFAULT (0) NOT NULL,
  [CreatedOn] DATETIME CONSTRAINT [DF_AllGames_CreatedOn] DEFAULT GETDATE() NOT NULL,
  [CreatedBy] NVARCHAR(50) NOT NULL,
  [EnabledCurrency] NVARCHAR(MAX) NOT NULL,
  [ModifiedOn] DATETIME CONSTRAINT [DF_AllGames_ModifiedOn] DEFAULT GETDATE() NOT NULL,
  [ModifiedBy] NVARCHAR(50) NOT NULL,
  [Platform] NVARCHAR(50) NULL,
  [Rank] INT NULL,
  [Device] INT NULL,
  [RTP] DECIMAL(5,4) NULL,
  [Rows] INT NULL,
  [Reels] INT NULL,
  [Lines] INT NULL,
  [SubProvider] NVARCHAR(255) NULL,
  [BlockCountry] NVARCHAR(MAX) NULL,
  [IsSpecialSelection] BIT CONSTRAINT [DF_AllGames_IsSpecialSelection] DEFAULT(0) NOT NULL,
  [IsPromotion] BIT DEFAULT(0),
  [IsProvideCommission] BIT DEFAULT(0),
  [IsEnabledByGameProvider] BIT CONSTRAINT [DF_AllGames_IsEnabledByGameProvider] DEFAULT (1) NOT NULL,
  [IsMaintainByGameProvider] BIT CONSTRAINT [DF_AllGames_IsMaintainByGameProvider] DEFAULT (0) NOT NULL,
  [SpecialSelectionExpireTime] DATETIME CONSTRAINT [DF_AllGames_SpecialSelectionExpireTime] DEFAULT GETDATE() NOT NULL,
  [SpecialSelectionPriority] INT CONSTRAINT [DF_AllGames_SpecialSelectionPriority] DEFAULT (0) NOT NULL,
  CONSTRAINT [PK_AllGames] PRIMARY KEY CLUSTERED ([APIGameProviderId],[GameProviderId],[GameId] ASC) WITH (FILLFACTOR = 85),
  CONSTRAINT UNIQUE_GameID UNIQUE(GameId)
)
GO
CREATE NONCLUSTERED INDEX [IX_AllGame_GameProvider_GetBettingRecord]
ON [dbo].[AllGames] ([GameIdAtGameProvider],[GameProviderId])
INCLUDE ([GameName])
```

| Column | Type | Description |
|--------|------|-------------|
| GameId | INT, PK | Unique identifier for the game |
| GameIdAtGameProvider | INT | Game ID as used by the provider |
| APIGameProviderId | INT | API game provider identifier |
| GameProviderId | INT | Game provider identifier |
| GameName | NVARCHAR(255) | Name of the game |
| GameCode | NVARCHAR(255) | Unique code for the game |
| GameCategory | INT | Internal game category classification |
| GameCategoryAtGameProvider | INT | Game category as defined by provider |
| GameType | INT | Type of game (slot, table, etc.) |
| PlayersiteTabType | INT | Tab type for player site display |
| IsEnabled | BIT | Whether the game is currently enabled |
| BSIPriority | INT | BSI priority setting |
| ASIPriority | INT | ASI priority setting |
| CreatedOn | DATETIME | When the game was added |
| CreatedBy | NVARCHAR(50) | Who added the game |
| EnabledCurrency | NVARCHAR(MAX) | Currencies supported by this game |
| ModifiedOn | DATETIME | Last modification timestamp |
| ModifiedBy | NVARCHAR(50) | Who last modified the game |
| Platform | NVARCHAR(50) | Platform compatibility |
| Rank | INT | Game ranking |
| Device | INT | Device compatibility flags |
| RTP | DECIMAL(5,4) | Return to Player percentage |
| Rows | INT | Number of rows (for slots) |
| Reels | INT | Number of reels (for slots) |
| Lines | INT | Number of pay lines (for slots) |
| SubProvider | NVARCHAR(255) | Sub-provider information |
| BlockCountry | NVARCHAR(MAX) | Countries where game is blocked |
| IsSpecialSelection | BIT | Whether included in special selection |
| IsPromotion | BIT | Whether this game has promotions |
| IsProvideCommission | BIT | Whether commission is provided |
| IsEnabledByGameProvider | BIT | Whether enabled by the provider |
| IsMaintainByGameProvider | BIT | Whether maintained by the provider |
| SpecialSelectionExpireTime | DATETIME | When special selection expires |
| SpecialSelectionPriority | INT | Priority in special selection |

### CompanyGames
Links games to specific companies with their individual settings.

```sql
CREATE TABLE [dbo].[CompanyGames] (
  [Id] INT PRIMARY KEY IDENTITY(1,1) NOT NULL,
  [APIGameProviderId] INT NOT NULL DEFAULT (1),
  [GameProviderId] INT NOT NULL,
  [GameId] INT NOT NULL,
  [BSIPriority] INT DEFAULT (0) NOT NULL,
  [ASIPriority] INT DEFAULT (0) NOT NULL,
  [IsEnabled] BIT DEFAULT (0) NOT NULL,
  [IsSpecialSelection] BIT DEFAULT (0) NOT NULL,
  [ModifiedOn] DATETIME DEFAULT GETDATE() NOT NULL,
  [ModifiedBy] NVARCHAR(50) NOT NULL,
  [WebId] INT NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Primary key |
| APIGameProviderId | INT | API game provider identifier |
| GameProviderId | INT | Game provider identifier |
| GameId | INT | Game identifier |
| BSIPriority | INT | BSI priority for this company |
| ASIPriority | INT | ASI priority for this company |
| IsEnabled | BIT | Whether enabled for this company |
| IsSpecialSelection | BIT | Whether in special selection for this company |
| ModifiedOn | DATETIME | Last modification timestamp |
| ModifiedBy | NVARCHAR(50) | Who last modified |
| WebId | INT | Website/company identifier |

### GameCategory
Defines game categories available in the system.

```sql
CREATE TABLE [dbo].[GameCategory] (
  [ID] INT IDENTITY (1, 1) NOT NULL PRIMARY KEY,
  [WebId] INT NOT NULL DEFAULT 0,
  [Name] NVARCHAR(50) NOT NULL CHECK ([Name] IN ('All Games', 'Special Selection', 'Hot Games', 'New Games', 'Favorite', 'Game Type', 'Provider')),
  [DisplayOrder] INT NOT NULL,
  [CreatedOn] DATETIME NULL,
  [CreatedBy] NVARCHAR(50) NULL,
  [ModifiedOn] DATETIME NULL,
  [ModifiedBy] NVARCHAR(50) NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| ID | INT, PK | Primary key |
| WebId | INT | Website identifier |
| Name | NVARCHAR(50) | Category name (constrained values) |
| DisplayOrder | INT | Order for display |
| CreatedOn | DATETIME | Creation timestamp |
| CreatedBy | NVARCHAR(50) | Creator |
| ModifiedOn | DATETIME | Modification timestamp |
| ModifiedBy | NVARCHAR(50) | Modifier |

### PromoteAllGames
Promotes specific games for enhanced visibility.

```sql
CREATE TABLE [dbo].[PromoteAllGames] (
  [Id] INT IDENTITY(1,1) NOT NULL PRIMARY KEY,
  [WebId] INT NOT NULL,
  [GameId] INT NOT NULL,
  [GameName] NVARCHAR(255) NOT NULL,
  [GameProviderId] INT NOT NULL,
  [DisplayOrder] INT NOT NULL DEFAULT 0,
  [IsEnabled] BIT NOT NULL DEFAULT 1,
  [CreatedOn] DATETIME NOT NULL DEFAULT GETDATE(),
  [CreatedBy] NVARCHAR(50) NOT NULL,
  [ModifiedOn] DATETIME NOT NULL DEFAULT GETDATE(),
  [ModifiedBy] NVARCHAR(50) NOT NULL
)
```

## Translation and Localization Tables

### GameTranslation
Stores game name translations for different languages.

```sql
CREATE TABLE [dbo].[GameTranslation] (
  [GameId] INT NOT NULL,
  [APIGameProviderId] INT NOT NULL,
  [GameProviderId] INT NOT NULL,
  [GameName] NVARCHAR(255) NOT NULL,
  [Language] NVARCHAR(255) NOT NULL,
  [GameIconUrl] NVARCHAR(255) NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| GameId | INT | Game identifier |
| APIGameProviderId | INT | API game provider identifier |
| GameProviderId | INT | Game provider identifier |
| GameName | NVARCHAR(255) | Translated game name |
| Language | NVARCHAR(255) | Language code |
| GameIconUrl | NVARCHAR(255) | URL to game icon |

### GameProviderTranslation
Translations for game provider names in different languages.

```sql
CREATE TABLE [dbo].[GameProviderTranslation] (
  [APIGameProviderId] INT NOT NULL,
  [GameProviderId] INT NOT NULL,
  [GameProviderName] NVARCHAR(255) NOT NULL,
  [Language] NVARCHAR(10) NOT NULL
)
```

## Betting Tables

### SboSportBets
Records sports betting transactions and their details.

```sql
CREATE TABLE [dbo].[SboSportBets] (
  [Id] BIGINT IDENTITY(1,1) NOT NULL,
  [WebId] INT NOT NULL,
  [RefNo] NVARCHAR(100) NOT NULL,
  [TransactionId] NVARCHAR(100) NOT NULL,
  [CustomerId] INT NOT NULL,
  [ParentId] INT NULL,
  [Username] NVARCHAR(50) NOT NULL,
  [UsernameAtGameProvider] NVARCHAR(50) NULL,
  [PromotionRequestId] INT DEFAULT ((0)) NOT NULL,
  [NewPromotionRequestId] INT DEFAULT ((0)) NOT NULL,
  [ReferralEventId] INT DEFAULT ((0)) NOT NULL,
  [WalletProductType] INT DEFAULT ((0)) NOT NULL,
  [WalletGameType] INT DEFAULT ((0)) NOT NULL,
  [SportsType] NVARCHAR(50) NULL,
  [BetType] INT NULL,
  [OrderTime] DATETIME NOT NULL,
  [ModifiedTime] DATETIME NOT NULL,
  [WinlostDate] DATETIME NULL,
  [Odds] DECIMAL(12, 3) DEFAULT ((0)) NULL,
  [OddsStyle] NVARCHAR(10) NULL,
  [Stake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ActualStake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Currency] CHAR(3) NULL,
  [ExchangeRate] DECIMAL(12, 8) NULL,
  [StatusAtGameProvider] VARCHAR(50) NULL,
  [Status] VARCHAR(50) NULL,
  [Winlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [TurnOver] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [IsLive] BIT NULL,
  [IsHalfWinLose] BIT DEFAULT ((0)) NULL,
  [MaxWinWithoutActualStake] DECIMAL(19,6),
  -- Commission and profit tracking fields
  [AgtPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [MaPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [SmaPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [SsmaPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [ShPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [TotalPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  -- Additional fields for comprehensive tracking
  [Ip] NVARCHAR(100) DEFAULT ('') NULL,
  [IsSystemTagRisky] BIT DEFAULT (0) NULL,
  [IsCashOut] BIT DEFAULT (0) NULL,
  [SyncTime] DATETIME,
  [BetFrom] NVARCHAR(20) DEFAULT '' NOT NULL,
  [ReferralLayerPercentageHistoryIds] NVARCHAR(1000),
  [VoidReason] NVARCHAR(500) DEFAULT ('') NULL,
  [Remark] NVARCHAR(500) DEFAULT ('') NULL,
  [WinlostDateByTimeZone] DATETIME NULL,
  [TimeZone] NVARCHAR(100) NULL,
  [SportGameId] NVARCHAR(100) NULL,
  [WithdrawalLimitId] INT DEFAULT(0) NOT NULL
)
```

### SboSeamlessBets
Records seamless gaming transactions across integrated game providers.

```sql
CREATE TABLE [dbo].[SboSeamlessBets] (
  [Id] BIGINT IDENTITY(1,1) NOT NULL,
  [GPID] INT NULL,
  [WebId] INT NOT NULL,
  [RefNo] NVARCHAR(255) NOT NULL,
  [TransactionId] NVARCHAR(255) NOT NULL,
  [CustomerId] INT NOT NULL,
  [ParentId] INT NULL,
  [MaId] INT CONSTRAINT [DF_SboSeamlessBets_MaId] DEFAULT ((0)) NOT NULL,
  [Username] NVARCHAR(50) NOT NULL,
  [UsernameAtGameProvider] NVARCHAR(50) NULL,
  [PromotionRequestId] INT CONSTRAINT [DF_SboSeamlessBets_PromotionRequestId] DEFAULT ((0)) NOT NULL,
  [NewPromotionRequestId] INT CONSTRAINT [DF_SboSeamlessBets_NewPromotionRequestId] DEFAULT ((0)) NOT NULL,
  [ReferralEventId] INT CONSTRAINT [DF_SboSeamlessBets_ReferralEventId] DEFAULT ((0)) NOT NULL,
  [WalletProductType] INT CONSTRAINT [DF_SboSeamlessBets_WalletProductType] DEFAULT ((0)) NOT NULL,
  [WalletGameType] INT CONSTRAINT [DF_SboSeamlessBets_WalletGameType] DEFAULT ((0)) NOT NULL,
  [OrderTime] DATETIME NOT NULL,
  [WinLostDate] DATETIME NOT NULL,
  [Stake] DECIMAL(19, 6) CONSTRAINT [DF_SboSeamlessBets_Stake] DEFAULT ((0)) NULL,
  [Currency] CHAR(3) NULL,
  [ExchangeRate] DECIMAL(12, 8) NULL,
  [TurnOver] DECIMAL(19, 6) CONSTRAINT [DF_SboSeamlessBets_TurnOver] DEFAULT ((0)) NULL,
  [StatusAtGameProvider] VARCHAR(50) NULL,
  [Status] VARCHAR(50) NULL,
  [IsFishGame] BIT CONSTRAINT [DF_SboSeamlessBets_IsFishGame] DEFAULT ((0)) NULL,
  [GameIdAtGameProvider] INT CONSTRAINT [DF_SboSeamlessBets_GameIdAtGameProvider] DEFAULT ((0)) NOT NULL,
  [GamePeriodId] NVARCHAR(100) NULL,
  [GameRoundId] NVARCHAR(100) NULL,
  [GameType] NVARCHAR(50) NULL,
  [OrderDetail] NVARCHAR(4000) NULL,
  [GameResult] NVARCHAR(4000) NULL,
  [Winlost] DECIMAL(19, 6) CONSTRAINT [DF_SboSeamlessBets_Winlost] DEFAULT ((0)) NULL,
  [AgtPT] DECIMAL(3, 2) CONSTRAINT [DF_SboSeamlessBets_AgtPt] DEFAULT ((0)) NULL,
  [MaPT] DECIMAL(3, 2) CONSTRAINT [DF_SboSeamlessBets_MaPt] DEFAULT ((0)) NULL,
  [SmaPT] DECIMAL(3, 2) CONSTRAINT [DF_SboSeamlessBets_SmaPt] DEFAULT ((0)) NULL,
  [SsmaPT] DECIMAL(3, 2) CONSTRAINT [DF_SboSeamlessBets_SsmaPt] DEFAULT ((0)) NULL,
  [ShPT] DECIMAL(3, 2) CONSTRAINT [DF_SboSeamlessBets_ShPt] DEFAULT ((0)) NULL,
  [TotalPT] DECIMAL(3, 2) CONSTRAINT [DF_SboSeamlessBets_TotalPT] DEFAULT ((0)) NULL,
  [AgtWinlost] DECIMAL(19, 6) CONSTRAINT [DF_SboSeamlessBets_AgtWinlost] DEFAULT ((0)) NULL,
  [MaWinlost] DECIMAL(19, 6) CONSTRAINT [DF_SboSeamlessBets_MaWinlost] DEFAULT ((0)) NULL,
  [SmaWinlost] DECIMAL(19, 6) CONSTRAINT [DF_SboSeamlessBets_SmaWinlost] DEFAULT ((0)) NULL,
  [SsmaWinlost] DECIMAL(19, 6) CONSTRAINT [DF_SboSeamlessBets_SsmaWinlost] DEFAULT ((0)) NULL,
  [ShWinlost] DECIMAL(19, 6) CONSTRAINT [DF_SboSeamlessBets_ShSwinlost] DEFAULT ((0)) NULL,
  [PlayerCommissionRate] DECIMAL(5, 4) CONSTRAINT [DF_SboSeamlessBets_PlayerDiscount] DEFAULT ((0)) NULL,
  [AgtCommissionRate] DECIMAL(5, 4) CONSTRAINT [DF_SboSeamlessBets_AgtDiscount] DEFAULT ((0)) NULL,
  [MaCommissionRate] DECIMAL(5, 4) CONSTRAINT [DF_SboSeamlessBets_MaDiscount] DEFAULT ((0)) NULL,
  [SmaCommissionRate] DECIMAL(5, 4) CONSTRAINT [DF_SboSeamlessBets_SmaDiscount] DEFAULT ((0)) NULL,
  [SsmaCommissionRate] DECIMAL(5, 4) CONSTRAINT [DF_SboSeamlessBets_SsmaDiscount] DEFAULT ((0)) NULL,
  [ShCommissionRate] DECIMAL(5, 4) CONSTRAINT [DF_SboSeamlessBets_ShDiscount] DEFAULT ((0)) NULL,
  [PlayerComm] DECIMAL(19, 6) CONSTRAINT [DF_SboSeamlessBets_PlayerComm] DEFAULT ((0)) NULL,
  [AgtComm] DECIMAL(19, 6) CONSTRAINT [DF_SboSeamlessBets_AgtComm] DEFAULT ((0)) NULL,
  [MaComm] DECIMAL(19, 6) CONSTRAINT [DF_SboSeamlessBets_MaComm] DEFAULT ((0)) NULL,
  [SmaComm] DECIMAL(19, 6) CONSTRAINT [DF_SboSeamlessBets_SmaComm] DEFAULT ((0)) NULL,
  [SsmaComm] DECIMAL(19, 6) CONSTRAINT [DF_SboSeamlessBets_SsmaComm] DEFAULT ((0)) NULL,
  [ShComm] DECIMAL(19, 6) CONSTRAINT [DF_SboSeamlessBets_ShComm] DEFAULT ((0)) NULL,
  [SyncTime] DATETIME,
  [ModifiedTime] DATETIME,
  [ReferralLayerPercentageHistoryIds] NVARCHAR(1000),
  [IsDailyStatementProceed] BIT CONSTRAINT [DF_SboSeamlessBets_IsDailyStatementProceed] DEFAULT (0) NOT NULL,
  [IsFeatureBuy] BIT DEFAULT(0),
  [NewGameType] INT,
  [MainBetId] NVARCHAR(255) NULL,
  [GameExtraInfo] NVARCHAR(4000) NULL,
  [WinlostDateByTimeZone] DATETIME NULL,
  [TimeZone] NVARCHAR(100) NULL,
  [WithdrawalLimitId] INT CONSTRAINT [DF_SboSeamlessBets_WithdrawalLimitId] DEFAULT(0) NOT NULL,
  CONSTRAINT [PK_SboSeamlessBets] PRIMARY KEY ([Id], [WinLostDate]) WITH (FILLFACTOR = 85)
)
GO
CREATE NONCLUSTERED INDEX [IX_SboSeamlessBets_WinLostDate]
ON [dbo].[SboSeamlessBets] ([WebId],[CustomerId],[Winlostdate],[Status])
INCLUDE ([GPID],[Username],[Stake],[Currency],[ExchangeRate],[Winlost],[TotalPT],[AgtWinlost],[MaWinlost],[SmaWinlost],[SsmaWinlost],[ShWinlost],[PlayerComm],[AgtComm],[MaComm],[SmaComm],[SsmaComm],[ShComm],[Refno],[Parentid],[GameRoundId],[GameType],[GamePeriodId],[OrderTime],[TransactionId])
GO
CREATE NONCLUSTERED INDEX [IX_SboSeamlessBets_RefNo] ON [dbo].[SboSeamlessBets] ([RefNo] ASC)
GO
CREATE NONCLUSTERED INDEX [IX_SboSeamlessBets_Report]
ON [dbo].[SboSeamlessBets] ([WebId],[CustomerId],[OrderTime],[Status])
INCLUDE ([GPID],[Username],[Stake],[Currency],[ExchangeRate],[Winlost],[TotalPT],[AgtWinlost],[MaWinlost],[SmaWinlost],[SsmaWinlost],[ShWinlost],[PlayerComm],[AgtComm],[MaComm],[SmaComm],[SsmaComm],[ShComm],[Refno],[Parentid],[GameRoundId],[GameType],[GamePeriodId],[Winlostdate])
GO
CREATE NONCLUSTERED INDEX [IX_SboSeamlessBets_Synctime]
ON [dbo].[SboSeamlessBets] ([WebId],[CustomerId],[SyncTime])
INCLUDE ([OrderTime],[RefNo],[Username],[GamePeriodId],[GameRoundId],[GameType],[Stake],[TurnOver],[Currency],[OrderDetail],[GameResult],[WinLostDate],[Status],[Winlost],[GPID])
GO
CREATE NONCLUSTERED INDEX [IX_SboSeamlessBets_Synctime_GetBetListByModifiedDate]
ON [dbo].[SboSeamlessBets] ([WebId],[MaId],[SyncTime])
INCLUDE ([OrderTime],[ModifiedTime],[RefNo],[Username],[GamePeriodId],[GameRoundId],[GameType],[Stake],[TurnOver],[Currency],[OrderDetail],[GameResult],[WinLostDate],[Status],[Winlost],[GPID],[GameIdAtGameProvider])
GO
CREATE NONCLUSTERED INDEX [IX_SboSeamlessBets_GetSeamlessGameBetListByTransIds]
ON [dbo].[SboSeamlessBets] ([WebId],[RefNo])
INCLUDE ([OrderTime],[SyncTime],[ModifiedTime],[Username],[GamePeriodId],[GameRoundId],[GameType],[Stake],[TurnOver],[Currency],[OrderDetail],[GameResult],[WinLostDate],[Status],[Winlost],[GPID],[GameIdAtGameProvider])
GO
CREATE NONCLUSTERED INDEX [IX_SboSeamlessBets_GetBettingRecord]
ON [dbo].[SboSeamlessBets] ([WebId])
INCLUDE ([GPID],[RefNo],[CustomerId],[OrderTime],[WinLostDate],[Stake],[Currency],[ExchangeRate],[Status],[GameIdAtGameProvider],[Winlost],[PlayerComm])
GO
CREATE NONCLUSTERED INDEX [IX_SboSeamlessBets_Bonus]
ON [dbo].[SboSeamlessBets] ([WebId],[TransactionId],[Status])
```

### SboSeamlessRunningBets
Tracks active/running seamless betting transactions that are not yet settled.

```sql
CREATE TABLE [dbo].[SboSeamlessRunningBets] (
  [Id] BIGINT IDENTITY(1,1) NOT NULL,
  [GPID] INT NULL,
  [WebId] INT NOT NULL,
  [RefNo] NVARCHAR(255) NOT NULL,
  [TransactionId] NVARCHAR(100) NOT NULL,
  [CustomerId] INT NOT NULL,
  [ParentId] INT NULL,
  [MaId] INT DEFAULT ((0)) NOT NULL,
  [Username] NVARCHAR(50) NOT NULL,
  [UsernameAtGameProvider] NVARCHAR(50) NULL,
  [PromotionRequestId] INT DEFAULT ((0)) NOT NULL,
  [NewPromotionRequestId] INT DEFAULT ((0)) NOT NULL,
  [ReferralEventId] INT DEFAULT ((0)) NOT NULL,
  [WalletProductType] INT DEFAULT ((0)) NOT NULL,
  [WalletGameType] INT DEFAULT ((0)) NOT NULL,
  [OrderTime] DATETIME NOT NULL,
  [WinLostDate] DATETIME NULL,
  [Stake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Currency] CHAR(3) NULL,
  [ExchangeRate] DECIMAL(12, 8) NULL,
  [TurnOver] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [StatusAtGameProvider] VARCHAR(50) NULL,
  [IsFishGame] BIT DEFAULT ((0)) NULL,
  [GameIdAtGameProvider] INT DEFAULT ((0)) NOT NULL,
  [GamePeriodId] NVARCHAR(100) NULL,
  [GameRoundId] NVARCHAR(100) NULL,
  [GameType] NVARCHAR(50) NULL,
  [OrderDetail] NVARCHAR(4000) NULL,
  [GameResult] NVARCHAR(4000) NULL,
  -- Commission percentage tracking
  [AgtPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [MaPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [SmaPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [SsmaPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [ShPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [TotalPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  -- Commission rates
  [PlayerCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [AgtCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [MaCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [SmaCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [SsmaCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [ShCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  -- Additional tracking
  [SyncTime] DATETIME,
  [ModifiedTime] DATETIME,
  [ReferralLayerPercentageHistoryIds] NVARCHAR(1000),
  [IsFeatureBuy] BIT DEFAULT(0),
  [NewGameType] INT,
  [WinlostDateByTimeZone] DATETIME NULL,
  [TimeZone] NVARCHAR(100) NULL,
  [WithdrawalLimitId] INT DEFAULT(0) NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | BIGINT, PK | Primary key |
| GPID | INT | Game provider identifier |
| WebId | INT | Website identifier |
| RefNo | NVARCHAR(255) | Reference number |
| TransactionId | NVARCHAR(100) | Transaction identifier |
| CustomerId | INT | Customer identifier |
| ParentId | INT | Parent customer identifier |
| MaId | INT | Master agent identifier |
| Username | NVARCHAR(50) | Username |
| UsernameAtGameProvider | NVARCHAR(50) | Username at game provider |
| OrderTime | DATETIME | Order timestamp |
| WinLostDate | DATETIME | Win/loss date (null for running bets) |
| Stake | DECIMAL(19, 6) | Stake amount |
| Currency | CHAR(3) | Currency code |
| ExchangeRate | DECIMAL(12, 8) | Exchange rate |
| TurnOver | DECIMAL(19, 6) | Turnover amount |
| StatusAtGameProvider | VARCHAR(50) | Status at game provider |
| IsFishGame | BIT | Whether this is a fish game |
| GameIdAtGameProvider | INT | Game ID at provider |
| GamePeriodId | NVARCHAR(100) | Game period identifier |
| GameRoundId | NVARCHAR(100) | Game round identifier |
| GameType | NVARCHAR(50) | Game type |
| OrderDetail | NVARCHAR(4000) | Order details |
| GameResult | NVARCHAR(4000) | Game result |
| Commission Fields | Various | Commission percentages and rates by hierarchy level |
| IsFeatureBuy | BIT | Whether this is a feature buy transaction |
| NewGameType | INT | New game type classification |

### SboCasinoBets
Records casino game betting transactions.

```sql
CREATE TABLE [dbo].[SboCasinoBets] (
  -- Similar structure to SboSportBets but specifically for casino games
  [Id] BIGINT IDENTITY(1,1) NOT NULL,
  [WebId] INT NOT NULL,
  [RefNo] NVARCHAR(100) NOT NULL,
  [TransactionId] NVARCHAR(100) NOT NULL,
  [CustomerId] INT NOT NULL,
  [ParentId] INT NULL,
  [Username] NVARCHAR(50) NOT NULL,
  [UsernameAtGameProvider] NVARCHAR(50) NULL,
  -- Casino-specific fields
  [GameType] NVARCHAR(50) NULL,
  [TableId] NVARCHAR(50) NULL,
  [GameCode] NVARCHAR(50) NULL,
  [OrderTime] DATETIME NOT NULL,
  [ModifiedTime] DATETIME NOT NULL,
  [WinlostDate] DATETIME NULL,
  [Stake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ActualStake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Currency] CHAR(3) NULL,
  [ExchangeRate] DECIMAL(12, 8) NULL,
  [StatusAtGameProvider] VARCHAR(50) NULL,
  [Status] VARCHAR(50) NULL,
  [Winlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [TurnOver] DECIMAL(19, 6) DEFAULT ((0)) NULL
)
```

### SboGamesBets
Records game betting transactions for SBO games platform.

```sql
CREATE TABLE [dbo].[SboGamesBets] (
  -- Similar structure focusing on SBO games
  [Id] BIGINT IDENTITY(1,1) NOT NULL,
  [WebId] INT NOT NULL,
  [RefNo] NVARCHAR(100) NOT NULL,
  [TransactionId] NVARCHAR(100) NOT NULL,
  [CustomerId] INT NOT NULL,
  [ParentId] INT NULL,
  [Username] NVARCHAR(50) NOT NULL,
  [UsernameAtGameProvider] NVARCHAR(50) NULL,
  [GameType] NVARCHAR(50) NULL,
  [OrderTime] DATETIME NOT NULL,
  [ModifiedTime] DATETIME NOT NULL,
  [WinlostDate] DATETIME NULL,
  [Stake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ActualStake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Currency] CHAR(3) NULL,
  [ExchangeRate] DECIMAL(12, 8) NULL,
  [StatusAtGameProvider] VARCHAR(50) NULL,
  [Status] VARCHAR(50) NULL,
  [Winlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [TurnOver] DECIMAL(19, 6) DEFAULT ((0)) NULL
)
```

### SboVirtualSportBets
Records virtual sports betting transactions.

```sql
CREATE TABLE [dbo].[SboVirtualSportBets] (
  -- Virtual sports specific betting records
  [Id] BIGINT IDENTITY(1,1) NOT NULL,
  [WebId] INT NOT NULL,
  [RefNo] NVARCHAR(100) NOT NULL,
  [TransactionId] NVARCHAR(100) NOT NULL,
  [CustomerId] INT NOT NULL,
  [ParentId] INT NULL,
  [Username] NVARCHAR(50) NOT NULL,
  [UsernameAtGameProvider] NVARCHAR(50) NULL,
  [SportsType] NVARCHAR(50) NULL,
  [BetType] INT NULL,
  [OrderTime] DATETIME NOT NULL,
  [ModifiedTime] DATETIME NOT NULL,
  [WinlostDate] DATETIME NULL,
  [Odds] DECIMAL(12, 3) DEFAULT ((0)) NULL,
  [OddsStyle] NVARCHAR(10) NULL,
  [Stake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ActualStake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Currency] CHAR(3) NULL,
  [ExchangeRate] DECIMAL(12, 8) NULL,
  [StatusAtGameProvider] VARCHAR(50) NULL,
  [Status] VARCHAR(50) NULL,
  [Winlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [TurnOver] DECIMAL(19, 6) DEFAULT ((0)) NULL
)
```

### SboThirdPartySportBets
Records third-party sports betting transactions.

```sql
CREATE TABLE [dbo].[SboThirdPartySportBets] (
  -- Third-party sports betting records
  [Id] BIGINT IDENTITY(1,1) NOT NULL,
  [WebId] INT NOT NULL,
  [RefNo] NVARCHAR(100) NOT NULL,
  [TransactionId] NVARCHAR(100) NOT NULL,
  [CustomerId] INT NOT NULL,
  [ParentId] INT NULL,
  [Username] NVARCHAR(50) NOT NULL,
  [UsernameAtGameProvider] NVARCHAR(50) NULL,
  [SportsType] NVARCHAR(50) NULL,
  [BetType] INT NULL,
  [OrderTime] DATETIME NOT NULL,
  [ModifiedTime] DATETIME NOT NULL,
  [WinlostDate] DATETIME NULL,
  [Odds] DECIMAL(12, 3) DEFAULT ((0)) NULL,
  [Stake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ActualStake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Currency] CHAR(3) NULL,
  [ExchangeRate] DECIMAL(12, 8) NULL,
  [StatusAtGameProvider] VARCHAR(50) NULL,
  [Status] VARCHAR(50) NULL,
  [Winlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [TurnOver] DECIMAL(19, 6) DEFAULT ((0)) NULL
)
```

### SboThirdPartySportBetDetails
Detailed information for third-party sports betting transactions.

```sql
CREATE TABLE [dbo].[SboThirdPartySportBetDetails] (
  [Id] INT NOT NULL PRIMARY KEY IDENTITY(1,1),
  [RefNo] NVARCHAR(100) NOT NULL,
  [TransactionId] NVARCHAR(100) NOT NULL,
  [SubbetId] NVARCHAR(100) NULL,
  [ParlayRefno] NVARCHAR(100) NULL,
  [BetOption] NVARCHAR(150) NULL,
  [IsLive] BIT NOT NULL,
  [MarketType] NVARCHAR(150) NULL,
  [SportType] NVARCHAR(50) NULL,
  [HandicapPoint] DECIMAL(12, 2) DEFAULT ((0)) NULL,
  [Odds] DECIMAL(12, 3) DEFAULT ((0)) NULL,
  [League] NVARCHAR(150) NULL,
  [HomeTeam] NVARCHAR(150) NULL,
  [AwayTeam] NVARCHAR(150) NULL,
  [Status] VARCHAR(50) NULL,
  [WinlostDate] DATETIME NULL,
  [ModifiedTime] DATETIME NULL,
  [LiveHomeScore] INT NULL,
  [LiveAwayScore] INT NULL,
  [WebId] INT NOT NULL,
  [SyncTime] DATETIME,
  [SettlementTime] DATETIME NULL,
  [Language] NVARCHAR(10) DEFAULT('en') NOT NULL,
  [FullTimeScore] NVARCHAR(100) NULL,
  [HalfTimeScore] NVARCHAR(100) NULL,
  [WinlostDateByTimeZone] DATETIME NULL,
  [TimeZone] NVARCHAR(100) NULL,
  [WithdrawalLimitId] INT DEFAULT(0) NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Primary key |
| RefNo | NVARCHAR(100) | Reference number |
| TransactionId | NVARCHAR(100) | Transaction identifier |
| SubbetId | NVARCHAR(100) | Sub-bet identifier for parlay bets |
| ParlayRefno | NVARCHAR(100) | Parlay reference number |
| BetOption | NVARCHAR(150) | Betting option selected |
| IsLive | BIT | Whether this is a live bet |
| MarketType | NVARCHAR(150) | Market type |
| SportType | NVARCHAR(50) | Sport type |
| HandicapPoint | DECIMAL(12, 2) | Handicap point value |
| Odds | DECIMAL(12, 3) | Betting odds |
| League | NVARCHAR(150) | League name |
| HomeTeam | NVARCHAR(150) | Home team name |
| AwayTeam | NVARCHAR(150) | Away team name |
| Status | VARCHAR(50) | Bet status |
| WinlostDate | DATETIME | Win/loss date |
| ModifiedTime | DATETIME | Last modification time |
| LiveHomeScore | INT | Live home team score |
| LiveAwayScore | INT | Live away team score |
| WebId | INT | Website identifier |
| SyncTime | DATETIME | Synchronization time |
| SettlementTime | DATETIME | Settlement time |
| Language | NVARCHAR(10) | Language code |
| FullTimeScore | NVARCHAR(100) | Full time score |
| HalfTimeScore | NVARCHAR(100) | Half time score |
| WinlostDateByTimeZone | DATETIME | Win/loss date by timezone |
| TimeZone | NVARCHAR(100) | Timezone |
| WithdrawalLimitId | INT | Withdrawal limit identifier |

### SboVirtualSportSubBets
Sub-bet details for virtual sports betting transactions.

```sql
CREATE TABLE [dbo].[SboVirtualSportSubBets] (
  [Id] BIGINT NOT NULL IDENTITY(1,1) PRIMARY KEY,
  [WebId] INT NOT NULL,
  [RefNo] NVARCHAR(100) NOT NULL,
  [TransactionId] NVARCHAR(100) NOT NULL DEFAULT(0),
  [Match] NVARCHAR(100) NOT NULL,
  [WinLostDate] DATETIME NOT NULL,
  [Status] NVARCHAR(15) NOT NULL,
  [BetOption] NVARCHAR(30) NOT NULL,
  [MarketType] NVARCHAR(50) NOT NULL,
  [Hdp] NVARCHAR(10) NOT NULL,
  [Odds] DECIMAL(12,3) NOT NULL,
  [HalfTimeScore] NVARCHAR(20) NOT NULL,
  [FullTimeScore] NVARCHAR(20) NOT NULL,
  [SyncTime] DATETIME NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | BIGINT, PK | Primary key |
| WebId | INT | Website identifier |
| RefNo | NVARCHAR(100) | Reference number |
| TransactionId | NVARCHAR(100) | Transaction identifier |
| Match | NVARCHAR(100) | Match identifier |
| WinLostDate | DATETIME | Win/loss date |
| Status | NVARCHAR(15) | Bet status |
| BetOption | NVARCHAR(30) | Betting option |
| MarketType | NVARCHAR(50) | Market type |
| Hdp | NVARCHAR(10) | Handicap value |
| Odds | DECIMAL(12,3) | Betting odds |
| HalfTimeScore | NVARCHAR(20) | Half time score |
| FullTimeScore | NVARCHAR(20) | Full time score |
| SyncTime | DATETIME | Synchronization time |

### SboLiveCoin
Records live coin transactions for gaming activities.

```sql
CREATE TABLE [dbo].[SboLiveCoin] (
  [Id] BIGINT IDENTITY(1,1) NOT NULL,
  [WebId] INT NOT NULL,
  [RefNo] NVARCHAR(100) NOT NULL,
  [TransactionId] NVARCHAR(100) NOT NULL,
  [WinlostDate] DATETIME NOT NULL,
  [OrderTime] DATETIME NOT NULL,
  [Status] NVARCHAR(50),
  [CustomerId] INT NOT NULL,
  [ParentId] INT NULL,
  [Username] NVARCHAR(50),
  [Detail] NVARCHAR(100),
  [Stake] DECIMAL(19,6) NOT NULL,
  [Winlose] DECIMAL(19,6) NOT NULL,
  [Currency] CHAR(3) NOT NULL,
  [ExchangeRate] DECIMAL(12, 8) NULL,
  [CreatedOn] DATETIME NOT NULL DEFAULT(GETDATE()),
  [ModifiedOn] DATETIME NOT NULL DEFAULT(GETDATE()),
  [PromotionRequestId] INT DEFAULT ((0)) NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | BIGINT, PK | Primary key |
| WebId | INT | Website identifier |
| RefNo | NVARCHAR(100) | Reference number |
| TransactionId | NVARCHAR(100) | Transaction identifier |
| WinlostDate | DATETIME | Win/loss date |
| OrderTime | DATETIME | Order timestamp |
| Status | NVARCHAR(50) | Transaction status |
| CustomerId | INT | Customer identifier |
| ParentId | INT | Parent customer identifier |
| Username | NVARCHAR(50) | Username |
| Detail | NVARCHAR(100) | Transaction details |
| Stake | DECIMAL(19,6) | Stake amount |
| Winlose | DECIMAL(19,6) | Win/loss amount |
| Currency | CHAR(3) | Currency code |
| ExchangeRate | DECIMAL(12, 8) | Exchange rate |
| CreatedOn | DATETIME | Creation timestamp |
| ModifiedOn | DATETIME | Modification timestamp |
| PromotionRequestId | INT | Promotion request identifier |

### LionKingMJP
Records Lion King MJP (Multi-Jackpot Progressive) transactions.

```sql
CREATE TABLE [dbo].[LionKingMJP] (
  [Id] BIGINT IDENTITY(1,1) NOT NULL,
  [WebId] INT NOT NULL,
  [RefNo] NVARCHAR(100) NOT NULL,
  [CustomerId] INT NOT NULL,
  [MjpWinlose] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [MjpCommission] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ModifiedTime] DATETIME
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | BIGINT, PK | Primary key |
| WebId | INT | Website identifier |
| RefNo | NVARCHAR(100) | Reference number |
| CustomerId | INT | Customer identifier |
| MjpWinlose | DECIMAL(19, 6) | Multi-jackpot progressive win/loss amount |
| MjpCommission | DECIMAL(19, 6) | Multi-jackpot progressive commission |
| ModifiedTime | DATETIME | Last modification time |

### SboResettledBet
Records bet resettlement transactions with old and new values.

```sql
CREATE TABLE [dbo].[SboResettledBet] (
  [Id] INT IDENTITY(1,1) PRIMARY KEY,
  [RefNo] NVARCHAR(100) NOT NULL,
  [TransactionId] NVARCHAR(100) NOT NULL,
  [ProductType] INT NULL,
  [GameProviderId] INT NULL,
  [WebId] INT NOT NULL,
  [CustomerId] INT NOT NULL,
  [Username] NVARCHAR(50) NOT NULL,
  [OldStatus] VARCHAR(50) NULL,
  [NewStatus] VARCHAR(50) NULL,
  [OldWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [OldAgtWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [OldMaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [OldSmaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [OldSsmaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [OldShWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [OldPlayerComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [OldAgtComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [OldMaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [OldSmaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [OldSsmaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [OldShComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [OldStatusRemark] VARCHAR(50) NULL CHECK ([OldStatusRemark] IN ('Cashout', NULL)),
  [NewWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [NewAgtWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [NewMaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [NewSmaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [NewSsmaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [NewShWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [NewPlayerComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [NewAgtComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [NewMaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [NewSmaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [NewSsmaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [NewShComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Currency] NVARCHAR(3) NULL,
  [ExchangeRate] DECIMAL(10,6) NULL,
  [ResettleDate] DATETIME NULL,
  [WinlostDate] DATETIME NULL,
  [ModifiedOn] DATETIME NULL,
  [CreatedOn] DATETIME NOT NULL
)
```

## Historical Data Tables (60-Day Archives)

### SboVirtualSportBets60
Historical archive of virtual sports betting transactions (60-day retention).

```sql
CREATE TABLE [dbo].[SboVirtualSportBets60] (
  [Id] BIGINT NOT NULL,
  [WebId] INT NOT NULL,
  [RefNo] NVARCHAR(100) NOT NULL,
  [TransactionId] NVARCHAR(100) NOT NULL,
  [CustomerId] INT NOT NULL,
  [ParentId] INT NULL,
  [Username] NVARCHAR(50) NOT NULL,
  [UsernameAtGameProvider] NVARCHAR(50) NULL,
  [PromotionRequestId] INT DEFAULT ((0)) NOT NULL,
  [NewPromotionRequestId] INT DEFAULT ((0)) NOT NULL,
  [ReferralEventId] INT DEFAULT ((0)) NOT NULL,
  [WalletProductType] INT DEFAULT ((0)) NOT NULL,
  [WalletGameType] INT DEFAULT ((0)) NOT NULL,
  [GameId] INT NULL,
  [ProductType] NVARCHAR(50) NULL,
  [OrderTime] DATETIME NOT NULL,
  [ModifiedTime] DATETIME NOT NULL,
  [WinLostDate] DATETIME NULL,
  [Odds] DECIMAL(12,3) NULL,
  [OddsStyle] NVARCHAR(10) NULL,
  [Stake] DECIMAL(19,6) NULL,
  [ActualStake] DECIMAL(19,6) NOT NULL,
  [Turnover] DECIMAL(19,6) NULL,
  [WinLost] DECIMAL(19,6) NULL,
  [StatusAtGameProvider] VARCHAR(50) NULL,
  [Status] VARCHAR(50) NULL,
  [Currency] CHAR(3) NOT NULL,
  [ActualRate] DECIMAL(12,8) NOT NULL,
  -- Commission percentage tracking
  [AgtPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [MaPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [SmaPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [SsmaPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [ShPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [TotalPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  -- Win/loss by hierarchy level
  [AgtWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [MaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SmaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SsmaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ShWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  -- Commission rates and amounts
  [PlayerCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [AgtCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [MaCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [SmaCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [SsmaCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [ShCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [PlayerComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [AgtComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [MaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SmaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SsmaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ShComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  -- Additional tracking fields
  [SyncTime] DATETIME,
  [IsDailyStatementProceed] BIT DEFAULT (0) NOT NULL,
  [ReferralLayerPercentageHistoryIds] NVARCHAR(1000),
  [WinlostDateByTimeZone] DATETIME NULL,
  [TimeZone] NVARCHAR(100) NULL,
  [WithdrawalLimitId] INT DEFAULT(0) NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | BIGINT | Original record ID (not identity in archive) |
| WebId | INT | Website identifier |
| RefNo | NVARCHAR(100) | Reference number |
| TransactionId | NVARCHAR(100) | Transaction identifier |
| CustomerId | INT | Customer identifier |
| ParentId | INT | Parent customer identifier |
| Username | NVARCHAR(50) | Username |
| UsernameAtGameProvider | NVARCHAR(50) | Username at game provider |
| PromotionRequestId | INT | Original promotion request ID |
| NewPromotionRequestId | INT | New promotion request ID |
| ReferralEventId | INT | Referral event identifier |
| WalletProductType | INT | Wallet product type |
| WalletGameType | INT | Wallet game type |
| GameId | INT | Game identifier |
| ProductType | NVARCHAR(50) | Product type name |
| OrderTime | DATETIME | Order timestamp |
| ModifiedTime | DATETIME | Modification timestamp |
| WinLostDate | DATETIME | Win/loss date |
| Odds | DECIMAL(12,3) | Betting odds |
| OddsStyle | NVARCHAR(10) | Odds display style |
| Stake | DECIMAL(19,6) | Stake amount |
| ActualStake | DECIMAL(19,6) | Actual stake amount |
| Turnover | DECIMAL(19,6) | Turnover amount |
| WinLost | DECIMAL(19,6) | Win/loss amount |
| StatusAtGameProvider | VARCHAR(50) | Status at game provider |
| Status | VARCHAR(50) | Current status |
| Currency | CHAR(3) | Currency code |
| ActualRate | DECIMAL(12,8) | Actual exchange rate |
| AgtPT to ShPT | DECIMAL(3, 2) | Commission percentages by hierarchy level |
| AgtWinlost to ShWinlost | DECIMAL(19, 6) | Win/loss amounts by hierarchy level |
| Commission Rates | DECIMAL(5, 4) | Commission rates by hierarchy level |
| Commission Amounts | DECIMAL(19, 6) | Commission amounts by hierarchy level |
| SyncTime | DATETIME | Synchronization timestamp |
| IsDailyStatementProceed | BIT | Whether daily statement was processed |
| ReferralLayerPercentageHistoryIds | NVARCHAR(1000) | Referral percentage history IDs |
| WinlostDateByTimeZone | DATETIME | Win/loss date by timezone |
| TimeZone | NVARCHAR(100) | Timezone |
| WithdrawalLimitId | INT | Withdrawal limit identifier |

### SboSportBets60
Historical archive of sports betting transactions (60-day retention).

```sql
CREATE TABLE [dbo].[SboSportBets60] (
  -- Similar structure to SboSportBets but for historical data
  [Id] BIGINT NOT NULL,
  [WebId] INT NOT NULL,
  [RefNo] NVARCHAR(100) NOT NULL,
  [TransactionId] NVARCHAR(100) NOT NULL,
  [CustomerId] INT NOT NULL,
  [ParentId] INT NULL,
  [Username] NVARCHAR(50) NOT NULL,
  [UsernameAtGameProvider] NVARCHAR(50) NULL,
  [PromotionRequestId] INT DEFAULT ((0)) NOT NULL,
  [NewPromotionRequestId] INT DEFAULT ((0)) NOT NULL,
  [ReferralEventId] INT DEFAULT ((0)) NOT NULL,
  [WalletProductType] INT DEFAULT ((0)) NOT NULL,
  [WalletGameType] INT DEFAULT ((0)) NOT NULL,
  [SportsType] NVARCHAR(50) NULL,
  [BetType] INT NULL,
  [OrderTime] DATETIME NOT NULL,
  [ModifiedTime] DATETIME NOT NULL,
  [WinlostDate] DATETIME NULL,
  [Odds] DECIMAL(12, 3) DEFAULT ((0)) NULL,
  [OddsStyle] NVARCHAR(10) NULL,
  [Stake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ActualStake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Currency] CHAR(3) NULL,
  [ExchangeRate] DECIMAL(12, 8) NULL,
  [StatusAtGameProvider] VARCHAR(50) NULL,
  [Status] VARCHAR(50) NULL,
  [Winlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [TurnOver] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [IsLive] BIT NULL,
  [IsHalfWinLose] BIT NULL,
  [MaxWinWithoutActualStake] DECIMAL(19,6),
  -- Commission and hierarchy tracking
  [AgtPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [MaPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [SmaPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [SsmaPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [ShPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [TotalPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [AgtWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [MaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SmaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SsmaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ShWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  -- Discount rates for sports betting
  [PlayerDiscount] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [AgtDiscount] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [MaDiscount] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [SmaDiscount] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [SsmaDiscount] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [ShDiscount] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  -- Commission amounts
  [PlayerComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [AgtComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [MaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SmaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SsmaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ShComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  -- Additional tracking
  [Ip] NVARCHAR(100) DEFAULT ('') NULL,
  [IsSystemTagRisky] BIT DEFAULT (0) NULL,
  [IsCashOut] BIT DEFAULT (0) NULL,
  [SyncTime] DATETIME,
  [BetFrom] NVARCHAR(20) DEFAULT '' NOT NULL,
  [ReferralLayerPercentageHistoryIds] NVARCHAR(1000),
  [VoidReason] NVARCHAR(500) DEFAULT ('') NULL,
  [Remark] NVARCHAR(500) DEFAULT ('') NULL,
  [WinlostDateByTimeZone] DATETIME NULL,
  [TimeZone] NVARCHAR(100) NULL,
  [SportGameId] NVARCHAR(100) NULL,
  [WithdrawalLimitId] INT DEFAULT(0) NOT NULL
)
```

### SboLiveCoin60
Historical archive of live coin transactions (60-day retention).

```sql
CREATE TABLE [dbo].[SboLiveCoin60] (
  -- Similar structure to SboLiveCoin but for historical data
  [Id] BIGINT NOT NULL,
  [WebId] INT NOT NULL,
  [RefNo] NVARCHAR(100) NOT NULL,
  [TransactionId] NVARCHAR(100) NOT NULL,
  [WinlostDate] DATETIME NOT NULL,
  [OrderTime] DATETIME NOT NULL,
  [Status] NVARCHAR(50),
  [CustomerId] INT NOT NULL,
  [ParentId] INT NULL,
  [Username] NVARCHAR(50),
  [Detail] NVARCHAR(100),
  [Stake] DECIMAL(19,6) NOT NULL,
  [Winlose] DECIMAL(19,6) NOT NULL,
  [Currency] CHAR(3) NOT NULL,
  [ExchangeRate] DECIMAL(12, 8) NULL,
  [CreatedOn] DATETIME NOT NULL,
  [ModifiedOn] DATETIME NOT NULL,
  [PromotionRequestId] INT DEFAULT ((0)) NOT NULL
)
```

### LionKingMJP60
Historical archive of Lion King MJP transactions (60-day retention).

```sql
CREATE TABLE [dbo].[LionKingMJP60] (
  -- Similar structure to LionKingMJP but for historical data
  [Id] BIGINT NOT NULL,
  [WebId] INT NOT NULL,
  [RefNo] NVARCHAR(100) NOT NULL,
  [CustomerId] INT NOT NULL,
  [MjpWinlose] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [MjpCommission] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ModifiedTime] DATETIME
)
```

### SboCasinoBets60
Historical archive of casino betting transactions (60-day retention).

```sql
CREATE TABLE [dbo].[SboCasinoBets60] (
  [Id] BIGINT NOT NULL,
  [WebId] INT NOT NULL,
  [RefNo] NVARCHAR(100) NOT NULL,
  [TransactionId] NVARCHAR(100) NOT NULL,
  [CustomerId] INT NOT NULL,
  [ParentId] INT NULL,
  [Username] NVARCHAR(50) NOT NULL,
  [UsernameAtGameProvider] NVARCHAR(50) NULL,
  [PromotionRequestId] INT DEFAULT ((0)) NOT NULL,
  [NewPromotionRequestId] INT DEFAULT ((0)) NOT NULL,
  [ReferralEventId] INT DEFAULT ((0)) NOT NULL,
  [WalletProductType] INT DEFAULT ((0)) NOT NULL,
  [WalletGameType] INT DEFAULT ((0)) NOT NULL,
  [ProductType] NVARCHAR(50) NULL,
  [GameId] BIGINT NULL,
  [TableName] NVARCHAR(100) NULL,
  [OrderTime] DATETIME NOT NULL,
  [WinLostDate] DATETIME NULL,
  [Stake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Currency] CHAR(3) NULL,
  [ExchangeRate] DECIMAL(12, 8) NULL,
  [TurnOver] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [StatusAtGameProvider] VARCHAR(50) NULL,
  [Status] VARCHAR(50) NULL,
  [Winlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  -- Commission and hierarchy tracking
  [AgtPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [MaPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [SmaPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [SsmaPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [ShPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [TotalPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [AgtWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [MaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SmaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SsmaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ShWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  -- Commission rates and amounts
  [PlayerCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [AgtCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [MaCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [SmaCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [SsmaCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [ShCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [PlayerComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [AgtComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [MaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SmaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SsmaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ShComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  -- Additional tracking
  [SyncTime] DATETIME,
  [ModifiedTime] DATETIME,
  [IsDailyStatementProceed] BIT DEFAULT (0) NOT NULL,
  [ReferralLayerPercentageHistoryIds] NVARCHAR(1000),
  [WinlostDateByTimeZone] DATETIME NULL,
  [TimeZone] NVARCHAR(100) NULL,
  [WithdrawalLimitId] INT DEFAULT(0) NOT NULL
)
```

### SboSeamlessBets60
Historical archive of seamless betting transactions (60-day retention).

```sql
CREATE TABLE [dbo].[SboSeamlessBets60] (
  [Id] BIGINT IDENTITY(1,1) NOT NULL,
  [GPID] INT NULL,
  [WebId] INT NOT NULL,
  [RefNo] NVARCHAR(255) NOT NULL,
  [TransactionId] NVARCHAR(255) NOT NULL,
  [CustomerId] INT NOT NULL,
  [ParentId] INT NULL,
  [MaId] INT DEFAULT ((0)) NOT NULL,
  [Username] NVARCHAR(50) NOT NULL,
  [UsernameAtGameProvider] NVARCHAR(50) NULL,
  [PromotionRequestId] INT DEFAULT ((0)) NOT NULL,
  [NewPromotionRequestId] INT DEFAULT ((0)) NOT NULL,
  [ReferralEventId] INT DEFAULT ((0)) NOT NULL,
  [WalletProductType] INT DEFAULT ((0)) NOT NULL,
  [WalletGameType] INT DEFAULT ((0)) NOT NULL,
  [OrderTime] DATETIME NOT NULL,
  [WinLostDate] DATETIME NOT NULL,
  [Stake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Currency] CHAR(3) NULL,
  [ExchangeRate] DECIMAL(12, 8) NULL,
  [TurnOver] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [StatusAtGameProvider] VARCHAR(50) NULL,
  [Status] VARCHAR(50) NULL,
  [IsFishGame] BIT DEFAULT ((0)) NULL,
  [GameIdAtGameProvider] INT DEFAULT ((0)) NOT NULL,
  [GamePeriodId] NVARCHAR(100) NULL,
  [GameRoundId] NVARCHAR(100) NULL,
  [GameType] NVARCHAR(50) NULL,
  [OrderDetail] NVARCHAR(4000) NULL,
  [GameResult] NVARCHAR(4000) NULL,
  [Winlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SyncTime] DATETIME,
  [ModifiedTime] DATETIME,
  [DataVersion] INT DEFAULT 2 -- Enhanced version indicator
)
```

### SboSeamlessBets90Temp
Extended 90-day temporary archive for seamless bets data.

```sql
CREATE TABLE [dbo].[SboSeamlessBets90Temp] (
  -- Extended retention temporary table for seamless bets
  -- Similar structure to SboSeamlessBets60 but with 90-day retention
  [Id] BIGINT NOT NULL,
  [GPID] INT NULL,
  [WebId] INT NOT NULL,
  [RefNo] NVARCHAR(255) NOT NULL,
  [TransactionId] NVARCHAR(255) NOT NULL,
  [CustomerId] INT NOT NULL,
  [ParentId] INT NULL,
  [MaId] INT DEFAULT ((0)) NOT NULL,
  [Username] NVARCHAR(50) NOT NULL,
  [UsernameAtGameProvider] NVARCHAR(50) NULL,
  [OrderTime] DATETIME NOT NULL,
  [WinLostDate] DATETIME NOT NULL,
  [Stake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Currency] CHAR(3) NULL,
  [ExchangeRate] DECIMAL(12, 8) NULL,
  [TurnOver] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [StatusAtGameProvider] VARCHAR(50) NULL,
  [Status] VARCHAR(50) NULL,
  [Winlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [TempRetentionDate] DATETIME DEFAULT DATEADD(DAY, 90, GETDATE())
)
```

## User and Registration Tables

### SboRegisteredUser
Tracks registered users for SBO services.

```sql
CREATE TABLE [dbo].[SboRegisteredUser] (
  [Id] INT NOT NULL IDENTITY(1,1),
  [WebId] INT NOT NULL,
  [CustomerId] INT NOT NULL,
  [AccountType] TINYINT NOT NULL,
  [Currency] CHAR(3) NOT NULL,
  [CreatedOn] DATETIME DEFAULT GETDATE() NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Primary key |
| WebId | INT | Website identifier |
| CustomerId | INT | Customer identifier |
| AccountType | TINYINT | Type of account |
| Currency | CHAR(3) | Default currency |
| CreatedOn | DATETIME | Registration timestamp |

## Summary and Reporting Tables

### DailyBetSummary
Aggregated daily betting summary across all product types and providers.

```sql
CREATE TABLE [dbo].[DailyBetSummary] (
  [Id] BIGINT IDENTITY(1,1) NOT NULL,
  [WebId] INT,
  [ProductType] INT,
  [DisplayName] NVARCHAR(50),
  [CustomerId] INT,
  [ParentId] INT,
  [Currency] CHAR(3),
  [Username] NVARCHAR(50),
  [Winlostdate] DATETIME,
  [BetCount] INT,
  [Stake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ActualStake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Turnover] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [NetTurnover] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [TurnoverByStake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [TurnoverByActualStake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [NetTurnoverByStake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [NetTurnoverByActualStake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Status] NVARCHAR(50),
  [ExchangeRate] DECIMAL(12, 8) NULL,
  [PlayerComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [AgtComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [MaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SmaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SsmaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ShComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Winlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [AgtWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [MaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SmaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SsmaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ShWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [CompanyWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [RowModifiedDate] DATETIME,
  [RowModifiedBy] NVARCHAR(50),
  [IsValidTurnover] BIT DEFAULT ((1)) NOT NULL,
  [NewPromotionRequestId] INT DEFAULT (0) NOT NULL,
  [WithdrawalLimitId] INT DEFAULT (0) NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | BIGINT, PK | Primary key |
| WebId | INT | Website identifier |
| ProductType | INT | Product type (1=Sports, 2=Casino, 3=Games, 4=Virtual Sports, 5=Seamless, 6=Third Party Sports, 7=Live Coin) |
| DisplayName | NVARCHAR(50) | Display name for the product/provider |
| CustomerId | INT | Customer identifier |
| ParentId | INT | Parent customer identifier |
| Currency | CHAR(3) | Currency code |
| Username | NVARCHAR(50) | Username |
| Winlostdate | DATETIME | Win/loss date |
| BetCount | INT | Number of bets |
| Stake | DECIMAL(19, 6) | Total stake amount |
| ActualStake | DECIMAL(19, 6) | Actual stake amount |
| Turnover | DECIMAL(19, 6) | Total turnover |
| NetTurnover | DECIMAL(19, 6) | Net turnover |
| TurnoverByStake | DECIMAL(19, 6) | Turnover calculated by stake |
| TurnoverByActualStake | DECIMAL(19, 6) | Turnover calculated by actual stake |
| NetTurnoverByStake | DECIMAL(19, 6) | Net turnover by stake |
| NetTurnoverByActualStake | DECIMAL(19, 6) | Net turnover by actual stake |
| Status | NVARCHAR(50) | Bet status |
| ExchangeRate | DECIMAL(12, 8) | Currency exchange rate |
| PlayerComm | DECIMAL(19, 6) | Player commission |
| AgtComm | DECIMAL(19, 6) | Agent commission |
| MaComm | DECIMAL(19, 6) | Master agent commission |
| SmaComm | DECIMAL(19, 6) | Super master agent commission |
| SsmaComm | DECIMAL(19, 6) | Super super master agent commission |
| ShComm | DECIMAL(19, 6) | Shareholder commission |
| Winlost | DECIMAL(19, 6) | Player win/loss amount |
| AgtWinlost | DECIMAL(19, 6) | Agent win/loss amount |
| MaWinlost | DECIMAL(19, 6) | Master agent win/loss amount |
| SmaWinlost | DECIMAL(19, 6) | Super master agent win/loss amount |
| SsmaWinlost | DECIMAL(19, 6) | Super super master agent win/loss amount |
| ShWinlost | DECIMAL(19, 6) | Shareholder win/loss amount |
| CompanyWinlost | DECIMAL(19, 6) | Company win/loss amount |
| RowModifiedDate | DATETIME | Row modification date |
| RowModifiedBy | NVARCHAR(50) | Who modified the row |
| IsValidTurnover | BIT | Whether turnover is valid |
| NewPromotionRequestId | INT | New promotion request identifier |
| WithdrawalLimitId | INT | Withdrawal limit identifier |

### DailyBetSummaryCurrent
Current daily betting summary for real-time reporting.

```sql
CREATE TABLE [dbo].[DailyBetSummaryCurrent] (
  -- Similar structure to DailyBetSummary but for current/real-time data
  [Id] BIGINT IDENTITY(1,1) NOT NULL,
  [WebId] INT,
  [ProductType] INT,
  [DisplayName] NVARCHAR(50),
  [CustomerId] INT,
  [ParentId] INT,
  [Currency] CHAR(3),
  [Username] NVARCHAR(50),
  [Winlostdate] DATETIME,
  [BetCount] INT,
  [Stake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ActualStake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Turnover] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [NetTurnover] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Status] NVARCHAR(50),
  [ExchangeRate] DECIMAL(12, 8) NULL,
  [PlayerComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Winlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [CompanyWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [RowModifiedDate] DATETIME,
  [RowModifiedBy] NVARCHAR(50),
  [IsValidTurnover] BIT DEFAULT ((1)) NOT NULL,
  [NewPromotionRequestId] INT DEFAULT (0) NOT NULL,
  [WithdrawalLimitId] INT DEFAULT (0) NOT NULL
)
```

### DailyBetSummaryByTimeZone
Timezone-aware daily betting summary for international operations.

```sql
CREATE TABLE [dbo].[DailyBetSummaryByTimeZone] (
  [Id] BIGINT IDENTITY(1,1) NOT NULL,
  [WebId] INT,
  [ProductType] INT,
  [DisplayName] NVARCHAR(50),
  [CustomerId] INT,
  [ParentId] INT,
  [Currency] CHAR(3),
  [Username] NVARCHAR(50),
  [BetCount] INT,
  [Stake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ActualStake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Turnover] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [NetTurnover] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [TurnoverByStake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [TurnoverByActualStake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [NetTurnoverByStake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [NetTurnoverByActualStake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Status] NVARCHAR(50),
  [ExchangeRate] DECIMAL(12, 8) NULL,
  [PlayerComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [AgtComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [MaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SmaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SsmaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ShComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Winlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [AgtWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [MaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SmaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SsmaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ShWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [CompanyWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [RowModifiedDate] DATETIME,
  [RowModifiedBy] NVARCHAR(50),
  [IsValidTurnover] BIT DEFAULT ((1)) NOT NULL,
  [NewPromotionRequestId] INT DEFAULT (0) NOT NULL,
  [WinlostDateByTimeZone] DATETIME NULL,
  [TimeZone] NVARCHAR(100) NULL,
  [WithdrawalLimitId] INT DEFAULT (0) NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | BIGINT, PK | Primary key |
| WebId | INT | Website identifier |
| ProductType | INT | Product type (1=Sports, 2=Casino, 3=Games, 4=Virtual Sports, 5=Seamless, 6=Third Party Sports, 7=Live Coin) |
| DisplayName | NVARCHAR(50) | Display name for the product/provider |
| CustomerId | INT | Customer identifier |
| ParentId | INT | Parent customer identifier |
| Currency | CHAR(3) | Currency code |
| Username | NVARCHAR(50) | Username |
| BetCount | INT | Number of bets |
| Stake | DECIMAL(19, 6) | Total stake amount |
| ActualStake | DECIMAL(19, 6) | Actual stake amount |
| Turnover | DECIMAL(19, 6) | Total turnover |
| NetTurnover | DECIMAL(19, 6) | Net turnover |
| TurnoverByStake | DECIMAL(19, 6) | Turnover calculated by stake |
| TurnoverByActualStake | DECIMAL(19, 6) | Turnover calculated by actual stake |
| NetTurnoverByStake | DECIMAL(19, 6) | Net turnover by stake |
| NetTurnoverByActualStake | DECIMAL(19, 6) | Net turnover by actual stake |
| Status | NVARCHAR(50) | Bet status |
| ExchangeRate | DECIMAL(12, 8) | Currency exchange rate |
| PlayerComm to ShComm | DECIMAL(19, 6) | Commission amounts by hierarchy level |
| Winlost to ShWinlost | DECIMAL(19, 6) | Win/loss amounts by hierarchy level |
| CompanyWinlost | DECIMAL(19, 6) | Company win/loss amount |
| RowModifiedDate | DATETIME | Row modification date |
| RowModifiedBy | NVARCHAR(50) | Who modified the row |
| IsValidTurnover | BIT | Whether turnover is valid |
| NewPromotionRequestId | INT | New promotion request identifier |
| WinlostDateByTimeZone | DATETIME | Win/loss date adjusted for timezone |
| TimeZone | NVARCHAR(100) | Timezone specification |
| WithdrawalLimitId | INT | Withdrawal limit identifier |

### DailySeamlessGamesBetsSummary
Daily summary specifically for seamless games betting.

```sql
CREATE TABLE [dbo].[DailySeamlessGamesBetsSummary] (
  [Id] BIGINT IDENTITY(1,1) NOT NULL,
  [WebId] INT,
  [GPID] INT,
  [DisplayName] NVARCHAR(50),
  [CustomerId] INT,
  [ParentId] INT,
  [Currency] CHAR(3),
  [Username] NVARCHAR(50),
  [Winlostdate] DATETIME,
  [BetCount] INT,
  [Stake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Turnover] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Status] NVARCHAR(50),
  [ExchangeRate] DECIMAL(12, 8) NULL,
  [PlayerComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Winlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [CompanyWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [RowModifiedDate] DATETIME,
  [RowModifiedBy] NVARCHAR(50),
  [IsValidTurnover] BIT DEFAULT ((1)) NOT NULL,
  [NewPromotionRequestId] INT DEFAULT (0) NOT NULL,
  [WithdrawalLimitId] INT DEFAULT (0) NOT NULL
)
```

## Compliance and Monitoring Tables

### GameComplianceThresholdSettings
Settings for game compliance thresholds to monitor player activity.

```sql
CREATE TABLE [dbo].[GameComplianceThresholdSettings] (
  [Id] BIGINT IDENTITY (1, 1),
  [CustomerId] NVARCHAR(20),
  [WebId] NVARCHAR(20),
  [AccountType] INT,
  [ProductType] INT,
  [ComplianceTypes] NVARCHAR(20),
  [GpId] NVARCHAR(20),
  [GameId] NVARCHAR(20),
  [CustomerMaxWin] DECIMAL(19,6),
  [CustomerMaxLose] DECIMAL(19,6),
  [TurnoverThreshold] DECIMAL(19,6),
  [CustomerMaxTurnover] DECIMAL(19,6),
  [CustomerMinTurnover] DECIMAL(19,6),
  [ActionType] NVARCHAR(50) CHECK (ActionType IN ('SuspendbyWinLost', 'AlertbyWinLost', 'SuspendbyTurnover', 'AlertbyTurnover')),
  [Remark] NVARCHAR(50) NULL,
  [CreatedOn] DATETIME,
  [CreatedBy] NVARCHAR(50),
  [ModifiedOn] DATETIME,
  [ModifiedBy] NVARCHAR(50)
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | BIGINT, PK | Primary key |
| CustomerId | NVARCHAR(20) | Customer identifier |
| WebId | NVARCHAR(20) | Website identifier |
| AccountType | INT | Account type |
| ProductType | INT | Product type |
| ComplianceTypes | NVARCHAR(20) | Compliance type identifier |
| GpId | NVARCHAR(20) | Game provider identifier |
| GameId | NVARCHAR(20) | Game identifier |
| CustomerMaxWin | DECIMAL(19,6) | Maximum win threshold for customer |
| CustomerMaxLose | DECIMAL(19,6) | Maximum loss threshold for customer |
| TurnoverThreshold | DECIMAL(19,6) | Turnover threshold |
| CustomerMaxTurnover | DECIMAL(19,6) | Maximum turnover threshold |
| CustomerMinTurnover | DECIMAL(19,6) | Minimum turnover threshold |
| ActionType | NVARCHAR(50) | Action to take when threshold is exceeded |
| Remark | NVARCHAR(50) | Additional remarks |
| CreatedOn | DATETIME | Creation timestamp |
| CreatedBy | NVARCHAR(50) | Creator |
| ModifiedOn | DATETIME | Modification timestamp |
| ModifiedBy | NVARCHAR(50) | Modifier |

### GameComplianceThresholdAction
Actions taken when compliance thresholds are triggered.

```sql
CREATE TABLE [dbo].[GameComplianceThresholdAction] (
  [Id] BIGINT IDENTITY (1, 1) PRIMARY KEY,
  [CustomerId] NVARCHAR(20),
  [WebId] NVARCHAR(20),
  [AccountType] INT,
  [ProductType] INT,
  [ComplianceTypes] NVARCHAR(20),
  [GpId] NVARCHAR(20),
  [GameId] NVARCHAR(20),
  [ActionType] NVARCHAR(50),
  [IsActive] BIT DEFAULT 1,
  [Remark] NVARCHAR(50) NULL,
  [CreatedOn] DATETIME,
  [CreatedBy] NVARCHAR(50),
  [ModifiedOn] DATETIME,
  [ModifiedBy] NVARCHAR(50)
)
```

### GameComplianceThresholdActionHistory
Historical record of compliance actions taken.

```sql
CREATE TABLE [dbo].[GameComplianceThresholdActionHistory] (
  [Id] BIGINT IDENTITY (1, 1) PRIMARY KEY,
  [GameComplianceThresholdActionId] BIGINT,
  [CustomerId] NVARCHAR(20),
  [WebId] NVARCHAR(20),
  [AccountType] INT,
  [ProductType] INT,
  [ComplianceTypes] NVARCHAR(20),
  [GpId] NVARCHAR(20),
  [GameId] NVARCHAR(20),
  [ActionType] NVARCHAR(50),
  [Remark] NVARCHAR(50) NULL,
  [CreatedOn] DATETIME,
  [CreatedBy] NVARCHAR(50)
)
```

### BonusThresholdAlertHistory
Historical record of bonus threshold alerts.

```sql
CREATE TABLE [dbo].[BonusThresholdAlertHistory] (
  [Id] BIGINT IDENTITY(1,1) NOT NULL PRIMARY KEY,
  [GameProviderId] INT NOT NULL,
  [RefNo] NVARCHAR(255) NOT NULL,
  [BonusThreshold] DECIMAL(19, 6) NOT NULL,
  [PlayerBonusSGD] DECIMAL(19, 6) NOT NULL,
  [IsNeedSuspendBonus] BIT NOT NULL DEFAULT 0,
  [CreatedOn] DATETIME NOT NULL DEFAULT GETDATE(),
  [CreatedBy] NVARCHAR(100) NOT NULL,
  [IsExceedThreshold] BIT NOT NULL DEFAULT 0
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | BIGINT, PK | Primary key |
| GameProviderId | INT | Game provider identifier |
| RefNo | NVARCHAR(255) | Reference number |
| BonusThreshold | DECIMAL(19, 6) | Bonus threshold amount |
| PlayerBonusSGD | DECIMAL(19, 6) | Player bonus amount in SGD |
| IsNeedSuspendBonus | BIT | Whether bonus needs to be suspended |
| CreatedOn | DATETIME | Creation timestamp |
| CreatedBy | NVARCHAR(100) | Creator |
| IsExceedThreshold | BIT | Whether threshold was exceeded |

### ProviderBounsThreshold
Bonus threshold settings for game providers.

```sql
CREATE TABLE [dbo].[ProviderBounsThreshold] (
  [Id] BIGINT IDENTITY(1,1) NOT NULL PRIMARY KEY,
  [GameProviderId] INT NOT NULL,
  [BonusThreshold] DECIMAL(19,6) NOT NULL,
  [IsEnabled] BIT NOT NULL DEFAULT 0,
  [IsNeedSuspendBonus] BIT NOT NULL DEFAULT 0,
  [Remark] NVARCHAR(100) NOT NULL,
  [CreatedBy] NVARCHAR(100) NOT NULL,
  [CreatedOn] DATETIME NOT NULL DEFAULT GETDATE(),
  [ModifiedBy] NVARCHAR(100) NOT NULL,
  [ModifiedOn] DATETIME NOT NULL DEFAULT GETDATE()
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | BIGINT, PK | Primary key |
| GameProviderId | INT | Game provider identifier |
| BonusThreshold | DECIMAL(19,6) | Bonus threshold amount |
| IsEnabled | BIT | Whether threshold is enabled |
| IsNeedSuspendBonus | BIT | Whether to suspend bonus when exceeded |
| Remark | NVARCHAR(100) | Remarks |
| CreatedBy | NVARCHAR(100) | Creator |
| CreatedOn | DATETIME | Creation timestamp |
| ModifiedBy | NVARCHAR(100) | Modifier |
| ModifiedOn | DATETIME | Modification timestamp |

## Specialized Regional Tables

### PeruPayoutId
Manages payout identifiers for Peru region.

```sql
CREATE TABLE [dbo].[PeruPayoutId] (
  [Id] INT PRIMARY KEY IDENTITY(1,1),
  [WebId] INT,
  [CustomerId] INT NOT NULL,
  [Username] NVARCHAR(100),
  [ModifiedOn] DATETIME,
  [ModifiedBy] NVARCHAR(50),
  [CreatedOn] DATETIME,
  [CreatedBy] NVARCHAR(50)
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Primary key |
| WebId | INT | Website identifier |
| CustomerId | INT | Customer identifier |
| Username | NVARCHAR(100) | Username |
| ModifiedOn | DATETIME | Last modification timestamp |
| ModifiedBy | NVARCHAR(50) | Who last modified |
| CreatedOn | DATETIME | Creation timestamp |
| CreatedBy | NVARCHAR(50) | Creator |

### PeruShopBetRecord
Records betting activities for Peru shop operations.

```sql
CREATE TABLE [dbo].[PeruShopBetRecord] (
  [Id] BIGINT IDENTITY(1,1) NOT NULL,
  [WebId] INT NOT NULL,
  [CustomerId] INT NOT NULL,
  [Username] NVARCHAR(100),
  [PayoutId] BIGINT NOT NULL,
  [Refno] NVARCHAR(100) NOT NULL,
  [ProductType] INT NOT NULL,
  [CreatedOn] DATETIME NOT NULL DEFAULT(GETDATE())
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | BIGINT, PK | Primary key |
| WebId | INT | Website identifier |
| CustomerId | INT | Customer identifier |
| Username | NVARCHAR(100) | Username |
| PayoutId | BIGINT | Payout identifier |
| Refno | NVARCHAR(100) | Reference number |
| ProductType | INT | Product type |
| CreatedOn | DATETIME | Creation timestamp |

### RedeemableRefno
Manages redeemable reference numbers for ticket operations.

```sql
CREATE TABLE [dbo].[RedeemableRefno] (
  [Id] BIGINT IDENTITY(1,1) NOT NULL,
  [WebId] INT NOT NULL,
  [RefNo] NVARCHAR(100) NOT NULL,
  [PrintCount] INT NOT NULL,
  [OperatorId] INT NOT NULL,
  [CreatedOn] DATETIME,
  [CreatedBy] NVARCHAR(100),
  [ModifiedOn] DATETIME,
  [ModifiedBy] NVARCHAR(100),
  [OperationStatus] NVARCHAR(100) NULL CHECK (OperationStatus IN ('Unpaid', 'Paid', 'Redeemed','Refunded'))
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | BIGINT, PK | Primary key |
| WebId | INT | Website identifier |
| RefNo | NVARCHAR(100) | Reference number |
| PrintCount | INT | Number of times printed |
| OperatorId | INT | Operator identifier |
| CreatedOn | DATETIME | Creation timestamp |
| CreatedBy | NVARCHAR(100) | Creator |
| ModifiedOn | DATETIME | Modification timestamp |
| ModifiedBy | NVARCHAR(100) | Modifier |
| OperationStatus | NVARCHAR(100) | Operation status (Unpaid, Paid, Redeemed, Refunded) |

## Supporting Tables

### PromotedLeague
Information about promoted leagues for sports betting.

```sql
CREATE TABLE [dbo].[PromotedLeague] (
  [Id] INT IDENTITY(1,1) NOT NULL PRIMARY KEY,
  [WebId] INT NOT NULL,
  [LeagueId] INT NOT NULL,
  [LeagueName] NVARCHAR(255) NOT NULL,
  [SportsType] NVARCHAR(50) NOT NULL,
  [DisplayOrder] INT NOT NULL DEFAULT 0,
  [IsEnabled] BIT NOT NULL DEFAULT 1,
  [CreatedOn] DATETIME NOT NULL DEFAULT GETDATE(),
  [CreatedBy] NVARCHAR(50) NOT NULL,
  [ModifiedOn] DATETIME NOT NULL DEFAULT GETDATE(),
  [ModifiedBy] NVARCHAR(50) NOT NULL
)
```

### SboSportsTypeOrder
Defines the display order for different sports types in the SBO platform.

```sql
CREATE TABLE [dbo].[SboSportsTypeOrder] (
  [ID] INT IDENTITY(1, 1) NOT NULL PRIMARY KEY,
  [WebId] INT NOT NULL DEFAULT 0,
  [SportType] NVARCHAR(50) NOT NULL,
  [SportTypeText] NVARCHAR(50) NOT NULL,
  [DisplayOrder] INT NOT NULL,
  [CreatedOn] DATETIME NULL,
  [CreatedBy] NVARCHAR(50) NULL,
  [ModifiedOn] DATETIME NULL,
  [ModifiedBy] NVARCHAR(50) NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| ID | INT, PK | Primary key |
| WebId | INT | Website identifier |
| SportType | NVARCHAR(50) | Sport type code |
| SportTypeText | NVARCHAR(50) | Display text for sport type |
| DisplayOrder | INT | Order for display purposes |
| CreatedOn | DATETIME | Creation timestamp |
| CreatedBy | NVARCHAR(50) | Creator |
| ModifiedOn | DATETIME | Modification timestamp |
| ModifiedBy | NVARCHAR(50) | Modifier |

### ReportPlayerList
List of players for reporting purposes.

```sql
CREATE TABLE [dbo].[ReportPlayerList] (
  [Id] BIGINT IDENTITY(1,1) NOT NULL PRIMARY KEY,
  [WebId] INT NOT NULL,
  [CustomerId] INT NOT NULL,
  [Username] NVARCHAR(50) NOT NULL,
  [ParentId] INT NULL,
  [Currency] CHAR(3) NULL,
  [CreatedOn] DATETIME NOT NULL DEFAULT GETDATE(),
  [CreatedBy] NVARCHAR(50) NOT NULL,
  [ModifiedOn] DATETIME NOT NULL DEFAULT GETDATE(),
  [ModifiedBy] NVARCHAR(50) NOT NULL,
  [IsActive] BIT NOT NULL DEFAULT 1
)
```

### ThirdPartyGameBetSettings
Settings for third-party game betting configurations.

```sql
CREATE TABLE [dbo].[ThirdPartyGameBetSettings] (
  [Id] INT IDENTITY(1,1) NOT NULL PRIMARY KEY,
  [WebId] INT NOT NULL,
  [GameProviderId] INT NOT NULL,
  [MinBet] DECIMAL(19, 6) NOT NULL DEFAULT 0,
  [MaxBet] DECIMAL(19, 6) NOT NULL DEFAULT 0,
  [MaxWin] DECIMAL(19, 6) NOT NULL DEFAULT 0,
  [IsEnabled] BIT NOT NULL DEFAULT 1,
  [CreatedOn] DATETIME NOT NULL DEFAULT GETDATE(),
  [CreatedBy] NVARCHAR(50) NOT NULL,
  [ModifiedOn] DATETIME NOT NULL DEFAULT GETDATE(),
  [ModifiedBy] NVARCHAR(50) NOT NULL
)
```

## Transfer and Wallet Tables

### SboGameproviderTransfer
Records transfers between game providers and main wallet.

```sql
CREATE TABLE [dbo].[SboGameproviderTransfer] (
  [Id] BIGINT IDENTITY(1,1) NOT NULL PRIMARY KEY,
  [WebId] INT NOT NULL,
  [CustomerId] INT NOT NULL,
  [Username] NVARCHAR(50) NOT NULL,
  [GameProviderId] INT NOT NULL,
  [TransferType] INT NOT NULL,
  [Amount] DECIMAL(19, 6) NOT NULL,
  [Currency] CHAR(3) NOT NULL,
  [ExchangeRate] DECIMAL(12, 8) NULL,
  [Status] NVARCHAR(50) NOT NULL,
  [RefNo] NVARCHAR(255) NOT NULL,
  [TransactionId] NVARCHAR(255) NOT NULL,
  [CreatedOn] DATETIME NOT NULL DEFAULT GETDATE(),
  [CreatedBy] NVARCHAR(50) NOT NULL,
  [ModifiedOn] DATETIME NOT NULL DEFAULT GETDATE(),
  [ModifiedBy] NVARCHAR(50) NOT NULL
)
```

### SboSeamlessWalletRequest
Wallet requests for seamless gaming integration.

```sql
CREATE TABLE [dbo].[SboSeamlessWalletRequest] (
  [Id] BIGINT IDENTITY(1,1) NOT NULL PRIMARY KEY,
  [WebId] INT NOT NULL,
  [CustomerId] INT NOT NULL,
  [Username] NVARCHAR(50) NOT NULL,
  [RequestType] NVARCHAR(50) NOT NULL,
  [Amount] DECIMAL(19, 6) NOT NULL,
  [Currency] CHAR(3) NOT NULL,
  [Status] NVARCHAR(50) NOT NULL,
  [RefNo] NVARCHAR(255) NOT NULL,
  [CreatedOn] DATETIME NOT NULL DEFAULT GETDATE(),
  [ProcessedOn] DATETIME NULL
)
```

### SboTransferGameProviderBetCheckSum
Checksum verification for transfer game provider betting data integrity.

```sql
CREATE TABLE [dbo].[SboTransferGameProviderBetCheckSum] (
  [Id] BIGINT IDENTITY(1,1) NOT NULL,
  [StartTime] DATETIME NOT NULL,
  [EndTime] DATETIME NOT NULL,
  [SboTotalBetCount] INT NOT NULL,
  [SboTotalWonCount] INT NOT NULL,
  [SboTotalDrawCount] INT NOT NULL,
  [SboTotalLoseCount] INT NOT NULL,
  [SboTotalTurnover] DECIMAL(19,6) NOT NULL,
  [SboTotalWonTurnover] DECIMAL(19,6) NOT NULL,
  [SboTotalDrawTurnover] DECIMAL(19,6) NOT NULL,
  [SboTotalLoseTurnover] DECIMAL(19,6) NOT NULL,
  [SboTotalWinlose] DECIMAL(19,6) NOT NULL,
  [TotalBetCount] INT NOT NULL,
  [TotalWonCount] INT NOT NULL,
  [TotalDrawCount] INT NOT NULL,
  [TotalLoseCount] INT NOT NULL,
  [TotalTurnover] DECIMAL(19,6) NOT NULL,
  [TotalWonTurnover] DECIMAL(19,6) NOT NULL,
  [TotalDrawTurnover] DECIMAL(19,6) NOT NULL,
  [TotalLoseTurnover] DECIMAL(19,6) NOT NULL,
  [TotalWinlose] DECIMAL(19,6) NOT NULL,
  [IsMapping] BIT NOT NULL,
  [CheckTime] DATETIME NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | BIGINT, PK | Primary key |
| StartTime | DATETIME | Start time for checksum period |
| EndTime | DATETIME | End time for checksum period |
| SboTotalBetCount | INT | Total SBO bet count |
| SboTotalWonCount | INT | Total SBO won bet count |
| SboTotalDrawCount | INT | Total SBO draw bet count |
| SboTotalLoseCount | INT | Total SBO lose bet count |
| SboTotalTurnover | DECIMAL(19,6) | Total SBO turnover |
| SboTotalWonTurnover | DECIMAL(19,6) | Total SBO won turnover |
| SboTotalDrawTurnover | DECIMAL(19,6) | Total SBO draw turnover |
| SboTotalLoseTurnover | DECIMAL(19,6) | Total SBO lose turnover |
| SboTotalWinlose | DECIMAL(19,6) | Total SBO win/loss amount |
| TotalBetCount | INT | Total bet count (all sources) |
| TotalWonCount | INT | Total won count (all sources) |
| TotalDrawCount | INT | Total draw count (all sources) |
| TotalLoseCount | INT | Total lose count (all sources) |
| TotalTurnover | DECIMAL(19,6) | Total turnover (all sources) |
| TotalWonTurnover | DECIMAL(19,6) | Total won turnover (all sources) |
| TotalDrawTurnover | DECIMAL(19,6) | Total draw turnover (all sources) |
| TotalLoseTurnover | DECIMAL(19,6) | Total lose turnover (all sources) |
| TotalWinlose | DECIMAL(19,6) | Total win/loss amount (all sources) |
| IsMapping | BIT | Whether data mapping is verified |
| CheckTime | DATETIME | Time when check was performed |

## Additional Supporting Tables

### GameProviderType
Classification types for game providers.

```sql
CREATE TABLE [dbo].[GameProviderType] (
  [Id] INT IDENTITY(1,1) NOT NULL PRIMARY KEY,
  [Name] NVARCHAR(100) NOT NULL,
  [Description] NVARCHAR(255) NULL,
  [IsEnabled] BIT NOT NULL DEFAULT 1,
  [CreatedOn] DATETIME NOT NULL DEFAULT GETDATE(),
  [CreatedBy] NVARCHAR(50) NOT NULL,
  [ModifiedOn] DATETIME NOT NULL DEFAULT GETDATE(),
  [ModifiedBy] NVARCHAR(50) NOT NULL
)
```

### GameProviderProductType
Maps game providers to their supported product types.

```sql
CREATE TABLE [dbo].[GameProviderProductType] (
  [Id] INT IDENTITY(1,1) NOT NULL PRIMARY KEY,
  [GameProviderId] INT NOT NULL,
  [ProductType] INT NOT NULL,
  [IsEnabled] BIT NOT NULL DEFAULT 1,
  [CreatedOn] DATETIME NOT NULL DEFAULT GETDATE(),
  [CreatedBy] NVARCHAR(50) NOT NULL,
  [ModifiedOn] DATETIME NOT NULL DEFAULT GETDATE(),
  [ModifiedBy] NVARCHAR(50) NOT NULL
)
```

### GameProviderWhitelist
Whitelist management for game providers.

```sql
CREATE TABLE [dbo].[GameProviderWhitelist] (
  [Id] INT IDENTITY(1,1) NOT NULL PRIMARY KEY,
  [GameProviderId] INT NOT NULL,
  [WebId] INT NOT NULL,
  [IsWhitelisted] BIT NOT NULL DEFAULT 1,
  [CreatedOn] DATETIME NOT NULL DEFAULT GETDATE(),
  [CreatedBy] NVARCHAR(50) NOT NULL
)
```

### GameProviderUMTime
Update/maintenance time tracking for game providers.

```sql
CREATE TABLE [dbo].[GameProviderUMTime] (
  [Id] INT IDENTITY(1,1) NOT NULL PRIMARY KEY,
  [GameProviderId] INT NOT NULL,
  [LastUpdateTime] DATETIME NOT NULL,
  [LastMaintenanceTime] DATETIME NULL,
  [NextScheduledMaintenance] DATETIME NULL
)
```

### ReportParentList
Parent account list for reporting hierarchy.

```sql
CREATE TABLE [dbo].[ReportParentList] (
  [Id] BIGINT IDENTITY(1,1) NOT NULL PRIMARY KEY,
  [WebId] INT NOT NULL,
  [ParentId] INT NOT NULL,
  [ParentUsername] NVARCHAR(50) NOT NULL,
  [Level] INT NOT NULL,
  [IsActive] BIT NOT NULL DEFAULT 1,
  [CreatedOn] DATETIME NOT NULL DEFAULT GETDATE(),
  [CreatedBy] NVARCHAR(50) NOT NULL
)
```

### ReportPlayerTransactionInfomation
Player transaction information for reporting.

```sql
CREATE TABLE [dbo].[ReportPlayerTransactionInfomation] (
  [Id] BIGINT IDENTITY(1,1) NOT NULL PRIMARY KEY,
  [WebId] INT NOT NULL,
  [CustomerId] INT NOT NULL,
  [Username] NVARCHAR(50) NOT NULL,
  [TransactionDate] DATETIME NOT NULL,
  [ProductType] INT NOT NULL,
  [TotalStake] DECIMAL(19,6) NOT NULL DEFAULT 0,
  [TotalWinlost] DECIMAL(19,6) NOT NULL DEFAULT 0,
  [TotalTurnover] DECIMAL(19,6) NOT NULL DEFAULT 0,
  [BetCount] INT NOT NULL DEFAULT 0,
  [Currency] CHAR(3) NOT NULL,
  [CreatedOn] DATETIME NOT NULL DEFAULT GETDATE()
)
```

### ReportPlayerBettingInfo
Detailed betting information for player reports.

```sql
CREATE TABLE [dbo].[ReportPlayerBettingInfo] (
  [Id] BIGINT IDENTITY(1,1) NOT NULL PRIMARY KEY,
  [WebId] INT NOT NULL,
  [CustomerId] INT NOT NULL,
  [Username] NVARCHAR(50) NOT NULL,
  [ParentId] INT NULL,
  [GameProviderId] INT NOT NULL,
  [ProductType] INT NOT NULL,
  [BettingDate] DATETIME NOT NULL,
  [TotalStake] DECIMAL(19,6) NOT NULL DEFAULT 0,
  [TotalWinlost] DECIMAL(19,6) NOT NULL DEFAULT 0,
  [TotalCommission] DECIMAL(19,6) NOT NULL DEFAULT 0,
  [BetCount] INT NOT NULL DEFAULT 0,
  [Currency] CHAR(3) NOT NULL,
  [IsActive] BIT NOT NULL DEFAULT 1,
  [CreatedOn] DATETIME NOT NULL DEFAULT GETDATE(),
  [CreatedBy] NVARCHAR(50) NOT NULL,
  [ModifiedOn] DATETIME NOT NULL DEFAULT GETDATE(),
  [ModifiedBy] NVARCHAR(50) NOT NULL
)
```

### SboTransferGameproviderBets
Records transfer game provider betting transactions.

```sql
CREATE TABLE [dbo].[SboTransferGameproviderBets] (
  [Id] BIGINT IDENTITY(1,1) NOT NULL PRIMARY KEY,
  [WebId] INT NOT NULL,
  [RefNo] NVARCHAR(100) NOT NULL,
  [TransactionId] NVARCHAR(100) NOT NULL,
  [CustomerId] INT NOT NULL,
  [ParentId] INT NULL,
  [MaId] INT NOT NULL,
  [Username] NVARCHAR(50) NOT NULL,
  [UsernameAtGameProvider] NVARCHAR(50) NULL,
  [GameProviderId] BIGINT NULL,
  [GameIdAtGameProvider] BIGINT NULL,
  [GameRoundId] NVARCHAR(100) NULL,
  [GamePeriodId] NVARCHAR(100) NULL,
  [GameType] NVARCHAR(50) NULL,
  [OrderDetail] NVARCHAR(4000) NULL,
  [GameResult] NVARCHAR(4000) NULL,
  [Currency] CHAR(3) NULL,
  [ExchangeRate] DECIMAL(12, 8) NULL,
  [Status] NVARCHAR(50) NULL,
  [StatusAtGameProvider] NVARCHAR(50) NULL,
  [Stake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [TurnOver] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Winlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [WinLostDate] DATETIME NULL,
  [OrderTime] DATETIME NOT NULL,
  [ModifiedTime] DATETIME,
  [SyncTime] DATETIME,
  [SettledTime] DATETIME,
  [OrderedTime] DATETIME,
  [IsDailyStatementProceed] BIT DEFAULT (0) NOT NULL,
  [IsCommissionCheck] BIT DEFAULT(0),
  [IsPromotionCheck] BIT DEFAULT(0),
  [IsReferralCheck] BIT DEFAULT(0),
  [IsWithdrawalLimitCheck] BIT DEFAULT(0),
  [ReferralLayerPercentageHistoryIds] NVARCHAR(1000),
  [PromotionRequestId] INT NOT NULL DEFAULT(0),
  [WalletProductType] INT NOT NULL DEFAULT(0),
  [WalletGameType] INT NOT NULL DEFAULT(0),
  [IsFishGame] BIT,
  -- Commission and hierarchy tracking
  [AgtPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [MaPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [SmaPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [SsmaPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [ShPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [TotalPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [AgtWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [MaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SmaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SsmaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ShWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  -- Commission rates and amounts
  [PlayerCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [AgtCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [MaCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [SmaCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [SsmaCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [ShCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [PlayerComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [AgtComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [MaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SmaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SsmaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ShComm] DECIMAL(19, 6) DEFAULT ((0)) NULL
)
```

### FailBetsRemovedForDuplicateIdentityGameProvider
Tracks failed bets that were removed due to duplicate identity issues.

```sql
CREATE TABLE [dbo].[FailBetsRemovedForDuplicateIdentityGameProvider] (
  [Id] BIGINT IDENTITY(1,1) NOT NULL PRIMARY KEY,
  [GPID] INT NULL,
  [WebId] INT NOT NULL,
  [RefNo] NVARCHAR(100) NOT NULL,
  [TransactionId] NVARCHAR(100) NOT NULL,
  [CustomerId] INT NOT NULL,
  [ParentId] INT NULL,
  [Username] NVARCHAR(50) NOT NULL,
  [UsernameAtGameProvider] NVARCHAR(50) NULL,
  [PromotionRequestId] INT NOT NULL,
  [WalletProductType] INT NOT NULL,
  [WalletGameType] INT NOT NULL,
  [OrderTime] DATETIME NOT NULL,
  [WinLostDate] DATETIME NULL,
  [Stake] DECIMAL(19, 6) NULL,
  [Currency] CHAR(3) NULL,
  [ExchangeRate] DECIMAL(12, 8) NULL,
  [TurnOver] DECIMAL(19, 6) NULL,
  [StatusAtGameProvider] VARCHAR(50) NULL,
  [Status] VARCHAR(50) NULL,
  [IsFishGame] BIT NULL,
  [GamePeriodId] NVARCHAR(100) NULL,
  [GameRoundId] NVARCHAR(100) NULL,
  [GameType] NVARCHAR(50) NULL,
  [OrderDetail] NVARCHAR(4000) NULL,
  [GameResult] NVARCHAR(4000) NULL,
  [Winlost] DECIMAL(19, 6) NULL,
  [SyncTime] DATETIME NULL,
  [ModifiedTime] DATETIME NULL,
  [ReferralLayerPercentageHistoryIds] NVARCHAR(1000) NULL,
  [IsDailyStatementProceed] BIT NOT NULL,
  [RemovalReason] NVARCHAR(500) NULL,
  [RemovedOn] DATETIME NOT NULL DEFAULT GETDATE(),
  [RemovedBy] NVARCHAR(50) NOT NULL
)
```

## Performance Indexes Summary

### Key Performance Indexes
The GameProvider database implements extensive indexing strategies for optimal query performance:

#### SboSeamlessBets Indexes
- **IX_SboSeamlessBets_WinLostDate**: Optimized for date-range queries with comprehensive INCLUDE columns
- **IX_SboSeamlessBets_RefNo**: Fast lookup by reference number
- **IX_SboSeamlessBets_Report**: Optimized for reporting queries
- **IX_SboSeamlessBets_Synctime**: Synchronization and data retrieval optimization
- **IX_SboSeamlessBets_GetBettingRecord**: Betting record retrieval optimization
- **IX_SboSeamlessBets_Bonus**: Bonus-related query optimization

#### AllGames Indexes
- **IX_AllGame_GameProvider_GetBettingRecord**: Optimized for game provider betting record queries

#### Primary Key Strategies
- **Composite Primary Keys**: Used for tables like SboSeamlessBets ([Id], [WinLostDate]) to support partitioning
- **Clustered Indexes**: Strategic placement for optimal data organization
- **Fill Factor**: Set to 85% for tables with frequent updates to reduce page splits

## Data Archival and Partitioning Strategy

### 60-Day Archive Pattern
The database implements a systematic 60-day archival strategy:

1. **Active Tables**: Current operational data (SboSeamlessBets, SboSportBets, etc.)
2. **60-Day Archives**: Historical data retention (SboSeamlessBets60, SboSportBets60, etc.)
3. **Extended Archives**: Special cases like SboSeamlessBets90Temp for extended retention
4. **Automated Jobs**: Scheduled jobs move data between tiers based on age

### Partitioning Strategy
- **Date-based Partitioning**: Primary tables partitioned by WinLostDate
- **Composite Primary Keys**: Support efficient partitioning and query performance
- **Archive Migration**: Automated processes move aged data to archive tables

## Commission Hierarchy Structure

### Commission Levels
The database supports a 6-level commission hierarchy:

1. **Player Level**: PlayerComm, PlayerCommissionRate
2. **Agent Level**: AgtComm, AgtCommissionRate, AgtPT, AgtWinlost
3. **Master Agent**: MaComm, MaCommissionRate, MaPT, MaWinlost
4. **Super Master Agent**: SmaComm, SmaCommissionRate, SmaPT, SmaWinlost
5. **Super Super Master Agent**: SsmaComm, SsmaCommissionRate, SsmaPT, SsmaWinlost
6. **Shareholder**: ShComm, ShCommissionRate, ShPT, ShWinlost

### Commission Calculation
- **PT Fields**: Percentage tracking for each level
- **Rate Fields**: Commission rates applied
- **Comm Fields**: Calculated commission amounts
- **Winlost Fields**: Win/loss amounts by hierarchy level

## Missing Tables from Actual Database

### MigratedSportsTransIdMapping
Mapping table for migrated sports transaction IDs.

```sql
CREATE TABLE [dbo].[MigratedSportsTransIdMapping] (
  [Id] INT IDENTITY(1,1) NOT NULL PRIMARY KEY,
  [OldTransactionId] NVARCHAR(100) NOT NULL,
  [NewTransactionId] NVARCHAR(100) NOT NULL,
  [ProductType] INT NOT NULL,
  [WebId] INT NOT NULL,
  [CreatedOn] DATETIME NOT NULL DEFAULT GETDATE()
)
```

### OldSystemBetSumarry
Legacy system bet summary data for migration purposes.

```sql
CREATE TABLE [dbo].[OldSystemBetSumarry] (
  [Id] BIGINT IDENTITY(1,1) NOT NULL,
  [WebId] INT,
  [ProductType] INT,
  [CustomerId] INT,
  [BetCount] INT,
  [TotalStake] DECIMAL(19, 6),
  [TotalWinlost] DECIMAL(19, 6),
  [CreatedOn] DATETIME DEFAULT GETDATE()
)
```

### SboSportsDelayBetList
List of delayed sports bets pending processing.

```sql
CREATE TABLE [dbo].[SboSportsDelayBetList] (
  [Id] INT IDENTITY(1,1) NOT NULL PRIMARY KEY,
  [RefNo] NVARCHAR(100) NOT NULL,
  [WebId] INT NOT NULL,
  [CustomerId] INT NOT NULL,
  [DelayReason] NVARCHAR(255),
  [OriginalOrderTime] DATETIME NOT NULL,
  [DelayedUntil] DATETIME,
  [Status] NVARCHAR(50),
  [CreatedOn] DATETIME NOT NULL DEFAULT GETDATE(),
  [ProcessedOn] DATETIME NULL
)
```

## Temporary and Extended Archive Tables

### SboSeamlessBets60new
New version of seamless bets 60-day archive with enhanced tracking.

```sql
CREATE TABLE [dbo].[SboSeamlessBets60new] (
  -- Enhanced version of SboSeamlessBets60 with additional fields
  -- Similar structure to SboSeamlessBets60 but with improved data integrity
  [Id] BIGINT IDENTITY(1,1) NOT NULL,
  [GPID] INT NULL,
  [WebId] INT NOT NULL,
  [RefNo] NVARCHAR(255) NOT NULL,
  [TransactionId] NVARCHAR(255) NOT NULL,
  [CustomerId] INT NOT NULL,
  [ParentId] INT NULL,
  [MaId] INT DEFAULT ((0)) NOT NULL,
  [Username] NVARCHAR(50) NOT NULL,
  [UsernameAtGameProvider] NVARCHAR(50) NULL,
  [PromotionRequestId] INT DEFAULT ((0)) NOT NULL,
  [NewPromotionRequestId] INT DEFAULT ((0)) NOT NULL,
  [ReferralEventId] INT DEFAULT ((0)) NOT NULL,
  [WalletProductType] INT DEFAULT ((0)) NOT NULL,
  [WalletGameType] INT DEFAULT ((0)) NOT NULL,
  [OrderTime] DATETIME NOT NULL,
  [WinLostDate] DATETIME NOT NULL,
  [Stake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Currency] CHAR(3) NULL,
  [ExchangeRate] DECIMAL(12, 8) NULL,
  [TurnOver] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [StatusAtGameProvider] VARCHAR(50) NULL,
  [Status] VARCHAR(50) NULL,
  [IsFishGame] BIT DEFAULT ((0)) NULL,
  [GameIdAtGameProvider] INT DEFAULT ((0)) NOT NULL,
  [GamePeriodId] NVARCHAR(100) NULL,
  [GameRoundId] NVARCHAR(100) NULL,
  [GameType] NVARCHAR(50) NULL,
  [OrderDetail] NVARCHAR(4000) NULL,
  [GameResult] NVARCHAR(4000) NULL,
  [Winlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SyncTime] DATETIME,
  [ModifiedTime] DATETIME,
  [DataVersion] INT DEFAULT 2 -- Enhanced version indicator
)
```

### SboSeamlessBets90Temp
Extended 90-day temporary archive for seamless bets data.

```sql
CREATE TABLE [dbo].[SboSeamlessBets90Temp] (
  -- Extended retention temporary table for seamless bets
  -- Similar structure to SboSeamlessBets60 but with 90-day retention
  [Id] BIGINT NOT NULL,
  [GPID] INT NULL,
  [WebId] INT NOT NULL,
  [RefNo] NVARCHAR(255) NOT NULL,
  [TransactionId] NVARCHAR(255) NOT NULL,
  [CustomerId] INT NOT NULL,
  [ParentId] INT NULL,
  [MaId] INT DEFAULT ((0)) NOT NULL,
  [Username] NVARCHAR(50) NOT NULL,
  [UsernameAtGameProvider] NVARCHAR(50) NULL,
  [OrderTime] DATETIME NOT NULL,
  [WinLostDate] DATETIME NOT NULL,
  [Stake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Currency] CHAR(3) NULL,
  [ExchangeRate] DECIMAL(12, 8) NULL,
  [TurnOver] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [StatusAtGameProvider] VARCHAR(50) NULL,
  [Status] VARCHAR(50) NULL,
  [Winlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [TempRetentionDate] DATETIME DEFAULT DATEADD(DAY, 90, GETDATE())
)
```

### SboThirdPartySportBets60
Historical archive of third-party sports betting transactions (60-day retention).

```sql
CREATE TABLE [dbo].[SboThirdPartySportBets60] (
  [Id] BIGINT NOT NULL,
  [WebId] INT NOT NULL,
  [RefNo] NVARCHAR(100) NOT NULL,
  [TransactionId] NVARCHAR(100) NOT NULL,
  [CustomerId] INT NOT NULL,
  [ParentId] INT NULL,
  [Username] NVARCHAR(50) NOT NULL,
  [UsernameAtGameProvider] NVARCHAR(50) NULL,
  [PromotionRequestId] INT DEFAULT ((0)) NOT NULL,
  [NewPromotionRequestId] INT DEFAULT ((0)) NOT NULL,
  [ReferralEventId] INT DEFAULT ((0)) NOT NULL,
  [WalletProductType] INT DEFAULT ((0)) NOT NULL,
  [WalletGameType] INT DEFAULT ((0)) NOT NULL,
  [SportsType] NVARCHAR(50) NULL,
  [BetType] INT NULL,
  [OrderTime] DATETIME NOT NULL,
  [ModifiedTime] DATETIME NOT NULL,
  [WinlostDate] DATETIME NULL,
  [Odds] DECIMAL(12, 3) DEFAULT ((0)) NULL,
  [Stake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ActualStake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Currency] CHAR(3) NULL,
  [ExchangeRate] DECIMAL(12, 8) NULL,
  [StatusAtGameProvider] VARCHAR(50) NULL,
  [Status] VARCHAR(50) NULL,
  [Winlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [TurnOver] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [IsLive] BIT NULL,
  [IsHalfWinLose] BIT NULL,
  -- Commission and hierarchy tracking
  [AgtPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [MaPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [SmaPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [SsmaPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [ShPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [TotalPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [AgtWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [MaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SmaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SsmaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ShWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  -- Commission rates and amounts
  [PlayerCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [AgtCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [MaCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [SmaCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [SsmaCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [ShCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [PlayerComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [AgtComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [MaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SmaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SsmaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ShComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  -- Additional tracking
  [SyncTime] DATETIME,
  [IsDailyStatementProceed] BIT DEFAULT (0) NOT NULL,
  [ReferralLayerPercentageHistoryIds] NVARCHAR(1000),
  [WinlostDateByTimeZone] DATETIME NULL,
  [TimeZone] NVARCHAR(100) NULL,
  [WithdrawalLimitId] INT DEFAULT(0) NOT NULL
)
```

### SboSportBetDetails
Detailed information for sports betting transactions.

```sql
CREATE TABLE [dbo].[SboSportBetDetails] (
  [Id] INT NOT NULL PRIMARY KEY IDENTITY(1,1),
  [RefNo] NVARCHAR(100) NOT NULL,
  [TransactionId] NVARCHAR(100) NOT NULL,
  [SubbetId] NVARCHAR(100) NULL,
  [ParlayRefno] NVARCHAR(100) NULL,
  [BetOption] NVARCHAR(150) NULL,
  [IsLive] BIT NOT NULL,
  [MarketType] NVARCHAR(150) NULL,
  [SportType] NVARCHAR(50) NULL,
  [HandicapPoint] DECIMAL(12, 2) DEFAULT ((0)) NULL,
  [Odds] DECIMAL(12, 3) DEFAULT ((0)) NULL,
  [League] NVARCHAR(150) NULL,
  [HomeTeam] NVARCHAR(150) NULL,
  [AwayTeam] NVARCHAR(150) NULL,
  [Status] VARCHAR(50) NULL,
  [WinlostDate] DATETIME NULL,
  [ModifiedTime] DATETIME NULL,
  [LiveHomeScore] INT NULL,
  [LiveAwayScore] INT NULL,
  [WebId] INT NOT NULL,
  [SyncTime] DATETIME,
  [SettlementTime] DATETIME NULL,
  [Language] NVARCHAR(10) DEFAULT('en') NOT NULL,
  [FullTimeScore] NVARCHAR(100) NULL,
  [HalfTimeScore] NVARCHAR(100) NULL,
  [WinlostDateByTimeZone] DATETIME NULL,
  [TimeZone] NVARCHAR(100) NULL,
  [WithdrawalLimitId] INT DEFAULT(0) NOT NULL
)
```

### SboSportBetDetails60
Historical archive of sports bet details (60-day retention).

```sql
CREATE TABLE [dbo].[SboSportBetDetails60] (
  [Id] INT NOT NULL PRIMARY KEY IDENTITY(1,1),
  [RefNo] NVARCHAR(100) NOT NULL,
  [TransactionId] NVARCHAR(100) NOT NULL,
  [SubbetId] NVARCHAR(100) NULL,
  [ParlayRefno] NVARCHAR(100) NULL,
  [BetOption] NVARCHAR(150) NULL,
  [IsLive] BIT NOT NULL,
  [MarketType] NVARCHAR(150) NULL,
  [SportType] NVARCHAR(50) NULL,
  [HandicapPoint] DECIMAL(12, 2) DEFAULT ((0)) NULL,
  [Odds] DECIMAL(12, 3) DEFAULT ((0)) NULL,
  [League] NVARCHAR(150) NULL,
  [HomeTeam] NVARCHAR(150) NULL,
  [AwayTeam] NVARCHAR(150) NULL,
  [Status] VARCHAR(50) NULL,
  [WinlostDate] DATETIME NULL,
  [ModifiedTime] DATETIME NULL,
  [LiveHomeScore] INT NULL,
  [LiveAwayScore] INT NULL,
  [WebId] INT NOT NULL,
  [SyncTime] DATETIME,
  [SettlementTime] DATETIME NULL,
  [Language] NVARCHAR(10) DEFAULT('en') NOT NULL,
  [FullTimeScore] NVARCHAR(100) NULL,
  [HalfTimeScore] NVARCHAR(100) NULL,
  [WinlostDateByTimeZone] DATETIME NULL,
  [TimeZone] NVARCHAR(100) NULL,
  [WithdrawalLimitId] INT DEFAULT(0) NOT NULL
)
```

### SboGamesBets60
Historical archive of SBO games betting transactions (60-day retention).

```sql
CREATE TABLE [dbo].[SboGamesBets60] (
  [Id] BIGINT NOT NULL,
  [WebId] INT NOT NULL,
  [RefNo] NVARCHAR(100) NOT NULL,
  [TransactionId] NVARCHAR(100) NOT NULL,
  [CustomerId] INT NOT NULL,
  [ParentId] INT NULL,
  [Username] NVARCHAR(50) NOT NULL,
  [UsernameAtGameProvider] NVARCHAR(50) NULL,
  [PromotionRequestId] INT DEFAULT ((0)) NOT NULL,
  [NewPromotionRequestId] INT DEFAULT ((0)) NOT NULL,
  [ReferralEventId] INT DEFAULT ((0)) NOT NULL,
  [WalletProductType] INT DEFAULT ((0)) NOT NULL,
  [WalletGameType] INT DEFAULT ((0)) NOT NULL,
  [ProductType] NVARCHAR(50) NULL,
  [GameId] BIGINT NULL,
  [GameName] NVARCHAR(100) NULL,
  [OrderTime] DATETIME NOT NULL,
  [WinLostDate] DATETIME NULL,
  [Stake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Currency] CHAR(3) NULL,
  [ExchangeRate] DECIMAL(12, 8) NULL,
  [TurnOver] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [StatusAtGameProvider] VARCHAR(50) NULL,
  [Status] VARCHAR(50) NULL,
  [Winlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  -- Commission and hierarchy tracking
  [AgtPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [MaPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [SmaPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [SsmaPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [ShPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [TotalPT] DECIMAL(3, 2) DEFAULT ((0)) NULL,
  [AgtWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [MaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SmaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SsmaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ShWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  -- Commission rates and amounts
  [PlayerCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [AgtCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [MaCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [SmaCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [SsmaCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [ShCommissionRate] DECIMAL(5, 4) DEFAULT ((0)) NULL,
  [PlayerComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [AgtComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [MaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SmaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SsmaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ShComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  -- Additional tracking
  [SyncTime] DATETIME,
  [ModifiedTime] DATETIME,
  [ReferralLayerPercentageHistoryIds] NVARCHAR(1000),
  [IsDailyStatementProceed] BIT DEFAULT (0) NOT NULL,
  [WinlostDateByTimeZone] DATETIME NULL,
  [TimeZone] NVARCHAR(100) NULL,
  [WithdrawalLimitId] INT DEFAULT(0) NOT NULL
)
```

### DailyBetSummaryCurrentByTimeZone
Current daily betting summary with timezone awareness for real-time reporting.

```sql
CREATE TABLE [dbo].[DailyBetSummaryCurrentByTimeZone] (
  [Id] BIGINT IDENTITY(1,1) NOT NULL,
  [WebId] INT,
  [ProductType] INT,
  [DisplayName] NVARCHAR(50),
  [CustomerId] INT,
  [ParentId] INT,
  [Currency] CHAR(3),
  [Username] NVARCHAR(50),
  [BetCount] INT,
  [Stake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ActualStake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Turnover] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [NetTurnover] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [TurnoverByStake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [TurnoverByActualStake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [NetTurnoverByStake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [NetTurnoverByActualStake] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Status] NVARCHAR(50),
  [ExchangeRate] DECIMAL(12, 8) NULL,
  [PlayerComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [AgtComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [MaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SmaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SsmaComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ShComm] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [Winlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [AgtWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [MaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SmaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [SsmaWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [ShWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [CompanyWinlost] DECIMAL(19, 6) DEFAULT ((0)) NULL,
  [RowModifiedDate] DATETIME,
  [RowModifiedBy] NVARCHAR(50),
  [IsValidTurnover] BIT DEFAULT ((1)) NOT NULL,
  [NewPromotionRequestId] INT DEFAULT (0) NOT NULL,
  [WinlostDateByTimeZone] DATETIME NULL,
  [TimeZone] NVARCHAR(100) NULL,
  [WithdrawalLimitId] INT DEFAULT (0) NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | BIGINT, PK | Primary key |
| WebId | INT | Website identifier |
| ProductType | INT | Product type (1=Sports, 2=Casino, 3=Games, 4=Virtual Sports, 5=Seamless, 6=Third Party Sports, 7=Live Coin) |
| DisplayName | NVARCHAR(50) | Display name for the product/provider |
| CustomerId | INT | Customer identifier |
| ParentId | INT | Parent customer identifier |
| Currency | CHAR(3) | Currency code |
| Username | NVARCHAR(50) | Username |
| BetCount | INT | Number of bets |
| Stake | DECIMAL(19, 6) | Total stake amount |
| ActualStake | DECIMAL(19, 6) | Actual stake amount |
| Turnover | DECIMAL(19, 6) | Total turnover |
| NetTurnover | DECIMAL(19, 6) | Net turnover |
| TurnoverByStake | DECIMAL(19, 6) | Turnover calculated by stake |
| TurnoverByActualStake | DECIMAL(19, 6) | Turnover calculated by actual stake |
| NetTurnoverByStake | DECIMAL(19, 6) | Net turnover by stake |
| NetTurnoverByActualStake | DECIMAL(19, 6) | Net turnover by actual stake |
| Status | NVARCHAR(50) | Bet status |
| ExchangeRate | DECIMAL(12, 8) | Currency exchange rate |
| PlayerComm to ShComm | DECIMAL(19, 6) | Commission amounts by hierarchy level |
| Winlost to ShWinlost | DECIMAL(19, 6) | Win/loss amounts by hierarchy level |
| CompanyWinlost | DECIMAL(19, 6) | Company win/loss amount |
| RowModifiedDate | DATETIME | Row modification date |
| RowModifiedBy | NVARCHAR(50) | Who modified the row |
| IsValidTurnover | BIT | Whether turnover is valid |
| NewPromotionRequestId | INT | New promotion request identifier |
| WinlostDateByTimeZone | DATETIME | Win/loss date adjusted for timezone |
| TimeZone | NVARCHAR(100) | Timezone specification |
| WithdrawalLimitId | INT | Withdrawal limit identifier |

## Database Schema Summary

### Total Tables: 90+ Tables
The GameProvider database contains over 90 tables organized into the following categories:

#### Core Provider Management (8 tables)
- `AllGameProviders` - Master game provider registry
- `CompanyGameProviders` - Company-specific provider configurations  
- `AllAPIGameProviders` - API provider master list
- `CompanyAPIGameProviders` - Company-API provider mappings
- `GameProviderType` - Provider type classifications
- `GameProviderProductType` - Provider-product type mappings
- `GameProviderWhitelist` - Provider whitelist management
- `GameProviderUMTime` - Update/maintenance time tracking

#### Game Management (6 tables)
- `AllGames` - Master game registry
- `CompanyGames` - Company-specific game configurations
- `GameCategory` - Game category definitions
- `PromoteAllGames` - Game promotion management
- `GameTranslation` - Multi-language game names
- `GameProviderTranslation` - Multi-language provider names

#### Betting Transaction Tables (20+ tables)
**Active Betting Tables:**
- `SboSportBets` - Sports betting transactions
- `SboSeamlessBets` - Seamless gaming transactions
- `SboSeamlessRunningBets` - Active seamless bets
- `SboCasinoBets` - Casino game transactions
- `SboGamesBets` - SBO games transactions
- `SboVirtualSportBets` - Virtual sports betting
- `SboThirdPartySportBets` - Third-party sports betting
- `SboLiveCoin` - Live coin transactions
- `SboTransferGameproviderBets` - Transfer provider bets
- `LionKingMJP` - Lion King MJP transactions

**Detailed Transaction Tables:**
- `SboSportBetDetails` - Sports bet details
- `SboThirdPartySportBetDetails` - Third-party sports bet details
- `SboVirtualSportSubBets` - Virtual sports sub-bet details

**Archive Tables (60-day retention):**
- `SboSportBets60`, `SboSeamlessBets60`, `SboCasinoBets60`
- `SboGamesBets60`, `SboVirtualSportBets60`, `SboLiveCoin60`
- `SboThirdPartySportBets60`, `LionKingMJP60`
- `SboSportBetDetails60` - Sports bet details archive

**Extended Archive Tables:**
- `SboSeamlessBets90Temp` - 90-day extended retention
- `SboSeamlessBets60new` - Enhanced 60-day archive

#### Summary and Reporting Tables (9 tables)
- `DailyBetSummary` - Daily betting aggregations
- `DailyBetSummaryCurrent` - Real-time daily summaries
- `DailyBetSummaryByTimeZone` - Timezone-aware summaries
- `DailyBetSummaryCurrentByTimeZone` - Real-time timezone-aware summaries
- `DailySeamlessGamesBetsSummary` - Seamless games summaries
- `ReportPlayerList` - Player reporting registry
- `ReportPlayerBettingInfo` - Detailed player betting info
- `ReportPlayerTransactionInfomation` - Transaction information
- `ReportParentList` - Parent account hierarchy

#### Compliance and Monitoring (5 tables)
- `GameComplianceThresholdSettings` - Compliance threshold configurations
- `GameComplianceThresholdAction` - Compliance actions
- `GameComplianceThresholdActionHistory` - Action history
- `BonusThresholdAlertHistory` - Bonus alert tracking
- `ProviderBounsThreshold` - Provider bonus thresholds

#### Transfer and Wallet Management (4 tables)
- `SboGameproviderTransfer` - Provider-wallet transfers
- `SboSeamlessWalletRequest` - Seamless wallet requests
- `SboTransferGameProviderBetCheckSum` - Data integrity checksums
- `SboRegisteredUser` - User registration tracking

#### Regional and Specialized Tables (6 tables)
- `PeruPayoutId` - Peru region payout management
- `PeruShopBetRecord` - Peru shop betting records
- `RedeemableRefno` - Ticket redemption management
- `PromotedLeague` - Sports league promotions
- `SboSportsTypeOrder` - Sports type display ordering
- `ThirdPartyGameBetSettings` - Third-party game configurations

#### Data Quality and Migration (4 tables)
- `FailBetsRemovedForDuplicateIdentityGameProvider` - Failed bet tracking
- `MigratedSportsTransIdMapping` - Migration ID mappings
- `OldSystemBetSumarry` - Legacy system data
- `SboSportsDelayBetList` - Delayed bet processing

### Key Architectural Features

#### 1. Commission Hierarchy System
Six-level commission structure supporting:
- Player → Agent → Master Agent → Super Master Agent → Super Super Master Agent → Shareholder
- Percentage tracking (PT fields)
- Commission rates and calculated amounts
- Win/loss distribution by hierarchy level

#### 2. Multi-Currency Support
- Currency fields in all transaction tables
- Exchange rate tracking
- Multi-currency reporting capabilities

#### 3. Multi-Tenant Architecture
- WebId field for company/website isolation
- Company-specific configurations for providers and games
- Isolated reporting and compliance per tenant

#### 4. Data Lifecycle Management
- Automated 60-day archival strategy
- Extended retention for special cases
- Partitioning by date for performance
- Data integrity checksums

#### 5. Performance Optimization
- Extensive indexing strategies (50+ indexes)
- Composite primary keys for partitioning
- Fill factor optimization (85%)
- Include columns for covering indexes

#### 6. Compliance and Risk Management
- Real-time threshold monitoring
- Automated compliance actions
- Bonus threshold management
- Risk tagging and tracking

#### 7. Internationalization Support
- Multi-language game and provider names
- Timezone-aware date handling
- Regional-specific features (Peru operations)

### Data Volume Considerations
- High-volume transaction tables with millions of records
- Partitioning strategy for performance at scale
- Archive tables for historical data retention
- Optimized for both OLTP and OLAP workloads

This comprehensive schema supports a full-featured gaming platform with robust transaction processing, compliance monitoring, multi-tenant operations, and extensive reporting capabilities.
