# History Database Schema

## Overview
The History database serves as the archival storage system for the Coloris Framework, maintaining historical records of transactions, betting activities, member actions, and system operations. It provides long-term data retention, supports historical reporting and auditing capabilities, and ensures comprehensive tracking of all system activities.

## Core Tables

### MemberActionLogHistory
Tracks member actions and interactions within the system for audit and analysis purposes.

```sql
CREATE TABLE [dbo].[MemberActionLogHistory] (
    [Id] INT IDENTITY(1, 1) PRIMARY KEY,
    [WebId] INT NOT NULL,
    [CustomerId] INT NOT NULL,
    [TraceId] NVARCHAR(50),
    [ActionType] NVARCHAR(50),
    [Target] NVARCHAR(100),
    [Ip] NVARCHAR(200),
    [Result] NVARCHAR(50),
    [Action] NVARCHAR(255),
    [Request] NVARCHAR(MAX),
    [Response] NVARCHAR(MAX),
    [ActionOn] DATETIME DEFAULT GETDATE()
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Primary key, auto-incrementing unique identifier |
| WebId | INT | Foreign key reference to the website/application |
| CustomerId | INT | Foreign key reference to the customer |
| TraceId | NVARCHAR(50) | Optional trace identifier for tracking related actions |
| ActionType | NVARCHAR(50) | Type/category of the action performed |
| Target | NVARCHAR(100) | Target object or resource of the action |
| Ip | NVARCHAR(200) | IP address from which the action was performed |
| Result | NVARCHAR(50) | Result or outcome of the action |
| Action | NVARCHAR(255) | Detailed description of the action |
| Request | NVARCHAR(MAX) | Request data associated with the action |
| Response | NVARCHAR(MAX) | Response data from the action |
| ActionOn | DATETIME | Timestamp when the action occurred |

### BetSummaryHistory
Comprehensive historical record of betting summaries including stakes, winnings, and commission details.

```sql
CREATE TABLE [dbo].[BetSummaryHistory] (
    [Id] BIGINT IDENTITY(1,1) NOT NULL PRIMARY KEY,
    [WebId] INT NOT NULL,
    [CustomerId] INT NOT NULL,
    [Currency] CHAR(3) NOT NULL,
    [Username] NVARCHAR(50) NOT NULL,
    [BetCount] INT NOT NULL DEFAULT (0),
    [Stake] DECIMAL(19,6) NOT NULL DEFAULT (0),
    [ActualStake] DECIMAL(19,6) NOT NULL DEFAULT (0),
    [Turnover] DECIMAL(19,6) NOT NULL DEFAULT (0),
    [NetTurnover] DECIMAL(19,6) NOT NULL DEFAULT (0),
    [TurnoverByStake] DECIMAL(19,6) NOT NULL DEFAULT (0),
    [TurnoverByActualStake] DECIMAL(19,6) NOT NULL DEFAULT (0),
    [NetTurnoverByStake] DECIMAL(19,6) NOT NULL DEFAULT (0),
    [NetTurnoverByActualStake] DECIMAL(19,6) NOT NULL DEFAULT (0),
    [ExchangeRate] DECIMAL(12,8) NULL,
    [PlayerComm] DECIMAL(19,6) NOT NULL DEFAULT (0),
    [AgtComm] DECIMAL(19,6) NOT NULL DEFAULT (0),
    [MaComm] DECIMAL(19,6) NOT NULL DEFAULT (0),
    [SmaComm] DECIMAL(19,6) NOT NULL DEFAULT (0),
    [SsmaComm] DECIMAL(19,6) NOT NULL DEFAULT (0),
    [ShComm] DECIMAL(19,6) NOT NULL DEFAULT (0),
    [Winlost] DECIMAL(19,6) NOT NULL DEFAULT (0),
    [AgtWinlost] DECIMAL(19,6) NOT NULL DEFAULT (0),
    [MaWinlost] DECIMAL(19,6) NOT NULL DEFAULT (0),
    [SmaWinlost] DECIMAL(19,6) NOT NULL DEFAULT (0),
    [SsmaWinlost] DECIMAL(19,6) NOT NULL DEFAULT (0),
    [ShWinlost] DECIMAL(19,6) NOT NULL DEFAULT (0),
    [CompanyWinlost] DECIMAL(19,6) NOT NULL DEFAULT (0),
    [RowModifiedDate] DATETIME NOT NULL DEFAULT (GETDATE()),
    [RowModifiedBy] NVARCHAR(50) NOT NULL,
    [CreatedDate] DATETIME NOT NULL DEFAULT (GETDATE()),
    [IsValidTurnover] BIT NOT NULL DEFAULT (1),
    [NewPromotionRequestId] INT NOT NULL DEFAULT (0)
)
```

### TransactionRequestHistory
Historical records of all financial transaction requests including deposits, withdrawals, and transfers.

```sql
CREATE TABLE [dbo].[TransactionRequestHistory] (
    [ID] INT IDENTITY (1, 1) NOT NULL,
    [TransactionType] INT NOT NULL,
    [ParentId] INT NOT NULL,
    [CustomerId] INT NULL,
    [Username] NVARCHAR (50) NULL,
    [Amount] DECIMAL (19, 6) NOT NULL,
    [Currency] CHAR (3) NULL,
    [ExchangeRate] DECIMAL (12, 8) NULL,
    [CompanyBankID] INT NULL,
    [PlayerBankName] NVARCHAR(100) NULL,
    [PlayerBankAccountNumber] VARCHAR (50) NULL,
    [PlayerBankAccountName] NVARCHAR (MAX) NULL,
    [TransactionStatus] INT DEFAULT ((0)) NOT NULL,
    [BeforeBalance] DECIMAL (19, 6) NULL,
    [AfterBalance] DECIMAL (19, 6) NULL,
    [VerifiedAmount] DECIMAL(19,6) NULL,
    [BankReference] VARCHAR (50) NULL,
    [IP] VARCHAR (50) NULL,
    [ModifiedBy] VARCHAR (100) NULL,
    [ModifiedOn] DATETIME NULL,
    [CreatedBy] VARCHAR (100) NULL,
    [CreatedOn] DATETIME DEFAULT (getdate()) NULL,
    [Description] NVARCHAR (250) NULL,
    [Remark] NVARCHAR (250) NULL,
    [PlayerRemark] NVARCHAR(250) NULL,
    [WebId] INT NOT NULL,
    [DepositTime] DATETIME NULL,
    [SlipImage] NVARCHAR(Max) NULL,
    [BranchName] NVARCHAR(100) NULL,
    [CustomizeBankName] NVARCHAR(100) NULL,
    [IsArbitragePlayer] BIT DEFAULT 0 NOT NULL,
    CONSTRAINT [PK_TransactionRequestHistory] PRIMARY KEY CLUSTERED ([ID] ASC)
)
```

### DailyBetSummaryHistory
Daily aggregated betting summaries grouped by product type, date, and status. Retains data for 61-90 days.

```sql
CREATE TABLE [dbo].[DailyBetSummaryHistory] (
    [Id] [bigint] IDENTITY(1,1) NOT NULL,
    [WebId] INT,
    [ProductType] INT,
    [DisplayName] NVARCHAR(50),
    [CustomerId] INT,
    [ParentId] INT,
    [Currency] CHAR(3),
    [Username] NVARCHAR(50),
    [Winlostdate] DATETIME,
    [BetCount] INT,
    [Stake] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [ActualStake] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [Turnover] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [NetTurnover] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [TurnoverByStake] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [TurnoverByActualStake] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [NetTurnoverByStake] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [NetTurnoverByActualStake] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [Status] NVARCHAR(50),
    [ExchangeRate] DECIMAL (12, 8) NULL,
    [PlayerComm] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [AgtComm] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [MaComm] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [SmaComm] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [SsmaComm] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [ShComm] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [Winlost] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [AgtWinlost] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [MaWinlost] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [SmaWinlost] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [SsmaWinlost] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [ShWinlost] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [CompanyWinlost] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [RowModifiedDate] DATETIME,
    [RowModifiedBy] NVARCHAR(50),
    [IsValidTurnover] BIT DEFAULT ((1)) NOT NULL,
    [NewPromotionRequestId] INT DEFAULT (0) NOT NULL,
    [WithdrawalLimitId] INT DEFAULT (0) NOT NULL,
    CONSTRAINT [PK_DailyBetSummaryHistory] PRIMARY KEY ([Id]) WITH (FILLFACTOR = 85)
)
```

**Product Types:**
- 1 = SBO Sports
- 2 = SBO Live Casino
- 3 = SBO Games
- 4 = SBO Virtual Sports
- 5 = Seamless Games
- 6 = Third Party Sports
- 7 = Live Coin Transaction

### DailyBetSummaryHistoryByTimeZone
Daily betting summaries organized by timezone for accurate regional reporting and multi-timezone support.

```sql
CREATE TABLE [dbo].[DailyBetSummaryHistoryByTimeZone] (
    [Id] BIGINT IDENTITY(1,1) NOT NULL,
    [WebId] INT,
    [ProductType] INT,
    [DisplayName] NVARCHAR(50),
    [CustomerId] INT,
    [ParentId] INT,
    [Currency] CHAR(3),
    [Username] NVARCHAR(50),
    [BetCount] INT,
    [Stake] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [ActualStake] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [Turnover] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [NetTurnover] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [TurnoverByStake] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [TurnoverByActualStake] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [NetTurnoverByStake] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [NetTurnoverByActualStake] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [Status] NVARCHAR(50),
    [ExchangeRate] DECIMAL(12,8) NULL,
    [PlayerComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [AgtComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [MaComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [SmaComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [SsmaComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [ShComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [Winlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [AgtWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [MaWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [SmaWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [SsmaWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [ShWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [CompanyWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [RowModifiedDate] DATETIME,
    [RowModifiedBy] NVARCHAR(50),
    [IsValidTurnover] BIT DEFAULT ((1)) NOT NULL,
    [NewPromotionRequestId] INT DEFAULT (0) NOT NULL,
    [WinlostDateByTimeZone] DATETIME NULL,
    [TimeZone] NVARCHAR(100) NULL,
    [WithdrawalLimitId] INT DEFAULT (0) NOT NULL,
    CONSTRAINT [PK_DailyBetSummaryHistoryByTimeZone] PRIMARY KEY ([Id]) WITH (FILLFACTOR = 85)
)
```

**Key Features:**
- **Timezone Support**: `WinlostDateByTimeZone` and `TimeZone` fields for accurate regional reporting
- **Product Type Categorization**: Same as DailyBetSummaryHistory (1=SBO Sports, 2=SBO Live Casino, etc.)
- **Multi-level Commission Tracking**: Complete hierarchy from Player to Super Head (SH)
- **Performance Optimization**: Includes covering index for reporting queries

**Product Types:**
- 1 = SBO Sports
- 2 = SBO Live Casino  
- 3 = SBO Games
- 4 = SBO Virtual Sports
- 5 = Seamless Games
- 6 = Third Party Sports
- 7 = Live Coin Transaction

## Sports Betting History Tables

### SboSportBetsHistory
Comprehensive history of sports betting activities including detailed bet information, odds, and commission structures.

```sql
CREATE TABLE [dbo].[SboSportBetsHistory] (
    [Id] BIGINT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [RefNo] NVARCHAR(100) NOT NULL,
    [TransactionId] NVARCHAR(100) NOT NULL,
    [CustomerId] INT NOT NULL,
    [ParentId] INT NULL,
    [Username] NVARCHAR(50) NOT NULL,
    [UsernameAtGameProvider] NVARCHAR(50) NULL,
    [PromotionRequestId] INT DEFAULT ((0)) NOT NULL,
    [NewPromotionRequestId] INT DEFAULT ((0)) NOT NULL,
    [ReferralEventId] INT DEFAULT ((0)) NOT NULL,
    [WalletProductType] INT DEFAULT ((0)) NOT NULL,
    [WalletGameType] INT DEFAULT ((0)) NOT NULL,
    [SportsType] NVARCHAR(50) NULL,
    [BetType] INT NULL,
    [OrderTime] DATETIME NOT NULL,
    [ModifiedTime] DATETIME NOT NULL,
    [WinlostDate] DATETIME NULL,
    [Odds] DECIMAL (12, 3) DEFAULT ((0)) NULL,
    [OddsStyle] NVARCHAR(10) NULL,
    [Stake] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [ActualStake] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [Currency] CHAR(3) NULL,
    [ExchangeRate] DECIMAL (12, 8) NULL,
    [StatusAtGameProvider] VARCHAR (50) NULL,
    [Status] VARCHAR (50) NULL,
    [Winlost] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [TurnOver] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [IsLive] BIT NULL,
    [IsHalfWinLose] BIT NULL,
    [MaxWinWithoutActualStake] DECIMAL (19,6),
    [AgtPT] DECIMAL (3, 2) DEFAULT ((0)) NULL,
    [MaPT] DECIMAL (3, 2) DEFAULT ((0)) NULL,
    [SmaPT] DECIMAL (3, 2) DEFAULT ((0)) NULL,
    [SsmaPT] DECIMAL (3, 2) DEFAULT ((0)) NULL,
    [ShPT] DECIMAL (3, 2) DEFAULT ((0)) NULL,
    [TotalPT] DECIMAL (3, 2) DEFAULT ((0)) NULL,
    [AgtWinlost] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [MaWinlost] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [SmaWinlost] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [SsmaWinlost] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [ShWinlost] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [PlayerDiscount] DECIMAL (5, 4) DEFAULT ((0)) NULL,
    [AgtDiscount] DECIMAL (5, 4) DEFAULT ((0)) NULL,
    [MaDiscount] DECIMAL (5, 4) DEFAULT ((0)) NULL,
    [SmaDiscount] DECIMAL (5, 4) DEFAULT ((0)) NULL,
    [SsmaDiscount] DECIMAL (5, 4) DEFAULT ((0)) NULL,
    [ShDiscount] DECIMAL (5, 4) DEFAULT ((0)) NULL,
    [PlayerComm] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [AgtComm] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [MaComm] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [SmaComm] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [SsmaComm] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [ShComm] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [Ip] NVARCHAR (100) DEFAULT ('') NULL,
    [IsSystemTagRisky] BIT,
    [SyncTime] DATETIME,
    [BetFrom] NVARCHAR(20) DEFAULT '' NOT NULL,
    [ReferralLayerPercentageHistoryIds] nvarchar (1000),
    [VoidReason] NVARCHAR(500) DEFAULT ('') NULL,
    [Remark] NVARCHAR(500) DEFAULT ('') NULL,
    [WinlostDateByTimeZone] DATETIME NULL,
    [TimeZone] NVARCHAR(100) NULL,
    [SportGameId] NVARCHAR(100) NULL,
    [WithdrawalLimitId] INT DEFAULT(0) NOT NULL,
    CONSTRAINT [PK_SboSportBetsHistory] PRIMARY KEY ([RefNo],[Webid]) WITH (FILLFACTOR = 85)
)
```

### SboCasinoBetsHistory
Historical records of casino betting activities with detailed game and commission information.

```sql
CREATE TABLE [dbo].[SboCasinoBetsHistory] (
    [Id] BIGINT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [RefNo] NVARCHAR(100) NOT NULL,
    [TransactionId] NVARCHAR(100) NOT NULL,
    [CustomerId] INT NOT NULL,
    [ParentId] INT NULL,
    [Username] NVARCHAR(50) NOT NULL,
    [UsernameAtGameProvider] NVARCHAR(50) NULL,
    [PromotionRequestId] INT DEFAULT ((0)) NOT NULL,
    [NewPromotionRequestId] INT DEFAULT ((0)) NOT NULL,
    [ReferralEventId] INT DEFAULT ((0)) NOT NULL,
    [WalletProductType] INT DEFAULT ((0)) NOT NULL,
    [WalletGameType] INT DEFAULT ((0)) NOT NULL,
    [ProductType] NVARCHAR(50) NULL,
    [GameId] BIGINT NULL,
    [TableName] NVARCHAR(100) NULL,
    [OrderTime] DATETIME NOT NULL,
    [WinLostDate] DATETIME NULL,
    [Stake] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [Currency] CHAR(3) NULL,
    [ExchangeRate] DECIMAL (12, 8) NULL,
    [TurnOver] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [StatusAtGameProvider] VARCHAR (50) NULL,
    [Status] VARCHAR (50) NULL,
    [Winlost] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [AgtPT] DECIMAL (3, 2) DEFAULT ((0)) NULL,
    [MaPT] DECIMAL (3, 2) DEFAULT ((0)) NULL,
    [SmaPT] DECIMAL (3, 2) DEFAULT ((0)) NULL,
    [SsmaPT] DECIMAL (3, 2) DEFAULT ((0)) NULL,
    [ShPT] DECIMAL (3, 2) DEFAULT ((0)) NULL,
    [TotalPT] DECIMAL (3, 2) DEFAULT ((0)) NULL,
    [AgtWinlost] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [MaWinlost] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [SmaWinlost] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [SsmaWinlost] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [ShWinlost] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [PlayerCommissionRate] DECIMAL (5, 4) DEFAULT ((0)) NULL,
    [AgtCommissionRate] DECIMAL (5, 4) DEFAULT ((0)) NULL,
    [MaCommissionRate] DECIMAL (5, 4) DEFAULT ((0)) NULL,
    [SmaCommissionRate] DECIMAL (5, 4) DEFAULT ((0)) NULL,
    [SsmaCommissionRate] DECIMAL (5, 4) DEFAULT ((0)) NULL,
    [ShCommissionRate] DECIMAL (5, 4) DEFAULT ((0)) NULL,
    [PlayerComm] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [AgtComm] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [MaComm] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [SmaComm] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [SsmaComm] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [ShComm] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [SyncTime] DATETIME,
    [ModifiedTime] DATETIME,
    [IsDailyStatementProceed] BIT DEFAULT (0) NOT NULL,
    [ReferralLayerPercentageHistoryIds] nvarchar (1000),
    [WinlostDateByTimeZone] DATETIME NULL,
    [TimeZone] NVARCHAR(100) NULL,
    [WithdrawalLimitId] INT DEFAULT(0) NOT NULL,
    CONSTRAINT [PK_SboCasinoBetsHistory] PRIMARY KEY ([RefNo],[Webid]) WITH (FILLFACTOR = 85)
)
```

### SboSeamlessBetsHistory
History of seamless betting activities with comprehensive game provider integration details.

```sql
CREATE TABLE [dbo].[SboSeamlessBetsHistory] (
    [Id] BIGINT IDENTITY(1,1) NOT NULL,
    [GPID] int NULL,
    [WebId] INT NOT NULL,
    [RefNo] NVARCHAR(100) NOT NULL,
    [TransactionId] NVARCHAR(100) NOT NULL,
    [CustomerId] INT NOT NULL,
    [ParentId] INT NULL,
    [Username] NVARCHAR(50) NOT NULL,
    [UsernameAtGameProvider] NVARCHAR(50) NULL,
    [PromotionRequestId] INT DEFAULT ((0)) NOT NULL,
    [NewPromotionRequestId] INT DEFAULT ((0)) NOT NULL,
    [ReferralEventId] INT DEFAULT ((0)) NOT NULL,
    [WalletProductType] INT DEFAULT ((0)) NOT NULL,
    [WalletGameType] INT DEFAULT ((0)) NOT NULL,
    [OrderTime] DATETIME NOT NULL,
    [WinLostDate] DATETIME NULL,
    [Stake] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [Currency] CHAR(3) NULL,
    [ExchangeRate] DECIMAL (12, 8) NULL,
    [TurnOver] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [StatusAtGameProvider] VARCHAR (50) NULL,
    [Status] VARCHAR (50) NULL,
    [IsFishGame] BIT DEFAULT ((0)) NULL,
    [GamePeriodId] NVARCHAR(100) NULL,
    [GameRoundId] NVARCHAR(100) NULL,
    [GameType] NVARCHAR(50) NULL,
    [OrderDetail] NVARCHAR(4000) NULL,
    [GameResult] NVARCHAR(4000) NULL,
    [Winlost] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [AgtPT] DECIMAL (3, 2) DEFAULT ((0)) NULL,
    [MaPT] DECIMAL (3, 2) DEFAULT ((0)) NULL,
    [SmaPT] DECIMAL (3, 2) DEFAULT ((0)) NULL,
    [SsmaPT] DECIMAL (3, 2) DEFAULT ((0)) NULL,
    [ShPT] DECIMAL (3, 2) DEFAULT ((0)) NULL,
    [TotalPT] DECIMAL (3, 2) DEFAULT ((0)) NULL,
    [AgtWinlost] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [MaWinlost] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [SmaWinlost] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [SsmaWinlost] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [ShWinlost] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [PlayerCommissionRate] DECIMAL (5, 4) DEFAULT ((0)) NULL,
    [AgtCommissionRate] DECIMAL (5, 4) DEFAULT ((0)) NULL,
    [MaCommissionRate] DECIMAL (5, 4) DEFAULT ((0)) NULL,
    [SmaCommissionRate] DECIMAL (5, 4) DEFAULT ((0)) NULL,
    [SsmaCommissionRate] DECIMAL (5, 4) DEFAULT ((0)) NULL,
    [ShCommissionRate] DECIMAL (5, 4) DEFAULT ((0)) NULL,
    [PlayerComm] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [AgtComm] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [MaComm] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [SmaComm] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [SsmaComm] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [ShComm] DECIMAL (19, 6) DEFAULT ((0)) NULL,
    [SyncTime] DATETIME,
    [ModifiedTime] DATETIME,
    [IsDailyStatementProceed] BIT DEFAULT (0) NOT NULL,
    [ReferralLayerPercentageHistoryIds] nvarchar (1000),
    [MaId] INT DEFAULT ((0)) NOT NULL,
    [GameIdAtGameProvider] INT DEFAULT ((0)) NOT NULL,
    [GameId] INT NULL,
    [IsFeatureBuy] BIT DEFAULT(0),
    [NewGameType] INT,
    [MainBetId] NVARCHAR(255) NULL,
    [GameExtraInfo] NVARCHAR(4000) NULL,
    [WinlostDateByTimeZone] DATETIME NULL,
    [TimeZone] NVARCHAR(100) NULL,
    [WithdrawalLimitId] INT DEFAULT(0) NOT NULL,
    CONSTRAINT [PK_SboSeamlessBetsHistory] PRIMARY KEY ([ID]) WITH (FILLFACTOR = 85)
)
```

## Game Provider History Tables

### SboVirtualSportBetsHistory
Historical records of virtual sports betting activities with comprehensive commission and profit tracking.

```sql
CREATE TABLE [dbo].[SboVirtualSportBetsHistory] (
    [Id] BIGINT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [RefNo] NVARCHAR(100) NOT NULL,
    [TransactionId] NVARCHAR(100) NOT NULL,
    [CustomerId] INT NOT NULL,
    [ParentId] INT NULL,
    [Username] NVARCHAR(50) NOT NULL,
    [UsernameAtGameProvider] NVARCHAR(50) NULL,
    [PromotionRequestId] INT DEFAULT ((0)) NOT NULL,
    [NewPromotionRequestId] INT DEFAULT ((0)) NOT NULL,
    [ReferralEventId] INT DEFAULT ((0)) NOT NULL,
    [WalletProductType] INT DEFAULT ((0)) NOT NULL,
    [WalletGameType] INT DEFAULT ((0)) NOT NULL,
    [GameId] INT NULL,
    [ProductType] NVARCHAR(50) NULL,
    [OrderTime] DATETIME NOT NULL,
    [ModifiedTime] DATETIME NOT NULL,
    [WinLostDate] DATETIME NULL,
    [Odds] DECIMAL(12,3) NULL,
    [OddsStyle] NVARCHAR(10) NULL,
    [Stake] DECIMAL(19,6) NULL,
    [ActualStake] DECIMAL(19,6) NOT NULL,
    [Turnover] DECIMAL(19,6) NULL,
    [WinLost] DECIMAL(19,6) NULL,
    [StatusAtGameProvider] VARCHAR(50) NULL,
    [Status] VARCHAR(50) NULL,
    [Currency] CHAR(3) NOT NULL,
    [ActualRate] DECIMAL(12,8) NOT NULL,
    [AgtPT] DECIMAL(3,2) DEFAULT ((0)) NULL,
    [MaPT] DECIMAL(3,2) DEFAULT ((0)) NULL,
    [SmaPT] DECIMAL(3,2) DEFAULT ((0)) NULL,
    [SsmaPT] DECIMAL(3,2) DEFAULT ((0)) NULL,
    [ShPT] DECIMAL(3,2) DEFAULT ((0)) NULL,
    [TotalPT] DECIMAL(3,2) DEFAULT ((0)) NULL,
    [AgtWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [MaWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [SmaWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [SsmaWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [ShWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [PlayerCommissionRate] DECIMAL(5,4) DEFAULT ((0)) NULL,
    [AgtCommissionRate] DECIMAL(5,4) DEFAULT ((0)) NULL,
    [MaCommissionRate] DECIMAL(5,4) DEFAULT ((0)) NULL,
    [SmaCommissionRate] DECIMAL(5,4) DEFAULT ((0)) NULL,
    [SsmaCommissionRate] DECIMAL(5,4) DEFAULT ((0)) NULL,
    [ShCommissionRate] DECIMAL(5,4) DEFAULT ((0)) NULL,
    [PlayerComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [AgtComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [MaComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [SmaComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [SsmaComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [ShComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [SyncTime] DATETIME,
    [IsDailyStatementProceed] BIT DEFAULT (0) NOT NULL,
    [ReferralLayerPercentageHistoryIds] NVARCHAR(1000),
    [WinlostDateByTimeZone] DATETIME NULL,
    [TimeZone] NVARCHAR(100) NULL,
    [WithdrawalLimitId] INT DEFAULT(0) NOT NULL,
    CONSTRAINT [PK_SboVirtualSportBetsHistory] PRIMARY KEY ([RefNo],[Webid]) WITH (FILLFACTOR = 85)
)
```

### SboGamesBetsHistory
Historical records of general game betting activities with detailed commission tracking.

```sql
CREATE TABLE [dbo].[SboGamesBetsHistory] (
    [Id] BIGINT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [RefNo] NVARCHAR(100) NOT NULL,
    [TransactionId] NVARCHAR(100) NOT NULL,
    [CustomerId] INT NOT NULL,
    [ParentId] INT NULL,
    [Username] NVARCHAR(50) NOT NULL,
    [UsernameAtGameProvider] NVARCHAR(50) NULL,
    [PromotionRequestId] INT DEFAULT ((0)) NOT NULL,
    [NewPromotionRequestId] INT DEFAULT ((0)) NOT NULL,
    [ReferralEventId] INT DEFAULT ((0)) NOT NULL,
    [WalletProductType] INT DEFAULT ((0)) NOT NULL,
    [WalletGameType] INT DEFAULT ((0)) NOT NULL,
    [ProductType] NVARCHAR(50) NULL,
    [GameId] BIGINT NULL,
    [GameName] NVARCHAR(100) NULL,
    [OrderTime] DATETIME NOT NULL,
    [WinLostDate] DATETIME NULL,
    [Stake] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [Currency] CHAR(3) NULL,
    [ExchangeRate] DECIMAL(12,8) NULL,
    [TurnOver] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [StatusAtGameProvider] VARCHAR(50) NULL,
    [Status] VARCHAR(50) NULL,
    [Winlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [AgtPT] DECIMAL(3,2) DEFAULT ((0)) NULL,
    [MaPT] DECIMAL(3,2) DEFAULT ((0)) NULL,
    [SmaPT] DECIMAL(3,2) DEFAULT ((0)) NULL,
    [SsmaPT] DECIMAL(3,2) DEFAULT ((0)) NULL,
    [ShPT] DECIMAL(3,2) DEFAULT ((0)) NULL,
    [TotalPT] DECIMAL(3,2) DEFAULT ((0)) NULL,
    [AgtWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [MaWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [SmaWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [SsmaWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [ShWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [PlayerCommissionRate] DECIMAL(5,4) DEFAULT ((0)) NULL,
    [AgtCommissionRate] DECIMAL(5,4) DEFAULT ((0)) NULL,
    [MaCommissionRate] DECIMAL(5,4) DEFAULT ((0)) NULL,
    [SmaCommissionRate] DECIMAL(5,4) DEFAULT ((0)) NULL,
    [SsmaCommissionRate] DECIMAL(5,4) DEFAULT ((0)) NULL,
    [ShCommissionRate] DECIMAL(5,4) DEFAULT ((0)) NULL,
    [PlayerComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [AgtComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [MaComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [SmaComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [SsmaComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [ShComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [SyncTime] DATETIME,
    [ModifiedTime] DATETIME,
    [IsDailyStatementProceed] BIT DEFAULT (0) NOT NULL,
    [ReferralLayerPercentageHistoryIds] NVARCHAR(1000),
    [WinlostDateByTimeZone] DATETIME NULL,
    [TimeZone] NVARCHAR(100) NULL,
    [WithdrawalLimitId] INT DEFAULT(0) NOT NULL,
    CONSTRAINT [PK_SboGamesBetsHistory] PRIMARY KEY ([RefNo],[Webid]) WITH (FILLFACTOR = 85)
)
```

### SboTransferGameproviderBetsHistory
Historical records of game provider transfer transactions with comprehensive commission and game details.

```sql
CREATE TABLE [dbo].[SboTransferGameproviderBetsHistory] (
    [Id] BIGINT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [RefNo] NVARCHAR(100) NOT NULL,
    [TransactionId] NVARCHAR(100) NOT NULL,
    [CustomerId] INT NOT NULL,
    [ParentId] INT NULL,
    [MaId] INT NOT NULL,
    [Username] NVARCHAR(50) NOT NULL,
    [UsernameAtGameProvider] NVARCHAR(50) NULL,
    [GameProviderId] BIGINT NULL,
    [GameIdAtGameProvider] BIGINT NULL,
    [GameRoundId] NVARCHAR(100) NULL,
    [GamePeriodId] NVARCHAR(100) NULL,
    [GameType] NVARCHAR(50) NULL,
    [OrderDetail] NVARCHAR(4000) NULL,
    [GameResult] NVARCHAR(4000) NULL,
    [Currency] CHAR(3) NULL,
    [ExchangeRate] DECIMAL(12,8) NULL,
    [Status] NVARCHAR(50) NULL,
    [StatusAtGameProvider] NVARCHAR(50) NULL,
    [Stake] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [TurnOver] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [Winlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [MjpWinlose] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [MjpCommission] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [WinLostDate] DATETIME NULL,
    [AgtPT] DECIMAL(3,2) DEFAULT ((0)) NULL,
    [MaPT] DECIMAL(3,2) DEFAULT ((0)) NULL,
    [SmaPT] DECIMAL(3,2) DEFAULT ((0)) NULL,
    [SsmaPT] DECIMAL(3,2) DEFAULT ((0)) NULL,
    [ShPT] DECIMAL(3,2) DEFAULT ((0)) NULL,
    [TotalPT] DECIMAL(3,2) DEFAULT ((0)) NULL,
    [AgtWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [MaWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [SmaWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [SsmaWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [ShWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [PlayerCommissionRate] DECIMAL(5,4) DEFAULT ((0)) NULL,
    [AgtCommissionRate] DECIMAL(5,4) DEFAULT ((0)) NULL,
    [MaCommissionRate] DECIMAL(5,4) DEFAULT ((0)) NULL,
    [SmaCommissionRate] DECIMAL(5,4) DEFAULT ((0)) NULL,
    [SsmaCommissionRate] DECIMAL(5,4) DEFAULT ((0)) NULL,
    [ShCommissionRate] DECIMAL(5,4) DEFAULT ((0)) NULL,
    [PlayerComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [AgtComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [MaComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [SmaComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [SsmaComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [ShComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [SyncTime] DATETIME,
    [ModifiedTime] DATETIME,
    [SettledTime] DATETIME,
    [OrderedTime] DATETIME,
    [IsDailyStatementProceed] BIT DEFAULT (0) NOT NULL,
    [IsCommissionCheck] BIT DEFAULT(0),
    [IsPromotionCheck] BIT DEFAULT(0),
    [IsReferralCheck] BIT DEFAULT(0),
    [IsWithdrawalLimitCheck] BIT DEFAULT(0),
    [ReferralLayerPercentageHistoryIds] NVARCHAR(1000),
    [PromotionRequestId] INT NOT NULL DEFAULT(0),
    [WalletProductType] INT NOT NULL DEFAULT(0),
    [WalletGameType] INT NOT NULL DEFAULT(0),
    [IsFishGame] BIT,
    CONSTRAINT [PK_SboTransferGameProviderBetsHistory] PRIMARY KEY ([RefNo],[Webid]) WITH (FILLFACTOR = 85)
)
```

**Key Features:**
- **Game Provider Integration**: Comprehensive tracking of transfers between game providers
- **Multi-Jackpot Progressive Support**: MJP win/loss and commission tracking
- **Complete Commission Structure**: Full hierarchy commission tracking and rates
- **Game Details**: Round ID, period ID, game type, and detailed game results
- **Status Tracking**: Multiple status fields for different processing stages
- **Audit Flags**: Various check flags for commission, promotion, referral, and withdrawal limit processing

### SboThirdPartySportBetsHistory
Historical records of third-party sports betting activities with comprehensive commission and parlay support.

```sql
CREATE TABLE [dbo].[SboThirdPartySportBetsHistory] (
    [Id] BIGINT NOT NULL,
    [WebId] INT NOT NULL,
    [GPID] INT NOT NULL,
    [RefNo] NVARCHAR(100) NOT NULL,
    [ProductType] NVARCHAR(50) NULL,
    [ParlayRefNo] NVARCHAR(100) NULL,
    [IsParlay] BIT DEFAULT ((0)) NOT NULL,
    [IsHalfWinlose] BIT DEFAULT ((0)) NOT NULL,
    [TransactionId] NVARCHAR(100) NOT NULL,
    [CustomerId] INT NOT NULL,
    [ParentId] INT NULL,
    [Username] NVARCHAR(50) NOT NULL,
    [UsernameAtGameProvider] NVARCHAR(50) NULL,
    [PromotionRequestId] INT DEFAULT ((0)) NOT NULL,
    [NewPromotionRequestId] INT DEFAULT ((0)) NOT NULL,
    [ReferralEventId] INT DEFAULT ((0)) NOT NULL,
    [WalletProductType] INT DEFAULT ((0)) NOT NULL,
    [WalletGameType] INT DEFAULT ((0)) NOT NULL,
    [OrderTime] DATETIME NOT NULL,
    [ModifiedTime] DATETIME NOT NULL,
    [WinlostDate] DATETIME NULL,
    [Odds] DECIMAL(12,3) DEFAULT ((0)) NULL,
    [OddsStyle] NVARCHAR(10) NULL,
    [Stake] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [ActualStake] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [Currency] CHAR(3) NULL,
    [ExchangeRate] DECIMAL(12,8) NULL,
    [StatusAtGameProvider] VARCHAR(50) NULL,
    [Status] VARCHAR(50) NULL,
    [Winlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [TurnOver] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [MaxWinWithoutActualStake] DECIMAL(19,6),
    [AgtPT] DECIMAL(3,2) DEFAULT ((0)) NULL,
    [MaPT] DECIMAL(3,2) DEFAULT ((0)) NULL,
    [SmaPT] DECIMAL(3,2) DEFAULT ((0)) NULL,
    [SsmaPT] DECIMAL(3,2) DEFAULT ((0)) NULL,
    [ShPT] DECIMAL(3,2) DEFAULT ((0)) NULL,
    [TotalPT] DECIMAL(3,2) DEFAULT ((0)) NULL,
    [AgtWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [MaWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [SmaWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [SsmaWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [ShWinlost] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [PlayerDiscount] DECIMAL(5,4) DEFAULT ((0)) NULL,
    [AgtDiscount] DECIMAL(5,4) DEFAULT ((0)) NULL,
    [MaDiscount] DECIMAL(5,4) DEFAULT ((0)) NULL,
    [SmaDiscount] DECIMAL(5,4) DEFAULT ((0)) NULL,
    [SsmaDiscount] DECIMAL(5,4) DEFAULT ((0)) NULL,
    [ShDiscount] DECIMAL(5,4) DEFAULT ((0)) NULL,
    [PlayerComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [AgtComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [MaComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [SmaComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [SsmaComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [ShComm] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [Ip] NVARCHAR(100) DEFAULT ('') NULL,
    [SyncTime] DATETIME,
    [ReferralLayerPercentageHistoryIds] NVARCHAR(1000),
    [GameRoundId] NVARCHAR(100) DEFAULT ('') NULL,
    [GamePeriodId] NVARCHAR(100) DEFAULT ('') NULL,
    [WinlostDateByTimeZone] DATETIME NULL,
    [TimeZone] NVARCHAR(100) NULL,
    [WithdrawalLimitId] INT DEFAULT(0) NOT NULL,
    CONSTRAINT [PK_SboThirdPartySportBetsHistory] PRIMARY KEY ([RefNo],[Webid]) WITH (FILLFACTOR = 85)
)
```

## System History Tables

### UserActionRecordHistory
Tracks user actions and interactions for audit purposes.

```sql
CREATE TABLE [dbo].[UserActionRecordHistory] (
    [Id] BIGINT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [CustomerId] INT NOT NULL,
    [Username] NVARCHAR(50),
    [ActionType] NVARCHAR(50),
    [ActionUrl] NVARCHAR(MAX),
    [ActionOn] DATETIME,
    [IP] NVARCHAR(200),
    [FingerPrint] NVARCHAR(MAX)
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | BIGINT, PK | Primary key, auto-incrementing unique identifier |
| WebId | INT | Foreign key reference to the website/application |
| CustomerId | INT | Foreign key reference to the customer |
| Username | NVARCHAR(50) | Username of the user who performed the action |
| ActionType | NVARCHAR(50) | Type/category of the action performed |
| ActionUrl | NVARCHAR(MAX) | URL where the action was performed |
| ActionOn | DATETIME | Timestamp when the action occurred |
| IP | NVARCHAR(200) | IP address from which the action was performed |
| FingerPrint | NVARCHAR(MAX) | Browser fingerprint for additional security tracking |

### PromotionSystemRejectRecordHistory
Historical records of rejected promotion system requests with detailed rejection information.

```sql
CREATE TABLE [dbo].[PromotionSystemRejectRecordHistory] (
    [Id] BIGINT IDENTITY(1,1) NOT NULL,
    [WebId] INT,
    [CustomerId] INT NOT NULL,
    [Username] NVARCHAR(100) NOT NULL,
    [ActionType] NVARCHAR(100) NOT NULL,
    [RejectSetting] NVARCHAR(100) NOT NULL,
    [PromotionRequestId] INT NOT NULL,
    [CompanyPromotionId] INT NOT NULL,
    [Remark] NVARCHAR(MAX) NOT NULL,
    [CreatedOn] DATETIME NOT NULL DEFAULT GETDATE()
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | BIGINT, PK | Primary key, auto-incrementing unique identifier |
| WebId | INT | Website identifier |
| CustomerId | INT | Customer identifier who was rejected |
| Username | NVARCHAR(100) | Username of the customer |
| ActionType | NVARCHAR(100) | Type of promotion action that was rejected |
| RejectSetting | NVARCHAR(100) | Specific setting that caused the rejection |
| PromotionRequestId | INT | Associated promotion request identifier |
| CompanyPromotionId | INT | Company promotion identifier |
| Remark | NVARCHAR(MAX) | Detailed rejection reason and remarks |
| CreatedOn | DATETIME | Record creation timestamp |

### CurrencyHistory
Historical exchange rate records for different currencies.

```sql
CREATE TABLE [dbo].[CurrencyHistory] (
    [WebId] INT NOT NULL,
    [Currency] NVARCHAR(10) NOT NULL,
    [ExchangeRate] DECIMAL (10, 6) NOT NULL,
    [StartDate] DATETIME NOT NULL,
    [EndDate] DATETIME NOT NULL,
    CONSTRAINT [PK_CurrencyHistory] PRIMARY KEY ([WebId],[Currency],[StartDate],[EndDate]) WITH (FILLFACTOR = 85)
)
```

| Column | Type | Description |
|--------|------|-------------|
| WebId | INT, PK | Website identifier |
| Currency | NVARCHAR(10), PK | Currency code |
| ExchangeRate | DECIMAL(10,6) | Exchange rate value |
| StartDate | DATETIME, PK | Start date of the exchange rate period |
| EndDate | DATETIME, PK | End date of the exchange rate period |

### CurrencyUsdHistory
Historical USD exchange rate records for currency conversion tracking.

```sql
CREATE TABLE [dbo].[CurrencyUsdHistory] (
    [Currency] CHAR(3) NOT NULL,
    [ExchangeRate] DECIMAL(10,6) NULL,
    [StartDate] DATETIME NOT NULL,
    [EndDate] DATETIME NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Currency | CHAR(3) | Currency code (3-letter ISO code) |
| ExchangeRate | DECIMAL(10,6) | Exchange rate against USD |
| StartDate | DATETIME | Start date of the exchange rate period |
| EndDate | DATETIME | End date of the exchange rate period |

## Specialized History Tables

### SboLiveCoinHistory
Historical records of live coin transactions and activities with comprehensive betting details.

```sql
CREATE TABLE [dbo].[SboLiveCoinHistory] (
    [Id] BIGINT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [RefNo] NVARCHAR(100) NOT NULL,
    [TransactionId] NVARCHAR(100) NOT NULL,
    [WinlostDate] DATETIME NOT NULL,
    [OrderTime] DATETIME NOT NULL,
    [Status] NVARCHAR(50),
    [CustomerId] INT NOT NULL,
    [ParentId] INT NULL,
    [Username] NVARCHAR(50),
    [Detail] NVARCHAR(100),
    [Stake] DECIMAL(19,6) NOT NULL,
    [Winlose] DECIMAL(19,6) NOT NULL,
    [Currency] CHAR(3) NOT NULL,
    [ExchangeRate] DECIMAL(12,8) NULL,
    [CreatedOn] DATETIME NOT NULL DEFAULT(GETDATE()),
    [ModifiedOn] DATETIME NOT NULL DEFAULT(GETDATE()),
    [PromotionRequestId] INT DEFAULT(0) NOT NULL,
    CONSTRAINT [PK_SboLiveCoinHistory] PRIMARY KEY ([Id]) WITH (FILLFACTOR = 85)
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | BIGINT, PK | Primary key, auto-incrementing unique identifier |
| WebId | INT | Website identifier |
| RefNo | NVARCHAR(100) | Reference number for the transaction |
| TransactionId | NVARCHAR(100) | Transaction identifier from game provider |
| WinlostDate | DATETIME | Date when the win/loss was determined |
| OrderTime | DATETIME | Timestamp when the order was placed |
| Status | NVARCHAR(50) | Current status of the transaction |
| CustomerId | INT | Customer identifier |
| ParentId | INT | Parent customer identifier (for hierarchy) |
| Username | NVARCHAR(50) | Username of the player |
| Detail | NVARCHAR(100) | Additional transaction details |
| Stake | DECIMAL(19,6) | Amount staked in the transaction |
| Winlose | DECIMAL(19,6) | Win/loss amount for the transaction |
| Currency | CHAR(3) | Currency code (3-letter ISO code) |
| ExchangeRate | DECIMAL(12,8) | Exchange rate used for currency conversion |
| CreatedOn | DATETIME | Record creation timestamp |
| ModifiedOn | DATETIME | Last modification timestamp |
| PromotionRequestId | INT | Associated promotion request identifier |

### LionKingMJPHistory
Historical records specific to Lion King MJP (Multi-Jackpot Progressive) game activities.

```sql
CREATE TABLE [dbo].[LionKingMJPHistory] (
    [Id] BIGINT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [RefNo] NVARCHAR(100) NOT NULL,
    [CustomerId] INT NOT NULL,
    [MjpWinlose] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [MjpCommission] DECIMAL(19,6) DEFAULT ((0)) NULL,
    [ModifiedTime] DATETIME,
    CONSTRAINT [PK_LionKingMJPHistory] PRIMARY KEY ([Id]) WITH (FILLFACTOR = 85)
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | BIGINT, PK | Primary key, auto-incrementing unique identifier |
| WebId | INT | Website identifier |
| RefNo | NVARCHAR(100) | Reference number for the MJP transaction |
| CustomerId | INT | Customer identifier |
| MjpWinlose | DECIMAL(19,6) | Multi-jackpot progressive win/loss amount |
| MjpCommission | DECIMAL(19,6) | Commission amount for MJP |
| ModifiedTime | DATETIME | Last modification timestamp |

### SboSportBetDetailsHistory
Detailed historical records of sports bet specifics with comprehensive match and betting information.

```sql
CREATE TABLE [dbo].[SboSportBetDetailsHistory] (
    [Id] BIGINT NOT NULL,
    [RefNo] NVARCHAR(50) NOT NULL,
    [IndentityString] NVARCHAR(200) NOT NULL,
    [BetOption] NVARCHAR(100) NULL,
    [MarketType] NVARCHAR(50) NULL,
    [BetType] INT DEFAULT ((0)) NULL,
    [SportType] VARCHAR(50) NULL,
    [HandicapPoint] DECIMAL(12,2) DEFAULT ((0)) NULL,
    [Odds] DECIMAL(12,3) DEFAULT ((0)) NULL,
    [League] NVARCHAR(150) NULL,
    [Match] NVARCHAR(150) NULL,
    [Status] VARCHAR(50) NULL,
    [IsHalfWinlose] BIT NULL,
    [WinlostDate] DATETIME NULL,
    [LiveScore] VARCHAR(50) NULL,
    [HalfTimeScore] VARCHAR(50) NULL,
    [FullTimeScore] VARCHAR(50) NULL,
    [CustomizedBetType] VARCHAR(50) NULL,
    [WebId] INT NOT NULL,
    [SyncTime] DATETIME,
    [KickOffTime] DATETIME NULL,
    [Language] NVARCHAR(10) DEFAULT('en') NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | BIGINT | Unique identifier for the bet detail record |
| RefNo | NVARCHAR(50) | Reference number linking to main bet record |
| IndentityString | NVARCHAR(200) | Identity string for the bet detail |
| BetOption | NVARCHAR(100) | Specific betting option selected |
| MarketType | NVARCHAR(50) | Type of betting market |
| BetType | INT | Numeric bet type identifier |
| SportType | VARCHAR(50) | Type of sport |
| HandicapPoint | DECIMAL(12,2) | Handicap points for spread betting |
| Odds | DECIMAL(12,3) | Betting odds for this selection |
| League | NVARCHAR(150) | League or competition name |
| Match | NVARCHAR(150) | Match description or teams |
| Status | VARCHAR(50) | Current status of the bet |
| IsHalfWinlose | BIT | Indicates if this is a half win/loss scenario |
| WinlostDate | DATETIME | Date when the bet was settled |
| LiveScore | VARCHAR(50) | Live score information |
| HalfTimeScore | VARCHAR(50) | Score at half-time |
| FullTimeScore | VARCHAR(50) | Final score of the match |
| CustomizedBetType | VARCHAR(50) | Customized bet type description |
| WebId | INT | Website identifier |
| SyncTime | DATETIME | Data synchronization timestamp |
| KickOffTime | DATETIME | Match kick-off time |
| Language | NVARCHAR(10) | Language code for localization |

### SboVirtualSportSubBetsHistory
Sub-bet details for virtual sports betting activities with match information and scoring details.

```sql
CREATE TABLE [dbo].[SboVirtualSportSubBetsHistory] (
    [Id] INT NOT NULL IDENTITY(1,1) PRIMARY KEY,
    [WebId] INT NOT NULL,
    [RefNo] NVARCHAR(100) NOT NULL,
    [Match] NVARCHAR(100) NOT NULL,
    [WinLostDate] DATETIME NOT NULL,
    [Status] NVARCHAR(15) NOT NULL,
    [BetOption] NVARCHAR(30) NOT NULL,
    [MarketType] NVARCHAR(50) NOT NULL,
    [Hdp] NVARCHAR(10) NOT NULL,
    [Odds] DECIMAL(12,3) NOT NULL,
    [HalfTimeScore] NVARCHAR(20) NOT NULL,
    [FullTimeScore] NVARCHAR(20) NOT NULL,
    [SyncTime] DATETIME NOT NULL
)
```

| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Primary key, auto-incrementing unique identifier |
| WebId | INT | Website identifier |
| RefNo | NVARCHAR(100) | Reference number linking to parent bet |
| Match | NVARCHAR(100) | Match identifier or description |
| WinLostDate | DATETIME | Date when the bet result was determined |
| Status | NVARCHAR(15) | Current status of the sub-bet |
| BetOption | NVARCHAR(30) | Specific betting option for this sub-bet |
| MarketType | NVARCHAR(50) | Type of betting market |
| Hdp | NVARCHAR(10) | Handicap value for the bet |
| Odds | DECIMAL(12,3) | Betting odds for this sub-bet |
| HalfTimeScore | NVARCHAR(20) | Score at half-time |
| FullTimeScore | NVARCHAR(20) | Final score of the match |
| SyncTime | DATETIME | Data synchronization timestamp |

### SboThirdPartySportBetDetailsHistory
Detailed historical records of third-party sports bet details, including match, odds, and team information for each sub-bet or parlay leg. This table is used to archive granular bet selections and outcomes from third-party sports providers for audit and reporting purposes.

```sql
CREATE TABLE [dbo].[SboThirdPartySportBetDetailsHistory] (
    [Id] INT NOT NULL PRIMARY KEY IDENTITY(1,1),
    [RefNo] NVARCHAR(100) NOT NULL,
    [TransactionId] NVARCHAR(100) NOT NULL,
    [SubbetId] NVARCHAR(100) NULL,
    [ParlayRefno] NVARCHAR(100) NULL,
    [BetOption] NVARCHAR(150) NULL,
    [IsLive] BIT NOT NULL,
    [MarketType] NVARCHAR(150) NULL,
    [SportType] NVARCHAR(50) NULL,
    [HandicapPoint] DECIMAL(12,2) DEFAULT(0) NULL,
    [Odds] DECIMAL(12,3) DEFAULT(0) NULL,
    [League] NVARCHAR(150) NULL,
    [HomeTeam] NVARCHAR(150) NULL,
    [AwayTeam] NVARCHAR(150) NULL,
    [Status] VARCHAR(50) NULL,
    [WinlostDate] DATETIME NULL,
    [HalfTimeScore] NVARCHAR(20) NULL,
    [FullTimeScore] NVARCHAR(20) NULL,
    [SyncTime] DATETIME NULL,
    [WebId] INT NOT NULL,
    [KickOffTime] DATETIME NULL,
    [Language] NVARCHAR(10) DEFAULT('en') NOT NULL
)
```

| Column         | Type                | Description                                      |
|---------------|---------------------|--------------------------------------------------|
| Id            | INT, PK, IDENTITY   | Primary key, auto-incrementing unique identifier  |
| RefNo         | NVARCHAR(100)       | Reference number for the main bet                 |
| TransactionId | NVARCHAR(100)       | Transaction identifier from provider              |
| SubbetId      | NVARCHAR(100)       | Sub-bet identifier (for parlays)                  |
| ParlayRefno   | NVARCHAR(100)       | Parlay reference number                           |
| BetOption     | NVARCHAR(150)       | Selected betting option                           |
| IsLive        | BIT                 | Indicates if the bet was placed live              |
| MarketType    | NVARCHAR(150)       | Type of betting market                            |
| SportType     | NVARCHAR(50)        | Sport type (e.g., Soccer, Basketball)             |
| HandicapPoint | DECIMAL(12,2)       | Handicap value                                    |
| Odds          | DECIMAL(12,3)       | Betting odds                                      |
| League        | NVARCHAR(150)       | League or competition name                        |
| HomeTeam      | NVARCHAR(150)       | Home team name                                    |
| AwayTeam      | NVARCHAR(150)       | Away team name                                    |
| Status        | VARCHAR(50)         | Current status of the bet                         |
| WinlostDate   | DATETIME            | Date when the bet was settled                     |
| HalfTimeScore | NVARCHAR(20)        | Score at half-time                                |
| FullTimeScore | NVARCHAR(20)        | Final score of the match                          |
| SyncTime      | DATETIME            | Data synchronization timestamp                    |
| WebId         | INT                 | Website identifier                                |
| KickOffTime   | DATETIME            | Match kick-off time                               |
| Language      | NVARCHAR(10)        | Language code for localization                    |

## Key Features

1. **Comprehensive Data Archival**
   - Long-term retention of all betting activities
   - Complete transaction history preservation
   - Member action tracking and audit trails
   - System operation logs and records
   - Multi-timezone support for global operations

2. **Detailed Betting Records**
   - Sports betting with odds and selections
   - Casino game history with table details
   - Virtual sports tracking
   - Seamless game provider integration
   - Third-party provider transactions
   - Commission and profit-sharing calculations

3. **Financial Transaction History**
   - Deposit and withdrawal records
   - Balance tracking with before/after amounts
   - Currency exchange history with rates
   - Bank transaction details and references
   - Payment verification and slip images
   - Arbitrage player detection

4. **Advanced Reporting Support**
   - Daily summaries with timezone support
   - Product type categorization
   - Commission structure tracking
   - Win/loss calculations at all levels
   - Exchange rate historical data
   - Promotion and referral tracking

5. **Audit and Compliance**
   - Complete member action logging
   - IP address tracking
   - Fingerprint recording for security
   - Risk management flags
   - Regulatory compliance support
   - Data integrity maintenance

## Indexing Strategy

The History database implements comprehensive indexing for optimal query performance:

1. **Primary Keys**: Optimized with FILLFACTOR = 85 for insert performance
2. **Reporting Indexes**: Covering indexes for common report queries
3. **Date-based Indexes**: Optimized for time-range queries
4. **Multi-column Indexes**: Supporting complex filtering requirements

## Data Retention Policy

- **DailyBetSummaryHistory**: 61-90 days retention
- **Transaction Records**: Long-term retention for compliance
- **Member Actions**: Extended retention for audit purposes
- **Betting Records**: Comprehensive historical preservation
- **Currency Data**: Complete exchange rate history

## Performance Optimization

1. **Partitioning Strategy**
   - Date-based partitioning for large tables
   - Efficient archival and purging processes
   - Optimized storage management

2. **Query Optimization**
   - Covering indexes for reporting queries
   - Proper join optimization
   - Resource-efficient aggregations

3. **Maintenance Procedures**
   - Regular index maintenance
   - Statistics updates
   - Data compression strategies
   - Archive management

## Security and Compliance

1. **Access Controls**
   - Role-based permissions
   - Audit trail protection
   - Data encryption at rest
   - Secure data transmission

2. **Regulatory Compliance**
   - Complete transaction trails
   - Anti-money laundering support
   - Responsible gaming tracking
   - Regulatory reporting capabilities

## Relationships

- Historical records maintain referential integrity with operational databases
- Foreign keys link to WebId, CustomerId, and other core identifiers
- Time-based relationships support temporal analysis
- Cross-database relationships enable comprehensive reporting
- Archive tables mirror operational table structures for consistency

## Best Practices

1. **Data Integrity**
   - Consistent archival processes
   - Validation rules enforcement
   - Error handling and recovery
   - Quality assurance procedures

2. **Performance Management**
   - Regular monitoring and optimization
   - Capacity planning and scaling
   - Resource utilization tracking
   - Query performance analysis

3. **Maintenance Operations**
   - Scheduled maintenance windows
   - Backup and recovery procedures
   - Data lifecycle management
   - Archive strategy implementation 