# Online Database Schema

## Overview
The Online database manages user sessions, single sign-on (SSO), and player activity tracking across the platform.

## Core Tables

### PlayerOnline
```sql
CREATE TABLE [dbo].[PlayerOnline] (
    [OnlineId]      BIGINT         IDENTITY (1, 1) NOT NULL,
    [WebId]         INT            NOT NULL,
    [CustomerId]    INT            NOT NULL,
    [PlayerStatus]  INT            NULL,
    [ReloadTime]    DATETIME       DEFAULT (dateadd(hour,(-1),getdate())) NOT NULL,
    [LoginCountry]  NVARCHAR(10)   NULL,
    [LoginUrl]      NVARCHAR(255)  NULL,
    [LoginIp]       NVARCHAR(100)  NULL,
    [CreatedOn]     DATETIME       DEFAULT (getdate()) NULL,
    [LogoutTime]    DATETIME       NULL,
    CONSTRAINT [PK_PlayerOnline] PRIMARY KEY CLUSTERED ([OnlineId] DESC)
)
```

| Column        | Type           | Description                                      |
|---------------|----------------|--------------------------------------------------|
| OnlineId      | BIGINT, PK     | Primary key, unique session identifier           |
| WebId         | INT            | Website/application identifier                   |
| CustomerId    | INT            | Player/customer identifier                       |
| PlayerStatus  | INT, NULL      | Player session status code                       |
| ReloadTime    | DATETIME       | Last session refresh time                        |
| LoginCountry  | NVARCHAR(10)   | Country code of login                            |
| LoginUrl      | NVARCHAR(255)  | URL used for login                               |
| LoginIp       | NVARCHAR(100)  | IP address used for login                        |
| CreatedOn     | DATETIME       | Session creation (login) time                    |
| LogoutTime    | DATETIME, NULL | Session end (logout) time                        |

### PlayerOnlineHistory
```sql
CREATE TABLE [dbo].[PlayerOnlineHistory] (
    [ID]           BIGINT         IDENTITY (1, 1) NOT NULL,
    [OnlineId]     BIGINT         NOT NULL,
    [WebId]        INT            NOT NULL,
    [CustomerId]   INT            NOT NULL,
    [PlayerStatus] INT            NULL,
    [LoginCountry] NVARCHAR(10)   NULL,
    [LoginUrl]     NVARCHAR(255)  NULL,
    [LoginIp]      NVARCHAR(100)  NULL,
    [LoginTime]    DATETIME       NOT NULL,
    [LogoutTime]   DATETIME       NULL,
    CONSTRAINT [PK_PlayerOnlineHistory] PRIMARY KEY CLUSTERED ([Id] DESC)
)
```

| Column        | Type           | Description                                      |
|---------------|----------------|--------------------------------------------------|
| ID            | BIGINT, PK     | Primary key, unique history record               |
| OnlineId      | BIGINT         | Reference to PlayerOnline session                |
| WebId         | INT            | Website/application identifier                   |
| CustomerId    | INT            | Player/customer identifier                       |
| PlayerStatus  | INT, NULL      | Player session status code                       |
| LoginCountry  | NVARCHAR(10)   | Country code of login                            |
| LoginUrl      | NVARCHAR(255)  | URL used for login                               |
| LoginIp       | NVARCHAR(100)  | IP address used for login                        |
| LoginTime     | DATETIME       | Session start (login) time                       |
| LogoutTime    | DATETIME, NULL | Session end (logout) time                        |

### PlayerSSO
```sql
CREATE TABLE [dbo].[PlayerSSO] (
    [SsoID]      BIGINT         IDENTITY(1,1) NOT NULL,
    [SsoKey]     NVARCHAR(50)   NOT NULL,
    [GameType]   INT            NULL,
    [IsConsumed] BIT            NOT NULL,
    [CustomerId] INT            NOT NULL,
    [IP]         NVARCHAR(100)  NULL,
    [FromUrl]    NVARCHAR(200)  NOT NULL,
    [LoginTime]  DATETIME       DEFAULT (GETDATE()) NOT NULL,
    [OtherInfo]  NVARCHAR(500)  NULL,
    [WebId]      INT            NOT NULL,
    [OnlineId]   BIGINT         DEFAULT (0) NOT NULL,
    CONSTRAINT [PK_PlayerSSO] PRIMARY KEY CLUSTERED ([SsoID] ASC)
)
```

| Column        | Type            | Description                                    |
|---------------|-----------------|------------------------------------------------|
| SsoID         | BIGINT, PK      | Primary key, unique SSO record                 |
| SsoKey        | NVARCHAR(50)    | SSO token/key                                  |
| GameType      | INT, NULL       | Game type for SSO                              |
| IsConsumed    | BIT             | 1=used, 0=not used                             |
| CustomerId    | INT             | Player/customer identifier                     |
| IP            | NVARCHAR(100)   | IP address for SSO                             |
| FromUrl       | NVARCHAR(200)   | URL initiating SSO                             |
| LoginTime     | DATETIME        | SSO creation time                              |
| OtherInfo     | NVARCHAR(500)   | Additional info                                |
| WebId         | INT             | Website/application identifier                 |
| OnlineId      | BIGINT          | Reference to PlayerOnline session              |

### PlayerLoginLog
```sql
CREATE TABLE [dbo].[PlayerLoginLog] (
    [Id]            BIGINT         IDENTITY(1,1) NOT NULL,
    [WebId]         INT            NOT NULL,
    [CustomerId]    INT            NOT NULL,
    [LoginIP]       NVARCHAR(50)   NULL,
    [LoginOn]       DATETIME       NULL,
    [LoginUrl]      NVARCHAR(255)  NULL,
    [IsSuccessLogin] BIT           NOT NULL,
    [LoginCountry]  CHAR(2)        NULL,
    [LoginRegion]   NVARCHAR(50)   NULL,
    [LoginCity]     NVARCHAR(50)   NULL,
    [LoginPlatform] NVARCHAR(50)   NULL,
    [LoginBrower]   NVARCHAR(50)   NULL,
    CONSTRAINT [PK_PlayerLoginLog] PRIMARY KEY ([Id])
)
```

| Column          | Type            | Description                                    |
|-----------------|-----------------|------------------------------------------------|
| Id              | BIGINT, PK      | Primary key, unique log entry                  |
| WebId           | INT             | Website/application identifier                 |
| CustomerId      | INT             | Player/customer identifier                     |
| LoginIP         | NVARCHAR(50)    | IP address used for login                      |
| LoginOn         | DATETIME        | Login attempt time                             |
| LoginUrl        | NVARCHAR(255)   | URL used for login                             |
| IsSuccessLogin  | BIT             | 1=success, 0=failure                           |
| LoginCountry    | CHAR(2)         | Country code of login                          |
| LoginRegion     | NVARCHAR(50)    | Region of login                                |
| LoginCity       | NVARCHAR(50)    | City of login                                  |
| LoginPlatform   | NVARCHAR(50)    | Platform used for login                        |
| LoginBrower     | NVARCHAR(50)    | Browser used for login                         |

### PageAccessSSO
```sql
CREATE TABLE [dbo].[PageAccessSSO] (
    [SsoID]       BIGINT         IDENTITY(1,1) NOT NULL,
    [SsoKey]      NVARCHAR(50)   NOT NULL,
    [PageType]    NVARCHAR(50)   NULL,
    [IsConsumed]  BIT            NOT NULL,
    [CustomerId]  INT            NOT NULL,
    [RequestTime] DATETIME       DEFAULT (GETDATE()) NOT NULL,
    [LoginTime]   DATETIME       NULL,
    [OtherInfo]   NVARCHAR(500)  NULL,
    [WebId]       INT            NOT NULL,
    CONSTRAINT [PK_PageAccessSSO] PRIMARY KEY CLUSTERED ([SsoID] ASC)
)
```

| Column        | Type            | Description                                    |
|---------------|-----------------|------------------------------------------------|
| SsoID         | BIGINT, PK      | Primary key, unique SSO record                 |
| SsoKey        | NVARCHAR(50)    | SSO token/key for page access                  |
| PageType      | NVARCHAR(50)    | Page type accessed                             |
| IsConsumed    | BIT             | 1=used, 0=not used                             |
| CustomerId    | INT             | Player/customer identifier                     |
| RequestTime   | DATETIME        | SSO request time                               |
| LoginTime     | DATETIME, NULL  | SSO consumption time                           |
| OtherInfo     | NVARCHAR(500)   | Additional info                                |
| WebId         | INT             | Website/application identifier                 |

## Views

### vPlayerOnline
```sql
CREATE VIEW [dbo].[vPlayerOnline]
AS
    SELECT OnlineId, WebId, CustomerId, PlayerStatus,
           ReloadTime, LoginCountry, LoginUrl, LoginIp,
           CreatedOn AS LoginTime, LogoutTime
    FROM [dbo].[PlayerOnline]
    UNION ALL
    SELECT OnlineId, WebId, CustomerId, PlayerStatus,
           NULL, LoginCountry, LoginUrl, LoginIp,
           LoginTime, LogoutTime
    FROM [dbo].[PlayerOnlineHistory]
```

## Key Features

1. **Session Management**
   - Active session tracking
   - Historical session records
   - Multiple login detection and handling
   - Session timeout management

2. **Single Sign-On (SSO)**
   - Game-specific SSO handling
   - Page access SSO support
   - SSO token expiration (120 seconds)
   - SSO consumption tracking

3. **Login Activity Tracking**
   - Detailed login history
   - Geolocation information
   - Platform and browser details
   - Success/failure tracking

4. **Security Features**
   - IP tracking
   - Session validation
   - Login attempt monitoring
   - Multiple login prevention

## Database Roles
- RoleColoris: Full access (EXECUTE, SELECT, UPDATE, INSERT)
- RoleYog: Full access (EXECUTE, SELECT, UPDATE, INSERT)
- RoleSchedudia: Full access (EXECUTE, SELECT, UPDATE, INSERT) 