# Payment Database Schema

## Overview
The Payment database manages payment transactions, payment providers, bank information, and wallet addresses across different payment methods and currencies. It supports multi-currency, multi-provider, and multi-method payment operations, including both fiat and cryptocurrency, and provides comprehensive audit and compliance tracking for all payment-related activities.

## Core Tables

### Transaction
Manages all payment transactions, including deposits, withdrawals, and transfers, across all supported payment methods and providers. Tracks transaction status, audit information, and supports both fiat and crypto operations.

```sql
CREATE TABLE [dbo].[Transaction] (
    [ExRefNo] NVARCHAR(250) NOT NULL PRIMARY KEY,
    [RefNo] NVARCHAR(250) NOT NULL,
    [WebId] INT NOT NULL,
    [CompanyKey] NVARCHAR(250) NOT NULL,
    [ActionType] INT NOT NULL,
    [PPId] INT NOT NULL,
    [UserBankAccount] NVARCHAR(100) NULL,
    [OperatorBankAccount] NVARCHAR(100) NULL,
    [Amount] DECIMAL(19, 6) NOT NULL,
    [UserAccount] NVARCHAR(100) NOT NULL,
    [Currency] CHAR(3) NOT NULL,
    [Status] INT NOT NULL,
    [Remark] NVARCHAR(250) NULL,
    [DeclineReason] NVARCHAR(250) NULL,
    [BankCode] NVARCHAR(50) NULL,
    [TransactionFee] DECIMAL(19, 6) CONSTRAINT [DF_Transaction_TransactionFee] DEFAULT (0) NOT NULL,
    [CallBackUrl] VARCHAR(MAX) NOT NULL,
    [CreatedDate] DATETIME NULL,
    [LastModifiedBy] NVARCHAR(100) NOT NULL,
    [LastModifiedDate] DATETIME NOT NULL,
    [TransactionToken] NVARCHAR(50) NOT NULL,
    [PaymentMethod] NVARCHAR(100) CONSTRAINT [DF_Transaction_PaymentMethod] DEFAULT ('InternetBanking') NOT NULL,
    [CryptoCurrencyNetwork] NVARCHAR(50) DEFAULT '',
    [ExtraParameter] NVARCHAR(MAX) DEFAULT '' NOT NULL
)
```

| Column                | Type              | Description                                                        |
|-----------------------|-------------------|--------------------------------------------------------------------|
| ExRefNo               | NVARCHAR(250), PK | External reference number, unique transaction identifier           |
| RefNo                 | NVARCHAR(250)     | Internal reference number                                          |
| WebId                 | INT               | Website/application identifier                                     |
| CompanyKey            | NVARCHAR(250)     | Company or merchant key                                           |
| ActionType            | INT               | Type of transaction (deposit, withdrawal, etc.)                   |
| PPId                  | INT               | Payment provider identifier                                       |
| UserBankAccount       | NVARCHAR(100)     | User's bank account (if applicable)                               |
| OperatorBankAccount   | NVARCHAR(100)     | Operator's bank account (if applicable)                           |
| Amount                | DECIMAL(19,6)     | Transaction amount                                                |
| UserAccount           | NVARCHAR(100)     | User's account identifier                                         |
| Currency              | CHAR(3)           | Transaction currency (ISO code)                                   |
| Status                | INT               | Transaction status code                                           |
| Remark                | NVARCHAR(250)     | Optional remarks                                                  |
| DeclineReason         | NVARCHAR(250)     | Reason for decline (if any)                                       |
| BankCode              | NVARCHAR(50)      | Bank code (if applicable)                                         |
| TransactionFee        | DECIMAL(19,6)     | Transaction fee (default 0)                                       |
| CallBackUrl           | VARCHAR(MAX)      | Callback URL for transaction status updates                       |
| CreatedDate           | DATETIME          | Transaction creation timestamp                                    |
| LastModifiedBy        | NVARCHAR(100)     | User who last modified the transaction                            |
| LastModifiedDate      | DATETIME          | Last modification timestamp                                       |
| TransactionToken      | NVARCHAR(50)      | Unique token for transaction validation                           |
| PaymentMethod         | NVARCHAR(100)     | Payment method (default 'InternetBanking')                        |
| CryptoCurrencyNetwork | NVARCHAR(50)      | Cryptocurrency network (if applicable)                            |
| ExtraParameter        | NVARCHAR(MAX)     | Additional parameters (JSON or key-value)                         |

**Key Features:**
- Supports all payment flows: deposit, withdrawal, transfer
- Multi-currency and multi-provider support
- Tracks both fiat and crypto transactions
- Full audit trail with modification tracking
- Status and error/decline reason tracking
- Extensible via ExtraParameter for provider-specific data

---

### PaymentUnitBanks
Defines the mapping between payment units, banks, and providers, including min/max limits, enablement, and audit information. Used to control which banks are available for each payment method and provider.

```sql
CREATE TABLE [dbo].[PaymentUnitBanks] (
    [Id] INT NOT NULL IDENTITY(1,1),
    [WebId] INT NOT NULL DEFAULT (0),
    [PaymentUnitCode] NVARCHAR(200) NOT NULL,
    [BankCode] NVARCHAR(100) NOT NULL,
    [BankName] NVARCHAR(100) NOT NULL,
    [PaymentMethod] NVARCHAR(50),
    [Currency] CHAR(3),
    [PaymentProviderId] INT,
    [DepositMin] DECIMAL(19,6),
    [DepositMax] DECIMAL(19,6),
    [WithdrawalMin] DECIMAL(19,6),
    [WithdrawalMax] DECIMAL(19,6),
    [IsUsingBank] BIT DEFAULT (1) NOT NULL,
    [TransactionMethod] INT,
    [IsEnabled] BIT CONSTRAINT [DF_PaymentUnitBanks_IsEnabled] DEFAULT((0)),
    [CreatedOn] DATETIME CONSTRAINT [DF_PaymentUnitBanks_CreatedOn] DEFAULT (GETDATE()) NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME CONSTRAINT [DF_PaymentUnitBanks_ModifiedOn] DEFAULT (GETDATE()) NOT NULL,
    [ModifiedBy] NVARCHAR(50) NOT NULL
)
```

| Column            | Type              | Description                                              |
|-------------------|-------------------|----------------------------------------------------------|
| Id                | INT, PK           | Primary key, auto-incrementing unique identifier         |
| WebId             | INT               | Website/application identifier                           |
| PaymentUnitCode   | NVARCHAR(200)     | Unique code for payment unit (currency_provider_method)  |
| BankCode          | NVARCHAR(100)     | Bank code                                                |
| BankName          | NVARCHAR(100)     | Bank name                                                |
| PaymentMethod     | NVARCHAR(50)      | Payment method (e.g., InternetBanking, E-Wallet)         |
| Currency          | CHAR(3)           | Currency code                                            |
| PaymentProviderId | INT               | Payment provider identifier                              |
| DepositMin        | DECIMAL(19,6)     | Minimum deposit amount                                   |
| DepositMax        | DECIMAL(19,6)     | Maximum deposit amount                                   |
| WithdrawalMin     | DECIMAL(19,6)     | Minimum withdrawal amount                                |
| WithdrawalMax     | DECIMAL(19,6)     | Maximum withdrawal amount                                |
| IsUsingBank       | BIT               | Whether this unit uses a bank (default 1)                |
| TransactionMethod | INT               | Transaction method type                                  |
| IsEnabled         | BIT               | Whether this mapping is enabled                          |
| CreatedOn         | DATETIME          | Creation timestamp                                       |
| CreatedBy         | NVARCHAR(50)      | User who created the record                              |
| ModifiedOn        | DATETIME          | Last modification timestamp                              |
| ModifiedBy        | NVARCHAR(50)      | User who last modified the record                        |

**Key Features:**
- Controls which banks are available for each payment method/provider
- Supports min/max limits for deposits and withdrawals
- Multi-currency and multi-provider support
- Enable/disable control for each mapping
- Full audit trail

---

### PaymentAggregators
Defines payment aggregator entities and their relationship to payment providers. Used for routing and managing aggregator-based payment flows.

```sql
CREATE TABLE [dbo].[PaymentAggregators] (
    [Id] INT NOT NULL IDENTITY(1,1),
    [PaymentProviderId] INT NOT NULL,
    [PaymentAggregatorId] INT NOT NULL,
    [PaymentAggregatorName] VARCHAR(100) NOT NULL,
    [IsEnabled] BIT CONSTRAINT [DF_PaymentAggregators_IsEnabled] DEFAULT((0)),
    [CreatedOn] DATETIME CONSTRAINT [DF_PaymentAggregators_CreatedOn] DEFAULT (GETDATE()) NOT NULL,
    [CreatedBy] VARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME CONSTRAINT [DF_PaymentAggregators_ModifiedOn] DEFAULT (GETDATE()) NOT NULL,
    [ModifiedBy] VARCHAR(50) NOT NULL,
    CONSTRAINT [UQ_PaymentProviderId] UNIQUE ([PaymentProviderId])
)
```

| Column                | Type              | Description                                      |
|-----------------------|-------------------|--------------------------------------------------|
| Id                    | INT, PK           | Primary key, auto-incrementing unique identifier |
| PaymentProviderId      | INT, Unique       | Payment provider identifier                      |
| PaymentAggregatorId    | INT               | Aggregator identifier                            |
| PaymentAggregatorName  | VARCHAR(100)      | Aggregator name                                  |
| IsEnabled              | BIT               | Whether this aggregator is enabled               |
| CreatedOn             | DATETIME          | Creation timestamp                               |
| CreatedBy             | VARCHAR(50)       | User who created the record                      |
| ModifiedOn            | DATETIME          | Last modification timestamp                      |
| ModifiedBy            | VARCHAR(50)       | User who last modified the record                |

**Key Features:**
- Manages aggregator-to-provider mapping
- Supports enable/disable for each aggregator
- Ensures one-to-one relationship with provider (unique constraint)
- Full audit trail

---

### PlayerWalletAddress
Stores player wallet addresses for cryptocurrency transactions, including network, provider, and usage status. Used for both deposit and withdrawal flows involving crypto.

```sql
CREATE TABLE [dbo].[PlayerWalletAddress] (
    [Id] INT NOT NULL IDENTITY(1,1),
    [WebId] INT NOT NULL,
    [CustomerId] NVARCHAR(50) NOT NULL,
    [Username] NVARCHAR(50) NOT NULL,
    [WalletAddress] NVARCHAR(MAX) NOT NULL,
    [CryptoCurrencyNetwork] NVARCHAR(50) NULL,
    [ActionType] INT NULL DEFAULT 0,
    [PaymentProviderId] INT NULL,
    [IsUsed] BIT NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL
)
```

| Column                | Type              | Description                                      |
|-----------------------|-------------------|--------------------------------------------------|
| Id                    | INT, PK           | Primary key, auto-incrementing unique identifier |
| WebId                 | INT               | Website/application identifier                   |
| CustomerId            | NVARCHAR(50)      | Customer identifier                              |
| Username              | NVARCHAR(50)      | Username                                         |
| WalletAddress         | NVARCHAR(MAX)     | Cryptocurrency wallet address                    |
| CryptoCurrencyNetwork | NVARCHAR(50)      | Crypto network (e.g., ERC20, TRC20)              |
| ActionType            | INT               | Action type (default 0)                          |
| PaymentProviderId     | INT               | Payment provider identifier                      |
| IsUsed                | BIT               | Whether this address is currently used           |
| CreatedBy             | NVARCHAR(50)      | User who created the record                      |
| CreatedOn             | DATETIME          | Creation timestamp                               |

**Key Features:**
- Stores player crypto wallet addresses for deposit/withdrawal
- Supports multiple crypto networks and providers
- Tracks usage status for address management
- Full audit trail

---

### PlayerCryptoWallet
Stores player cryptocurrency wallet addresses, similar to PlayerWalletAddress, but may be used for legacy or alternate crypto flows. Tracks address, network, and usage status.

```sql
CREATE TABLE [dbo].[PlayerCryptoWallet] (
    [Id] INT NOT NULL IDENTITY(1,1),
    [WebId] INT NOT NULL,
    [CustomerId] NVARCHAR(50) NOT NULL,
    [Username] NVARCHAR(50) NOT NULL,
    [WalletAddress] NVARCHAR(MAX) NOT NULL,
    [CryptoCurrencyNetwork] NVARCHAR(50) NULL,
    [IsUsed] BIT NOT NULL,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL
)
```

| Column                | Type              | Description                                      |
|-----------------------|-------------------|--------------------------------------------------|
| Id                    | INT, PK           | Primary key, auto-incrementing unique identifier |
| WebId                 | INT               | Website/application identifier                   |
| CustomerId            | NVARCHAR(50)      | Customer identifier                              |
| Username              | NVARCHAR(50)      | Username                                         |
| WalletAddress         | NVARCHAR(MAX)     | Cryptocurrency wallet address                    |
| CryptoCurrencyNetwork | NVARCHAR(50)      | Crypto network (e.g., ERC20, TRC20)              |
| IsUsed                | BIT               | Whether this address is currently used           |
| CreatedBy             | NVARCHAR(50)      | User who created the record                      |
| CreatedOn             | DATETIME          | Creation timestamp                               |

**Key Features:**
- Stores player crypto wallet addresses for legacy or alternate flows
- Supports multiple crypto networks
- Tracks usage status for address management
- Full audit trail

---

## User-Defined Types

### PaymentUnitBank
Defines a table type for bulk operations on payment unit banks.

```sql
CREATE TYPE [dbo].[PaymentUnitBank] AS TABLE (
    [PaymentUnitCode] NVARCHAR(200) NOT NULL,
    [BankCode] NVARCHAR(100) NOT NULL,
    [BankName] NVARCHAR(100) NOT NULL,
    [IsEnabled] BIT,
    [DepositMin] DECIMAL(19,6),
    [DepositMax] DECIMAL(19,6),
    [WithdrawalMin] DECIMAL(19,6),
    [WithdrawalMax] DECIMAL(19,6),
    [TransactionMethod] INT,
    [Currency] CHAR(3),
    [PaymentProviderId] INT,
    [PaymentMethod] NVARCHAR(50),
    [IsUsingBank] BIT DEFAULT (1) NOT NULL
)
```

### PaymentStatuses
Defines a table type for bulk status operations on payment methods.

```sql
CREATE TYPE [dbo].[PaymentStatuses] AS TABLE (
    [TransactionMethod] INT,
    [Currency] NVARCHAR(5),
    [PaymentMethod] NVARCHAR(50),
    [PaymentProviderId] INT,
    [IsEnabled] BIT,
    [BankCode] NVARCHAR(100)
)
```

### BankInfo
Defines a table type for bulk bank information operations.

```sql
CREATE TYPE [dbo].[BankInfo] AS TABLE (
    [BankName] NVARCHAR(100),
    [PpBankCode] NVARCHAR(100),
    [PpId] INT,
    [PaymentMethod] NVARCHAR(50),
    [Currency] NVARCHAR(5),
    [CountryCode] NVARCHAR(5),
    [IsEnable] BIT
)
```

## Key Features

1. **Comprehensive Payment Management**
   - Supports all major payment methods: Internet Banking, E-Wallets, Bank Transfer, Crypto, etc.
   - Multi-currency and multi-provider support
   - Full audit and compliance tracking
   - Min/max and enablement controls for all payment units
   - Crypto wallet management for both player and operator

2. **Audit and Security**
   - All tables include audit fields (CreatedBy, CreatedOn, ModifiedBy, ModifiedOn)
   - Status and enablement fields for operational control
   - Unique and primary key constraints for data integrity

3. **Extensibility**
   - ExtraParameter and flexible table types for future payment methods
   - User-defined types for bulk and batch operations

4. **Relationships**
   - Transactions link to providers, banks, and wallets
   - Aggregators maintain one-to-one mapping with providers
   - Wallet addresses are linked to customers and providers

5. **Performance and Compliance**
   - Indexed primary and unique keys for fast lookups
   - Audit trail for all payment operations
   - Compliance with multi-jurisdictional requirements

## Payment Methods
The system supports various payment methods including:
- Internet Banking
- E-Wallets (OVO, DANA, GOPAY, SHOPEEPAY, LINKAJA, etc.)
- Bank Transfer
- PIX (Brazil)
- CashApp
- Cryptocurrency
- Online Banking
- Offline Banking
- QR Code/Scan Code
- TrueMoney
- PromptPay

## Supported Currencies
The system handles multiple currencies including:
- IDR (Indonesian Rupiah)
- THB (Thai Baht)
- PHP (Philippine Peso)
- BRL (Brazilian Real)
- USD (US Dollar)
- MMK (Myanmar Kyat)
- CNY (Chinese Yuan)
- ZAR (South African Rand)
- KRW (Korean Won)
- ARS (Argentine Peso)

## Payment Providers
The system integrates with multiple payment providers/aggregators:
- PayTrust
- FattyPay
- GCX
- PayPro
- 58Payment
- TbetPayment
- HTPay
- PrincePay
- NewPrincePay
- VnHotPay
- CopoPay
- CapitalPay
- VcPay
- WPay
- PTYPay
- WakaPay
- AuPay
- WowPay
- BabPay
- SojuPay
- Gtrpay
- OPay
- SafePay
- Pay969
- BetcatPay
- LuckyPay
- StarPay
- CallPay
- HIPAY
- OKPay
- SupefinaPay
- RMPay
- CriptoPay
- WowoWallet

## Common Fields
Most payment tables share these common fields:
- `WebId`: Website identifier
- `PaymentProviderId`: Payment provider identifier
- `Currency`: Transaction currency
- `IsEnabled`: Status flag
- `CreatedOn/ModifiedOn`: Audit timestamps
- `CreatedBy/ModifiedBy`: Audit user information

## Relationships
- Transactions are linked to payment providers through `PPId`
- Payment unit banks are associated with payment providers and currencies
- Player wallet addresses are linked to customers and payment providers
- Payment aggregators maintain a one-to-one relationship with payment providers

## Transaction Methods
The system supports different transaction methods (represented by `TransactionMethod`):
1. General Transaction
2. Bank Transfer
3. E-wallet/Digital Payment

## Constraints and Validations
- Each payment provider has unique configurations
- Transaction amounts must be within min/max limits per payment method
- Payment methods are currency-specific
- Bank codes must be valid for the selected payment provider
- Payment unit codes follow a specific format: `{CURRENCY}_{PROVIDER}_{METHOD}` 