# Promotion Database Schema

## Overview
The Promotion database manages all promotional activities, campaigns, and rewards for the platform. It handles various types of promotions including registration bonuses, deposit bonuses, rebates, referrals, lucky wheel promotions, and coupon rewards.

## Core Tables

### CompanyPromotion
```sql
CREATE TABLE [CompanyPromotion] (
  [Id] int IDENTITY(1,1) PRIMARY KEY,
  [WebId] int NOT NULL,
  [PromotionTypeId] int NOT NULL,
  [DefaultLanguage] nvarchar(10) NOT NULL,
  [Currency] nvarchar(3) NOT NULL,
  [StartDate] datetime NOT NULL,
  [EndDate] datetime NOT NULL,
  [Status] nvarchar(255) NOT NULL,
  [PromotionGameProviderIds] NVARCHAR(MAX) NULL,
  [IsShowToGuest] bit,
  [DisplayOrder] int NOT NULL,
  [ModifiedOn] datetime NOT NULL,
  [ModifiedBy] NVARCHAR(100) NOT NULL,
  [CreatedOn] datetime NOT NULL,
  [CreatedBy] NVARCHAR(100) NOT NULL,
  [IsCreatePromotionWallet] BIT DEFAULT(1) NOT NULL,
  CONSTRAINT CK_CompanyPromotions_Status CHECK ([Status] IN ('Active', 'Deleted', 'Disabled', 'ActiveAndOpenForApply', 'Finished')),
  CONSTRAINT CK_CompanyPromotions_Finished_Status CHECK ([dbo].[CheckPromotionsFinishedStatus]([Id],[Status],[WebId])=1)
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | int, PK | Primary key, auto-incrementing unique identifier |
| WebId | int | Website/application identifier |
| PromotionTypeId | int | Foreign key to PromotionType table |
| DefaultLanguage | nvarchar(10) | Default language code for the promotion |
| Currency | nvarchar(3) | Currency code (ISO 4217) |
| StartDate | datetime | Promotion start date/time |
| EndDate | datetime | Promotion end date/time |
| Status | nvarchar(255) | Promotion status (Active, Deleted, Disabled, etc.) |
| PromotionGameProviderIds | nvarchar(MAX) | List of game provider IDs eligible for this promotion |
| IsShowToGuest | bit | Whether the promotion is visible to guests |
| DisplayOrder | int | Display order for sorting promotions |
| ModifiedOn | datetime | Last modification timestamp |
| ModifiedBy | nvarchar(100) | User who last modified the record |
| CreatedOn | datetime | Creation timestamp |
| CreatedBy | nvarchar(100) | User who created the record |
| IsCreatePromotionWallet | bit | Whether to create a promotion wallet (default 1) |

### PromotionType
```sql
CREATE TABLE [PromotionType] (
  [Id] INT PRIMARY KEY IDENTITY(1,1),
  [Name] NVARCHAR(100),
  [ClassName] NVARCHAR(100),
  [Type] NVARCHAR(100),
  [CreatedOn] DATETIME,
  CONSTRAINT CK_PromotionType_Type CHECK ([Type] IN ('RegisterPromotion', 'DepositPromotion', 'RebatePromotion', 'Referral', 'LuckyWheelPromotion', 'CouponPromotion'))
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | int, PK | Primary key, auto-incrementing unique identifier |
| Name | nvarchar(100) | Name of the promotion type |
| ClassName | nvarchar(100) | Class name for backend logic |
| Type | nvarchar(100) | Type of promotion (RegisterPromotion, DepositPromotion, etc.) |
| CreatedOn | datetime | Creation timestamp |

### PromotionRequest
```sql
CREATE TABLE [PromotionRequest] (
  [Id] int PRIMARY KEY IDENTITY(1,1),
  [CompanyPromotionId] int,
  [PromotionTypeId] int,
  [CustomerId] int,
  [ParentId] INT,
  [Username] nvarchar(100),
  [DepositId] int,
  [DepositAmount] decimal(19,6),
  [IsPayment] bit,
  [SuggestAmount] decimal(19,6),
  [ApproveAmount] decimal(19,6),
  [MinTurnoverToWithdrawal] decimal(19,6),
  [CurrentTurnover] decimal(19,6),
  [Remark] nvarchar(max),
  [ModifiedOn] datetime,
  [VerifiedOn] datetime NULL,
  [ModifiedBy] nvarchar(100),
  [CreatedOn] datetime,
  [CreatedBy] nvarchar(100),
  [Status] nvarchar(255) NOT NULL,
  [Ip] nvarchar(50) NULL,
  [Phone] nvarchar(50) NULL,
  [Email] nvarchar(100) NULL,
  [WebId] int NOT NULL,
  [Device] nvarchar(max) NULL,
  [Agent] nvarchar(50) NULL,
  [BankAccountNumber] NCHAR(50) NULL,
  [FingerPrint] NVARCHAR(MAX),
  [FreezeId] INT,
  [IsSelected] BIT,
  [EligibleAmount] DECIMAL(19, 6),
  [CompleteAmount] DECIMAL(19, 6),
  [IsCreatePromotionWallet] BIT DEFAULT(1) NOT NULL,
  CONSTRAINT CK_PromotionRequest_Status CHECK ([Status] IN ('Waiting', 'Rejected', 'Approved', 'Completed','Cancelled'))
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | int, PK | Primary key, auto-incrementing unique identifier |
| CompanyPromotionId | int | Foreign key to CompanyPromotion table |
| PromotionTypeId | int | Foreign key to PromotionType table |
| CustomerId | int | Customer identifier |
| ParentId | int | Parent customer identifier (for referral/agent hierarchy) |
| Username | nvarchar(100) | Username of the applicant |
| DepositId | int | Related deposit transaction ID |
| DepositAmount | decimal(19,6) | Deposit amount for the promotion |
| IsPayment | bit | Whether payment is completed |
| SuggestAmount | decimal(19,6) | Suggested bonus amount |
| ApproveAmount | decimal(19,6) | Approved bonus amount |
| MinTurnoverToWithdrawal | decimal(19,6) | Minimum turnover required for withdrawal |
| CurrentTurnover | decimal(19,6) | Current turnover achieved |
| Remark | nvarchar(max) | Additional remarks |
| ModifiedOn | datetime | Last modification timestamp |
| VerifiedOn | datetime | Verification timestamp |
| ModifiedBy | nvarchar(100) | User who last modified the record |
| CreatedOn | datetime | Creation timestamp |
| CreatedBy | nvarchar(100) | User who created the record |
| Status | nvarchar(255) | Request status (Waiting, Rejected, Approved, Completed, Cancelled) |
| Ip | nvarchar(50) | IP address of the applicant |
| Phone | nvarchar(50) | Phone number of the applicant |
| Email | nvarchar(100) | Email address of the applicant |
| WebId | int | Website/application identifier |
| Device | nvarchar(max) | Device information |
| Agent | nvarchar(50) | Agent identifier or code |
| BankAccountNumber | nchar(50) | Bank account number for payout |
| FingerPrint | nvarchar(MAX) | Device/browser fingerprint |
| FreezeId | int | Freeze record identifier |
| IsSelected | bit | Whether this request is selected for processing |
| EligibleAmount | decimal(19,6) | Amount eligible for bonus |
| CompleteAmount | decimal(19,6) | Amount completed for bonus |
| IsCreatePromotionWallet | bit | Whether to create a promotion wallet (default 1) |

## Settings Tables

### PromotionSetting
```sql
CREATE TABLE [PromotionSetting] (
  [Id] INT PRIMARY KEY IDENTITY(1,1),
  [Name] NVARCHAR(100) NOT NULL,
  [ClassName] NVARCHAR(100) NOT NULL,
  CONSTRAINT UQ_PromotionSettings_Name_ClassName UNIQUE ([Name], ClassName)
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | int, PK | Primary key, auto-incrementing unique identifier |
| Name | nvarchar(100) | Name of the promotion setting |
| ClassName | nvarchar(100) | Class name for backend logic |

### CompanyPromotionSetting
```sql
CREATE TABLE [CompanyPromotionSetting] (
  [Id] int PRIMARY KEY IDENTITY(1,1),
  [CompanyPromotionId] int,
  [Name] nvarchar(100) NOT NULL,
  [CanBeEdit] bit,
  [Value] nvarchar(100),
  [ModifiedOn] datetime,
  [ModifiedBy] nvarchar(100),
  [CreatedOn] datetime,
  [CreatedBy] nvarchar(100) NOT NULL,
  CONSTRAINT CK_CompanyPromotionSettings_Name CHECK ([dbo].[CheckPromotionSettingAvailable]([Name])=1),
  CONSTRAINT UQ_CompanyPromotionId_Name UNIQUE ([CompanyPromotionId], [Name])
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | int, PK | Primary key, auto-incrementing unique identifier |
| CompanyPromotionId | int | Foreign key to CompanyPromotion table |
| Name | nvarchar(100) | Name of the setting |
| CanBeEdit | bit | Whether the setting can be edited |
| Value | nvarchar(100) | Value of the setting |
| ModifiedOn | datetime | Last modification timestamp |
| ModifiedBy | nvarchar(100) | User who last modified the record |
| CreatedOn | datetime | Creation timestamp |
| CreatedBy | nvarchar(100) | User who created the record |

### CompanyPromotionSettingChangeLog
```sql
CREATE TABLE [CompanyPromotionSettingChangeLog] (
  [Id] int PRIMARY KEY IDENTITY(1,1),
  [CompanyPromotionId] int,
  [SettingId] int NOT NULL,
  [Name] nvarchar(255) NOT NULL,
  [OldValue] nvarchar(100),
  [NewValue] nvarchar(100),
  [ModifiedOn] datetime,
  [ModifiedBy] nvarchar(100),
  CONSTRAINT CK_CompanyPromotionSettingsChangeLog_Name CHECK ([dbo].[CheckPromotionSettingAvailable]([Name])=1)
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | int, PK | Primary key, auto-incrementing unique identifier |
| CompanyPromotionId | int | Foreign key to CompanyPromotion table |
| SettingId | int | Foreign key to CompanyPromotionSetting table |
| Name | nvarchar(255) | Name of the setting changed |
| OldValue | nvarchar(100) | Previous value |
| NewValue | nvarchar(100) | New value |
| ModifiedOn | datetime | Modification timestamp |
| ModifiedBy | nvarchar(100) | User who modified the record |

### PromotionTypeSetting
```sql
CREATE TABLE [PromotionTypeSetting] (
  [Id] int PRIMARY KEY IDENTITY(1,1),
  [PromotionTypeId] int,
  [Name] nvarchar(100) NOT NULL,
  [Value] nvarchar(100),
  [CanBeEdit] bit,
  CONSTRAINT CK_PromotionTypeSetting_Name CHECK ([dbo].[CheckPromotionSettingAvailable]([Name])=1)
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | int, PK | Primary key, auto-incrementing unique identifier |
| PromotionTypeId | int | Foreign key to PromotionType table |
| Name | nvarchar(100) | Name of the type setting |
| Value | nvarchar(100) | Value of the type setting |
| CanBeEdit | bit | Whether the setting can be edited |

## Special Promotion Types

### DepositPromotionOption
```sql
CREATE TABLE [DepositPromotionOption] (
    [Id] INT IDENTITY(1,1) PRIMARY KEY,
    [WebId] INT NOT NULL,
    [Currency] NVARCHAR(3) NOT NULL,
    [IsDepositOptionAmountEnabled] BIT DEFAULT 0,
    [IsDepositOptionPromotionEnabled] BIT DEFAULT 0,
    [CompanyPromotionIds] NVARCHAR(MAX) NULL,
    [ModifiedOn] DATETIME NOT NULL,
    [ModifiedBy] NVARCHAR(100) NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [CreatedBy] NVARCHAR(100) NOT NULL
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | int, PK | Primary key, auto-incrementing unique identifier |
| WebId | int | Website/application identifier |
| Currency | nvarchar(3) | Currency code (ISO 4217) |
| IsDepositOptionAmountEnabled | bit | Whether deposit option amount is enabled |
| IsDepositOptionPromotionEnabled | bit | Whether deposit option promotion is enabled |
| CompanyPromotionIds | nvarchar(MAX) | List of related company promotion IDs |
| ModifiedOn | datetime | Last modification timestamp |
| ModifiedBy | nvarchar(100) | User who last modified the record |
| CreatedOn | datetime | Creation timestamp |
| CreatedBy | nvarchar(100) | User who created the record |

### DepositPromotionOptionAmount
```sql
CREATE TABLE [DepositPromotionOptionAmount] (
    [Id] INT IDENTITY(1,1) PRIMARY KEY,
    [DepositPromotionOptionId] INT NOT NULL,
    [Amount] DECIMAL(19, 6) NOT NULL, 
    [IsHot] BIT DEFAULT 0,
    [Order] INT DEFAULT 0,
    [Status] NVARCHAR(10) NOT NULL DEFAULT 'Active',
    [ModifiedOn] datetime NOT NULL,
    [ModifiedBy] NVARCHAR(100) NOT NULL,
    [CreatedOn] datetime NOT NULL,
    [CreatedBy] NVARCHAR(100) NOT NULL,
    CONSTRAINT CK_DepositPromotionOptionAmount_Status CHECK ([Status] IN ('Active', 'Deleted'))
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | int, PK | Primary key, auto-incrementing unique identifier |
| DepositPromotionOptionId | int | Foreign key to DepositPromotionOption table |
| Amount | decimal(19,6) | Deposit amount option |
| IsHot | bit | Whether this amount is marked as hot (highlighted) |
| Order | int | Display order |
| Status | nvarchar(10) | Status (Active, Deleted) |
| ModifiedOn | datetime | Last modification timestamp |
| ModifiedBy | nvarchar(100) | User who last modified the record |
| CreatedOn | datetime | Creation timestamp |
| CreatedBy | nvarchar(100) | User who created the record |

### LuckyWheelOption
```sql
CREATE TABLE LuckyWheelOption (
  [Id] INT IDENTITY(1,1) PRIMARY KEY,
  [CompanyPromotionId] INT NOT NULL,
  [BonusAmount] DECIMAL(19, 6) NOT NULL,
  [Percentage] DECIMAL(19, 6) NOT NULL,
  [PoolTickets] INT NOT NULL,
  [UsedTickets] INT NOT NULL,
  [MaxTickets] INT NOT NULL,
  [IsLimitMax] BIT NOT NULL DEFAULT(0)
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | int, PK | Primary key, auto-incrementing unique identifier |
| CompanyPromotionId | int | Foreign key to CompanyPromotion table |
| BonusAmount | decimal(19,6) | Bonus amount for this wheel option |
| Percentage | decimal(19,6) | Percentage chance for this option |
| PoolTickets | int | Total tickets in the pool |
| UsedTickets | int | Number of tickets used |
| MaxTickets | int | Maximum tickets allowed |
| IsLimitMax | bit | Whether there is a max ticket limit |

### LuckyWheelOptionSetting
```sql
CREATE TABLE LuckyWheelOptionSetting (
  [Id] INT IDENTITY(1,1) PRIMARY KEY,
  [BonusAmount] DECIMAL(19, 6) NOT NULL,
  [Percentage] DECIMAL(19, 6) NOT NULL,
  [MaxTickets] INT NOT NULL,
  [IsLimitMax] BIT NOT NULL DEFAULT(0),
  [OptionType] INT NOT NULL
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | int, PK | Primary key, auto-incrementing unique identifier |
| BonusAmount | decimal(19,6) | Bonus amount for this wheel option setting |
| Percentage | decimal(19,6) | Percentage chance for this option |
| MaxTickets | int | Maximum tickets allowed |
| IsLimitMax | bit | Whether there is a max ticket limit |
| OptionType | int | Type of wheel option |

### LuckyWheelPerformanceForTicket
```sql
CREATE TABLE LuckyWheelPerformanceForTicket (
  [Id] INT IDENTITY(1,1) PRIMARY KEY,
  [CompanyPromotionId] INT NOT NULL,
  [CustomerId] INT NOT NULL,
  [WinLoseDate] DATETIME NOT NULL,
  [PerformanceType] INT NOT NULL, 
  [CurrentAmount] DECIMAL(19, 6) NOT NULL,
  [AmountToTicket] DECIMAL(19, 6) NOT NULL,
  [CreatedOn] DATETIME NOT NULL,
  [CreatedBy] NVARCHAR(100) NOT NULL,
  [ModifiedOn] DATETIME NOT NULL,
  [ModifiedBy] NVARCHAR(100)
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | int, PK | Primary key, auto-incrementing unique identifier |
| CompanyPromotionId | int | Foreign key to CompanyPromotion table |
| CustomerId | int | Customer identifier |
| WinLoseDate | datetime | Date of win/loss event |
| PerformanceType | int | Type of performance metric |
| CurrentAmount | decimal(19,6) | Current amount for performance calculation |
| AmountToTicket | decimal(19,6) | Amount converted to ticket(s) |
| CreatedOn | datetime | Creation timestamp |
| CreatedBy | nvarchar(100) | User who created the record |
| ModifiedOn | datetime | Last modification timestamp |
| ModifiedBy | nvarchar(100) | User who last modified the record |

### LuckyWheelTicket
```sql
CREATE TABLE LuckyWheelTicket (
  [Id] INT IDENTITY(1,1) PRIMARY KEY,
  [CompanyPromotionId] INT NOT NULL,
  [CustomerId] INT NOT NULL,
  [TicketDate] DATETIME NOT NULL,
  [TicketType] INT NOT NULL,
  [IsUsed] BIT NOT NULL DEFAULT(0),
  [Ip] NVARCHAR(50),
  [Phone] NVARCHAR(50),
  [Email] NVARCHAR(100),
  [BankAccountNumber] NVARCHAR(50),
  [FingerPrint] NVARCHAR(MAX),
  [BonusAmountEarned] DECIMAL(19, 6) NOT NULL DEFAULT 0,
  [ExpireTime] DATETIME NOT NULL,
  [CreatedOn] DATETIME NOT NULL,
  [CreatedBy] NVARCHAR(100) NOT NULL,
  [ModifiedOn] DATETIME,
  [ModifiedBy] NVARCHAR(100) 
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | int, PK | Primary key, auto-incrementing unique identifier |
| CompanyPromotionId | int | Foreign key to CompanyPromotion table |
| CustomerId | int | Customer identifier |
| TicketDate | datetime | Date the ticket was issued |
| TicketType | int | Type of ticket |
| IsUsed | bit | Whether the ticket has been used |
| Ip | nvarchar(50) | IP address of the user |
| Phone | nvarchar(50) | Phone number of the user |
| Email | nvarchar(100) | Email address of the user |
| BankAccountNumber | nvarchar(50) | Bank account number for payout |
| FingerPrint | nvarchar(MAX) | Device/browser fingerprint |
| BonusAmountEarned | decimal(19,6) | Bonus amount earned with this ticket |
| ExpireTime | datetime | Expiry date/time of the ticket |
| CreatedOn | datetime | Creation timestamp |
| CreatedBy | nvarchar(100) | User who created the record |
| ModifiedOn | datetime | Last modification timestamp |
| ModifiedBy | nvarchar(100) | User who last modified the record |

### CouponCode
```sql
CREATE TABLE CouponCode (
    [Id] INT PRIMARY KEY IDENTITY(1,1),
    [WebId] INT NOT NULL,
    [Currency] NVARCHAR(3) NOT NULL,
    [CompanyPromotionId] INT NOT NULL,
    [CouponType] INT NOT NULL,
    [CustomerId] INT,
    [Username] NVARCHAR(50),
    [CouponCode] NVARCHAR(255) NOT NULL,
    [Bonus] DECIMAL(19,6),
    [Remark] NVARCHAR(255),
    [IsUsed] BIT NOT NULL DEFAULT 0,
    [UsedDate] DATETIME NULL,
    [CreatedOn] DATETIME NOT NULL DEFAULT GETDATE(),
    [CreatedBy] NVARCHAR(255) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL DEFAULT GETDATE(),
    [ModifiedBy] NVARCHAR(255) NOT NULL
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | int, PK | Primary key, auto-incrementing unique identifier |
| WebId | int | Website/application identifier |
| Currency | nvarchar(3) | Currency code (ISO 4217) |
| CompanyPromotionId | int | Foreign key to CompanyPromotion table |
| CouponType | int | Type of coupon |
| CustomerId | int | Customer identifier (if assigned) |
| Username | nvarchar(50) | Username of the coupon recipient |
| CouponCode | nvarchar(255) | Unique coupon code |
| Bonus | decimal(19,6) | Bonus amount for the coupon |
| Remark | nvarchar(255) | Additional remarks |
| IsUsed | bit | Whether the coupon has been used |
| UsedDate | datetime | Date/time the coupon was used |
| CreatedOn | datetime | Creation timestamp |
| CreatedBy | nvarchar(255) | User who created the record |
| ModifiedOn | datetime | Last modification timestamp |
| ModifiedBy | nvarchar(255) | User who last modified the record |

### ReferralCash
```sql
CREATE TABLE [ReferralCash] (
    [Id] int PRIMARY KEY IDENTITY(1,1),
    [CustomerId] INT,
    [Username] NVARCHAR(100) NOT NULL,
    [Currency] NVARCHAR(10) NOT NULL,
    [CompanyPromotionId] INT,
    [CashRedeemed] DECIMAL(19,6),
    [ModifiedOn] DATETIME NOT NULL,
    [ModifiedBy] NVARCHAR(100) NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [CreatedBy] NVARCHAR(100) NOT NULL,
    CONSTRAINT UQ_ReferralCash UNIQUE ([CustomerId], [CompanyPromotionId])
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | int, PK | Primary key, auto-incrementing unique identifier |
| CustomerId | int | Customer identifier |
| Username | nvarchar(100) | Username of the customer |
| Currency | nvarchar(10) | Currency code (ISO 4217) |
| CompanyPromotionId | int | Foreign key to CompanyPromotion table |
| CashRedeemed | decimal(19,6) | Amount of cash redeemed |
| ModifiedOn | datetime | Last modification timestamp |
| ModifiedBy | nvarchar(100) | User who last modified the record |
| CreatedOn | datetime | Creation timestamp |
| CreatedBy | nvarchar(100) | User who created the record |

### ReferralMember
```sql
CREATE TABLE [ReferralMember] (
    [Id] int PRIMARY key IDENTITY(1,1) NOT NULL,
    [CustomerId] INT NOT NULL,
    [Username] NVARCHAR(100) NOT NULL,
    [ParentId] INT NOT NULL,
    [CompanyPromotionId] INT NOT NULL,
    [PromotionRequestId] INT DEFAULT 0 NOT NULL,
    [Currency] NVARCHAR(10) NOT NULL,
    [ContributeAmount] DECIMAL(19,6) DEFAULT 0 NOT NULL,
    [CurrentTurnover] DECIMAL(19,6) DEFAULT 0 NOT NULL,
    [CurrentDepositAmount] DECIMAL(19,6) DEFAULT 0 NOT NULL,
    [CurrentWithdrawalAmount] DECIMAL(19,6) DEFAULT 0 NOT NULL,
    [DepositCount] INT DEFAULT 0 NOT NULL,
    [WithdrawalCount] INT DEFAULT 0 NOT NULL,
    [CurrentWinlost] DECIMAL(19,6) DEFAULT 0 NOT NULL, 
    [QualifiedDepositLimit] DECIMAL(19,6) DEFAULT 0 NOT NULL,
    [QualifiedTurnoverLimit] DECIMAL(19,6) DEFAULT 0 NOT NULL,
    [IsQualifiedReferee] BIT DEFAULT 0 NOT NULL,
    [IsValidReferee] BIT DEFAULT 0 NOT NULL,
    [InValidReason] NVARCHAR(100) DEFAULT '' NOT NULL,
    [ModifiedOn] DATETIME NOT NULL,
    [ModifiedBy] NVARCHAR(100) NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [CreatedBy] NVARCHAR(100) NOT NULL,
    CONSTRAINT UC_ReferralMember UNIQUE (CustomerId),
    CONSTRAINT CK_ReferralMember_InValidReason CHECK (InValidReason IN ('','IP','Email','Phone','Not eligible as a referee: Same IP','Not eligible as a referee: Same phone number','Not eligible as a referee: Same Email'))
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | int, PK | Primary key, auto-incrementing unique identifier |
| CustomerId | int | Customer identifier |
| Username | nvarchar(100) | Username of the customer |
| ParentId | int | Parent customer identifier (referrer) |
| CompanyPromotionId | int | Foreign key to CompanyPromotion table |
| PromotionRequestId | int | Related promotion request ID |
| Currency | nvarchar(10) | Currency code (ISO 4217) |
| ContributeAmount | decimal(19,6) | Amount contributed by the referee |
| CurrentTurnover | decimal(19,6) | Current turnover achieved |
| CurrentDepositAmount | decimal(19,6) | Current deposit amount |
| CurrentWithdrawalAmount | decimal(19,6) | Current withdrawal amount |
| DepositCount | int | Number of deposits made |
| WithdrawalCount | int | Number of withdrawals made |
| CurrentWinlost | decimal(19,6) | Current win/loss amount |
| QualifiedDepositLimit | decimal(19,6) | Qualified deposit limit |
| QualifiedTurnoverLimit | decimal(19,6) | Qualified turnover limit |
| IsQualifiedReferee | bit | Whether the referee is qualified |
| IsValidReferee | bit | Whether the referee is valid |
| InValidReason | nvarchar(100) | Reason for invalid referee |
| ModifiedOn | datetime | Last modification timestamp |
| ModifiedBy | nvarchar(100) | User who last modified the record |
| CreatedOn | datetime | Creation timestamp |
| CreatedBy | nvarchar(100) | User who created the record |

### PromotionApplicationFilterDetail
```sql
CREATE TABLE [dbo].[PromotionApplicationFilterDetail] (
  [WebId] INT NOT NULL,
  [CustomerId] INT NOT NULL,
  [CompanyPromotionId] INT NOT NULL,
  [WalletType] NVARCHAR(100),
  [Currency] NVARCHAR(3) NOT NULL,
  [RecordId] INT NOT NULL,
  [RecordTable] NVARCHAR(100) NOT NULL,
  [RecordStatus] NVARCHAR(100) NOT NULL CHECK ([RecordStatus] IN ('Success', 'Fail')),
  [EligibleProvider] NVARCHAR(MAX),
  [TransactionType] NVARCHAR(500),
  [FilterStartDate] DATETIME,
  [FilterEndDate] DATETIME,
  [CreatedOn] DATETIME,
  [CreatedBy] NVARCHAR(100)
)
```
| Column | Type | Description |
|--------|------|-------------|
| WebId | int | Website/application identifier |
| CustomerId | int | Customer identifier |
| CompanyPromotionId | int | Foreign key to CompanyPromotion table |
| WalletType | nvarchar(100) | Type of wallet (if applicable) |
| Currency | nvarchar(3) | Currency code (ISO 4217) |
| RecordId | int | Related record identifier |
| RecordTable | nvarchar(100) | Name of the related record table |
| RecordStatus | nvarchar(100) | Status of the record (Success, Fail) |
| EligibleProvider | nvarchar(MAX) | Eligible game providers (if any) |
| TransactionType | nvarchar(500) | Type of transaction |
| FilterStartDate | datetime | Filter start date/time |
| FilterEndDate | datetime | Filter end date/time |
| CreatedOn | datetime | Creation timestamp |
| CreatedBy | nvarchar(100) | User who created the record |

### CompanyEditPromotionGameProviderRecord
```sql
CREATE TABLE [CompanyEditPromotionGameProviderRecord] (
  [Id] INT IDENTITY(1,1) PRIMARY KEY,
  [WebId] INT NOT NULL,
  [CompanyPromotionId] INT NOT NULL,
  [PromotionTypeId] INT NOT NULL,
  [PromotionGameProviderIds] NVARCHAR(MAX) NOT NULL,
  [ModifiedBy] NVARCHAR(100) NOT NULL,
  [ModifiedOn] datetime NOT NULL
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | int, PK | Primary key, auto-incrementing unique identifier |
| WebId | int | Website/application identifier |
| CompanyPromotionId | int | Foreign key to CompanyPromotion table |
| PromotionTypeId | int | Foreign key to PromotionType table |
| PromotionGameProviderIds | nvarchar(MAX) | List of game provider IDs |
| ModifiedBy | nvarchar(100) | User who last modified the record |
| ModifiedOn | datetime | Last modification timestamp |

### PromotionSystemRejectRecord
```sql
CREATE TABLE [PromotionSystemRejectRecord] (
    [Id] int PRIMARY KEY IDENTITY(1,1) NOT NULL,
    [WebId] int,
    [CustomerId] INT NOT NULL,
    [Username] NVARCHAR(100) NOT NULL,
    [ActionType] NVARCHAR(100) NOT NULL,
    [RejectSetting] NVARCHAR(100) NOT NULL,
    [PromotionRequestId] INT NOT NULL,
    [CompanyPromotionId] INT NOT NULL,
    [Remark] NVARCHAR(MAX) NOT NULL,
    [CreatedOn] DATETIME NOT NULL DEFAULT GETDATE(),
    CONSTRAINT [CK_PromotionSystemRejectRecord_RejectSetting] CHECK ([dbo].[CheckPromotionSettingAvailable]([RejectSetting])=1 OR [RejectSetting] = 'Deposit' OR [RejectSetting] = 'ApplyPromotion'OR [RejectSetting] = 'RebateDeposit' OR [RejectSetting] = 'RebateOnWinlose'),
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | int, PK | Primary key, auto-incrementing unique identifier |
| WebId | int | Website/application identifier |
| CustomerId | int | Customer identifier |
| Username | nvarchar(100) | Username of the customer |
| ActionType | nvarchar(100) | Type of action rejected |
| RejectSetting | nvarchar(100) | Setting that caused rejection |
| PromotionRequestId | int | Related promotion request ID |
| CompanyPromotionId | int | Foreign key to CompanyPromotion table |
| Remark | nvarchar(MAX) | Reason for rejection |
| CreatedOn | datetime | Creation timestamp |

### RebateEligibleGames
```sql
CREATE TABLE [dbo].[RebateEligibleGames](
    [CompanyPromotionId] INT NOT NULL,
    [GameProviderId] INT NOT NULL,
    [GameId] NVARCHAR(MAX) NOT NULL,
    [CreatedOn] DATETIME,
    [CreatedBy] NVARCHAR(50),
    [ModifiedOn] DATETIME,
    [ModifiedBy] NVARCHAR(50)
)
```
| Column | Type | Description |
|--------|------|-------------|
| CompanyPromotionId | int | Foreign key to CompanyPromotion table |
| GameProviderId | int | Game provider identifier |
| GameId | nvarchar(MAX) | Game identifier(s) |
| CreatedOn | datetime | Creation timestamp |
| CreatedBy | nvarchar(50) | User who created the record |
| ModifiedOn | datetime | Last modification timestamp |
| ModifiedBy | nvarchar(50) | User who last modified the record |

## Multilingual and Type Language Tables

### PromotionTypeLanguage
```sql
CREATE TABLE [PromotionTypeLanguage] (
  [Id] INT PRIMARY KEY IDENTITY(1,1),
  [PromotionTypeId] INT,
  [Language] NVARCHAR(100),
  [Name] NVARCHAR(100),
  [Content] NVARCHAR(max),
  [CreatedOn] DATETIME,
  CONSTRAINT UC_PromotionTypeLanguage UNIQUE ([Language],PromotionTypeId)
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | int, PK | Primary key, auto-incrementing unique identifier |
| PromotionTypeId | int | Foreign key to PromotionType table |
| Language | nvarchar(100) | Language code |
| Name | nvarchar(100) | Name in the specified language |
| Content | nvarchar(max) | Localized content |
| CreatedOn | datetime | Creation timestamp |

### CompanyPromotionLanguage
```sql
CREATE TABLE [CompanyPromotionLanguage] (
  [Id] int PRIMARY KEY IDENTITY(1,1),
  [CompanyPromotionId] int NOT NULL,
  [Language] nvarchar(10) NOT NULL,
  [Title] nvarchar(MAX) NULL,
  [Code] nvarchar(MAX) NULL,
  [Content] nvarchar(MAX) NULL,
  [Remark] nvarchar(MAX) NULL,
  [ImagePath] nvarchar(MAX) NULL,
  [ModifiedOn] datetime,
  [ModifiedBy] nvarchar(100),
  [CreatedOn] datetime,
  [CreatedBy] nvarchar(100)
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | int, PK | Primary key, auto-incrementing unique identifier |
| CompanyPromotionId | int | Foreign key to CompanyPromotion table |
| Language | nvarchar(10) | Language code |
| Title | nvarchar(MAX) | Localized title |
| Code | nvarchar(MAX) | Localized code or identifier |
| Content | nvarchar(MAX) | Localized content |
| Remark | nvarchar(MAX) | Localized remark |
| ImagePath | nvarchar(MAX) | Path to localized image |
| ModifiedOn | datetime | Last modification timestamp |
| ModifiedBy | nvarchar(100) | User who last modified the record |
| CreatedOn | datetime | Creation timestamp |
| CreatedBy | nvarchar(100) | User who created the record |

## Arbitrage and Miscellaneous Tables

### ArbitrageAction
```sql
CREATE TABLE [ArbitrageAction] (
  [Id] int IDENTITY(1,1) PRIMARY KEY,
  [WebId] int NOT NULL,
  [CustomerId] INT NOT NULL,
  [Username] NVARCHAR(100) NOT NULL,
  [CompanyPromotionId] INT NOT NULL,
  [WalletId] INT NOT NULL,
  [GameProviderId]  INT NOT NULL,
  [GameId]  INT NOT NULL,
  [CreatedOn] datetime NOT NULL,
  [ModifiedOn] datetime NOT NULL
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | int, PK | Primary key, auto-incrementing unique identifier |
| WebId | int | Website/application identifier |
| CustomerId | int | Customer identifier |
| Username | nvarchar(100) | Username of the customer |
| CompanyPromotionId | int | Foreign key to CompanyPromotion table |
| WalletId | int | Wallet identifier |
| GameProviderId | int | Game provider identifier |
| GameId | int | Game identifier |
| CreatedOn | datetime | Creation timestamp |
| ModifiedOn | datetime | Last modification timestamp |

### PromotionTransfer
```sql
CREATE TABLE [PromotionTransfer] (
  [Id] INT PRIMARY KEY IDENTITY(1,1),
  [WebId] INT,
  [CustomerId] INT,
  [FromPromotionRequestId] INT,
  [ToPromotionRequestId] INT,
  [Amount] DECIMAL(19,6),
  [CreatedOn] DATETIME,
  [CreatedBy] NVARCHAR(MAX),
  [Remark] NVARCHAR(Max)
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | int, PK | Primary key, auto-incrementing unique identifier |
| WebId | int | Website/application identifier |
| CustomerId | int | Customer identifier |
| FromPromotionRequestId | int | Source promotion request ID |
| ToPromotionRequestId | int | Target promotion request ID |
| Amount | decimal(19,6) | Amount transferred |
| CreatedOn | datetime | Creation timestamp |
| CreatedBy | nvarchar(MAX) | User who created the record |
| Remark | nvarchar(MAX) | Additional remarks |

## Indexes

### CompanyPromotion
- `IX_CompanyPromotion_WebId` on `WebId`
- `IX_CompanyPromotion_Status` on `Status`
- `IX_CompanyPromotion_PromotionTypeId` on `PromotionTypeId`

### PromotionRequest
- `IX_PromotionReqeust_GetPromotionRequestList` on `CompanyPromotionId` (includes multiple columns)
- `IX_PromotionReqeust_VerifiedOn_CustomerId_WebId_Status_PromotionTypeId` on multiple columns

### DepositPromotionOption
- `IX_DepositPromotionOption_WebId_Currency` on `WebId, Currency`

## Relationships
- `CompanyPromotion` references `PromotionType` through `PromotionTypeId`
- `PromotionRequest` references `CompanyPromotion` through `CompanyPromotionId`
- All promotion-specific settings tables reference their parent tables
- All tables maintain audit trails with creation and modification timestamps
- Most tables are linked to website configuration through the `WebId` field 

## Key Features

1. **Multilingual Support**
   - Full support for multiple languages through CompanyPromotionLanguage table
   - Default language setting per promotion
   - Language-specific content including titles, descriptions, and remarks

2. **Flexible Promotion Types**
   - Multiple promotion types supporting various business scenarios:
     - Registration bonuses
     - First deposit bonuses
     - Daily deposit bonuses
     - Rebate systems (turnover, lose-only, deposit/withdrawal)
     - Referral programs
     - Lucky wheel/draw promotions
     - Coupon reward systems

3. **Status Management**
   - Comprehensive status tracking for promotions:
     - Active
     - Deleted
     - Disabled
     - ActiveAndOpenForApply
     - Finished
   - Request status workflow:
     - Waiting
     - Rejected
     - Approved
     - Completed
     - Cancelled

4. **Security Features**
   - IP tracking for promotion requests
   - Device fingerprinting
   - System reject record tracking
   - Duplicate promotion prevention
   - Validation checks on settings and statuses

5. **Promotion Filtering**
   - Game provider filtering
   - Currency-based filtering
   - Date range filtering
   - Eligibility tracking

6. **Bonus Management**
   - Turnover tracking
   - Withdrawal requirements
   - Multiple bonus calculation methods
   - Automatic completion checks

## Common Workflows

1. **Promotion Creation**
   - Set basic promotion details (currency, dates, type)
   - Configure promotion settings
   - Add multilingual content
   - Set game provider restrictions
   - Define bonus calculation rules

2. **Promotion Application**
   - User submits request
   - System validates eligibility
   - Request approval/rejection
   - Bonus credit and turnover tracking
   - Withdrawal requirement monitoring

3. **Coupon Management**
   - Coupon code generation
   - Validation and redemption
   - Usage tracking
   - User assignment and tracking

4. **Lucky Wheel/Draw**
   - Prize pool management
   - Ticket allocation
   - Winner selection
   - Reward distribution
   - Usage limits enforcement

5. **Referral Program**
   - Referrer tracking
   - Bonus calculation
   - Multi-level referral support
   - Cash redemption management

## Best Practices
1. Always check for existing active promotions of the same type before creating new ones
2. Implement proper validation for promotion settings
3. Monitor turnover requirements and completion status
4. Maintain audit trails for all promotion-related actions
5. Use appropriate indexes for performance optimization
6. Implement proper currency handling and conversions 