# Record Database Schema

## Overview
The Record database maintains historical records, audit trails, and system logs for various operations across the platform.

## Core Tables

### BalanceHistory
Tracks all changes to player balances, including cash and outstanding balances, for audit and reporting purposes.

```sql
CREATE TABLE [dbo].[BalanceHistory] (
    [ID] Bigint IDENTITY(1,1),
    [WebId] INT,
    [WalletId] INT,
    [ProductType] INT,
    [CustomerId] INT,
    [Username] NVARCHAR(50),
    [Currency] CHAR(3),
    [TransactionId] NVARCHAR(200),
    [BeforeCashBalance] DECIMAL(19,6),
    [BeforeOutstanding] DECIMAL(19,6),
    [CashBalanceDelta] DECIMAL(19,6),
    [OutstandingDelta] DECIMAL(19,6),
    [ActionType] NVARCHAR(50),
    [Remark] NVARCHAR(100),
    [SpName] NVARCHAR(200),
    [ModifiedDate] DATETIME,
    [ModifiedBy] NVARCHAR(100),
    [GameProvider] NVARCHAR(100),
    [GameName] NVARCHAR(255),
    CONSTRAINT [PK_BalanceHistory] PRIMARY KEY ([ID])
)
```

| Column              | Type             | Description                                                      |
|---------------------|------------------|------------------------------------------------------------------|
| ID                  | Bigint, PK       | Primary key, auto-incrementing unique identifier                 |
| WebId               | INT              | Website identifier                                               |
| WalletId            | INT              | Wallet identifier                                                |
| ProductType         | INT              | Product type (game, casino, etc.)                                |
| CustomerId          | INT              | Customer identifier                                              |
| Username            | NVARCHAR(50)     | Username of the customer                                         |
| Currency            | CHAR(3)          | Currency code (3-letter ISO code)                                |
| TransactionId       | NVARCHAR(200)    | Transaction identifier                                           |
| BeforeCashBalance   | DECIMAL(19,6)    | Cash balance before the transaction                              |
| BeforeOutstanding   | DECIMAL(19,6)    | Outstanding balance before the transaction                       |
| CashBalanceDelta    | DECIMAL(19,6)    | Change in cash balance                                           |
| OutstandingDelta    | DECIMAL(19,6)    | Change in outstanding balance                                    |
| ActionType          | NVARCHAR(50)     | Type of action (deposit, withdrawal, etc.)                       |
| Remark              | NVARCHAR(100)    | Additional remarks                                               |
| SpName              | NVARCHAR(200)    | Stored procedure or process name                                 |
| ModifiedDate        | DATETIME         | Timestamp when the record was modified                           |
| ModifiedBy          | NVARCHAR(100)    | User who modified the record                                     |
| GameProvider        | NVARCHAR(100)    | Game provider name                                               |
| GameName            | NVARCHAR(255)    | Game name                                                        |

**Indexes:**
- `IX_BalanceHistory_WebId_Username_ModifiedDate`: For efficient balance history queries
- `IX_BalanceHistory_WebId_ModifiedDate`: For reporting and auditing

### DailyStatementHistory
Tracks daily financial statements for each customer, including cash flow, commissions, discounts, and status, supporting financial reconciliation and auditing.

```sql
CREATE TABLE [dbo].[DailyStatementHistory] (
    [Transid] BIGINT IDENTITY(1,1) NOT NULL,
    [Refno] NVARCHAR(100) NOT NULL,
    [CustID] INT NOT NULL,
    [UserName] NVARCHAR(50) NOT NULL,
    [WinLostDate] SMALLDATETIME NULL,
    [AgtID] INT NOT NULL,
    [MaID] INT NOT NULL,
    [SmaID] INT NOT NULL,
    [SSmaID] INT NOT NULL,
    [SHID] INT NOT NULL,
    [StatementType] TINYINT NOT NULL,
    [Currency] CHAR(3) NULL,
    [CustomerStatus] INT NOT NULL,
    [ProductType] INT NOT NULL,
    [StatementStatus] INT NULL,
    [CashIn] DECIMAL(19,6) NOT NULL,
    [CashOut] DECIMAL(19,6) NOT NULL,
    [CommIn] DECIMAL(19,6) NOT NULL,
    [CommOut] DECIMAL(19,6) NOT NULL,
    [DiscountIn] DECIMAL(19,6) NOT NULL,
    [DiscountOut] DECIMAL(19,6) NOT NULL,
    [ActualRate] DECIMAL(12,8) NULL,
    [TotalCashIn] DECIMAL(19,6) NOT NULL,
    [TotalCashOut] DECIMAL(19,6) NOT NULL,
    [TransDate] DATETIME NOT NULL,
    [CasinoTotalCashIn] DECIMAL(19,6) NULL,
    [CasinoTotalCashOut] DECIMAL(19,6) NULL,
    [NetTurnover] DECIMAL(19,6) NULL,
    [tstamp] ROWVERSION NULL,
    [WebId] INT NOT NULL,
    CONSTRAINT [PK_WinlostDateIsNull] PRIMARY KEY NONCLUSTERED ([Transid] ASC)
)
```

| Column              | Type             | Description                                      |
|---------------------|------------------|--------------------------------------------------|
| Transid             | BIGINT, PK       | Primary key, auto-incrementing unique identifier |
| Refno               | NVARCHAR(100)    | Reference number for the statement               |
| CustID              | INT              | Customer identifier                              |
| UserName            | NVARCHAR(50)     | Username                                         |
| WinLostDate         | SMALLDATETIME    | Date of win/loss                                 |
| AgtID               | INT              | Agent identifier                                 |
| MaID                | INT              | Master agent identifier                          |
| SmaID               | INT              | Sub-master agent identifier                      |
| SSmaID              | INT              | Super sub-master agent identifier                |
| SHID                | INT              | Super head identifier                            |
| StatementType       | TINYINT          | Type of statement                                |
| Currency            | CHAR(3)          | Currency code                                    |
| CustomerStatus      | INT              | Customer status                                  |
| ProductType         | INT              | Product type                                     |
| StatementStatus     | INT              | Status of the statement                          |
| CashIn              | DECIMAL(19,6)    | Cash in amount                                   |
| CashOut             | DECIMAL(19,6)    | Cash out amount                                  |
| CommIn              | DECIMAL(19,6)    | Commission in                                    |
| CommOut             | DECIMAL(19,6)    | Commission out                                   |
| DiscountIn          | DECIMAL(19,6)    | Discount in                                      |
| DiscountOut         | DECIMAL(19,6)    | Discount out                                     |
| ActualRate          | DECIMAL(12,8)    | Actual exchange rate                             |
| TotalCashIn         | DECIMAL(19,6)    | Total cash in                                    |
| TotalCashOut        | DECIMAL(19,6)    | Total cash out                                   |
| TransDate           | DATETIME         | Transaction date                                 |
| CasinoTotalCashIn   | DECIMAL(19,6)    | Casino total cash in                             |
| CasinoTotalCashOut  | DECIMAL(19,6)    | Casino total cash out                            |
| NetTurnover         | DECIMAL(19,6)    | Net turnover                                     |
| tstamp              | ROWVERSION       | Row version for concurrency                      |
| WebId               | INT              | Website identifier                               |

### PromotionCashSettledHistory
Tracks settled promotion cash transactions for each player, including before/after values and provider breakdowns.

```sql
CREATE TABLE [dbo].[PromotionCashSettledHistory](
    [Id] BIGINT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [CustomerId] INT NOT NULL,
    [PromotionRequestId] INT DEFAULT(0) NOT NULL,
    [TransferIn] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [TransferOut] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [SBPlayerCashSettled] DECIMAL(19,6) DEFAULT(0) NULL,
    [SBPlayerCashReturn] DECIMAL(19,6) DEFAULT(0) NULL,
    [LCPlayerCashSettled] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [LCPlayerCashReturn] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [GMPlayerCashSettled] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [GMPlayerCashReturn] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [AfterSBPlayerCashSettled] DECIMAL(19,6) DEFAULT(0) NULL,
    [AfterSBPlayerCashReturn] DECIMAL(19,6) DEFAULT(0) NULL,
    [AfterLCPlayerCashSettled] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [AfterLCPlayerCashReturn] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [AfterGMPlayerCashSettled] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [AfterGMPlayerCashReturn] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [ModifiedBy] NVARCHAR(50) NULL,
    [ModifiedOn] DATETIME NULL,
    CONSTRAINT [PK_PromotionCashSettledHistory] PRIMARY KEY CLUSTERED ([ID])
)
```

| Column                   | Type             | Description                                      |
|--------------------------|------------------|--------------------------------------------------|
| Id                       | BIGINT, PK       | Primary key, auto-incrementing unique identifier |
| WebId                    | INT              | Website identifier                               |
| CustomerId               | INT              | Customer identifier                              |
| PromotionRequestId       | INT              | Promotion request identifier                      |
| TransferIn               | DECIMAL(19,6)    | Amount transferred in                            |
| TransferOut              | DECIMAL(19,6)    | Amount transferred out                           |
| SBPlayerCashSettled      | DECIMAL(19,6)    | Settled cash for SB player                       |
| SBPlayerCashReturn       | DECIMAL(19,6)    | Returned cash for SB player                      |
| LCPlayerCashSettled      | DECIMAL(19,6)    | Settled cash for LC player                       |
| LCPlayerCashReturn       | DECIMAL(19,6)    | Returned cash for LC player                      |
| GMPlayerCashSettled      | DECIMAL(19,6)    | Settled cash for GM player                       |
| GMPlayerCashReturn       | DECIMAL(19,6)    | Returned cash for GM player                      |
| AfterSBPlayerCashSettled | DECIMAL(19,6)    | After-settle cash for SB player                  |
| AfterSBPlayerCashReturn  | DECIMAL(19,6)    | After-settle returned cash for SB player         |
| AfterLCPlayerCashSettled | DECIMAL(19,6)    | After-settle cash for LC player                  |
| AfterLCPlayerCashReturn  | DECIMAL(19,6)    | After-settle returned cash for LC player         |
| AfterGMPlayerCashSettled | DECIMAL(19,6)    | After-settle cash for GM player                  |
| AfterGMPlayerCashReturn  | DECIMAL(19,6)    | After-settle returned cash for GM player         |
| ModifiedBy               | NVARCHAR(50)     | User who last modified the record                |
| ModifiedOn               | DATETIME         | Timestamp of last modification                   |

### PromotionCashUsedHistory
Tracks usage of promotion cash by players, including before/after values and provider breakdowns, for audit and compliance.

```sql
CREATE TABLE [dbo].[PromotionCashUsedHistory](
    [Id] BIGINT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [CustomerId] INT NOT NULL,
    [PromotionRequestId] INT DEFAULT(0) NOT NULL,
    [Username] NVARCHAR(50) NOT NULL,
    [Currency] NVARCHAR(3) NULL,
    [SBPlayerCashUsed] DECIMAL(19,6) DEFAULT(0) NULL,
    [LCPlayerCashUsed] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [GMPlayerCashUsed] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [AfterSBPlayerCashUsed] DECIMAL(19,6) DEFAULT(0) NULL,
    [AfterLCPlayerCashUsed] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [AfterGMPlayerCashUsed] DECIMAL(19,6) DEFAULT(0) NOT NULL,
    [ModifiedBy] NVARCHAR(50) NULL,
    [ModifiedOn] DATETIME NULL,
    CONSTRAINT [PK_PromotionCashUsed] PRIMARY KEY CLUSTERED ([ID])
)
```

| Column                | Type             | Description                                      |
|-----------------------|------------------|--------------------------------------------------|
| Id                    | BIGINT, PK       | Primary key, auto-incrementing unique identifier |
| WebId                 | INT              | Website identifier                               |
| CustomerId            | INT              | Customer identifier                              |
| PromotionRequestId    | INT              | Promotion request identifier                      |
| Username              | NVARCHAR(50)     | Username of the player                           |
| Currency              | NVARCHAR(3)      | Currency code                                    |
| SBPlayerCashUsed      | DECIMAL(19,6)    | Promotion cash used for SB player                |
| LCPlayerCashUsed      | DECIMAL(19,6)    | Promotion cash used for LC player                |
| GMPlayerCashUsed      | DECIMAL(19,6)    | Promotion cash used for GM player                |
| AfterSBPlayerCashUsed | DECIMAL(19,6)    | After-use cash for SB player                     |
| AfterLCPlayerCashUsed | DECIMAL(19,6)    | After-use cash for LC player                     |
| AfterGMPlayerCashUsed | DECIMAL(19,6)    | After-use cash for GM player                     |
| ModifiedBy            | NVARCHAR(50)     | User who last modified the record                |
| ModifiedOn            | DATETIME         | Timestamp of last modification                   |

## System Monitoring Tables

### ApplicationError
Tracks application-level errors and exceptions for system monitoring and debugging.

```sql
CREATE TABLE [dbo].[ApplicationError] (
    [ID] Bigint IDENTITY(1,1),
    [HostName] NVARCHAR(100) NOT NULL,
    [Url] NVARCHAR(1000) NULL,
    [Username] NVARCHAR(100) NULL,
    [Application] NVARCHAR(100) NOT NULL,
    [ErrorFilePath] NVARCHAR(1000) NULL,
    [ErrorLine] INT NULL,
    [CallStake] NVARCHAR(MAX) NULL,
    [ErrorMessage] NVARCHAR(1000) NULL,
    [CreatedOn] DATETIME NULL,
    CONSTRAINT [PK_ApplicationError] PRIMARY KEY ([ID])
)
```

| Column         | Type             | Description                                                      |
|---------------|------------------|------------------------------------------------------------------|
| ID            | Bigint, PK       | Primary key, auto-incrementing unique identifier                 |
| HostName      | NVARCHAR(100)    | Host where error occurred (server name)                          |
| Url           | NVARCHAR(1000)   | URL at the time of error (if applicable)                         |
| Username      | NVARCHAR(100)    | Username involved in the error (if any)                          |
| Application   | NVARCHAR(100)    | Application name where the error occurred                        |
| ErrorFilePath | NVARCHAR(1000)   | File path of the error (if available)                            |
| ErrorLine     | INT              | Line number in the file where the error occurred                 |
| CallStake     | NVARCHAR(MAX)    | Call stack at the time of error                                  |
| ErrorMessage  | NVARCHAR(1000)   | Error message                                                    |
| CreatedOn     | DATETIME         | Timestamp when the error was logged                              |

### OperatorActivityLog
```sql
CREATE TABLE [dbo].[OperatorActivityLog] (
    [Id] INT NOT NULL PRIMARY KEY IDENTITY(1,1),
    [WebId] INT,
    [SimulateLoginId] int,
    [SimulateLoginName] nvarchar(50),
    [OperatorId] int,
    [OperatorName] nvarchar(50),
    [RequestActionUrl] nvarchar(200),
    [RequestData] nvarchar(max),
    [ActionTime] datetime,
    [Duration] nvarchar(50),
    [Ip] NVARCHAR(100) NULL
)
```

## Game-Related Tables

### PlayerGameEntranceLocation
Tracks where and how players enter games, including device and provider details, for analytics and troubleshooting.

```sql
CREATE TABLE [dbo].[PlayerGameEntranceLocation](
    [WebId] INT,
    [Username] NVARCHAR(50),
    [EntranceLocation] NVARCHAR(50),
    [ApiGameProviderId] INT,
    [GameProviderId] INT,
    [GameIdAtGameProvider] INT,
    [Device] INT,
    [LoginOn] DATETIME
)
```

| Column              | Type           | Description                                      |
|---------------------|----------------|--------------------------------------------------|
| WebId               | INT            | Website identifier                               |
| Username            | NVARCHAR(50)   | Username of the player                           |
| EntranceLocation    | NVARCHAR(50)   | Where the player entered the game                |
| ApiGameProviderId   | INT            | API game provider identifier                     |
| GameProviderId      | INT            | Game provider identifier                         |
| GameIdAtGameProvider| INT            | Game ID at the provider                          |
| Device              | INT            | Device type (e.g., mobile, desktop)              |
| LoginOn             | DATETIME       | Timestamp of login                               |

### PlayerGameSearch
Logs player search actions for games, including device and search content, for UX and analytics.

```sql
CREATE TABLE [dbo].[PlayerGameSearch](
    [WebId] INT,
    [Username] NVARCHAR(50),
    [SearchContent] NVARCHAR(MAX),
    [SearchPage] NVARCHAR(200),
    [Device] INT,
    [SearchOn] DATETIME
)
```

| Column         | Type           | Description                                      |
|---------------|----------------|--------------------------------------------------|
| WebId         | INT            | Website identifier                               |
| Username      | NVARCHAR(50)   | Username of the player                           |
| SearchContent | NVARCHAR(MAX)  | Content searched by the player                   |
| SearchPage    | NVARCHAR(200)  | Page where the search was performed              |
| Device        | INT            | Device type (e.g., mobile, desktop)              |
| SearchOn      | DATETIME       | Timestamp of the search                          |

## Error Tracking Tables

### SboSeamlessBetsNotTally
Tracks seamless bets that do not tally between internal and provider records, for reconciliation and error handling.

```sql
CREATE TABLE [dbo].[SboSeamlessBetsNotTally](
    [Id] BIGINT IDENTITY(1,1) NOT NULL,
    [GPID] INT NULL,
    [WebId] INT NOT NULL,
    [RefNo] NVARCHAR(100) NOT NULL,
    [CustomerId] INT NOT NULL,
    [ParentId] INT NULL,
    [Username] NVARCHAR(50) NOT NULL,
    [OrderTime] DATETIME NOT NULL,
    [WinLostDate] DATETIME NULL,
    [Stake] DECIMAL(19,6) NOT NULL,
    [StakeAtGameProvider] DECIMAL(19,6) NOT NULL,
    [Currency] CHAR(3) NULL,
    [ExchangeRate] DECIMAL(12,8) NULL,
    [Status] VARCHAR(50) NOT NULL,
    [Winlost] DECIMAL(19,6) NOT NULL,
    [WinlostAtGameProvider] DECIMAL(19,6) NOT NULL,
    [CreatedOn] DATETIME DEFAULT GETDATE() NOT NULL,
    [isSendMail] BIT DEFAULT 0 NOT NULL,
    CONSTRAINT [PK_SboSeamlessBetsNotTally] PRIMARY KEY ([Webid],[RefNo])
)
```

| Column                | Type             | Description                                      |
|-----------------------|------------------|--------------------------------------------------|
| Id                    | BIGINT, PK       | Primary key, auto-incrementing unique identifier |
| GPID                  | INT              | Game provider ID (nullable)                      |
| WebId                 | INT              | Website identifier                               |
| RefNo                 | NVARCHAR(100)    | Reference number for the bet                     |
| CustomerId            | INT              | Customer identifier                              |
| ParentId              | INT              | Parent customer identifier (nullable)            |
| Username              | NVARCHAR(50)     | Username of the player                           |
| OrderTime             | DATETIME         | Time the bet was placed                          |
| WinLostDate           | DATETIME         | Date of win/loss                                 |
| Stake                 | DECIMAL(19,6)    | Stake amount                                     |
| StakeAtGameProvider   | DECIMAL(19,6)    | Stake amount at game provider                    |
| Currency              | CHAR(3)          | Currency code                                    |
| ExchangeRate          | DECIMAL(12,8)    | Exchange rate                                    |
| Status                | VARCHAR(50)      | Status of the bet                                |
| Winlost               | DECIMAL(19,6)    | Win/loss amount                                  |
| WinlostAtGameProvider | DECIMAL(19,6)    | Win/loss amount at game provider                 |
| CreatedOn             | DATETIME         | Timestamp when the record was created            |
| isSendMail            | BIT              | Whether a notification email was sent            |

### SboSeamlessWalletError
Logs errors related to seamless wallet operations for SBO, including error codes and messages, for troubleshooting and audit.

```sql
CREATE TABLE [dbo].[SboSeamlessWalletError](
    [Id] BIGINT IDENTITY(1,1) NOT NULL,
    [ProductType] INT NULL,
    [WebId] INT NOT NULL,
    [RefNo] NVARCHAR(100) NOT NULL,
    [Username] NVARCHAR(50) NOT NULL,
    [ErrorCode] INT NOT NULL,
    [Action] NVARCHAR(100) NOT NULL,
    [ErrorMessage] NVARCHAR(200) NOT NULL,
    [CreatedOn] DATETIME DEFAULT GETDATE() NOT NULL,
    CONSTRAINT [PK_SboSeamlessWalletError] PRIMARY KEY ([Id])
)
```

| Column         | Type           | Description                                      |
|---------------|----------------|--------------------------------------------------|
| Id            | BIGINT, PK     | Primary key, auto-incrementing unique identifier |
| ProductType   | INT            | Product type (nullable)                          |
| WebId         | INT            | Website identifier                               |
| RefNo         | NVARCHAR(100)  | Reference number for the transaction             |
| Username      | NVARCHAR(50)   | Username of the player                           |
| ErrorCode     | INT            | Error code                                       |
| Action        | NVARCHAR(100)  | Action attempted                                 |
| ErrorMessage  | NVARCHAR(200)  | Error message                                    |
| CreatedOn     | DATETIME       | Timestamp when the error was logged              |

### BetStatusNotTally
Tracks discrepancies between internal and game provider bet statuses for auditing and reconciliation.

```sql
CREATE TABLE [dbo].[BetStatusNotTally](
    [WebId] INT NOT NULL,
    [CustomerId] INT NOT NULL,
    [RefNo] NVARCHAR(100) NOT NULL,
    [OrderTime] DATETIME NOT NULL,
    [Status] VARCHAR(50) NOT NULL,
    [StatusAtGameProvider] VARCHAR(50) NOT NULL,
    [IsFirstTime] BIT NOT NULL
)
```

| Column              | Type           | Description                                      |
|---------------------|----------------|--------------------------------------------------|
| WebId               | INT            | Website identifier                               |
| CustomerId          | INT            | Customer identifier                              |
| RefNo               | NVARCHAR(100)  | Reference number for the bet                     |
| OrderTime           | DATETIME       | Time the bet was placed                          |
| Status              | VARCHAR(50)    | Internal system status of the bet                |
| StatusAtGameProvider| VARCHAR(50)    | Status as reported by the game provider          |
| IsFirstTime         | BIT            | Indicates if this is the first occurrence        |

### Conversation
Tracks chat or support conversations between users and the system.

```sql
CREATE TABLE [dbo].[Conversation](
    [Id] INT IDENTITY(1,1) NOT NULL,
    [WebId] INT NOT NULL,
    [CustomerId] INT NOT NULL,
    [CreatedOn] DATETIME
)
```

| Column      | Type         | Description                              |
|-------------|--------------|------------------------------------------|
| Id          | INT, PK      | Primary key, auto-incrementing identifier|
| WebId       | INT          | Website identifier                       |
| CustomerId  | INT          | Customer identifier                      |
| CreatedOn   | DATETIME     | Timestamp when the conversation started  |

### DailyStatementCheckRecord
Tracks the correctness and correction status of daily statements for auditing purposes.

```sql
CREATE TABLE [dbo].[DailyStatementCheckRecord](
    [WebId] INT NOT NULL,
    [Winlostdate] DATETIME NOT NULL,
    [ProductType] INT NOT NULL,
    [IsCorrect] BIT NULL,
    [HasBeenCorrected] BIT NULL,
    [ModifiedOn] DATETIME DEFAULT (getdate()) NOT NULL,
    [ModifiedBy] NVARCHAR(100) NOT NULL,
    CONSTRAINT [PK_DailyStatementCheckRecord] PRIMARY KEY CLUSTERED ([WebId], [Winlostdate], [ProductType])
)
```

| Column           | Type         | Description                                      |
|------------------|--------------|--------------------------------------------------|
| WebId            | INT          | Website identifier                               |
| Winlostdate      | DATETIME     | Date of win/loss                                 |
| ProductType      | INT          | Product type                                     |
| IsCorrect        | BIT          | Indicates if the statement is correct            |
| HasBeenCorrected | BIT          | Indicates if the statement has been corrected    |
| ModifiedOn       | DATETIME     | Timestamp of last modification                   |
| ModifiedBy       | NVARCHAR(100)| User who last modified the record                |

## Communication Tables

### Message
Tracks individual messages within a conversation, supporting chat and support features.

```sql
CREATE TABLE [dbo].[Message](
    [Id] INT IDENTITY(1,1) NOT NULL,
    [ConversationId] INT NOT NULL,
    [MessageType] INT NOT NULL,
    [Content] NVARCHAR(MAX),
    [CreatedOn] DATETIME
)
```

| Column         | Type         | Description                                      |
|---------------|--------------|--------------------------------------------------|
| Id            | INT, PK      | Primary key, auto-incrementing unique identifier |
| ConversationId| INT          | Reference to the conversation                    |
| MessageType   | INT          | Type of message (e.g., user, system, etc.)       |
| Content       | NVARCHAR(MAX)| Message content                                  |
| CreatedOn     | DATETIME     | Timestamp when the message was created           |

---

### NegativeOutstandingRecord
Logs cases where a player's outstanding balance is negative, for risk and error monitoring.

```sql
CREATE TABLE [dbo].[NegativeOutstandingRecord](
    [WebId] INT,
    [CustomerId] INT,
    [Username] NVARCHAR(50),
    [CashBalance] DECIMAL(19,6),
    [Outstanding] DECIMAL(19,6),
    [Remark] NVARCHAR(100),
    [CreatedOn] DATETIME,
    [IsSendMail] BIT,
    [Count] INT CONSTRAINT [DF_NegativeOutstandingRecord_Count] DEFAULT ((0)) NOT NULL
)
```

| Column         | Type           | Description                                      |
|---------------|----------------|--------------------------------------------------|
| WebId         | INT            | Website identifier                               |
| CustomerId    | INT            | Customer identifier                              |
| Username      | NVARCHAR(50)   | Username of the customer                         |
| CashBalance   | DECIMAL(19,6)  | Player's cash balance                            |
| Outstanding   | DECIMAL(19,6)  | Player's outstanding balance                     |
| Remark        | NVARCHAR(100)  | Additional remarks                               |
| CreatedOn     | DATETIME       | Timestamp when the record was created            |
| IsSendMail    | BIT            | Whether a notification email was sent            |
| Count         | INT            | Number of occurrences (default 0)                |

---

### OperatorActivityLog
Tracks actions performed by operators for audit and monitoring purposes.

```sql
CREATE TABLE [dbo].[OperatorActivityLog](
    [Id] INT NOT NULL PRIMARY KEY IDENTITY(1,1),
    [WebId] INT,
    [SimulateLoginId] INT,
    [SimulateLoginName] NVARCHAR(50),
    [OperatorId] INT,
    [OperatorName] NVARCHAR(50),
    [RequestActionUrl] NVARCHAR(200),
    [RequestData] NVARCHAR(MAX),
    [ActionTime] DATETIME,
    [Duration] NVARCHAR(50),
    [Ip] NVARCHAR(100) NULL
)
```

| Column            | Type           | Description                                      |
|-------------------|----------------|--------------------------------------------------|
| Id                | INT, PK        | Primary key, auto-incrementing unique identifier |
| WebId             | INT            | Website identifier                               |
| SimulateLoginId   | INT            | Simulated login user ID (if any)                 |
| SimulateLoginName | NVARCHAR(50)   | Simulated login username (if any)                |
| OperatorId        | INT            | Operator's user ID                               |
| OperatorName      | NVARCHAR(50)   | Operator's username                              |
| RequestActionUrl  | NVARCHAR(200)  | URL of the action performed                      |
| RequestData       | NVARCHAR(MAX)  | Data sent with the request                       |
| ActionTime        | DATETIME       | Timestamp of the action                          |
| Duration          | NVARCHAR(50)   | Duration of the action                           |
| Ip                | NVARCHAR(100)  | IP address of the operator (nullable)            |

## Indexes
- `IX_BalanceHistory_WebId_Username_ModifiedDate`: For efficient balance history queries
- `IX_BalanceHistory_WebId_ModifiedDate`: For reporting and auditing
- `IX_PromotionCashUsedHistory_vPlayerBalance_CustomerId`: For promotion cash usage tracking

## Relationships
- All record tables are linked to website configuration through the `WebId` field
- Balance history tracks both cash and outstanding balance changes
- Daily statements track financial transactions and commissions
- Promotion cash history tracks both settled and used amounts
- Error tracking tables maintain relationships with transactions and game providers

## Common Fields
Most record tables share these common fields:
- `WebId`: Website identifier
- `CustomerId`: Customer identifier
- `Username`: Customer username
- `CreatedOn/ModifiedOn`: Audit timestamps
- `CreatedBy/ModifiedBy`: Audit user information

### DetectGetBalanceOver4SecondsForWhoLoginIn30MinutesLog
Logs cases where balance retrieval takes over 4 seconds for users who logged in within the last 30 minutes, for performance monitoring.

```sql
CREATE TABLE [dbo].[DetectGetBalanceOver4SecondsForWhoLoginIn30MinutesLog](
    [Id] INT IDENTITY(1,1) PRIMARY KEY,
    [ServerNameGetBalanceFrom] VARCHAR(50),
    [RequestBody] NVARCHAR(1000),
    [WebId] INT,
    [Username] NVARCHAR(50),
    [SecondTaken] INT,
    [CreatedOn] DATETIME CONSTRAINT [DF_DetectGetBalanceOver4SecondsForWhoLoginIn30MinutesLog_CreatedOn] DEFAULT (GETDATE())
)
```

| Column                    | Type           | Description                                              |
|---------------------------|----------------|----------------------------------------------------------|
| Id                        | INT, PK        | Primary key, auto-incrementing unique identifier         |
| ServerNameGetBalanceFrom  | VARCHAR(50)    | Server name where balance was retrieved                  |
| RequestBody               | NVARCHAR(1000) | Request body content                                     |
| WebId                     | INT            | Website identifier                                       |
| Username                  | NVARCHAR(50)   | Username of the customer                                 |
| SecondTaken               | INT            | Seconds taken to retrieve balance                        |
| CreatedOn                 | DATETIME       | Timestamp when the log was created                       |

---

### InCorrectBalancecPlayer
Tracks players with incorrect balance or outstanding values for reconciliation and investigation.

```sql
CREATE TABLE [dbo].[InCorrectBalancecPlayer](
    [Webid] INT NOT NULL,
    [Customerid] INT NULL,
    [Username] NVARCHAR(50) NOT NULL,
    [CorrectCashBalance] DECIMAL(38,6) NULL,
    [Cashbalance] DECIMAL(23,6) NULL,
    [CorrectOutstanding] DECIMAL(38,6) NOT NULL,
    [Outstanding] DECIMAL(22,6) NULL,
    [Remark] VARCHAR(100) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL,
    [CreatedOn] DATETIME NOT NULL,
    [IsFirstTime] BIT NOT NULL
)
```

| Column              | Type             | Description                                      |
|---------------------|------------------|--------------------------------------------------|
| Webid               | INT              | Website identifier                               |
| Customerid          | INT              | Customer identifier                              |
| Username            | NVARCHAR(50)     | Username of the customer                         |
| CorrectCashBalance  | DECIMAL(38,6)    | Correct cash balance value                       |
| Cashbalance         | DECIMAL(23,6)    | Current cash balance value                       |
| CorrectOutstanding  | DECIMAL(38,6)    | Correct outstanding value                        |
| Outstanding         | DECIMAL(22,6)    | Current outstanding value                        |
| Remark              | VARCHAR(100)     | Additional remarks                               |
| ModifiedOn          | DATETIME         | Timestamp of last modification                   |
| CreatedOn           | DATETIME         | Timestamp when the record was created            |
| IsFirstTime         | BIT              | Indicates if this is the first occurrence        |

---

### IncorrectDailyStatementRecord
Logs records of daily statements with detected discrepancies for audit and correction.

```sql
CREATE TABLE [dbo].[IncorrectDailyStatementRecord] (
    [CustomerId] INT NOT NULL,
    [Username] NVARCHAR(100) NOT NULL,
    [agtid] INT NOT NULL,
    [diffamountcash] DECIMAL(19,6) NOT NULL,
    [diffamountcomm] DECIMAL(19,6) NOT NULL,
    [diffamounttotalcash] DECIMAL(19,6) NOT NULL,
    [winlostdate] DATETIME,
    [dswinlostdate] DATETIME,
    [producttype] INT NOT NULL,
    [ModifiedOn] DATETIME CONSTRAINT [IncorrectDailyStatementRecord_ModifiedOn] DEFAULT (getdate()) NOT NULL,
    [transid] INT IDENTITY(1,1),
    CONSTRAINT [PK_IncorrectDailyStatementRecord] PRIMARY KEY CLUSTERED (transid ASC)
);
```

| Column              | Type             | Description                                      |
|---------------------|------------------|--------------------------------------------------|
| CustomerId          | INT              | Customer identifier                              |
| Username            | NVARCHAR(100)    | Username                                         |
| agtid               | INT              | Agent identifier                                 |
| diffamountcash      | DECIMAL(19,6)    | Difference in cash amount                        |
| diffamountcomm      | DECIMAL(19,6)    | Difference in commission amount                  |
| diffamounttotalcash | DECIMAL(19,6)    | Total difference in cash                         |
| winlostdate         | DATETIME         | Win/loss date                                    |
| dswinlostdate       | DATETIME         | Daily statement win/loss date                    |
| producttype         | INT              | Product type                                     |
| ModifiedOn          | DATETIME         | Timestamp of last modification                   |
| transid             | INT, PK          | Primary key, auto-incrementing unique identifier |

## Indexes
- `IX_BalanceHistory_WebId_Username_ModifiedDate`: For efficient balance history queries
- `IX_BalanceHistory_WebId_ModifiedDate`: For reporting and auditing
- `IX_PromotionCashUsedHistory_vPlayerBalance_CustomerId`: For promotion cash usage tracking

## Relationships
- All record tables are linked to website configuration through the `WebId` field
- Balance history tracks both cash and outstanding balance changes
- Daily statements track financial transactions and commissions
- Promotion cash history tracks both settled and used amounts
- Error tracking tables maintain relationships with transactions and game providers

## Common Fields
Most record tables share these common fields:
- `WebId`: Website identifier
- `CustomerId`: Customer identifier
- `Username`: Customer username
- `CreatedOn/ModifiedOn`: Audit timestamps
- `CreatedBy/ModifiedBy`: Audit user information

### SchedulerExecution
Logs the execution of scheduled jobs, including project and scheduler names, for monitoring and audit.

```sql
CREATE TABLE [dbo].[SchedulerExecution](
    [Id] INT PRIMARY KEY IDENTITY(1, 1) NOT NULL,
    [ProjectName] NVARCHAR(50) NOT NULL,
    [SchedulerName] NVARCHAR(50) NOT NULL,
    [ExecutionTime] DATETIME NOT NULL DEFAULT (GETDATE())
)
```

| Column        | Type           | Description                                      |
|--------------|----------------|--------------------------------------------------|
| Id           | INT, PK        | Primary key, auto-incrementing unique identifier |
| ProjectName  | NVARCHAR(50)   | Name of the project                              |
| SchedulerName| NVARCHAR(50)   | Name of the scheduler                            |
| ExecutionTime| DATETIME       | Time the scheduler was executed                  |

---

### SchedulerExecutionByTimeZone
Logs scheduled job executions with timezone information for regional monitoring and reporting.

```sql
CREATE TABLE [dbo].[SchedulerExecutionByTimeZone](
    [Id] INT PRIMARY KEY IDENTITY(1, 1) NOT NULL,
    [ProjectName] NVARCHAR(50) NOT NULL,
    [SchedulerName] NVARCHAR(50) NOT NULL,
    [ExecutionTime] DATETIME NOT NULL DEFAULT (GETDATE()),
    [ExecutionTimeByTimeZone] DATETIME NOT NULL DEFAULT (GETDATE()),
    [TimeZone] NVARCHAR(50) NOT NULL,
    [UTCOffset] NVARCHAR(50)
)
```

| Column                 | Type           | Description                                      |
|------------------------|----------------|--------------------------------------------------|
| Id                     | INT, PK        | Primary key, auto-incrementing unique identifier |
| ProjectName            | NVARCHAR(50)   | Name of the project                              |
| SchedulerName          | NVARCHAR(50)   | Name of the scheduler                            |
| ExecutionTime          | DATETIME       | Time the scheduler was executed                  |
| ExecutionTimeByTimeZone| DATETIME       | Execution time in the specified timezone         |
| TimeZone               | NVARCHAR(50)   | Timezone of the execution                        |
| UTCOffset              | NVARCHAR(50)   | UTC offset for the timezone                      |

---

### SettleDailyStockRevenueFailLog
Logs failed attempts to settle daily stock revenue, including error codes and messages, for troubleshooting and audit.

```sql
CREATE TABLE [dbo].[SettleDailyStockRevenueFailLog](
    [ID] BIGINT IDENTITY(1,1),
    [WebId] INT NULL,
    [FromCustomerId] INT NULL,
    [ToCustomerId] INT NULL,
    [ToAmount] INT NULL,
    [ErrorCode] INT NULL,
    [ErrorMessage] NVARCHAR(1000) NULL,
    [CreatedOn] DATETIME NULL,
    CONSTRAINT [PK_SettleDailyStockRevenueFailLog] PRIMARY KEY ([ID])
)
```

| Column         | Type           | Description                                      |
|---------------|----------------|--------------------------------------------------|
| ID            | BIGINT, PK     | Primary key, auto-incrementing unique identifier |
| WebId         | INT            | Website identifier (nullable)                    |
| FromCustomerId| INT            | Source customer identifier (nullable)            |
| ToCustomerId  | INT            | Target customer identifier (nullable)            |
| ToAmount      | INT            | Amount to be settled (nullable)                  |
| ErrorCode     | INT            | Error code (nullable)                            |
| ErrorMessage  | NVARCHAR(1000) | Error message (nullable)                         |
| CreatedOn     | DATETIME       | Timestamp when the record was created (nullable) |

## Indexes
- `IX_BalanceHistory_WebId_Username_ModifiedDate`: For efficient balance history queries
- `IX_BalanceHistory_WebId_ModifiedDate`: For reporting and auditing
- `IX_PromotionCashUsedHistory_vPlayerBalance_CustomerId`: For promotion cash usage tracking

## Relationships
- All record tables are linked to website configuration through the `WebId` field
- Balance history tracks both cash and outstanding balance changes
- Daily statements track financial transactions and commissions
- Promotion cash history tracks both settled and used amounts
- Error tracking tables maintain relationships with transactions and game providers

## Common Fields
Most record tables share these common fields:
- `WebId`: Website identifier
- `CustomerId`: Customer identifier
- `Username`: Customer username
- `CreatedOn/ModifiedOn`: Audit timestamps
- `CreatedBy/ModifiedBy`: Audit user information

### TransferCashAgentPlayerFailedInfo
Logs failed attempts to transfer funds between cash agents and players, including error details, for troubleshooting and audit.

```sql
CREATE TABLE [dbo].[TransferCashAgentPlayerFailedInfo](
    [Id] INT IDENTITY(1,1) NOT NULL,
    [ErrorCode] INT NOT NULL,
    [PlayerId] INT NOT NULL,
    [FromCashAgentId] INT NOT NULL,
    [ToCashAgentId] INT NOT NULL,
    [FailedDB] NVARCHAR(100),
    [FailedTable] NVARCHAR(100),
    [FailedTime] DATETIME DEFAULT (GETDATE())
)
```

| Column         | Type           | Description                                      |
|---------------|----------------|--------------------------------------------------|
| Id            | INT, PK        | Primary key, auto-incrementing unique identifier |
| ErrorCode     | INT            | Error code for the failure                       |
| PlayerId      | INT            | Player identifier                                |
| FromCashAgentId| INT           | Source cash agent identifier                     |
| ToCashAgentId | INT            | Target cash agent identifier                     |
| FailedDB      | NVARCHAR(100)  | Database where the failure occurred              |
| FailedTable   | NVARCHAR(100)  | Table where the failure occurred                 |
| FailedTime    | DATETIME       | Timestamp when the failure occurred              |

## Indexes
- `IX_BalanceHistory_WebId_Username_ModifiedDate`: For efficient balance history queries
- `IX_BalanceHistory_WebId_ModifiedDate`: For reporting and auditing
- `IX_PromotionCashUsedHistory_vPlayerBalance_CustomerId`: For promotion cash usage tracking

## Relationships
- All record tables are linked to website configuration through the `WebId` field
- Balance history tracks both cash and outstanding balance changes
- Daily statements track financial transactions and commissions
- Promotion cash history tracks both settled and used amounts
- Error tracking tables maintain relationships with transactions and game providers

## Common Fields
Most record tables share these common fields:
- `WebId`: Website identifier
- `CustomerId`: Customer identifier
- `Username`: Customer username
- `CreatedOn/ModifiedOn`: Audit timestamps
- `CreatedBy/ModifiedBy`: Audit user information 