# Theme Schema Documentation

## Overview
This document describes the theme-related tables, their structures, and relationships. All tables are listed in alphabetical order, with detailed field descriptions and constraints.

## Core Tables

### CustomizeThemeProperties
Defines custom theme properties for specific HTML elements and themes.

```sql
CREATE TABLE [dbo].[CustomizeThemeProperties](
    [Id] INT IDENTITY (1, 1)  NOT NULL PRIMARY KEY, 
    [HtmlId] NVARCHAR(100) NOT NULL, 
    [ThemeId] INT NOT NULL, 
    [IsAppTheme] bit NOT NULL default 0,
    [Type] INT NULL, 
    [Path] NVARCHAR(MAX) NULL,
    [Height] NVARCHAR(10) NULL,
    [Width] NVARCHAR(10) NULL,
    [HyperLink] NVARCHAR(Max) NULL,
    [Text] NVARCHAR(Max) NULL,
    [ResourceKey] NVARCHAR(MAX) NULL,
    [IsHidden] bit NULL default 0,
    [Status] bit NOT NULL default 1,
    [WebId] INT NOT NULL,
    [DomainId] INT NOT NULL CONSTRAINT [DF_CustomizeThemeProperties_DomainId] DEFAULT ((0)),
    [Language] NVARCHAR (50) NOT NULL CONSTRAINT [DF_CustomizeThemeProperties_Language] DEFAULT 'en',
    [CreatedOn] DATETIME NULL,
    [CreatedBy] NVARCHAR (50) NULL,
    [ModifiedOn] DATETIME NULL,
    [ModifiedBy] NVARCHAR (50) NULL,
    [ImgCloudServiceUrl] NVARCHAR(Max) NULL,
    CONSTRAINT UQ_CustomizeThemeProperties UNIQUE ([HtmlId],[ThemeId], [DomainId], [WebId], [IsAppTheme], [Language])
)
```
| Column              | Type             | Description                                                                                   |
|---------------------|------------------|-----------------------------------------------------------------------------------------------|
| Id                  | INT, IDENTITY    | **Primary Key**. Auto-incremented unique identifier for the record.                           |
| HtmlId              | NVARCHAR(100)    | HTML element identifier. **Unique constraint** (with ThemeId, DomainId, WebId, IsAppTheme, Language). |
| ThemeId             | INT              | Theme identifier.                                                                             |
| IsAppTheme          | BIT              | Indicates if this is for app theme. Default: 0.                                               |
| Type                | INT, NULL        | Type of the property.                                                                         |
| Path                | NVARCHAR(MAX)    | Path to resource.                                                                             |
| Height              | NVARCHAR(10)     | Height value.                                                                                 |
| Width               | NVARCHAR(10)     | Width value.                                                                                  |
| HyperLink           | NVARCHAR(MAX)    | Hyperlink associated.                                                                         |
| Text                | NVARCHAR(MAX)    | Text content.                                                                                 |
| ResourceKey         | NVARCHAR(MAX)    | Resource key.                                                                                 |
| IsHidden            | BIT, NULL        | Indicates if hidden. Default: 0.                                                              |
| Status              | BIT              | Status flag. Default: 1.                                                                      |
| WebId               | INT              | Website identifier.                                                                           |
| DomainId            | INT              | Domain identifier. Default: 0.                                                                |
| Language            | NVARCHAR(50)     | Language code. Default: 'en'.                                                                 |
| CreatedOn           | DATETIME, NULL   | Creation timestamp.                                                                           |
| CreatedBy           | NVARCHAR(50),NULL| Creator.                                                                                      |
| ModifiedOn          | DATETIME, NULL   | Last modification timestamp.                                                                  |
| ModifiedBy          | NVARCHAR(50),NULL| Last modifier.                                                                                |
| ImgCloudServiceUrl  | NVARCHAR(MAX)    | Image cloud service URL.                                                                      |

**Constraints:**
- Unique: (HtmlId, ThemeId, DomainId, WebId, IsAppTheme, Language)
- Default values for DomainId (0), Language ('en'), IsAppTheme (0), IsHidden (0), Status (1)

---

### DefaultThemeProperties
Defines default theme properties for HTML elements and themes.

```sql
CREATE TABLE [dbo].[DefaultThemeProperties](
    [Id] INT IDENTITY (1, 1)  NOT NULL PRIMARY KEY, 
    [HtmlId] NVARCHAR(100) NULL, 
    [ThemeId] INT NOT NULL, 
    [Type] INT NULL, 
    [Path] NVARCHAR(MAX) NULL,
    [Height] NVARCHAR(10) NULL,
    [Width] NVARCHAR(10) NULL,
    [HyperLink] NVARCHAR(Max) NULL,
    [Text] NVARCHAR(max) NULL,
    [ResourceKey] NVARCHAR(50) NULL,
    [IsHidden] bit NULL default 0,
    [Status] bit NOT NULL default 1,
    [CreatedOn]  DATETIME      NULL,
    [CreatedBy]  NVARCHAR (50)  NULL,
    [ModifiedOn] DATETIME      NULL,
    [ModifiedBy] NVARCHAR (50)  NULL,
    [ImgCloudServiceUrl] NVARCHAR(Max) NULL
)
```
| Column             | Type             | Description                                                      |
|--------------------|------------------|------------------------------------------------------------------|
| Id                 | INT, IDENTITY    | **Primary Key**. Auto-incremented unique identifier.             |
| HtmlId             | NVARCHAR(100)    | HTML element identifier.                                         |
| ThemeId            | INT              | Theme identifier.                                                |
| Type               | INT, NULL        | Type of the property.                                            |
| Path               | NVARCHAR(MAX)    | Path to resource.                                                |
| Height             | NVARCHAR(10)     | Height value.                                                    |
| Width              | NVARCHAR(10)     | Width value.                                                     |
| HyperLink          | NVARCHAR(MAX)    | Hyperlink associated.                                            |
| Text               | NVARCHAR(MAX)    | Text content.                                                    |
| ResourceKey        | NVARCHAR(50)     | Resource key.                                                    |
| IsHidden           | BIT, NULL        | Indicates if hidden. Default: 0.                                 |
| Status             | BIT              | Status flag. Default: 1.                                         |
| CreatedOn          | DATETIME, NULL   | Creation timestamp.                                              |
| CreatedBy          | NVARCHAR(50),NULL| Creator.                                                         |
| ModifiedOn         | DATETIME, NULL   | Last modification timestamp.                                     |
| ModifiedBy         | NVARCHAR(50),NULL| Last modifier.                                                   |
| ImgCloudServiceUrl | NVARCHAR(MAX)    | Image cloud service URL.                                         |

**Constraints:**
- Default values for IsHidden (0), Status (1)

---

### LobbyBannerSetting
Defines lobby banner settings for domains and websites.

```sql
CREATE TABLE [dbo].[LobbyBannerSetting](
    [Id]            INT PRIMARY KEY IDENTITY(1, 1),
    [HtmlId]        NVARCHAR(500),
    [DomainId]      INT,
    [WebId]         INT,
    [IsDefault]     BIT,
    [IsMobileTheme] BIT,
    [Language]      NVARCHAR(5)   NOT NULL,
    [ImageName]     NVARCHAR(MAX) NOT NULL,
    [Hyperlink]     NVARCHAR(MAX),
    [StartDate]     DATETIME,
    [EndDate]       DATETIME,
    [IsForever]     BIT,
    [Status]        NVARCHAR(50),
    [DisplayOrder]  INT,
    [IsDefaultImage] BIT,
    [CreatedOn]     DATETIME,
    [ModifiedOn]    DATETIME,
    [ModifiedBy]    NVARCHAR(50),
    CONSTRAINT UQ_LobbyBannerSetting UNIQUE (HtmlId, DomainId, WebId, IsMobileTheme, Language),
    CONSTRAINT chk_status CHECK (Status IN ('Enabled', 'Disabled', 'Deleted'))
)
```
| Column          | Type             | Description                                                                                   |
|-----------------|------------------|-----------------------------------------------------------------------------------------------|
| Id              | INT, IDENTITY    | **Primary Key**. Auto-incremented unique identifier.                                          |
| HtmlId          | NVARCHAR(500)    | HTML element identifier. **Unique constraint** (with DomainId, WebId, IsMobileTheme, Language)|
| DomainId        | INT              | Domain identifier.                                                                            |
| WebId           | INT              | Website identifier.                                                                           |
| IsDefault       | BIT              | Indicates if this is the default banner.                                                      |
| IsMobileTheme   | BIT              | Indicates if this is for mobile theme.                                                        |
| Language        | NVARCHAR(5)      | Language code. Not null.                                                                      |
| ImageName       | NVARCHAR(MAX)    | Name of the image. Not null.                                                                  |
| Hyperlink       | NVARCHAR(MAX)    | Hyperlink associated.                                                                         |
| StartDate       | DATETIME         | Banner start date.                                                                            |
| EndDate         | DATETIME         | Banner end date.                                                                              |
| IsForever       | BIT              | Indicates if the banner is shown forever.                                                     |
| Status          | NVARCHAR(50)     | Status of the banner. Must be 'Enabled', 'Disabled', or 'Deleted'.                            |
| DisplayOrder    | INT              | Display order.                                                                                |
| IsDefaultImage  | BIT              | Indicates if this is the default image.                                                       |
| CreatedOn       | DATETIME         | Creation timestamp.                                                                           |
| ModifiedOn      | DATETIME         | Last modification timestamp.                                                                  |
| ModifiedBy      | NVARCHAR(50)     | Last modifier.                                                                                |

**Constraints:**
- Unique: (HtmlId, DomainId, WebId, IsMobileTheme, Language)
- Status must be one of: 'Enabled', 'Disabled', 'Deleted'

---

### ThemeOptions
Defines available theme options for each website.

```sql
CREATE TABLE [dbo].[ThemeOptions](
    [Id] INT IDENTITY (1, 1)  NOT NULL PRIMARY KEY, 
    [WebId] INT NOT NULL, 
    [ThemeId] INT NOT NULL, 
    [Status] INT NOT NULL default 1,
    [IsAppTheme] INT NOT NULL default 0,
    [CreatedOn]  DATETIME      NULL,
    [CreatedBy]  NVARCHAR (50)  NULL,
    [ModifiedOn] DATETIME      NULL,
    [ModifiedBy] NVARCHAR (50)  NULL
)
```
| Column      | Type             | Description                                                      |
|-------------|------------------|------------------------------------------------------------------|
| Id          | INT, IDENTITY    | **Primary Key**. Auto-incremented unique identifier.             |
| WebId       | INT              | Website identifier.                                              |
| ThemeId     | INT              | Theme identifier.                                                |
| Status      | INT              | Status flag. Default: 1.                                         |
| IsAppTheme  | INT              | Indicates if this is for app theme. Default: 0.                  |
| CreatedOn   | DATETIME, NULL   | Creation timestamp.                                              |
| CreatedBy   | NVARCHAR(50),NULL| Creator.                                                         |
| ModifiedOn  | DATETIME, NULL   | Last modification timestamp.                                     |
| ModifiedBy  | NVARCHAR(50),NULL| Last modifier.                                                   |

**Constraints:**
- Default values for Status (1), IsAppTheme (0)

---

### ThemeSelected
Defines which theme is selected for each website and domain.

```sql
CREATE TABLE [dbo].[ThemeSelected](
    [Id] INT IDENTITY (1, 1)  NOT NULL PRIMARY KEY, 
    [WebId] INT NOT NULL, 
    [DomainId] INT NOT NULL default 0,
    [ThemeId] INT NOT NULL, 
    [IsMobileTheme] BIT NOT NULL default 0,
    [IsAppTheme] BIT NOT NULL default 0,
    [CreatedOn]  DATETIME      NULL,
    [CreatedBy]  NVARCHAR (50)  NULL,
    [ModifiedOn] DATETIME      NULL,
    [ModifiedBy] NVARCHAR (50)  NULL
)
```
| Column         | Type             | Description                                                      |
|----------------|------------------|------------------------------------------------------------------|
| Id             | INT, IDENTITY    | **Primary Key**. Auto-incremented unique identifier.             |
| WebId          | INT              | Website identifier.                                              |
| DomainId       | INT              | Domain identifier. Default: 0.                                   |
| ThemeId        | INT              | Theme identifier.                                                |
| IsMobileTheme  | BIT              | Indicates if this is for mobile theme. Default: 0.               |
| IsAppTheme     | BIT              | Indicates if this is for app theme. Default: 0.                  |
| CreatedOn      | DATETIME, NULL   | Creation timestamp.                                              |
| CreatedBy      | NVARCHAR(50),NULL| Creator.                                                         |
| ModifiedOn     | DATETIME, NULL   | Last modification timestamp.                                     |
| ModifiedBy     | NVARCHAR(50),NULL| Last modifier.                                                   |

**Constraints:**
- Default values for DomainId (0), IsMobileTheme (0), IsAppTheme (0)

---

### Themes
Defines available themes.

```sql
CREATE TABLE [dbo].[Themes](
    [ID] INT IDENTITY (1, 1) NOT NULL PRIMARY KEY,
    [ThemeName] NVARCHAR(50) NOT NULL, 
    [IsMobileTheme] BIT NOT NULL default 0,
    [IsAppTheme] BIT NOT NULL default 0,
    [Status] INT NOT NULL default 1,
    [CreatedOn]  DATETIME      NULL,
    [CreatedBy]  NVARCHAR (50)  NULL,
    [ModifiedOn] DATETIME      NULL,
    [ModifiedBy] NVARCHAR (50)  NULL
)
```
| Column         | Type             | Description                                                      |
|----------------|------------------|------------------------------------------------------------------|
| ID             | INT, IDENTITY    | **Primary Key**. Auto-incremented unique identifier.             |
| ThemeName      | NVARCHAR(50)     | Name of the theme.                                               |
| IsMobileTheme  | BIT              | Indicates if this is for mobile theme. Default: 0.               |
| IsAppTheme     | BIT              | Indicates if this is for app theme. Default: 0.                  |
| Status         | INT              | Status flag. Default: 1.                                         |
| CreatedOn      | DATETIME, NULL   | Creation timestamp.                                              |
| CreatedBy      | NVARCHAR(50),NULL| Creator.                                                         |
| ModifiedOn     | DATETIME, NULL   | Last modification timestamp.                                     |
| ModifiedBy     | NVARCHAR(50),NULL| Last modifier.                                                   |

**Constraints:**
- Default values for IsMobileTheme (0), IsAppTheme (0), Status (1)

---

## Relationships
- Most tables reference `ThemeId` and/or `WebId` as logical foreign keys, but explicit foreign key constraints are not declared in the schema.
- `CustomizeThemeProperties` and `DefaultThemeProperties` are property tables for themes and HTML elements.
- `ThemeSelected` and `ThemeOptions` associate themes with websites and domains.
- `LobbyBannerSetting` is related to banners for specific domains, websites, and themes.

## Notes
- All tables and columns are documented in accordance with the monika-schema-rule.mdc.
- If you add or modify tables, update this documentation accordingly. 