# VIP Schema Documentation

## Overview
The VIP system in the Coloris Framework manages player VIP levels, benefits, performance tracking, and rewards. It provides a comprehensive system for VIP program management, level progression, and benefit distribution.

## Core VIP Tables

### Level
Defines the available VIP levels in the system.

```sql
CREATE TABLE Level (
    Id INT PRIMARY KEY IDENTITY(1,1),
    Name NVARCHAR(100)
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for the VIP level. Used as a foreign key in other tables. |
| Name | NVARCHAR(100) | Human-readable name for the level (e.g., "Gold", "Platinum"). |

### LevelSetting
Configures requirements and settings for each VIP level, including upgrade/maintenance thresholds and renewal rules.

```sql
CREATE TABLE LevelSetting (
    Id INT PRIMARY KEY IDENTITY(1,1),
    EventId INT,
    LevelId INT,
    MembershipRenewalDays INT,
    IsMembershipRenewalLiftTime BIT,
    UpgradeToThisLevelTurnover DECIMAL(19,6),
    UpgradeToThisLevelDeposit DECIMAL(19,6),
    MaintainedThisLevelTurnover DECIMAL(19,6),
    MaintainedThisLevelDeposit DECIMAL(19,6),
    Currency NVARCHAR(3),
    IsEnabled BIT,
    ModifiedOn DATETIME,
    ModifiedBy NVARCHAR(100),
    CreatedOn DATETIME,
    CreatedBy NVARCHAR(100)
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for this setting. |
| EventId | INT | Links to a specific VIP event (see `Event`). |
| LevelId | INT | The VIP level this setting applies to. |
| MembershipRenewalDays | INT | Number of days before membership renewal is required. |
| IsMembershipRenewalLiftTime | BIT | If true, membership is lifetime and does not require renewal. |
| UpgradeToThisLevelTurnover | DECIMAL(19,6) | Turnover required to upgrade to this level. |
| UpgradeToThisLevelDeposit | DECIMAL(19,6) | Deposit required to upgrade to this level. |
| MaintainedThisLevelTurnover | DECIMAL(19,6) | Turnover required to maintain this level. |
| MaintainedThisLevelDeposit | DECIMAL(19,6) | Deposit required to maintain this level. |
| Currency | NVARCHAR(3) | Currency code (e.g., USD, EUR). |
| IsEnabled | BIT | If false, this level is not available for upgrade/maintenance. |
| ModifiedOn | DATETIME | Last modification timestamp. |
| ModifiedBy | NVARCHAR(100) | Who last modified this setting. |
| CreatedOn | DATETIME | When this setting was created. |
| CreatedBy | NVARCHAR(100) | Who created this setting. |

### BenefitsSetting
Defines the benefits (such as commission rates or upgrade bonuses) for each level and event.

```sql
CREATE TABLE BenefitsSetting (
    Id INT PRIMARY KEY IDENTITY(1,1),
    EventId INT,
    LevelId INT,
    Type NVARCHAR(100),
    ProductType INT,
    Commission DECIMAL(19,6),
    Currency NVARCHAR(3),
    ModifiedOn DATETIME,
    ModifiedBy NVARCHAR(100),
    CreatedOn DATETIME,
    CreatedBy NVARCHAR(100),
    CONSTRAINT CK_BenefitsSetting_Type CHECK ([Type] IN ('UpgradeBonus', 'Commission'))
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for this benefit setting. |
| EventId | INT | Links to a specific VIP event. |
| LevelId | INT | The VIP level this benefit applies to. |
| Type | NVARCHAR(100) | 'UpgradeBonus' or 'Commission'. Indicates the type of benefit. |
| ProductType | INT | Product type (e.g., sports, casino, etc.). |
| Commission | DECIMAL(19,6) | Commission or bonus amount. |
| Currency | NVARCHAR(3) | Currency code. |
| ModifiedOn | DATETIME | Last modification timestamp. |
| ModifiedBy | NVARCHAR(100) | Who last modified this setting. |
| CreatedOn | DATETIME | When this setting was created. |
| CreatedBy | NVARCHAR(100) | Who created this setting. |

**Constraints:**
- `CK_BenefitsSetting_Type`: Only allows 'UpgradeBonus' or 'Commission' for `Type`.

### MemberPerformance
Tracks each member's performance for VIP progression, including deposits, turnover, and related settings.

```sql
CREATE TABLE MemberPerformance (
    Id INT PRIMARY KEY IDENTITY(1,1),
    Webid INT,
    CustomerId INT,
    LevelId INT,
    LevelSettingId INT,
    EventId INT,
    Deposit DECIMAL(19,6),
    Turnover DECIMAL(19,6),
    StartDate DATETIME,
    OperatorId INT NOT NULL DEFAULT 0
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for this record. |
| Webid | INT | Website identifier (multi-tenant support). |
| CustomerId | INT | Member's unique identifier. |
| LevelId | INT | Current VIP level. |
| LevelSettingId | INT | The level setting this performance is measured against. |
| EventId | INT | The event this performance is part of. |
| Deposit | DECIMAL(19,6) | Total deposit amount in the period. |
| Turnover | DECIMAL(19,6) | Total turnover in the period. |
| StartDate | DATETIME | When this performance period started. |
| OperatorId | INT | Operator who last updated this record (default 0 = system). |

### TransferHistory
Logs all VIP-related transfers, such as bonuses, commissions, or other benefit distributions.

```sql
CREATE TABLE TransferHistory (
    Id INT PRIMARY KEY IDENTITY(1,1),
    WebId INT,
    Type NVARCHAR(100),
    EventId INT,
    CustomerId INT,
    ParentId INT,
    BenefitsSettingId INT,
    LevelId INT,
    TransactionDate DATETIME,
    Amount DECIMAL(19,6),
    Interval NVARCHAR(50) NULL
)
```
| Column | Type | Description |
|--------|------|-------------|
| Id | INT, PK | Unique identifier for this transfer. |
| WebId | INT | Website identifier. |
| Type | NVARCHAR(100) | Type of transfer (e.g., bonus, commission). |
| EventId | INT | Related event. |
| CustomerId | INT | Member receiving the transfer. |
| ParentId | INT | Parent member (for hierarchy, if any). |
| BenefitsSettingId | INT | Which benefit setting this transfer is based on. |
| LevelId | INT | VIP level at the time of transfer. |
| TransactionDate | DATETIME | When the transfer occurred. |
| Amount | DECIMAL(19,6) | Amount transferred. |
| Interval | NVARCHAR(50) | Interval (e.g., monthly, weekly), if applicable. |

**Indexes:**
- `IX_TransferHistory_WebId_CustomerId_TransactionDate_EventId`: (WebId, CustomerId, TransactionDate, EventId; includes ParentId, Interval, Amount, Type, BenefitsSettingId, LevelId)

### MemberLevelChangeLog
Tracks all changes to a member's VIP level, including the source and target levels, operator, and event type for audit and progression analysis.

```sql
CREATE TABLE MemberLevelChangeLog (
    Id INT PRIMARY KEY IDENTITY(1,1),
    Webid INT,
    CustomerId INT,
    FromLevelId INT,
    ToLevelId INT,
    CreatedOn DATETIME,
    CreatedBy NVARCHAR(100),
    OperatorId INT NOT NULL DEFAULT 0,
    EventType NVARCHAR(100) NULL
)
```
| Column      | Type           | Description                                      |
|-------------|----------------|--------------------------------------------------|
| Id          | INT, PK        | Unique identifier for this log entry. |
| Webid       | INT            | Website identifier. |
| CustomerId  | INT            | Member whose level changed. |
| FromLevelId | INT            | Previous VIP level. |
| ToLevelId   | INT            | New VIP level. |
| CreatedOn   | DATETIME       | When the change occurred. |
| CreatedBy   | NVARCHAR(100)  | Who/what triggered the change. |
| OperatorId  | INT            | Operator involved (0 = system). |
| EventType   | NVARCHAR(100)  | Type of event (e.g., upgrade, downgrade, manual adjustment). |

### BenefitsSettingChangeLog
Logs all changes to VIP benefit settings, including type, product, commission, and audit information for compliance and rollback.

```sql
CREATE TABLE BenefitsSettingChangeLog (
    Id INT,
    EventId INT,
    LevelId INT,
    Type NVARCHAR(100),
    ProductType INT,
    Commission DECIMAL(19,6),
    ModifiedOn DATETIME,
    ModifiedBy NVARCHAR(100),
    CreatedOn DATETIME,
    CreatedBy NVARCHAR(100),
    CONSTRAINT CK_BenefitsSettingChangeLog_Type CHECK ([Type] IN ('UpgradeBonus', 'Commission'))
)
```
| Column      | Type           | Description                                      |
|-------------|----------------|--------------------------------------------------|
| Id          | INT            | Identifier (not PK, see usage). |
| EventId     | INT            | Event identifier. |
| LevelId     | INT            | VIP level identifier. |
| Type        | NVARCHAR(100)  | Benefit type ('UpgradeBonus' or 'Commission'). |
| ProductType | INT            | Product type. |
| Commission  | DECIMAL(19,6)  | Commission value. |
| ModifiedOn  | DATETIME       | Timestamp of last modification. |
| ModifiedBy  | NVARCHAR(100)  | User who last modified. |
| CreatedOn   | DATETIME       | Timestamp of creation. |
| CreatedBy   | NVARCHAR(100)  | User who created the record. |

**Constraints:**
- `CK_BenefitsSettingChangeLog_Type`: Only allows 'UpgradeBonus' or 'Commission' for `Type`.

### Event
Defines a VIP event, including currency, rebate settings, and audit fields. Used to group VIP activities and settings.

```sql
CREATE TABLE Event (
    Id INT PRIMARY KEY IDENTITY(1,1),
    WebId INT,
    Currency NVARCHAR(3),
    IsEnabled BIT,
    RebateCalculation INT,
    RebateInterval NVARCHAR(100),
    IsWithdrawalLimitEnabled BIT,
    ModifiedOn DATETIME,
    ModifiedBy NVARCHAR(100),
    CreatedOn DATETIME,
    CreatedBy NVARCHAR(100),
    CONSTRAINT CK_Event_RebateInterval CHECK ([RebateInterval] IN ('Daily', 'Weekly', 'Monthly'))
)
```
| Column                   | Type           | Description                                      |
|--------------------------|----------------|--------------------------------------------------|
| Id                       | INT, PK        | Unique identifier for the event. |
| WebId                    | INT            | Website identifier. |
| Currency                 | NVARCHAR(3)    | Currency code. |
| IsEnabled                | BIT            | Whether the event is active. |
| RebateCalculation        | INT            | Method for calculating rebates. |
| RebateInterval           | NVARCHAR(100)  | Interval for rebates ('Daily', 'Weekly', 'Monthly'). |
| IsWithdrawalLimitEnabled | BIT            | Whether withdrawal limits are enforced. |
| ModifiedOn               | DATETIME       | Last modification timestamp. |
| ModifiedBy               | NVARCHAR(100)  | Who last modified. |
| CreatedOn                | DATETIME       | Creation timestamp. |
| CreatedBy                | NVARCHAR(100)  | Who created the event. |

**Constraints:**
- `CK_Event_RebateInterval`: Only allows 'Daily', 'Weekly', or 'Monthly' for `RebateInterval`.

## Localization Tables

### LevelLanguage
Stores translations for VIP level names for multilingual support.

```sql
CREATE TABLE LevelLanguage (
    Id INT PRIMARY KEY IDENTITY(1,1),
    EventId INT,
    LevelId INT,
    Name NVARCHAR(100),
    Lang NVARCHAR(100)
)
```
| Column   | Type           | Description |
|----------|----------------|-------------|
| Id       | INT, PK        | Unique identifier for this translation. |
| EventId  | INT            | Event reference. |
| LevelId  | INT            | VIP level reference. |
| Name     | NVARCHAR(100)  | Translated level name. |
| Lang     | NVARCHAR(100)  | Language code (e.g., 'en', 'zh'). |

### EventLanguage
Stores translations for event titles and content for multilingual support.

```sql
CREATE TABLE EventLanguage (
    Id INT PRIMARY KEY IDENTITY(1,1),
    EventId INT,
    Lang NVARCHAR(100),
    Title NVARCHAR(MAX),
    Content NVARCHAR(MAX),
    ModifiedOn DATETIME,
    ModifiedBy NVARCHAR(100),
    CreatedOn DATETIME,
    CreatedBy NVARCHAR(100)
)
```
| Column     | Type           | Description |
|------------|----------------|-------------|
| Id         | INT, PK        | Unique identifier for this translation. |
| EventId    | INT            | Event reference. |
| Lang       | NVARCHAR(100)  | Language code. |
| Title      | NVARCHAR(MAX)  | Translated event title. |
| Content    | NVARCHAR(MAX)  | Translated event content. |
| ModifiedOn | DATETIME       | Last modification timestamp. |
| ModifiedBy | NVARCHAR(100)  | Last modifier. |
| CreatedOn  | DATETIME       | Creation timestamp. |
| CreatedBy  | NVARCHAR(100)  | Creator. |

## Key Features

1. **VIP Level Management**
   - Multiple VIP levels
   - Level progression rules
   - Membership renewal settings
   - Level maintenance requirements
   - Manual and automatic upgrades

2. **Benefits System**
   - Upgrade bonuses
   - Commission rates
   - Product-specific benefits
   - Currency-based settings
   - Benefit distribution tracking

3. **Performance Tracking**
   - Deposit monitoring
   - Turnover tracking
   - Level progression
   - Historical performance
   - Event-based tracking

4. **Multilingual Support**
   - Level name translations
   - Event content localization
   - Multiple language support
   - Dynamic content management

## Best Practices

1. **Level Configuration**
   - Set appropriate upgrade requirements
   - Configure maintenance thresholds
   - Enable/disable levels as needed
   - Monitor level distribution
   - Regular requirement reviews

2. **Benefit Management**
   - Regular commission updates
   - Bonus amount reviews
   - Distribution monitoring
   - Compliance checks
   - Performance analysis

3. **Member Management**
   - Regular performance reviews
   - Level adjustment monitoring
   - Benefit distribution tracking
   - Communication management
   - Status updates

4. **System Maintenance**
   - Regular data cleanup
   - Performance optimization
   - Setting updates
   - Audit trail maintenance
   - Security reviews

## Relationships
- VIP levels are linked to events through `EventId`
- Members are tracked through `CustomerId` and `WebId`
- Benefits are associated with levels through `LevelId`
- Performance tracking through `MemberPerformance`
- History maintained through `TransferHistory`

**Indexes:**
- `IX_TransferHistory_WebId_CustomerId_TransactionDate_EventId`: (WebId, CustomerId, TransactionDate, EventId; includes ParentId, Interval, Amount, Type, BenefitsSettingId, LevelId) 