# Promodia Architecture Structure Documentation

## 📋 Overview
This document outlines the architecture and relationships between the Promodia ecosystem components, specifically showing how **Proxydia** (gRPC Client), **Promodia** (gRPC Server), and **Monika.sln** (Database Layer) work together to provide promotion and VIP services.

## 🏗️ Architecture Overview

```
┌─────────────────┐    HTTP/REST API     ┌─────────────────┐    gRPC Calls    ┌─────────────────┐
│                 │ ◄─────────────────── │                 │ ◄──────────────► │                 │
│   Frontend      │                      │   Proxydia      │                  │   Promodia      │
│   (Kirby/Tycho) │                      │   (API Gateway) │                  │   (Business     │
│                 │ ──────────────────► │                 │ ──────────────► │    Logic)       │
└─────────────────┘                      └─────────────────┘                  └─────────────────┘
                                                   │                                     │
                                                   │                                     │
                                                   ▼                                     ▼
                                         ┌─────────────────┐                  ┌─────────────────┐
                                         │   Proxydia      │                  │   Promodia      │
                                         │   Models &      │                  │   Repositories  │
                                         │   DTOs          │                  │   & Services    │
                                         └─────────────────┘                  └─────────────────┘
                                                                                        │
                                                                                        │
                                                                                        ▼
                                                                              ┌─────────────────┐
                                                                              │   Monika.sln    │
                                                                              │   (Database     │
                                                                              │    Layer)       │
                                                                              └─────────────────┘
```

## 🔗 Component Relationships

### 1. **Proxydia** - API Gateway & gRPC Client
**Location**: `/proxydia/Proxydia/`

**Role**: Acts as a **gRPC Client** and API gateway that:
- Exposes HTTP/REST APIs to frontend applications
- Translates HTTP requests to gRPC calls to Promodia
- Handles authentication, validation, and request/response formatting
- Provides a clean abstraction layer for frontend consumption

**Key Components**:
```
├── Controllers/
│   ├── PromotionController.cs          # HTTP API endpoints for promotions
│   ├── VipController.cs                # HTTP API endpoints for VIP services
│   └── ReferralController.cs           # HTTP API endpoints for referrals
├── Services/
│   └── Promotion/
│       ├── PromotionService.cs         # gRPC client service implementation
│       └── IPromotionService.cs        # Service interface
├── Models/                             # DTOs and request/response models
├── Protos/                             # gRPC client definitions
│   ├── model.proto                     # Shared data models
│   ├── promotion.proto                 # Promotion service contract
│   └── vip.proto                       # VIP service contract
└── Proxydia.csproj                     # gRPC Client configuration
```

**gRPC Configuration**: `<GrpcServices>Client</GrpcServices>`

### 2. **Promodia** - Core Business Logic & gRPC Server
**Location**: `/promodia/Promodia/`

**Role**: Acts as a **gRPC Server** that:
- Implements core promotion and VIP business logic
- Handles complex promotion calculations and rules
- Manages promotion workflows and state transitions
- Directly interfaces with the database layer
- Provides caching and performance optimizations

**Key Components**:
```
├── Services/
│   ├── PromotionService.cs             # Core promotion business logic (85KB, 1862 lines)
│   └── VipService.cs                   # VIP management logic
├── Repositories/                       # Data access layer
├── Strategy/                           # Strategy pattern implementations
├── Factory/                            # Factory patterns for promotion types
├── Cache/                              # Caching implementations
├── Enums/                              # Business enumerations
├── Exceptions/                         # Custom exception handling
├── Helper/                             # Utility and helper classes
├── Interceptor/                        # gRPC interceptors
├── Model/                              # Business models and entities
├── Protos/                             # gRPC server definitions
│   ├── model.proto                     # Shared data models
│   ├── promotion.proto                 # Promotion service contract
│   └── vip.proto                       # VIP service contract
└── Promodia.csproj                     # gRPC Server configuration
```

**gRPC Configuration**: `<GrpcServices>Server</GrpcServices>`

### 3. **Monika.sln** - Database Layer
**Location**: `/monika/Monika/`

**Role**: Provides the **database infrastructure** with:
- Separation of concerns through multiple database projects
- Structured schema management
- Stored procedures with versioning
- Data integrity and relationships

**Database Projects**:
```
├── Promotion/                          # Promotion-specific database
│   ├── dbo/
│   │   ├── Tables/
│   │   │   ├── CompanyPromotion.sql            # Main promotion table
│   │   │   ├── PromotionRequest.sql            # Promotion application requests
│   │   │   ├── CompanyPromotionSetting.sql     # Promotion configuration
│   │   │   ├── PromotionType.sql               # Promotion type definitions
│   │   │   ├── LuckyWheelTicket.sql           # Lucky wheel functionality
│   │   │   ├── CouponCode.sql                  # Coupon management
│   │   │   ├── ReferralMember.sql              # Referral system
│   │   │   └── DepositPromotionOption.sql      # Deposit promotion options
│   │   ├── Stored Procedures/                  # Business logic procedures
│   │   ├── Functions/                          # Database functions
│   │   └── User Defined Types/                 # Custom data types
│   └── Promotion.sqlproj
├── Account/                            # Account & Payment database
│   ├── dbo/
│   │   ├── Tables/
│   │   │   ├── PaymentGatewayCompanyBankSettings.sql  # Payment account settings
│   │   │   ├── Company.sql                     # Company information
│   │   │   ├── AllDomain.sql                   # Domain management
│   │   │   ├── PaymentProviders.sql            # Payment provider configuration
│   │   │   └── CompanyPaymentProviders.sql     # Company-provider relationships
│   │   └── Stored Procedures/
│   └── Account.sqlproj
├── VIP/                                # VIP-specific database
├── Main/                               # Core system database
├── ApplicationSetting/                 # Configuration database
├── Theme/                              # Theme and UI database
├── GameProvider/                       # Game provider database
├── Payment/                            # Payment processing database
├── Record/                             # Transaction records database
├── History/                            # Historical data database
└── Other specialized databases...
```

## 🔄 Data Flow & Communication

### 1. **Promotion Creation Flow**
```
Frontend Request → Proxydia HTTP API → Proxydia gRPC Client → 
Promodia gRPC Server → Promodia Repository → Monika Database
```

### 2. **Domain Filtering Implementation Points**
Based on the acceptance criteria, the domain filtering feature will touch:

**Proxydia Layer**:
- HTTP request validation
- DTO transformation
- gRPC call preparation

**Promodia Layer**:
- Business logic validation
- Domain-specific filtering
- Database query construction

**Monika Layer**:
- Schema modifications (JSON columns)
- Stored procedure updates
- Data persistence

### 3. **gRPC Communication Contract**
Both Proxydia and Promodia share identical `.proto` files:
- `model.proto` - Common data structures
- `promotion.proto` - Promotion service operations
- `vip.proto` - VIP service operations

## 📊 Database Schema Relationships

### Promotion Domain Tables
```sql
-- Core Promotion Tables
CompanyPromotion              # Main promotion definitions
├── CompanyPromotionSetting   # Promotion configuration settings
├── CompanyPromotionLanguage  # Multi-language support
├── PromotionRequest          # Player promotion applications
├── PromotionType            # Available promotion types
└── PromotionTypeSetting     # Type-specific settings

-- Supporting Tables
DepositPromotionOption        # Deposit promotion configurations
├── DepositPromotionOptionAmount  # Amount-based options
LuckyWheelTicket             # Lucky wheel functionality
ReferralMember               # Referral system
CouponCode                   # Coupon management
```

### Account & Payment Domain Tables
```sql
-- Payment Account Tables
PaymentGatewayCompanyBankSettings  # Company payment account settings
├── PaymentProviders              # Available payment providers
├── CompanyPaymentProviders       # Company-provider relationships
└── PaymentMethodDisplayOrder     # UI display configuration

-- Domain Management
AllDomain                    # Domain definitions and configuration
Company                      # Company/brand information
```

## 🛠️ Technology Stack

### Proxydia (gRPC Client)
- **.NET 7.0** - Runtime framework
- **ASP.NET Core** - Web API framework
- **gRPC Client** - Service communication
- **Core.Api** - Shared API library
- **Newtonsoft.Json** - JSON serialization

### Promodia (gRPC Server)
- **.NET 7.0** - Runtime framework
- **gRPC Server** - Service hosting
- **Hangfire** - Background job processing
- **FluentValidation** - Input validation
- **Core.Api** - Shared API library
- **NLog + ElasticSearch** - Logging infrastructure

### Monika (Database)
- **SQL Server** - Database engine
- **SQLPROJ** - Database project management
- **Stored Procedures** - Business logic encapsulation
- **Version Control** - Schema versioning

## 🔐 Security & Configuration

### Environment Configuration
Both services support multiple environments:
- **Local/Development**
- **Staging**
- **Demo**
- **Pre-production**
- **Production**
- **Regional variants** (SA, LATAM)

### Logging & Monitoring
- **NLog** configuration per environment
- **ElasticSearch** integration for log aggregation
- **Environment-specific** log levels and targets

## 📋 Implementation Guidelines for Domain Filtering

### 1. Database Layer (Monika)
- Add `Domains` JSON column to relevant tables
- Update stored procedures with versioning
- Implement backward compatibility for legacy data

### 2. Business Logic Layer (Promodia)
- Implement domain filtering in repository methods
- Add domain validation in service layers
- Update gRPC service implementations

### 3. API Layer (Proxydia)
- Add domain selection to HTTP endpoints
- Implement request/response DTOs
- Handle domain validation and transformation

### 4. Frontend Integration
- Domain dropdown components
- Validation rules for domain selection
- API integration for domain management

## 📄 Version Control & Deployment

### Database Versioning
- **Stored Procedures**: Follow `ProjectName_FeatureName_Version` pattern
- **Schema Changes**: Coordinate across service deployments
- **Migration Scripts**: Include in deployment pipelines

### Service Deployment
- **Container-based** deployment (Docker)
- **Environment-specific** configurations
- **gRPC compatibility** maintained across versions

## Proto Contract Synchronization Policy

Whenever you add, remove, or modify any field in `model.proto` (e.g., adding a parameter like `Domain` for promotion filtering), you **must**:
- Synchronize the change in both Promodia and Proxydia projects.
- Record the field name, type, and field number in this documentation.
- Ensure both sides (server and client) are regenerated and tested with the updated proto.

**Reviewer Checklist:**
- [ ] Have all relevant proto/model.proto changes been applied to both Promodia and Proxydia?
- [ ] Is the change documented in this section (field name, type, number, and context)?
- [ ] Has the proto been recompiled and tested on both sides?

> This ensures gRPC contract consistency and prevents integration issues between backend services and the API Gateway.

---

**Last Updated**: 2025-01-21  
**Version**: 1.0.0  
**Maintainer**: Development Team 