# Promodia Technical Detail Documentation

## 📋 Overview
**Promodia** is a **gRPC Server** microservice that provides the core business logic for promotion, VIP, and referral management systems. It serves as the central hub for complex promotion calculations, validation, and workflow management in the platform ecosystem.

**Technology Stack**: .NET 7.0, gRPC Server, Entity Framework, Hangfire, FluentValidation, Protocol Buffers, NLog, Elastic APM, Docker-ready

---

## 🗂️ Project Structure Overview

```
Promodia/
├── Services/         # gRPC service implementations (PromotionService, VipService)
├── Repositories/     # Data access and repository pattern
├── Protos/           # Protocol Buffers (gRPC service and message definitions)
├── Model/            # Data models, DTOs, and request/response objects
├── Factory/          # Factory classes for promotion and setting logic
├── Helper/           # Utility/helper classes (validation, math, reflection, etc.)
├── Enums/            # Enum definitions for business logic
├── Extension/        # Extension methods for C# types
├── Interceptor/      # gRPC interceptors (logging, exception handling)
├── Exceptions/       # Custom exception classes
├── Cache/            # Caching logic and interfaces
├── Strategy/         # Strategy pattern implementations for promotion/referral logic
├── Filters/          # Custom filters (e.g., authorization)
├── Properties/       # Project properties (launchSettings.json)
├── wwwroot/          # Static files/assets (e.g., CSS, SVG)
├── Program.cs        # Application entry point and DI setup
├── Promodia.csproj   # Project file and dependencies
├── appsettings*.json # Configuration files for different environments
```

### Folder/Module Purposes
- **Services/**: Main business logic, gRPC service implementations
- **Repositories/**: Data access, database queries, repository interfaces/implementations
- **Protos/**: `.proto` files for gRPC service and message contracts
- **Model/**: DTOs, request/response objects, and business models (organized by feature)
- **Factory/**: Factories for creating promotion types/settings
- **Helper/**: Utility classes for validation, math, reflection, etc.
- **Enums/**: Centralized enums for business rules and status codes
- **Extension/**: Extension methods for C# types and business logic
- **Interceptor/**: gRPC interceptors for logging and exception handling
- **Exceptions/**: Custom exception types for domain-specific error handling
- **Cache/**: Caching logic and interfaces for performance
- **Strategy/**: Strategy pattern implementations for extensible business rules
- **Filters/**: Custom filters (e.g., authorization)
- **Properties/**: Project-level settings (launchSettings.json)
- **wwwroot/**: Static files/assets (CSS, SVG, etc.)

---

## 🏗️ Architecture Design

### Service Architecture Pattern
```
┌─────────────────────────────────────────────────────────────────┐
│                        Promodia gRPC Server                    │
├─────────────────────────────────────────────────────────────────┤
│  Controllers Layer (gRPC Service Implementations)              │
│  ├── PromotionService.cs (85KB, 1862 lines)                   │
│  └── VipService.cs (10KB, 218 lines)                          │
├─────────────────────────────────────────────────────────────────┤
│  Business Logic Layer                                          │
│  ├── Strategy/ (Strategy Pattern Implementations)              │
│  ├── Factory/ (Factory Pattern for Promotion Types)           │
│  ├── Cache/ (Performance Optimization)                         │
│  └── Interceptor/ (Cross-cutting Concerns)                     │
├─────────────────────────────────────────────────────────────────┤
│  Data Access Layer                                             │
│  ├── Repositories/ (Data Access Abstraction)                   │
│  │   ├── PromotionRepository.cs (46KB, 1244 lines)            │
│  │   ├── VipRepository.cs (6.3KB, 173 lines)                  │
│  │   └── Supporting Repositories                              │
│  └── Models/ (Domain Models and DTOs)                          │
├─────────────────────────────────────────────────────────────────┤
│  Infrastructure Layer                                          │
│  ├── Exceptions/ (Custom Exception Handling)                   │
│  ├── Helper/ (Utility Functions)                              │
│  ├── Enums/ (Business Enumerations)                           │
│  └── Extension/ (Extension Methods)                            │
└─────────────────────────────────────────────────────────────────┘
                                │
                                ▼
┌─────────────────────────────────────────────────────────────────┐
│                    Monika Database Layer                       │
│  ├── Promotion Database (Promotion-specific tables)            │
│  ├── Account Database (Payment & Domain tables)                │
│  ├── VIP Database (VIP-related tables)                        │
│  └── Supporting Databases (Main, Payment, Record, etc.)        │
└─────────────────────────────────────────────────────────────────┘
```

## 🚀 Core Components

### 1. **gRPC Service Layer**
**Location**: `/Services/`

#### PromotionService.cs (85KB, 1862 lines)
**Purpose**: Implements the core promotion business logic via gRPC

**Key Responsibilities**:
- **Promotion Lifecycle Management**: Create, update, approve, reject promotions
- **Complex Business Rules**: Multi-layered validation and calculation logic
- **Lucky Wheel System**: Spin mechanics, ticket management, reward distribution
- **Coupon Management**: Validation, redemption, and usage tracking
- **Auto-Apply Logic**: Automatic promotion application for eligible users
- **Domain Filtering**: (To be implemented) Domain-specific promotion visibility

**Major Methods**:
```csharp
// Core Promotion Operations
public override Task<ProtoCreateCompanyPromotionResponse> CreateCompanyPromotion()
public override Task<ProtoGeneralResponse> ApplyPromotion()
public override Task<ProtoGeneralResponse> ProceedPromotion()

// Lucky Wheel Operations  
public override Task<ProtoPromodiaGetLuckyWheelSpinResultResponse> GetLuckyWheelSpinResult()
public override Task<ProtoGetAvailableLuckyWheelTicketsResponse> GetAvailableLuckyWheelTickets()

// Coupon Operations
public override Task<ProtoCouponRedeemPromotionResponse> CouponRedeemPromotion()
public override Task<ProtoCheckAvailableCouponPromotionResponse> CheckAvailableCouponPromotion()

// Batch Operations
public override Task<ProtoGeneralResponse> UpdateWaitingPromotionByBatch()
```

#### VipService.cs (10KB, 218 lines)
**Purpose**: Handles VIP member management and tier progression

### 2. **Repository Layer**
**Location**: `/Repositories/`

#### PromotionRepository.cs (46KB, 1244 lines)
**Purpose**: Provides data access abstraction for promotion-related operations

**Key Features**:
- **Complex Query Building**: Dynamic SQL generation based on business requirements
- **Transaction Management**: Ensures data consistency across multiple operations
- **Performance Optimization**: Caching strategies and optimized queries
- **Stored Procedure Integration**: Direct calls to versioned stored procedures

**Critical Methods**:
```csharp
// Core Data Operations
public DbResult CreateCompanyPromotion(ProtoCompanyPromotion request, List<LuckyWheelOptionDto> luckyWheelOptions, List<CouponCodeDto> couponCodes)
public GetCompanyAutoApplyPromotionDto GetCompanyAutoApplyRegisterPromotion(int webId, string currency)
public (IEnumerable<CompanyPromotionDto>, IEnumerable<CompanyPromotionContentDto>) GetCompanyPromotionList(GetCompanyPromotionListRequest request)

// Lucky Wheel Operations
public List<GetAvailableLuckyWheelTicketsDto> GetAvailableLuckyWheelTickets(GetAvailableLuckyWheelTicketsRequest request)
public DbResult UpdateLuckyWheelTicketQuantity(UpdateLuckyWheelTicketQuantity request)

// Domain-Specific Operations (To be implemented)
public List<CompanyPromotionDto> GetPromotionsByDomains(int webId, List<string> domains)
```

### 3. **Business Logic Patterns**

#### Strategy Pattern (`/Strategy/`)
**Purpose**: Handles different promotion types with specialized logic
```csharp
public interface IPromotionTypeStrategy
{
    void CreatePromotion(ProtoCompanyPromotion request);
    DbResult UpdateCompanyPromotion(ProtoUpdateCompanyPromotionRequest request);
    bool PlayerCanSeeInPlayerSite(PlayerInfoDto player);
}

// Implementations:
- FirstDepositBonusStrategy
- LuckyWheelBonusStrategy  
- CouponRewardStrategy
- RebatePromotionStrategy
```

#### Factory Pattern (`/Factory/`)
**Purpose**: Creates appropriate promotion type instances
```csharp
public class PromotionTypeFactory
{
    public IPromotionTypeStrategy GetPromotionTypeStrategy(int promotionTypeId)
    public ICompanyPromotionInterface GetCompanyPromotion(int companyPromotionId)
}
```

### 4. **Data Models**
**Location**: `/Model/`

#### Core Entities:
```csharp
// Promotion Models
public class CompanyPromotionDto
{
    public int Id { get; set; }
    public string Title { get; set; }
    public List<string> Domains { get; set; } // Domain filtering
    public List<CompanyPromotionSettingDto> Settings { get; set; }
    public EnumPromotionStatus Status { get; set; }
}

// Request/Response Models
public class ApplyPromotionRequest
{
    public int CompanyPromotionId { get; set; }
    public int CustomerId { get; set; }
    public int WebId { get; set; }
    public string Username { get; set; }
    public List<string> ApplicableDomains { get; set; } // Domain context
}
```

## 🔄 Integration Patterns

### 1. **gRPC Server Configuration**
**File**: `Promodia.csproj`
```xml
<Protobuf Include="Protos/promotion.proto">
  <GrpcServices>Server</GrpcServices>
  <Access>Public</Access>
</Protobuf>
```

### 2. **Client Communication Flow**
```
1. Coloris.Core.PromodiaService (HTTP Client)
   ↓ HTTP/REST calls
2. Proxydia.PromotionController (API Gateway)
   ↓ gRPC client calls  
3. Promodia.PromotionService (gRPC Server)
   ↓ Repository calls
4. Monika.Promotion Database
```

### 3. **Dependency Injection Setup**
**File**: `Program.cs`
```csharp
// gRPC Services
builder.Services.AddGrpc();
builder.Services.AddGrpcReflection();

// Business Services  
builder.Services.AddScoped<IPromotionRepository, PromotionRepository>();
builder.Services.AddScoped<PromotionTypeFactory>();

// Background Jobs
builder.Services.AddHangfire(configuration => configuration.UseSqlServerStorage(connectionString));
```

## 💾 Database Integration

### 1. **Connection Management**
- **Multiple Database Support**: Promotion, Account, VIP, Main databases
- **Connection String Strategy**: Environment-specific configurations
- **Transaction Scope**: Distributed transaction support across databases

### 2. **Stored Procedure Naming Convention**
```sql
-- Pattern: [ProjectName_FeatureName_Version]
Promodia_CreateCompanyPromotion_2.0.0
Promodia_GetPromotionsByDomains_1.0.0
Promodia_ApplyPromotionWithValidation_1.5.1
```

### 3. **Domain Filtering Implementation Points**
```csharp
// Repository Method for Domain Filtering
public List<CompanyPromotionDto> GetPromotionsByDomains(int webId, List<string> domains)
{
    return GetData<CompanyPromotionDto>("Promodia_GetPromotionsByDomains", new
    {
        webId,
        domains = JsonConvert.SerializeObject(domains)
    }).ToList();
}
```

## 🎯 Performance Optimization

### 1. **Caching Strategy** (`/Cache/`)
```csharp
public interface IPromotionCache
{
    T Get<T>(string key);
    void Set<T>(string key, T value, TimeSpan expiration);
    void Remove(string key);
    void Reload(); // Force cache refresh
}
```

### 2. **Background Processing** (Hangfire)
```csharp
// Auto Rebate Processing
public override Task<ProtoGeneralResponse> AutoApplyRebate(ProtoTimezoneRequest request)
```

## 🧩 Notable Patterns & Practices

- **gRPC-first architecture**: All business logic is exposed via gRPC services, defined in Protos and implemented in Services
- **Strong separation of concerns**: Data access, business logic, and transport layers are clearly separated
- **Extensibility**: Factory and strategy patterns allow for easy addition of new promotion/referral types
- **Observability**: Integrated logging (NLog), distributed tracing (Elastic APM), and Swagger for HTTP endpoints
- **Environment-based configuration**: Multiple appsettings files for different deployment environments

## 📚 References
- [Services-PromotionService.md](./Services-PromotionService.md): Full list of PromotionService endpoints
- [Services-VipService.md](./Services-VipService.md): Full list of VipService endpoints

---

**Last Updated**: 2025-01-21  
**Version**: 1.0.0  
**Maintainer**: Development Team 