"""
Analysis Agent - Error detection, pattern matching, and root cause analysis.
"""

import re
from typing import Dict, Any, List, Optional
from dataclasses import dataclass
from pathlib import Path

import sys
sys.path.insert(0, str(Path(__file__).parent.parent))

from .base import BaseAgent
from swarm.bus import MessageBus
from swarm.store import ResultStore
from swarm.context import QueryResult, AnalysisResult

# Error patterns for detection
ERROR_PATTERNS = {
    "SameFP": {
        "pattern": r"SameFP|Same.*Fingerprint|fingerprint.*conflict",
        "cause": "Fingerprint conflict - device already used",
        "solution": "Device fingerprint was previously used by another account to claim this promotion"
    },
    "SameIP": {
        "pattern": r"SameIP|Same.*IP|IP.*conflict",
        "cause": "IP address conflict",
        "solution": "Same IP address was used by another account"
    },
    "MaxClaim": {
        "pattern": r"MaxClaim|maximum.*claim|claim.*limit",
        "cause": "Maximum claim limit reached",
        "solution": "Player has reached the maximum number of claims for this promotion"
    },
    "Expired": {
        "pattern": r"Expired|promotion.*ended|no longer.*available",
        "cause": "Promotion expired",
        "solution": "The promotion period has ended"
    },
    "NotEligible": {
        "pattern": r"NotEligible|not.*eligible|ineligible",
        "cause": "Player not eligible",
        "solution": "Player does not meet the eligibility criteria"
    },
    "InsufficientDeposit": {
        "pattern": r"insufficient.*deposit|minimum.*deposit|deposit.*required",
        "cause": "Deposit requirement not met",
        "solution": "Player has not met the minimum deposit requirement"
    },
    "TurnoverPending": {
        "pattern": r"turnover|wagering.*requirement|rollover",
        "cause": "Turnover requirement pending",
        "solution": "Player must complete wagering requirements before withdrawal"
    },
    "DuplicateRequest": {
        "pattern": r"duplicate|already.*applied|pending.*request",
        "cause": "Duplicate request",
        "solution": "A request for this promotion is already pending"
    },
    "VIPLevelMismatch": {
        "pattern": r"VIP.*level|level.*mismatch|upgrade.*pending",
        "cause": "VIP level mismatch",
        "solution": "Player's VIP level does not match expected level for this benefit"
    },
    "PaymentFailed": {
        "pattern": r"payment.*failed|transaction.*declined|gateway.*error",
        "cause": "Payment processing failed",
        "solution": "Payment gateway returned an error"
    }
}

# Status code mappings by query type
STATUS_CODES = {
    "promotion": {
        0: "Pending",
        1: "Approved",
        2: "Rejected",
        3: "Cancelled",
        4: "Processing"
    },
    "transaction": {
        0: "Pending",
        1: "Success",
        2: "Failed",
        3: "Cancelled",
        4: "Processing",
        5: "Manual Review"
    },
    "vip": {
        0: "Inactive",
        1: "Active",
        2: "Upgraded",
        3: "Downgraded",
        4: "Pending"
    }
}


class AnalysisAgent(BaseAgent):
    """
    Analysis Agent - Detects errors, matches patterns, determines root cause.

    Features:
    - Error pattern detection
    - Status code interpretation
    - Root cause determination
    - Summary generation
    """

    def __init__(self, bus: MessageBus, store: ResultStore):
        super().__init__("analysis_agent", bus, store)
        self._patterns = ERROR_PATTERNS
        self._status_codes = STATUS_CODES

    async def _on_start(self):
        """Initialize analysis agent."""
        self.log(f"Loaded {len(self._patterns)} error patterns")

    async def _on_stop(self):
        """Cleanup."""
        pass

    async def _execute_task(self, task: str, **kwargs) -> Any:
        """Execute an analysis task."""
        tasks = {
            "analyze_results": self._analyze_results,
            "detect_errors": self._detect_errors,
            "determine_root_cause": self._determine_root_cause,
            "generate_summary": self._generate_summary,
            "interpret_status": self._interpret_status
        }

        if task not in tasks:
            raise ValueError(f"Unknown task: {task}")

        return await tasks[task](**kwargs)

    async def _analyze_results(self, query_results: List[QueryResult],
                                ticket_type: str) -> AnalysisResult:
        """
        Analyze all query results and produce findings.

        Args:
            query_results: List of query results
            ticket_type: Type of investigation

        Returns:
            AnalysisResult with findings, errors, root_cause, conclusion
        """
        all_findings = []
        all_errors = []

        for qr in query_results:
            # Detect errors in this result
            errors = await self._detect_errors(qr.name, qr.rows)
            if errors.get("detected"):
                all_errors.extend(errors["detected"])

            # Generate findings
            findings = self._extract_findings(qr, ticket_type)
            all_findings.extend(findings)

        # Determine root cause
        root_cause_info = await self._determine_root_cause(all_findings, all_errors)

        # Generate conclusion
        conclusion = self._generate_conclusion(root_cause_info, all_findings)

        return AnalysisResult(
            findings=all_findings,
            errors=all_errors,
            root_cause=root_cause_info.get("cause", "Unable to determine"),
            conclusion=conclusion,
            confidence=root_cause_info.get("confidence", "low")
        )

    async def _detect_errors(self, query_type: str, data: List[Dict]) -> Dict[str, List[str]]:
        """
        Detect error patterns in query results.

        Args:
            query_type: Type of query (for context)
            data: Query result rows

        Returns:
            Dict with 'detected' list of error messages
        """
        detected = []

        # Convert data to searchable text
        text = str(data).lower()

        for error_name, error_info in self._patterns.items():
            pattern = error_info["pattern"]
            if re.search(pattern, text, re.IGNORECASE):
                detected.append({
                    "type": error_name,
                    "cause": error_info["cause"],
                    "solution": error_info["solution"]
                })

        # Check for specific field values
        for row in data:
            # Check RejectSetting field
            if "RejectSetting" in row:
                reject = row["RejectSetting"]
                if reject and reject != "None":
                    detected.append({
                        "type": "RejectSetting",
                        "cause": f"Promotion rejected: {reject}",
                        "solution": self._get_reject_solution(reject)
                    })

            # Check Status field
            if "Status" in row or "TransactionStatus" in row:
                status = row.get("Status") or row.get("TransactionStatus")
                if status in [2, "2", "Failed", "Rejected"]:
                    detected.append({
                        "type": "StatusFailed",
                        "cause": f"Transaction/Request status: {status}",
                        "solution": "Check related logs for failure reason"
                    })

        return {"detected": detected}

    async def _determine_root_cause(self, findings: List[str],
                                     errors: List[Dict]) -> Dict[str, str]:
        """
        Determine root cause from findings and errors.

        Args:
            findings: List of finding strings
            errors: List of detected errors

        Returns:
            Dict with 'cause', 'solution', 'confidence'
        """
        if not errors and not findings:
            return {
                "cause": "No issues detected in database records",
                "solution": "Data appears normal - may need manual review",
                "confidence": "low"
            }

        # Prioritize specific errors over generic findings
        if errors:
            # Use most specific error
            priority_order = ["SameFP", "SameIP", "MaxClaim", "RejectSetting",
                           "VIPLevelMismatch", "PaymentFailed", "TurnoverPending"]

            for priority in priority_order:
                for error in errors:
                    if error.get("type") == priority:
                        return {
                            "cause": error["cause"],
                            "solution": error["solution"],
                            "confidence": "high"
                        }

            # Return first error if no priority match
            first_error = errors[0]
            return {
                "cause": first_error["cause"],
                "solution": first_error["solution"],
                "confidence": "medium"
            }

        # Analyze findings for patterns
        findings_text = " ".join(findings).lower()

        if "no record" in findings_text or "not found" in findings_text:
            return {
                "cause": "No matching records found",
                "solution": "Verify player details (username, WebId, date range)",
                "confidence": "medium"
            }

        if "pending" in findings_text:
            return {
                "cause": "Request is still pending processing",
                "solution": "Wait for system processing or escalate if overdue",
                "confidence": "high"
            }

        return {
            "cause": "Issue requires manual analysis",
            "solution": "Review query results and apply domain knowledge",
            "confidence": "low"
        }

    async def _generate_summary(self, context: Dict[str, Any],
                                 analysis: AnalysisResult) -> str:
        """
        Generate investigation summary.

        Args:
            context: Investigation context
            analysis: Analysis results

        Returns:
            Formatted summary string
        """
        summary = f"""
INVESTIGATION SUMMARY
{'='*50}
Ticket: {context.get('ticket_key', 'N/A')}
Player: {context.get('username', 'N/A')}
WebId: {context.get('webid', 'N/A')}
Type: {context.get('ticket_type', 'N/A')}

KEY FINDINGS:
"""
        for i, finding in enumerate(analysis.findings[:5], 1):
            summary += f"  {i}. {finding}\n"

        if analysis.errors:
            summary += f"\nERRORS DETECTED:\n"
            for error in analysis.errors[:3]:
                summary += f"  - {error.get('cause', 'Unknown')}\n"

        summary += f"""
{'='*50}
ROOT CAUSE: {analysis.root_cause}
CONFIDENCE: {analysis.confidence}

CONCLUSION: {analysis.conclusion}
"""
        return summary

    async def _interpret_status(self, status_code: Any, category: str) -> str:
        """Interpret a status code."""
        codes = self._status_codes.get(category, {})
        return codes.get(int(status_code), f"Unknown ({status_code})")

    def _extract_findings(self, qr: QueryResult, ticket_type: str) -> List[str]:
        """Extract key findings from a query result."""
        findings = []

        if qr.count == 0:
            findings.append(f"{qr.name}: No records found")
            return findings

        findings.append(f"{qr.name}: Found {qr.count} record(s)")

        # Extract specific findings based on query type
        for row in qr.rows[:3]:  # First 3 rows
            if "RejectSetting" in row and row["RejectSetting"]:
                findings.append(f"Rejection reason: {row['RejectSetting']}")

            if "Status" in row:
                status = row["Status"]
                findings.append(f"Status: {status}")

            if "CreatedOn" in row:
                findings.append(f"Created: {row['CreatedOn']}")

            if "Amount" in row:
                findings.append(f"Amount: {row['Amount']}")

        return findings

    def _generate_conclusion(self, root_cause_info: Dict, findings: List[str]) -> str:
        """Generate a conclusion statement."""
        cause = root_cause_info.get("cause", "Unknown issue")
        solution = root_cause_info.get("solution", "Manual review required")

        if root_cause_info.get("confidence") == "high":
            return f"{cause}. Recommended action: {solution}"
        elif root_cause_info.get("confidence") == "medium":
            return f"Likely cause: {cause}. Suggested: {solution}"
        else:
            return f"Possible cause: {cause}. Please verify and {solution.lower()}"

    def _get_reject_solution(self, reject_setting: str) -> str:
        """Get solution for a specific reject setting."""
        solutions = {
            "SameFP": "Device fingerprint was used by another account",
            "SameIP": "IP address was used by another account",
            "MaxClaim": "Player reached maximum claims for this promotion",
            "Expired": "Promotion has expired",
            "NotEligible": "Player does not meet eligibility criteria",
            "Manual": "Rejected by manual review",
        }

        for key, solution in solutions.items():
            if key.lower() in reject_setting.lower():
                return solution

        return f"Rejected due to: {reject_setting}"
