"""
Base Agent - Common functionality for all agents.
"""

import asyncio
from abc import ABC, abstractmethod
from enum import Enum
from datetime import datetime
from typing import Dict, Any, Optional, List
from dataclasses import dataclass, field

import sys
from pathlib import Path
sys.path.insert(0, str(Path(__file__).parent.parent))

from swarm.bus import MessageBus, Message, MessageType
from swarm.context import InvestigationContext
from swarm.store import ResultStore, AgentResult


class AgentState(Enum):
    """Agent lifecycle states."""
    CREATED = "created"
    STARTING = "starting"
    READY = "ready"
    BUSY = "busy"
    STOPPING = "stopping"
    STOPPED = "stopped"
    ERROR = "error"


@dataclass
class TaskResult:
    """Result from an agent task."""
    success: bool
    data: Any = None
    error: Optional[str] = None
    duration_ms: int = 0


class BaseAgent(ABC):
    """
    Base class for all swarm agents.
    Provides common functionality for lifecycle, messaging, and logging.
    """

    def __init__(self, name: str, bus: MessageBus, store: ResultStore):
        """
        Initialize agent.

        Args:
            name: Unique agent name
            bus: Message bus for communication
            store: Result store for aggregation
        """
        self.name = name
        self.bus = bus
        self.store = store
        self.state = AgentState.CREATED
        self._queue: Optional[asyncio.Queue] = None
        self._task: Optional[asyncio.Task] = None
        self._context: Optional[InvestigationContext] = None
        self._started_at: Optional[datetime] = None
        self._task_count = 0

    async def start(self):
        """Start the agent."""
        self.state = AgentState.STARTING
        self._queue = await self.bus.register(self.name)
        self._started_at = datetime.now()

        # Initialize agent-specific resources
        await self._on_start()

        self.state = AgentState.READY
        self.log(f"Agent started")

    async def stop(self):
        """Stop the agent."""
        self.state = AgentState.STOPPING

        # Cleanup agent-specific resources
        await self._on_stop()

        await self.bus.unregister(self.name)
        self.state = AgentState.STOPPED
        self.log(f"Agent stopped")

    def set_context(self, context: InvestigationContext):
        """Set the investigation context."""
        self._context = context

    async def execute(self, task: str, **kwargs) -> TaskResult:
        """
        Execute a task.

        Args:
            task: Task name/type
            **kwargs: Task parameters

        Returns:
            TaskResult with success/failure and data
        """
        self.state = AgentState.BUSY
        self._task_count += 1
        start_time = datetime.now()

        try:
            self.log(f"Executing: {task}")
            result = await self._execute_task(task, **kwargs)

            duration = int((datetime.now() - start_time).total_seconds() * 1000)

            # Store result
            await self.store.store_result(
                investigation_id=self._context.ticket_key or "default",
                result=AgentResult(
                    agent=self.name,
                    task=task,
                    success=True,
                    data=result,
                    duration_ms=duration
                )
            )

            self.state = AgentState.READY
            return TaskResult(success=True, data=result, duration_ms=duration)

        except Exception as e:
            duration = int((datetime.now() - start_time).total_seconds() * 1000)
            error_msg = str(e)

            self.log(f"Error in {task}: {error_msg}", level="error")

            # Store error
            await self.store.store_result(
                investigation_id=self._context.ticket_key or "default",
                result=AgentResult(
                    agent=self.name,
                    task=task,
                    success=False,
                    error=error_msg,
                    duration_ms=duration
                )
            )

            self.state = AgentState.READY
            return TaskResult(success=False, error=error_msg, duration_ms=duration)

    async def send_message(self, recipient: str, msg_type: MessageType, payload: Dict[str, Any]):
        """Send a message to another agent."""
        await self.bus.send(self.name, recipient, msg_type, payload)

    async def broadcast(self, msg_type: MessageType, payload: Dict[str, Any]):
        """Broadcast a message to all agents."""
        await self.bus.send(self.name, "broadcast", msg_type, payload)

    async def request(self, recipient: str, payload: Dict[str, Any],
                      timeout: float = 30.0) -> Optional[Message]:
        """Send a request and wait for response."""
        return await self.bus.request(self.name, recipient, payload, timeout)

    async def receive(self, timeout: Optional[float] = None) -> Optional[Message]:
        """Receive a message."""
        return await self.bus.receive(self.name, timeout)

    def log(self, message: str, level: str = "info"):
        """Log a message."""
        timestamp = datetime.now().strftime("%H:%M:%S")
        prefix = {
            "info": "",
            "warn": "",
            "error": "",
            "debug": ""
        }.get(level, "")
        print(f"  [{timestamp}] [{self.name}] {prefix}{message}")

    def get_stats(self) -> Dict[str, Any]:
        """Get agent statistics."""
        uptime = 0
        if self._started_at:
            uptime = (datetime.now() - self._started_at).total_seconds()

        return {
            "name": self.name,
            "state": self.state.value,
            "task_count": self._task_count,
            "uptime_seconds": uptime
        }

    # Abstract methods for subclasses

    @abstractmethod
    async def _on_start(self):
        """Called when agent starts. Initialize resources here."""
        pass

    @abstractmethod
    async def _on_stop(self):
        """Called when agent stops. Cleanup resources here."""
        pass

    @abstractmethod
    async def _execute_task(self, task: str, **kwargs) -> Any:
        """
        Execute a specific task.

        Args:
            task: Task name
            **kwargs: Task parameters

        Returns:
            Task result data
        """
        pass
