"""
Jira Agent - Handles all Jira ticket operations.
"""

import re
from datetime import datetime
from typing import Dict, Any, List, Optional
from pathlib import Path

import sys
sys.path.insert(0, str(Path(__file__).parent.parent))

from .base import BaseAgent
from swarm.bus import MessageBus
from swarm.store import ResultStore
from swarm.context import SimilarTicket

# WebId mappings
WEBID_MAP = {
    "saffaluck": "20154", "sfl": "20154",
    "nocmakatiinc": "20107", "nmi": "20107", "nocmakati": "20107",
    "bet25": "20120",
    "gbw": "20109", "gbw777": "20109",
    "lucky7": "20132",
    "jilibet": "20140",
    "betso88": "20145",
    "peso88": "20150",
    "pnxbet": "20155",
}

# Type detection keywords
TYPE_KEYWORDS = {
    "promotion": ["promotion", "promo", "bonus", "reject", "fp", "fingerprint", "lucky wheel", "cannot apply"],
    "payment": ["deposit", "withdrawal", "pending", "payment", "not credited", "transfer", "bank"],
    "vip": ["vip", "upgrade", "level", "benefits", "downgrade", "vip bonus"],
    "betting": ["bet", "settlement", "winning", "void", "resettlement", "odds", "stake"],
    "login": ["login", "locked", "2fa", "suspended", "password", "access", "cannot login"]
}


class JiraAgent(BaseAgent):
    """
    Jira Agent - Handles ticket operations.

    Features:
    - Fetch ticket details
    - Parse username/webid/type from description
    - Search similar tickets
    - Post investigation comments
    """

    def __init__(self, bus: MessageBus, store: ResultStore):
        super().__init__("jira_agent", bus, store)

    async def _on_start(self):
        """Initialize Jira agent."""
        self.log("Ready for Jira operations")

    async def _on_stop(self):
        """Cleanup."""
        pass

    async def _execute_task(self, task: str, **kwargs) -> Any:
        """Execute a Jira task."""
        tasks = {
            "parse_ticket": self._parse_ticket,
            "detect_type": self._detect_type,
            "detect_webid": self._detect_webid,
            "build_similar_jql": self._build_similar_jql,
            "format_comment": self._format_comment,
            "format_to_cas": self._format_to_cas
        }

        if task not in tasks:
            raise ValueError(f"Unknown task: {task}")

        return await tasks[task](**kwargs)

    async def _parse_ticket(self, summary: str, description: str) -> Dict[str, Any]:
        """
        Parse ticket to extract investigation parameters.

        Returns:
            Dict with username, webid, ticket_type, date_range, confidence
        """
        text = f"{summary}\n{description}".lower()
        result = {
            "username": None,
            "webid": None,
            "ticket_type": None,
            "date_start": None,
            "date_end": None,
            "confidence": {}
        }

        # Extract username
        username_patterns = [
            r'(?:username|user|player|account|玩家|账号)[:\s]*["\']?([A-Za-z0-9_]+)["\']?',
            r'(?:member|用户)[:\s]*["\']?([A-Za-z0-9_]+)["\']?',
        ]
        for pattern in username_patterns:
            match = re.search(pattern, text, re.IGNORECASE)
            if match:
                username = match.group(1)
                if username.lower() not in ['the', 'and', 'for', 'this', 'that']:
                    result["username"] = username
                    result["confidence"]["username"] = "high"
                    break

        # Extract WebId from site name
        for site, webid in WEBID_MAP.items():
            if site in text:
                result["webid"] = webid
                result["confidence"]["webid"] = "high"
                break

        # Detect ticket type
        type_scores = {}
        for ttype, keywords in TYPE_KEYWORDS.items():
            score = sum(1 for kw in keywords if kw in text)
            if score > 0:
                type_scores[ttype] = score

        if type_scores:
            result["ticket_type"] = max(type_scores, key=type_scores.get)
            result["confidence"]["ticket_type"] = "high" if type_scores[result["ticket_type"]] >= 2 else "medium"

        # Extract dates
        date_patterns = [
            r'(\d{4}[-/]\d{1,2}[-/]\d{1,2})',
            r'(\d{1,2}[-/]\d{1,2}[-/]\d{4})',
        ]
        for pattern in date_patterns:
            matches = re.findall(pattern, text)
            if matches:
                result["date_start"] = matches[0]
                result["date_end"] = matches[-1] if len(matches) > 1 else matches[0]
                break

        return result

    async def _detect_type(self, text: str) -> str:
        """Detect ticket type from text."""
        text_lower = text.lower()
        scores = {}

        for ttype, keywords in TYPE_KEYWORDS.items():
            score = sum(1 for kw in keywords if kw in text_lower)
            if score > 0:
                scores[ttype] = score

        return max(scores, key=scores.get) if scores else "promotion"

    async def _detect_webid(self, text: str) -> Optional[str]:
        """Detect WebId from text."""
        text_lower = text.lower()
        for site, webid in WEBID_MAP.items():
            if site in text_lower:
                return webid
        return None

    async def _build_similar_jql(self, ticket_type: str, status: str = "Done") -> str:
        """Build JQL for similar ticket search."""
        keywords = TYPE_KEYWORDS.get(ticket_type, [])[:3]

        conditions = ["project = TCP"]
        if keywords:
            keyword_cond = " OR ".join([f'summary ~ "{kw}"' for kw in keywords])
            conditions.append(f"({keyword_cond})")
        if status:
            conditions.append(f"status = {status}")
        conditions.append("created >= -30d")

        return " AND ".join(conditions) + " ORDER BY created DESC"

    async def _format_comment(self, ticket_key: str, username: str, webid: str,
                               findings: List[str], root_cause: str, conclusion: str,
                               queries_run: List[str], similar_tickets: List[Dict] = None) -> str:
        """Format investigation results as Jira comment."""
        now = datetime.now().strftime("%Y-%m-%d %H:%M")

        comment = f"""
*DATABASE INVESTIGATION RESULTS*
{'='*50}
*Ticket:* {ticket_key}
*Player:* {username}
*WebId:* {webid}
*Investigated:* {now}

*Queries Executed:*
"""
        for q in queries_run:
            comment += f"- {q}\n"

        comment += f"""
*Key Findings:*
"""
        for f in findings[:5]:
            comment += f"- {f}\n"

        if similar_tickets:
            comment += f"""
*Similar Past Tickets:*
"""
            for t in similar_tickets[:3]:
                comment += f"- {t.get('key', 'N/A')}: {t.get('summary', '')[:50]}...\n"

        comment += f"""
{'='*50}
*ROOT CAUSE:* {root_cause}

*CONCLUSION:* {conclusion}
"""
        return comment

    async def _format_to_cas(self, root_cause: str, conclusion: str, details: str = "") -> str:
        """Format To CAS response (500 char max)."""
        response = f"{root_cause}. {conclusion}."
        if details:
            response += f" {details}"

        # Truncate to 500 chars
        if len(response) > 500:
            response = response[:497] + "..."

        return response

    def parse_similar_tickets(self, jira_results: List[Dict]) -> List[SimilarTicket]:
        """Convert Jira search results to SimilarTicket objects."""
        similar = []
        for ticket in jira_results:
            fields = ticket.get("fields", {})
            similar.append(SimilarTicket(
                key=ticket.get("key", ""),
                summary=fields.get("summary", ""),
                status=fields.get("status", {}).get("name", ""),
                resolution=fields.get("resolution", {}).get("name") if fields.get("resolution") else None
            ))
        return similar
