"""
Learning Agent - Brain operations, pattern learning, and solution matching.
"""

import json
import asyncio
import aiofiles
from datetime import datetime
from typing import Dict, Any, List, Optional
from dataclasses import dataclass, asdict
from pathlib import Path

import sys
sys.path.insert(0, str(Path(__file__).parent.parent))

from .base import BaseAgent
from swarm.bus import MessageBus
from swarm.store import ResultStore

SKILL_DIR = Path.home() / ".claude/skills/artemis-debug-secure"
BRAIN_FILE = SKILL_DIR / "brain" / "learned_patterns.json"
STATS_FILE = SKILL_DIR / "brain" / "statistics.json"


@dataclass
class LearnedPattern:
    """A learned investigation pattern."""
    id: str
    ticket_type: str
    error_pattern: str
    root_cause: str
    solution: str
    queries_used: List[str]
    success_count: int = 1
    last_used: str = ""
    webids: List[str] = None

    def __post_init__(self):
        if self.webids is None:
            self.webids = []
        if not self.last_used:
            self.last_used = datetime.now().isoformat()


class LearningAgent(BaseAgent):
    """
    Learning Agent - Manages pattern learning and solution matching.

    Features:
    - Load/save learned patterns
    - Find similar past solutions
    - Learn from new investigations
    - Track success statistics
    """

    def __init__(self, bus: MessageBus, store: ResultStore):
        super().__init__("learning_agent", bus, store)
        self._patterns: Dict[str, LearnedPattern] = {}
        self._stats: Dict[str, Any] = {}
        self._dirty = False

    async def _on_start(self):
        """Load brain data."""
        await self._load_brain()
        self.log(f"Loaded {len(self._patterns)} learned patterns")

    async def _on_stop(self):
        """Save brain data if modified."""
        if self._dirty:
            await self._save_brain()
            self.log("Brain data saved")

    async def _execute_task(self, task: str, **kwargs) -> Any:
        """Execute a learning task."""
        tasks = {
            "find_similar": self._find_similar,
            "learn_pattern": self._learn_pattern,
            "get_suggestions": self._get_suggestions,
            "update_stats": self._update_stats,
            "get_stats": self._get_stats
        }

        if task not in tasks:
            raise ValueError(f"Unknown task: {task}")

        return await tasks[task](**kwargs)

    async def _load_brain(self):
        """Load patterns from brain file."""
        if not BRAIN_FILE.exists():
            BRAIN_FILE.parent.mkdir(parents=True, exist_ok=True)
            self._patterns = {}
            return

        try:
            async with aiofiles.open(BRAIN_FILE, 'r') as f:
                content = await f.read()
                data = json.loads(content)

                for pid, pdata in data.get("patterns", {}).items():
                    self._patterns[pid] = LearnedPattern(**pdata)

        except Exception as e:
            self.log(f"Error loading brain: {e}", level="error")
            self._patterns = {}

        # Load stats
        if STATS_FILE.exists():
            try:
                async with aiofiles.open(STATS_FILE, 'r') as f:
                    content = await f.read()
                    self._stats = json.loads(content)
            except Exception:
                self._stats = {}

    async def _save_brain(self):
        """Save patterns to brain file."""
        BRAIN_FILE.parent.mkdir(parents=True, exist_ok=True)

        # Convert patterns to dict
        patterns_dict = {
            pid: asdict(pattern) for pid, pattern in self._patterns.items()
        }

        data = {
            "version": "2.0",
            "updated": datetime.now().isoformat(),
            "patterns": patterns_dict
        }

        try:
            async with aiofiles.open(BRAIN_FILE, 'w') as f:
                await f.write(json.dumps(data, indent=2))

            # Save stats
            async with aiofiles.open(STATS_FILE, 'w') as f:
                await f.write(json.dumps(self._stats, indent=2))

            self._dirty = False

        except Exception as e:
            self.log(f"Error saving brain: {e}", level="error")

    async def _find_similar(self, ticket_type: str, error_pattern: str = "",
                            webid: str = "") -> List[Dict[str, Any]]:
        """
        Find similar past patterns.

        Args:
            ticket_type: Type of investigation
            error_pattern: Detected error pattern
            webid: WebId for site-specific patterns

        Returns:
            List of matching patterns with relevance scores
        """
        matches = []

        for pid, pattern in self._patterns.items():
            score = 0

            # Exact type match
            if pattern.ticket_type == ticket_type:
                score += 50

            # Error pattern match
            if error_pattern and pattern.error_pattern:
                if error_pattern.lower() in pattern.error_pattern.lower():
                    score += 30
                elif pattern.error_pattern.lower() in error_pattern.lower():
                    score += 20

            # WebId match
            if webid and webid in pattern.webids:
                score += 10

            # Boost by success count
            score += min(pattern.success_count * 2, 20)

            if score >= 50:  # Minimum threshold
                matches.append({
                    "id": pid,
                    "ticket_type": pattern.ticket_type,
                    "error_pattern": pattern.error_pattern,
                    "root_cause": pattern.root_cause,
                    "solution": pattern.solution,
                    "queries_used": pattern.queries_used,
                    "success_count": pattern.success_count,
                    "relevance_score": score
                })

        # Sort by relevance
        matches.sort(key=lambda x: x["relevance_score"], reverse=True)
        return matches[:5]  # Top 5

    async def _learn_pattern(self, ticket_type: str, error_pattern: str,
                              root_cause: str, solution: str,
                              queries_used: List[str], webid: str = "") -> str:
        """
        Learn a new pattern or update existing.

        Args:
            ticket_type: Type of investigation
            error_pattern: Error pattern detected
            root_cause: Determined root cause
            solution: Solution/resolution
            queries_used: Queries that helped
            webid: WebId for the site

        Returns:
            Pattern ID
        """
        # Generate pattern ID
        pattern_id = f"{ticket_type}_{error_pattern[:20]}".replace(" ", "_").lower()

        if pattern_id in self._patterns:
            # Update existing
            pattern = self._patterns[pattern_id]
            pattern.success_count += 1
            pattern.last_used = datetime.now().isoformat()
            if webid and webid not in pattern.webids:
                pattern.webids.append(webid)
            # Update solution if provided
            if solution:
                pattern.solution = solution
        else:
            # Create new
            self._patterns[pattern_id] = LearnedPattern(
                id=pattern_id,
                ticket_type=ticket_type,
                error_pattern=error_pattern,
                root_cause=root_cause,
                solution=solution,
                queries_used=queries_used,
                webids=[webid] if webid else []
            )

        self._dirty = True
        self.log(f"Learned pattern: {pattern_id}")
        return pattern_id

    async def _get_suggestions(self, ticket_type: str,
                                findings: List[str]) -> List[str]:
        """
        Get investigation suggestions based on patterns.

        Args:
            ticket_type: Type of investigation
            findings: Current findings

        Returns:
            List of suggestion strings
        """
        suggestions = []

        # Find relevant patterns
        similar = await self._find_similar(ticket_type)

        if similar:
            # Suggest queries from successful patterns
            suggested_queries = set()
            for match in similar[:3]:
                for q in match.get("queries_used", []):
                    suggested_queries.add(q)

            if suggested_queries:
                suggestions.append(f"Recommended queries: {', '.join(list(suggested_queries)[:3])}")

            # Suggest solutions
            top_match = similar[0]
            if top_match["relevance_score"] >= 70:
                suggestions.append(f"Similar past issue: {top_match['root_cause']}")
                suggestions.append(f"Previous solution: {top_match['solution']}")

        # Default suggestions by type
        type_suggestions = {
            "promotion": [
                "Check PromotionSystemRejectRecord for rejection reasons",
                "Verify player eligibility in CustomerPromotionRequest"
            ],
            "vip": [
                "Check MemberLevelChangeLog for level transitions",
                "Verify BenefitsSetting for bonus configurations"
            ],
            "payment": [
                "Check TransactionRequest for payment status",
                "Verify with Payment gateway logs"
            ],
            "betting": [
                "Check bet settlement status",
                "Verify void/resettlement records"
            ],
            "login": [
                "Check CustomerLoginHistory for failed attempts",
                "Verify account status and 2FA settings"
            ]
        }

        if ticket_type in type_suggestions:
            suggestions.extend(type_suggestions[ticket_type])

        return suggestions[:5]

    async def _update_stats(self, ticket_type: str, duration_ms: int,
                             success: bool, webid: str = ""):
        """Update investigation statistics."""
        today = datetime.now().strftime("%Y-%m-%d")

        if "daily" not in self._stats:
            self._stats["daily"] = {}
        if today not in self._stats["daily"]:
            self._stats["daily"][today] = {
                "total": 0,
                "success": 0,
                "by_type": {},
                "avg_duration_ms": 0
            }

        day_stats = self._stats["daily"][today]
        day_stats["total"] += 1
        if success:
            day_stats["success"] += 1

        # Update by type
        if ticket_type not in day_stats["by_type"]:
            day_stats["by_type"][ticket_type] = 0
        day_stats["by_type"][ticket_type] += 1

        # Update average duration
        prev_avg = day_stats["avg_duration_ms"]
        prev_count = day_stats["total"] - 1
        day_stats["avg_duration_ms"] = (prev_avg * prev_count + duration_ms) / day_stats["total"]

        # Total stats
        if "total" not in self._stats:
            self._stats["total"] = {"investigations": 0, "success": 0}
        self._stats["total"]["investigations"] += 1
        if success:
            self._stats["total"]["success"] += 1

        self._dirty = True

    async def _get_stats(self) -> Dict[str, Any]:
        """Get current statistics."""
        today = datetime.now().strftime("%Y-%m-%d")

        return {
            "patterns_learned": len(self._patterns),
            "today": self._stats.get("daily", {}).get(today, {}),
            "total": self._stats.get("total", {}),
            "top_patterns": self._get_top_patterns()
        }

    def _get_top_patterns(self, limit: int = 5) -> List[Dict]:
        """Get most successful patterns."""
        sorted_patterns = sorted(
            self._patterns.values(),
            key=lambda p: p.success_count,
            reverse=True
        )

        return [
            {
                "id": p.id,
                "ticket_type": p.ticket_type,
                "success_count": p.success_count,
                "root_cause": p.root_cause[:50]
            }
            for p in sorted_patterns[:limit]
        ]
