#!/usr/bin/env python3
"""
Artemis Database Query Tool using Playwright

Usage:
    # With environment variables
    export ARTEMIS_USER="your_username"
    export ARTEMIS_PASS="your_password"
    python scripts/artemis_query.py --query "SELECT TOP 10 ..." --database Promotion

    # With credentials file
    python scripts/artemis_query.py --query "SELECT TOP 10 ..." --database VIP

    # Run with --help for all options
    python scripts/artemis_query.py --help
"""

import argparse
import os
import sys
import time
from datetime import datetime

try:
    from playwright.sync_api import sync_playwright, TimeoutError as PlaywrightTimeout
except ImportError:
    print("Error: playwright not installed. Run: pip install playwright && playwright install chromium")
    sys.exit(1)


# Configuration
MAX_RETRIES = 3
RETRY_DELAY = 2  # seconds
DEFAULT_TIMEOUT = 30000  # 30 seconds
LONG_QUERY_TIMEOUT = 120000  # 2 minutes


class ArtemisQueryError(Exception):
    """Base exception for Artemis query errors."""
    pass


class QueryValidationError(ArtemisQueryError):
    """Query validation failed."""
    pass


class LoginFailedError(ArtemisQueryError):
    """Login to Artemis failed."""
    pass


class QueryExecutionError(ArtemisQueryError):
    """Query execution failed."""
    pass


def validate_query(query):
    """
    Validate SQL query before execution.

    Rules:
    - Must include WITH(NOLOCK) for all tables
    - Must not use SELECT *
    - Must not include dangerous operations

    Returns True if valid, raises QueryValidationError if not.
    """
    query_upper = query.upper()

    # Check for WITH(NOLOCK) - required for all SELECT queries
    if 'SELECT' in query_upper:
        # Simple check - if there's a FROM clause, there should be WITH(NOLOCK)
        if 'FROM' in query_upper and 'WITH(NOLOCK)' not in query_upper:
            raise QueryValidationError(
                "Query must include WITH(NOLOCK) for all tables.\n"
                "Example: SELECT [Id], [Name] FROM [Table] WITH(NOLOCK)"
            )

    # Check for SELECT *
    # Handle various whitespace patterns
    import re
    if re.search(r'SELECT\s+\*\s+FROM', query_upper):
        raise QueryValidationError(
            "Query must not use SELECT * - specify column names explicitly.\n"
            "Example: SELECT [Id], [Name] FROM [Table] WITH(NOLOCK)"
        )

    # Check for dangerous operations
    dangerous_keywords = ['DROP', 'DELETE', 'TRUNCATE', 'UPDATE', 'INSERT', 'ALTER', 'CREATE', 'EXEC']
    words = query_upper.split()
    for keyword in dangerous_keywords:
        if keyword in words:
            raise QueryValidationError(
                f"Dangerous operation detected: {keyword}\n"
                "Only SELECT queries are allowed."
            )

    return True


def determine_timeout(query):
    """Determine appropriate timeout based on query complexity."""
    query_upper = query.upper()

    # Complex queries need more time
    if any(kw in query_upper for kw in ['JOIN', 'GROUP BY', 'UNION', 'HAVING', 'SUBQUERY']):
        return LONG_QUERY_TIMEOUT

    # Large result sets need more time
    if 'TOP 100' in query_upper or 'TOP 200' in query_upper:
        return LONG_QUERY_TIMEOUT

    return DEFAULT_TIMEOUT


def load_credentials():
    """Load credentials from environment or file."""
    # Try environment variables first
    user = os.environ.get('ARTEMIS_USER')
    password = os.environ.get('ARTEMIS_PASS')

    if user and password:
        return user, password

    # Try credentials file
    creds_file = os.path.expanduser('~/.claude/skills/artemis-debug-secure/.credentials')
    if os.path.exists(creds_file):
        creds = {}
        with open(creds_file, 'r') as f:
            for line in f:
                if '=' in line:
                    key, value = line.strip().split('=', 1)
                    creds[key] = value
        if 'ARTEMIS_USER' in creds and 'ARTEMIS_PASS' in creds:
            return creds['ARTEMIS_USER'], creds['ARTEMIS_PASS']

    return None, None


def run_artemis_query(query, database='Promotion', customer='YY3_Ns_Prod',
                      output_path=None, headless=False, zoom=0.75,
                      skip_validation=False):
    """
    Run a query on Artemis and capture screenshot.

    Args:
        query: SQL query to execute
        database: Database name (Promotion, VIP, Main, Account, etc.)
        customer: Server/Customer (YY3_Ns_Prod, YY3_Ns_Stg, etc.)
        output_path: Path to save screenshot
        headless: Run browser in headless mode
        zoom: Browser zoom level (0.75 = 75%)
        skip_validation: Skip query validation (not recommended)

    Returns:
        Path to saved screenshot

    Raises:
        QueryValidationError: If query fails validation
        LoginFailedError: If login fails
        QueryExecutionError: If query execution fails
    """

    # Validate query first (unless skipped)
    if not skip_validation:
        print("Validating query...")
        validate_query(query)
        print("Query validation passed.")

    user, password = load_credentials()
    if not user or not password:
        print("Error: Credentials not found.")
        print("Set ARTEMIS_USER and ARTEMIS_PASS environment variables")
        print("Or create ~/.claude/skills/artemis-debug-secure/.credentials")
        sys.exit(1)

    if not output_path:
        timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
        output_path = f'/tmp/artemis_{database}_{timestamp}.png'

    # Determine timeout based on query complexity
    timeout = determine_timeout(query)
    print(f"Using timeout: {timeout // 1000}s")

    with sync_playwright() as p:
        browser = p.chromium.launch(headless=headless)
        page = browser.new_page(viewport={'width': 1400, 'height': 900})
        page.set_default_timeout(timeout)

        print(f"Connecting to Artemis...")

        try:
            # 1. Login
            page.goto('https://artemis.568winex.com')
            page.wait_for_load_state('networkidle')

            # Check if login form exists
            if page.locator('input[type="text"]').count() > 0:
                print(f"Logging in as {user}...")
                page.fill('input[type="text"]', user)
                page.fill('input[type="password"]', password)
                page.click('button:has-text("Login")')
                page.wait_for_load_state('networkidle')
                page.wait_for_timeout(1000)

                # Verify login success
                if page.locator('input[type="password"]').count() > 0:
                    raise LoginFailedError("Login failed - check credentials")

            # 2. Select Customer (Database Server)
            print(f"Selecting server: {customer}")
            page.select_option('#Customer', customer)
            page.wait_for_timeout(500)

            # 3. Select Database
            db_option = f'{database} ({customer})'
            print(f"Selecting database: {db_option}")
            page.select_option('#Database', db_option)
            page.wait_for_timeout(500)

            # 4. Enter SQL Query (CodeMirror editor)
            print("Entering query...")
            escaped_query = query.replace('`', '\\`').replace('$', '\\$')
            page.evaluate(f'''() => {{
                const cm = document.querySelector('.CodeMirror').CodeMirror;
                cm.setValue(`{escaped_query}`);
            }}''')

            # 5. Submit Query
            print("Executing query...")
            page.click('button:has-text("Submit")')
            page.wait_for_load_state('networkidle')
            page.wait_for_timeout(2000)

            # Check for errors
            error_msg = page.locator('text=Error Message').first
            if error_msg.is_visible():
                error_text = page.locator('.error-message, .alert-danger').text_content()
                print(f"Query Error: {error_text}")
                raise QueryExecutionError(f"Query failed: {error_text}")

            # 6. Zoom out for more rows
            if zoom != 1.0:
                print(f"Zooming to {int(zoom * 100)}%...")
                page.evaluate(f'document.body.style.zoom = "{zoom}"')
                page.wait_for_timeout(500)

            # 7. Take Screenshot
            page.screenshot(path=output_path, full_page=True)
            print(f"Screenshot saved: {output_path}")

        except PlaywrightTimeout as e:
            print(f"Timeout error: {e}")
            # Take screenshot anyway for debugging
            page.screenshot(path=output_path.replace('.png', '_timeout.png'), full_page=True)
            raise QueryExecutionError(f"Query timed out after {timeout // 1000}s")

        finally:
            browser.close()

    return output_path


def run_with_retry(query, database='Promotion', customer='YY3_Ns_Prod',
                   output_path=None, headless=False, zoom=0.75,
                   skip_validation=False, max_retries=MAX_RETRIES):
    """
    Run query with automatic retry on transient failures.

    Args:
        Same as run_artemis_query, plus:
        max_retries: Maximum number of retry attempts

    Returns:
        Path to saved screenshot
    """
    last_error = None

    for attempt in range(max_retries):
        try:
            print(f"\n--- Attempt {attempt + 1}/{max_retries} ---")
            return run_artemis_query(
                query=query,
                database=database,
                customer=customer,
                output_path=output_path,
                headless=headless,
                zoom=zoom,
                skip_validation=skip_validation
            )
        except (QueryExecutionError, PlaywrightTimeout) as e:
            last_error = e
            if attempt < max_retries - 1:
                wait_time = RETRY_DELAY * (attempt + 1)
                print(f"Retrying in {wait_time}s...")
                time.sleep(wait_time)
            continue
        except QueryValidationError:
            # Don't retry validation errors
            raise
        except LoginFailedError:
            # Don't retry login errors
            raise

    raise QueryExecutionError(f"Query failed after {max_retries} attempts: {last_error}")


def main():
    parser = argparse.ArgumentParser(
        description='Run SQL queries on Artemis database',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Query Promotion database
  python artemis_query.py -q "SELECT TOP 10 [Id], [Status] FROM [PromotionRequest] WITH(NOLOCK)" -d Promotion

  # Query VIP database
  python artemis_query.py -q "SELECT TOP 10 [Id], [LevelId] FROM [BenefitsSetting] WITH(NOLOCK)" -d VIP

  # Query with custom output
  python artemis_query.py -q "SELECT ..." -d Main -o /tmp/result.png

  # Query with retry
  python artemis_query.py -q "SELECT ..." -d Main --retry

  # Skip validation (not recommended)
  python artemis_query.py -q "SELECT ..." -d Main --skip-validation

Environment Variables:
  ARTEMIS_USER - Artemis username
  ARTEMIS_PASS - Artemis password

Query Validation:
  - Must include WITH(NOLOCK) for all tables
  - Must not use SELECT *
  - Only SELECT queries allowed (no INSERT, UPDATE, DELETE, etc.)
        """
    )
    parser.add_argument('-q', '--query', required=True, help='SQL query to execute')
    parser.add_argument('-d', '--database', default='Promotion',
                        help='Database name (default: Promotion)')
    parser.add_argument('-c', '--customer', default='YY3_Ns_Prod',
                        help='Customer/Server (default: YY3_Ns_Prod)')
    parser.add_argument('-o', '--output', help='Output screenshot path')
    parser.add_argument('--headless', action='store_true', help='Run in headless mode')
    parser.add_argument('--zoom', type=float, default=0.75,
                        help='Browser zoom level (default: 0.75)')
    parser.add_argument('--retry', action='store_true',
                        help='Enable automatic retry on failure (3 attempts)')
    parser.add_argument('--skip-validation', action='store_true',
                        help='Skip query validation (not recommended)')
    parser.add_argument('--validate-only', action='store_true',
                        help='Only validate query, do not execute')

    args = parser.parse_args()

    # Validate only mode
    if args.validate_only:
        try:
            validate_query(args.query)
            print("Query validation PASSED")
            sys.exit(0)
        except QueryValidationError as e:
            print(f"Query validation FAILED:\n{e}")
            sys.exit(1)

    # Choose execution method
    if args.retry:
        run_with_retry(
            query=args.query,
            database=args.database,
            customer=args.customer,
            output_path=args.output,
            headless=args.headless,
            zoom=args.zoom,
            skip_validation=args.skip_validation
        )
    else:
        run_artemis_query(
            query=args.query,
            database=args.database,
            customer=args.customer,
            output_path=args.output,
            headless=args.headless,
            zoom=args.zoom,
            skip_validation=args.skip_validation
        )


if __name__ == '__main__':
    main()
