#!/usr/bin/env python3
"""
Artemis Auto-Screenshot Script
Takes screenshot of query results with auto-zoom for more rows.
"""

import os
import sys
import time
from datetime import datetime
from playwright.sync_api import sync_playwright

# Load credentials from .credentials file
CREDS_FILE = os.path.expanduser("~/.claude/skills/artemis-debug-secure/.credentials")
SCREENSHOT_DIR = os.path.expanduser("~/.playwright-mcp")

def load_credentials():
    """Load credentials from .credentials file"""
    creds = {}
    if os.path.exists(CREDS_FILE):
        with open(CREDS_FILE) as f:
            for line in f:
                if '=' in line and not line.startswith('#'):
                    key, val = line.strip().split('=', 1)
                    creds[key] = val
    return creds

def take_artemis_screenshot(query: str, database: str = "Promotion",
                           customer: str = "YY3_Ns_Prod",
                           zoom: float = 0.67,
                           filename: str = None) -> str:
    """
    Execute query in Artemis and take screenshot.

    Args:
        query: SQL query to execute
        database: Database name (Promotion, Account, VIP, etc.)
        customer: Database server (YY3_Ns_Prod, YY3_Ns_Stg)
        zoom: Browser zoom level (0.5-1.0, default 0.67 for more rows)
        filename: Output filename (auto-generated if None)

    Returns:
        Path to screenshot file
    """
    creds = load_credentials()
    if not creds.get('ARTEMIS_USER') or not creds.get('ARTEMIS_PASS'):
        raise ValueError("Missing credentials in .credentials file")

    os.makedirs(SCREENSHOT_DIR, exist_ok=True)

    if filename is None:
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        filename = f"artemis_{database}_{timestamp}.png"

    screenshot_path = os.path.join(SCREENSHOT_DIR, filename)

    with sync_playwright() as p:
        browser = p.chromium.launch(headless=False)
        page = browser.new_page(viewport={'width': 1600, 'height': 1000})

        try:
            # 1. Login
            print("Logging in to Artemis...")
            page.goto(creds.get('ARTEMIS_URL', 'https://artemis.568winex.com'))
            page.wait_for_load_state('networkidle')

            page.fill('input[type="text"]', creds['ARTEMIS_USER'])
            page.fill('input[type="password"]', creds['ARTEMIS_PASS'])
            page.click('button:has-text("Login")')
            page.wait_for_load_state('networkidle')
            time.sleep(1)

            # 2. Select Customer (Server)
            print(f"Selecting server: {customer}")
            page.select_option('#Customer', customer)
            time.sleep(0.5)

            # 3. Select Database
            print(f"Selecting database: {database}")
            # Database dropdown value format: "Database (Server)"
            db_value = f"{database} ({customer})"
            page.select_option('#Database', db_value)
            time.sleep(0.5)

            # 4. Enter SQL Query (CodeMirror editor)
            print("Entering query...")
            page.evaluate(f'''() => {{
                const cm = document.querySelector('.CodeMirror').CodeMirror;
                cm.setValue(`{query.replace('`', '\\`')}`);
            }}''')

            # 5. Submit Query
            print("Executing query...")
            page.click('button:has-text("Submit")')
            page.wait_for_load_state('networkidle')
            time.sleep(2)  # Wait for results

            # 6. Zoom out for more rows
            print(f"Setting zoom to {int(zoom * 100)}%...")
            page.evaluate(f'document.body.style.zoom = "{zoom}"')
            time.sleep(0.5)

            # 7. Take Screenshot
            print(f"Taking screenshot: {screenshot_path}")
            page.screenshot(path=screenshot_path, full_page=False)

            print(f"Screenshot saved: {screenshot_path}")

        except Exception as e:
            print(f"Error: {e}")
            # Take error screenshot
            error_path = os.path.join(SCREENSHOT_DIR, f"error_{datetime.now().strftime('%H%M%S')}.png")
            page.screenshot(path=error_path)
            print(f"Error screenshot saved: {error_path}")
            raise
        finally:
            browser.close()

    return screenshot_path


def main():
    """CLI usage"""
    import argparse

    parser = argparse.ArgumentParser(description='Artemis Screenshot Tool')
    parser.add_argument('-q', '--query', required=True, help='SQL query to execute')
    parser.add_argument('-d', '--database', default='Promotion', help='Database name')
    parser.add_argument('-c', '--customer', default='YY3_Ns_Prod', help='Server/Customer')
    parser.add_argument('-z', '--zoom', type=float, default=0.67, help='Zoom level (0.5-1.0)')
    parser.add_argument('-o', '--output', help='Output filename')

    args = parser.parse_args()

    try:
        path = take_artemis_screenshot(
            query=args.query,
            database=args.database,
            customer=args.customer,
            zoom=args.zoom,
            filename=args.output
        )
        print(f"\nSuccess! Screenshot: {path}")
    except Exception as e:
        print(f"\nFailed: {e}")
        sys.exit(1)


if __name__ == '__main__':
    main()
