#!/usr/bin/env python3
"""
Artemis Auto-Investigation Script
Automatically runs all queries for a ticket type and generates summary.
"""

import os
import sys
import json
import time
import re
from datetime import datetime
from playwright.sync_api import sync_playwright

# Paths
SKILL_DIR = os.path.expanduser("~/.claude/skills/artemis-debug-secure")
CREDS_FILE = os.path.join(SKILL_DIR, ".credentials")
QUERIES_FILE = os.path.join(SKILL_DIR, "scripts/queries.json")
SCREENSHOT_DIR = os.path.expanduser("~/.playwright-mcp")
WEBIDS_FILE = os.path.join(SKILL_DIR, "docs/webids.md")

# Ticket type keywords
TYPE_KEYWORDS = {
    "promotion": ["promotion", "promo", "bonus", "lucky wheel", "cannot apply", "reject", "claim"],
    "payment": ["deposit", "withdrawal", "payment", "pending", "callback", "refno", "transaction"],
    "vip": ["vip", "upgrade", "level", "diamond", "gold", "platinum", "benefits"],
    "betting": ["bet", "settlement", "winning", "losing", "void", "cashout", "game"],
    "login": ["login", "access", "locked", "password", "2fa", "otp", "suspended"]
}

def load_credentials():
    """Load credentials from .credentials file"""
    creds = {}
    if os.path.exists(CREDS_FILE):
        with open(CREDS_FILE) as f:
            for line in f:
                if '=' in line and not line.startswith('#'):
                    key, val = line.strip().split('=', 1)
                    creds[key] = val
    return creds

def load_queries():
    """Load query definitions from JSON"""
    with open(QUERIES_FILE) as f:
        return json.load(f)

def load_webids():
    """Load WebId mapping from docs/webids.md"""
    webids = {}
    if os.path.exists(WEBIDS_FILE):
        with open(WEBIDS_FILE) as f:
            for line in f:
                # Parse markdown table: | Site | WebId |
                match = re.match(r'\|\s*(\w+)\s*\|\s*(\d+)\s*\|', line)
                if match:
                    site, webid = match.groups()
                    webids[site.lower()] = webid
    return webids

def detect_ticket_type(text: str) -> str:
    """Detect ticket type from text keywords"""
    text_lower = text.lower()
    scores = {}

    for ticket_type, keywords in TYPE_KEYWORDS.items():
        score = sum(1 for kw in keywords if kw in text_lower)
        if score > 0:
            scores[ticket_type] = score

    if scores:
        return max(scores, key=scores.get)
    return "promotion"  # Default

def detect_webid(text: str, webids: dict) -> str:
    """Detect WebId from site name in text"""
    text_lower = text.lower()
    for site, webid in webids.items():
        if site in text_lower:
            return webid
    return None

def fill_query_params(sql: str, params: dict) -> str:
    """Fill query parameters"""
    result = sql
    for key, val in params.items():
        result = result.replace(f"{{{key}}}", str(val))
        result = result.replace(f"'{{{key}}}'", f"'{val}'")
    return result

def parse_table_results(page) -> list:
    """Parse results table from Artemis DOM"""
    try:
        # Get table headers
        headers = page.evaluate('''() => {
            const ths = document.querySelectorAll('table th');
            return Array.from(ths).map(th => th.innerText.trim());
        }''')

        # Get table rows
        rows = page.evaluate('''() => {
            const trs = document.querySelectorAll('table tbody tr');
            return Array.from(trs).map(tr => {
                const tds = tr.querySelectorAll('td');
                return Array.from(tds).map(td => td.innerText.trim());
            });
        }''')

        if headers and rows:
            return [dict(zip(headers, row)) for row in rows[:20]]  # Limit to 20 rows
    except:
        pass
    return []

def run_investigation(ticket_type: str, params: dict, headless: bool = False) -> dict:
    """
    Run all queries for a ticket type.

    Args:
        ticket_type: Type of ticket (promotion, payment, vip, betting, login)
        params: Dict with webId, username, customerId (optional)
        headless: Run browser in headless mode

    Returns:
        Dict with results for each query
    """
    creds = load_credentials()
    if not creds.get('ARTEMIS_USER'):
        raise ValueError("Missing credentials")

    queries_def = load_queries()
    if ticket_type not in queries_def:
        raise ValueError(f"Unknown ticket type: {ticket_type}")

    playbook = queries_def[ticket_type]
    results = {
        "ticket_type": ticket_type,
        "playbook": playbook["name"],
        "params": params,
        "queries": [],
        "screenshots": [],
        "timestamp": datetime.now().isoformat()
    }

    os.makedirs(SCREENSHOT_DIR, exist_ok=True)

    with sync_playwright() as p:
        browser = p.chromium.launch(headless=headless)
        page = browser.new_page(viewport={'width': 1600, 'height': 1000})

        try:
            # Login
            print("🔐 Logging in to Artemis...")
            page.goto(creds.get('ARTEMIS_URL', 'https://artemis.568winex.com'))
            page.wait_for_load_state('networkidle')
            page.fill('input[type="text"]', creds['ARTEMIS_USER'])
            page.fill('input[type="password"]', creds['ARTEMIS_PASS'])
            page.click('button:has-text("Login")')
            page.wait_for_load_state('networkidle')
            time.sleep(1)

            # Select server
            customer = "YY3_Ns_Prod"
            page.select_option('#Customer', customer)
            time.sleep(0.5)

            # Run each query
            for i, query_def in enumerate(playbook["queries"]):
                query_name = query_def["name"]
                database = query_def["database"]
                sql = fill_query_params(query_def["sql"], params)

                print(f"\n📊 [{i+1}/{len(playbook['queries'])}] {query_name}")
                print(f"   Database: {database}")

                # Select database
                db_value = f"{database} ({customer})"
                try:
                    page.select_option('#Database', db_value)
                except:
                    print(f"   ⚠️ Database {database} not found, trying without suffix")
                    page.select_option('#Database', database)
                time.sleep(0.5)

                # Enter query
                escaped_sql = sql.replace('\\', '\\\\').replace('`', '\\`').replace('${', '\\${')
                page.evaluate(f'''() => {{
                    const cm = document.querySelector('.CodeMirror').CodeMirror;
                    cm.setValue(`{escaped_sql}`);
                }}''')

                # Execute
                page.click('button:has-text("Submit")')
                page.wait_for_load_state('networkidle')
                time.sleep(2)

                # Parse results
                data = parse_table_results(page)
                row_count = len(data)
                print(f"   ✅ {row_count} rows returned")

                # Zoom and screenshot
                page.evaluate('document.body.style.zoom = "0.67"')
                time.sleep(0.3)

                timestamp = datetime.now().strftime("%H%M%S")
                filename = f"{ticket_type}_{query_def['id']}_{timestamp}.png"
                screenshot_path = os.path.join(SCREENSHOT_DIR, filename)
                page.screenshot(path=screenshot_path)

                # Reset zoom
                page.evaluate('document.body.style.zoom = "1"')

                results["queries"].append({
                    "id": query_def["id"],
                    "name": query_name,
                    "database": database,
                    "sql": sql,
                    "row_count": row_count,
                    "data": data[:10],  # First 10 rows
                    "screenshot": screenshot_path
                })
                results["screenshots"].append(screenshot_path)

        except Exception as e:
            print(f"\n❌ Error: {e}")
            error_path = os.path.join(SCREENSHOT_DIR, f"error_{datetime.now().strftime('%H%M%S')}.png")
            page.screenshot(path=error_path)
            results["error"] = str(e)
            results["error_screenshot"] = error_path
        finally:
            browser.close()

    return results

def generate_summary(results: dict) -> str:
    """Generate To CAS summary from results"""
    lines = []
    lines.append(f"Investigation: {results['playbook']}")
    lines.append(f"WebId: {results['params'].get('webId', 'N/A')}")
    lines.append(f"Username: {results['params'].get('username', 'N/A')}")
    lines.append("")

    for q in results["queries"]:
        lines.append(f"[{q['name']}] {q['row_count']} rows")
        if q["data"]:
            # Show first row key fields
            first_row = q["data"][0]
            key_fields = ["Status", "RejectSetting", "VipLevel", "TransactionStatus", "Result"]
            for field in key_fields:
                if field in first_row:
                    lines.append(f"  → {field}: {first_row[field]}")

    return "\n".join(lines)

def generate_to_cas(results: dict, queries_def: dict) -> str:
    """Generate short To CAS format"""
    ticket_type = results["ticket_type"]
    playbook = queries_def.get(ticket_type, {})

    # Analyze results for key findings
    findings = []

    for q in results["queries"]:
        if not q["data"]:
            continue

        first_row = q["data"][0]

        # Check for common issues
        if "RejectSetting" in first_row and first_row["RejectSetting"]:
            reject = first_row["RejectSetting"]
            flags = playbook.get("flags", {})
            meaning = flags.get(reject, reject)
            findings.append(f"Rejected: {meaning}")

        if "TransactionStatus" in first_row:
            status = first_row["TransactionStatus"]
            status_codes = playbook.get("status_codes", {})
            meaning = status_codes.get(str(status), f"Status {status}")
            findings.append(f"Transaction: {meaning}")

        if "VipLevel" in first_row:
            level = first_row["VipLevel"]
            levels = playbook.get("levels", {})
            meaning = levels.get(str(level), f"Level {level}")
            findings.append(f"VIP: {meaning}")

        if "Result" in first_row:
            findings.append(f"Login: {first_row['Result']}")

    if findings:
        return f"To CAS:\n{findings[0]}. Please check attached screenshots."
    else:
        return f"To CAS:\nNo issues found in {results['playbook']}. Data verified."

def main():
    """CLI usage"""
    import argparse

    parser = argparse.ArgumentParser(description='Artemis Auto-Investigation')
    parser.add_argument('-t', '--type', help='Ticket type (promotion, payment, vip, betting, login)')
    parser.add_argument('-w', '--webid', required=True, help='WebId')
    parser.add_argument('-u', '--username', required=True, help='Player username')
    parser.add_argument('-c', '--customerid', help='CustomerId (auto-fetched if not provided)')
    parser.add_argument('-s', '--site', help='Site name (alternative to webid)')
    parser.add_argument('--text', help='Ticket text for auto-detection')
    parser.add_argument('--headless', action='store_true', help='Run headless')
    parser.add_argument('-o', '--output', help='Output JSON file')

    args = parser.parse_args()

    # Detect type from text or use provided
    ticket_type = args.type
    if not ticket_type and args.text:
        ticket_type = detect_ticket_type(args.text)
        print(f"🔍 Detected ticket type: {ticket_type}")
    elif not ticket_type:
        ticket_type = "promotion"

    # Get WebId from site name if provided
    webid = args.webid
    if args.site and not webid:
        webids = load_webids()
        webid = detect_webid(args.site, webids)
        if webid:
            print(f"🔍 Detected WebId: {webid} (from {args.site})")

    params = {
        "webId": webid,
        "username": args.username,
        "customerId": args.customerid or "{customerId}"  # Placeholder if not provided
    }

    print(f"\n{'='*60}")
    print(f"🚀 Auto-Investigation: {ticket_type.upper()}")
    print(f"{'='*60}")
    print(f"WebId: {params['webId']}")
    print(f"Username: {params['username']}")
    print(f"{'='*60}\n")

    # Run investigation
    results = run_investigation(ticket_type, params, headless=args.headless)

    # Load queries for summary generation
    queries_def = load_queries()

    # Generate summaries
    print(f"\n{'='*60}")
    print("📋 SUMMARY")
    print(f"{'='*60}")
    print(generate_summary(results))

    print(f"\n{'='*60}")
    print("📝 TO CAS")
    print(f"{'='*60}")
    print(generate_to_cas(results, queries_def))

    print(f"\n{'='*60}")
    print("📸 SCREENSHOTS")
    print(f"{'='*60}")
    for path in results["screenshots"]:
        print(f"  {path}")

    # Save JSON output
    if args.output:
        with open(args.output, 'w') as f:
            json.dump(results, f, indent=2, default=str)
        print(f"\n💾 Results saved to: {args.output}")

if __name__ == '__main__':
    main()
