#!/usr/bin/env python3
"""
Brain Module - Auto-learning from resolved tickets.
Stores patterns, solutions, and learns from each investigation.
"""

import json
import os
from datetime import datetime
from pathlib import Path
from typing import Dict, List, Optional, Any

SKILL_DIR = Path.home() / ".claude/skills/artemis-debug-secure"
BRAIN_FILE = SKILL_DIR / "memory/brain.json"
LEARNINGS_FILE = SKILL_DIR / "memory/learnings.json"


class Brain:
    """Skill memory with auto-learning capability."""

    def __init__(self):
        self._data = None
        self._learnings = None
        self._load()

    def _load(self):
        """Load brain data."""
        if BRAIN_FILE.exists():
            with open(BRAIN_FILE) as f:
                self._data = json.load(f)
        else:
            self._data = self._default_brain()

        if LEARNINGS_FILE.exists():
            with open(LEARNINGS_FILE) as f:
                self._learnings = json.load(f)
        else:
            self._learnings = {"patterns": [], "solutions": {}, "stats": {}}

    def _default_brain(self) -> Dict:
        """Default brain structure."""
        return {
            "version": "2.0",
            "ticket_patterns": {},
            "status_codes": {},
            "webid_shortcuts": {},
            "response_templates": {},
            "error_patterns": {}
        }

    def save(self):
        """Save brain and learnings."""
        with open(BRAIN_FILE, 'w') as f:
            json.dump(self._data, f, indent=2)
        with open(LEARNINGS_FILE, 'w') as f:
            json.dump(self._learnings, f, indent=2)

    # ==================== LEARNING ====================

    def learn_from_investigation(self, ticket_key: str, ticket_type: str,
                                  root_cause: str, solution: str,
                                  query_results: List[Dict]):
        """
        Learn from a completed investigation.
        Stores the pattern for future reference.
        """
        pattern = {
            "ticket_key": ticket_key,
            "ticket_type": ticket_type,
            "root_cause": root_cause,
            "solution": solution,
            "learned_at": datetime.now().isoformat(),
            "key_indicators": self._extract_indicators(query_results)
        }

        self._learnings["patterns"].append(pattern)

        # Update solution frequency
        cause_key = f"{ticket_type}:{root_cause}"
        if cause_key not in self._learnings["solutions"]:
            self._learnings["solutions"][cause_key] = {
                "count": 0,
                "solution": solution,
                "examples": []
            }
        self._learnings["solutions"][cause_key]["count"] += 1
        self._learnings["solutions"][cause_key]["examples"].append(ticket_key)

        # Update stats
        if ticket_type not in self._learnings["stats"]:
            self._learnings["stats"][ticket_type] = 0
        self._learnings["stats"][ticket_type] += 1

        self.save()
        return pattern

    def _extract_indicators(self, query_results: List[Dict]) -> Dict:
        """Extract key indicators from query results."""
        indicators = {}
        for q in query_results:
            if not q.get("data"):
                continue
            first_row = q["data"][0] if q["data"] else {}

            # Extract key fields
            for key in ["RejectSetting", "Status", "TransactionStatus",
                        "VipLevel", "Result", "FailReason"]:
                if key in first_row and first_row[key]:
                    indicators[key] = first_row[key]

        return indicators

    def get_similar_solutions(self, ticket_type: str, indicators: Dict) -> List[Dict]:
        """Find similar past solutions based on indicators."""
        similar = []

        for pattern in self._learnings.get("patterns", []):
            if pattern["ticket_type"] != ticket_type:
                continue

            # Check indicator match
            match_score = 0
            for key, val in indicators.items():
                if pattern.get("key_indicators", {}).get(key) == val:
                    match_score += 1

            if match_score > 0:
                similar.append({
                    "ticket_key": pattern["ticket_key"],
                    "root_cause": pattern["root_cause"],
                    "solution": pattern["solution"],
                    "match_score": match_score
                })

        # Sort by match score
        similar.sort(key=lambda x: x["match_score"], reverse=True)
        return similar[:5]  # Top 5

    def get_common_causes(self, ticket_type: str) -> List[Dict]:
        """Get most common causes for a ticket type."""
        causes = []
        for key, data in self._learnings.get("solutions", {}).items():
            if key.startswith(f"{ticket_type}:"):
                cause = key.split(":", 1)[1]
                causes.append({
                    "cause": cause,
                    "count": data["count"],
                    "solution": data["solution"]
                })

        causes.sort(key=lambda x: x["count"], reverse=True)
        return causes[:5]

    def get_stats(self) -> Dict:
        """Get learning statistics."""
        return {
            "total_patterns": len(self._learnings.get("patterns", [])),
            "by_type": self._learnings.get("stats", {}),
            "unique_solutions": len(self._learnings.get("solutions", {}))
        }

    # ==================== DETECTION ====================

    def detect_ticket_type(self, text: str) -> str:
        """Detect ticket type from text."""
        text_lower = text.lower()
        patterns = self._data.get("ticket_patterns", {})

        scores = {}
        for ticket_type, info in patterns.items():
            keywords = info.get("keywords", [])
            score = sum(1 for kw in keywords if kw in text_lower)
            if score > 0:
                scores[ticket_type] = score

        return max(scores, key=scores.get) if scores else "promotion"

    def get_webid(self, site: str) -> Optional[str]:
        """Get WebId from site name."""
        shortcuts = self._data.get("webid_shortcuts", {})
        return shortcuts.get(site.lower())

    def get_status_meaning(self, category: str, code: str) -> str:
        """Get status code meaning."""
        codes = self._data.get("status_codes", {}).get(category, {})
        return codes.get(str(code), f"Unknown ({code})")

    def get_template(self, key: str) -> str:
        """Get response template."""
        return self._data.get("response_templates", {}).get(key, "")


# Singleton
_brain = None

def get_brain() -> Brain:
    global _brain
    if _brain is None:
        _brain = Brain()
    return _brain


if __name__ == "__main__":
    brain = get_brain()
    stats = brain.get_stats()
    print(f"Brain Stats:")
    print(f"  Total patterns learned: {stats['total_patterns']}")
    print(f"  By type: {stats['by_type']}")
    print(f"  Unique solutions: {stats['unique_solutions']}")
