#!/usr/bin/env python3
"""
Artemis Core - Fast, reliable database investigation.
User-specific credentials, caching, retry logic, brain integration.
"""

import os
import sys
import json
import time
import getpass
import socket
import hashlib
from datetime import datetime, timedelta
from pathlib import Path
from typing import Optional, Dict, List, Any

# Paths
SKILL_DIR = Path.home() / ".claude/skills/artemis-debug-secure"
USERS_DIR = SKILL_DIR / "users"
MEMORY_DIR = SKILL_DIR / "memory"
CACHE_DIR = SKILL_DIR / ".cache"
SCREENSHOT_DIR = Path.home() / ".playwright-mcp"

# Current user detection
CURRENT_USER = getpass.getuser()
CURRENT_HOST = socket.gethostname().split('.')[0]

class UserConfig:
    """User-specific configuration and credentials."""

    def __init__(self):
        self.user_dir = USERS_DIR / CURRENT_USER
        self.creds_file = self.user_dir / ".credentials"
        self.config_file = self.user_dir / "config.json"
        self._creds = None
        self._config = None

    def is_authorized(self) -> bool:
        """Check if current user has credentials set up."""
        return self.user_dir.exists() and self.creds_file.exists()

    def setup_user(self):
        """Interactive setup for new user."""
        print(f"\n🔐 Setting up credentials for: {CURRENT_USER}@{CURRENT_HOST}")
        self.user_dir.mkdir(parents=True, exist_ok=True)

        url = input("Artemis URL [https://artemis.568winex.com]: ").strip()
        if not url:
            url = "https://artemis.568winex.com"

        username = input("Artemis username: ").strip()
        password = input("Artemis password: ").strip()

        # Save credentials
        with open(self.creds_file, 'w') as f:
            f.write(f"ARTEMIS_URL={url}\n")
            f.write(f"ARTEMIS_USER={username}\n")
            f.write(f"ARTEMIS_PASS={password}\n")
        os.chmod(self.creds_file, 0o600)

        # Save config
        config = {
            "user": CURRENT_USER,
            "hostname": CURRENT_HOST,
            "created": datetime.now().isoformat(),
            "settings": {
                "default_server": "YY3_Ns_Prod",
                "default_zoom": 0.67,
                "headless": False,
                "timeout": 30,
                "retry_count": 3
            }
        }
        with open(self.config_file, 'w') as f:
            json.dump(config, f, indent=2)

        print(f"✅ Credentials saved to: {self.creds_file}")

    @property
    def credentials(self) -> Dict[str, str]:
        """Load credentials for current user."""
        if self._creds is None:
            self._creds = {}
            if self.creds_file.exists():
                with open(self.creds_file) as f:
                    for line in f:
                        if '=' in line and not line.startswith('#'):
                            key, val = line.strip().split('=', 1)
                            self._creds[key] = val
        return self._creds

    @property
    def config(self) -> Dict:
        """Load config for current user."""
        if self._config is None:
            self._config = {"settings": {}}
            if self.config_file.exists():
                with open(self.config_file) as f:
                    self._config = json.load(f)
        return self._config

    def get_setting(self, key: str, default: Any = None) -> Any:
        """Get a setting value."""
        return self.config.get("settings", {}).get(key, default)


class Brain:
    """Skill memory and intelligence."""

    def __init__(self):
        self.brain_file = MEMORY_DIR / "brain.json"
        self._data = None

    @property
    def data(self) -> Dict:
        """Load brain data."""
        if self._data is None:
            if self.brain_file.exists():
                with open(self.brain_file) as f:
                    self._data = json.load(f)
            else:
                self._data = {}
        return self._data

    def detect_ticket_type(self, text: str) -> str:
        """Detect ticket type from text using brain patterns."""
        text_lower = text.lower()
        patterns = self.data.get("ticket_patterns", {})

        scores = {}
        for ticket_type, info in patterns.items():
            keywords = info.get("keywords", [])
            score = sum(1 for kw in keywords if kw in text_lower)
            if score > 0:
                scores[ticket_type] = score

        return max(scores, key=scores.get) if scores else "promotion"

    def get_webid(self, site: str) -> Optional[str]:
        """Get WebId from site name shortcut."""
        shortcuts = self.data.get("webid_shortcuts", {})
        return shortcuts.get(site.lower())

    def get_status_meaning(self, category: str, code: str) -> str:
        """Get status code meaning."""
        codes = self.data.get("status_codes", {}).get(category, {})
        return codes.get(str(code), f"Unknown ({code})")

    def get_template(self, key: str) -> str:
        """Get response template."""
        templates = self.data.get("response_templates", {})
        return templates.get(key, "")

    def get_common_causes(self, ticket_type: str) -> List[str]:
        """Get common causes for ticket type."""
        patterns = self.data.get("ticket_patterns", {})
        return patterns.get(ticket_type, {}).get("common_causes", [])

    def learn_pattern(self, ticket_type: str, cause: str, solution: str):
        """Learn a new pattern from resolved ticket."""
        if "learned_patterns" not in self.data:
            self.data["learned_patterns"] = []

        self.data["learned_patterns"].append({
            "type": ticket_type,
            "cause": cause,
            "solution": solution,
            "learned_at": datetime.now().isoformat()
        })

        # Save brain
        with open(self.brain_file, 'w') as f:
            json.dump(self.data, f, indent=2)


class Cache:
    """Query result caching for speed."""

    def __init__(self, ttl_minutes: int = 5):
        self.cache_dir = CACHE_DIR
        self.cache_dir.mkdir(exist_ok=True)
        self.ttl = timedelta(minutes=ttl_minutes)

    def _key(self, query: str, params: Dict) -> str:
        """Generate cache key."""
        data = f"{query}:{json.dumps(params, sort_keys=True)}"
        return hashlib.md5(data.encode()).hexdigest()[:16]

    def get(self, query: str, params: Dict) -> Optional[Dict]:
        """Get cached result if valid."""
        key = self._key(query, params)
        cache_file = self.cache_dir / f"{key}.json"

        if cache_file.exists():
            with open(cache_file) as f:
                cached = json.load(f)

            cached_time = datetime.fromisoformat(cached["timestamp"])
            if datetime.now() - cached_time < self.ttl:
                return cached["data"]
        return None

    def set(self, query: str, params: Dict, data: Dict):
        """Cache query result."""
        key = self._key(query, params)
        cache_file = self.cache_dir / f"{key}.json"

        with open(cache_file, 'w') as f:
            json.dump({
                "timestamp": datetime.now().isoformat(),
                "query": query,
                "params": params,
                "data": data
            }, f)

    def clear(self):
        """Clear all cache."""
        for f in self.cache_dir.glob("*.json"):
            f.unlink()


class Artemis:
    """Fast, reliable Artemis connection with retry logic."""

    def __init__(self, user_config: UserConfig, headless: bool = False):
        self.config = user_config
        self.headless = headless
        self.page = None
        self.browser = None
        self.cache = Cache()
        self._logged_in = False

    def connect(self) -> bool:
        """Connect to Artemis with retry."""
        from playwright.sync_api import sync_playwright

        creds = self.config.credentials
        if not creds.get('ARTEMIS_USER'):
            raise ValueError("No credentials configured")

        retry_count = self.config.get_setting('retry_count', 3)
        timeout = self.config.get_setting('timeout', 30) * 1000

        for attempt in range(retry_count):
            try:
                if self.browser is None:
                    self._pw = sync_playwright().start()
                    self.browser = self._pw.chromium.launch(headless=self.headless)
                    self.page = self.browser.new_page(viewport={'width': 1600, 'height': 1000})
                    self.page.set_default_timeout(timeout)

                if not self._logged_in:
                    print(f"🔐 Connecting to Artemis... (attempt {attempt + 1})")
                    self.page.goto(creds.get('ARTEMIS_URL', 'https://artemis.568winex.com'))
                    self.page.wait_for_load_state('networkidle')

                    self.page.fill('input[type="text"]', creds['ARTEMIS_USER'])
                    self.page.fill('input[type="password"]', creds['ARTEMIS_PASS'])
                    self.page.click('button:has-text("Login")')
                    self.page.wait_for_load_state('networkidle')
                    time.sleep(0.5)

                    self._logged_in = True
                    print("✅ Connected")

                return True

            except Exception as e:
                print(f"⚠️ Attempt {attempt + 1} failed: {e}")
                if attempt < retry_count - 1:
                    time.sleep(2)
                else:
                    raise

        return False

    def query(self, sql: str, database: str = "Main",
              use_cache: bool = True) -> Dict:
        """Execute query with caching and retry."""
        server = self.config.get_setting('default_server', 'YY3_Ns_Prod')

        # Check cache
        params = {"sql": sql, "database": database, "server": server}
        if use_cache:
            cached = self.cache.get(sql, params)
            if cached:
                print(f"📦 Cache hit for {database}")
                return cached

        # Ensure connected
        self.connect()

        # Select database
        db_value = f"{database} ({server})"
        try:
            self.page.select_option('#Database', db_value)
        except:
            self.page.select_option('#Database', database)
        time.sleep(0.3)

        # Enter and execute query
        escaped_sql = sql.replace('\\', '\\\\').replace('`', '\\`').replace('${', '\\${')
        self.page.evaluate(f'''() => {{
            const cm = document.querySelector('.CodeMirror').CodeMirror;
            cm.setValue(`{escaped_sql}`);
        }}''')

        self.page.click('button:has-text("Submit")')
        self.page.wait_for_load_state('networkidle')
        time.sleep(1)

        # Parse results
        result = self._parse_results()

        # Cache result
        if use_cache and result.get("rows"):
            self.cache.set(sql, params, result)

        return result

    def _parse_results(self) -> Dict:
        """Parse results from Artemis DOM."""
        try:
            headers = self.page.evaluate('''() => {
                const ths = document.querySelectorAll('table th');
                return Array.from(ths).map(th => th.innerText.trim());
            }''')

            rows = self.page.evaluate('''() => {
                const trs = document.querySelectorAll('table tbody tr');
                return Array.from(trs).slice(0, 50).map(tr => {
                    const tds = tr.querySelectorAll('td');
                    return Array.from(tds).map(td => td.innerText.trim());
                });
            }''')

            if headers and rows:
                return {
                    "headers": headers,
                    "rows": [dict(zip(headers, row)) for row in rows],
                    "count": len(rows)
                }
        except:
            pass

        return {"headers": [], "rows": [], "count": 0}

    def screenshot(self, filename: str = None) -> str:
        """Take screenshot with optimal zoom."""
        SCREENSHOT_DIR.mkdir(exist_ok=True)

        zoom = self.config.get_setting('default_zoom', 0.67)
        self.page.evaluate(f'document.body.style.zoom = "{zoom}"')
        time.sleep(0.3)

        if filename is None:
            filename = f"artemis_{datetime.now().strftime('%H%M%S')}.png"

        path = SCREENSHOT_DIR / filename
        self.page.screenshot(path=str(path))

        # Reset zoom
        self.page.evaluate('document.body.style.zoom = "1"')

        return str(path)

    def close(self):
        """Close browser."""
        if self.browser:
            self.browser.close()
            self._pw.stop()
            self.browser = None
            self.page = None
            self._logged_in = False


def get_customer_id(artemis: Artemis, webid: str, username: str) -> Optional[str]:
    """Get CustomerId for a player (cached)."""
    sql = f"""SELECT TOP 1 [CustomerId], [Username], [VipLevel]
FROM [Main].[dbo].[Customer] WITH(NOLOCK)
WHERE [WebId] = {webid} AND [Username] = '{username}'"""

    result = artemis.query(sql, "Main", use_cache=True)
    if result["rows"]:
        return result["rows"][0].get("CustomerId")
    return None


# Singleton instances
_user_config = None
_brain = None

def get_user_config() -> UserConfig:
    global _user_config
    if _user_config is None:
        _user_config = UserConfig()
    return _user_config

def get_brain() -> Brain:
    global _brain
    if _brain is None:
        _brain = Brain()
    return _brain


if __name__ == '__main__':
    # Test user setup
    config = get_user_config()

    if not config.is_authorized():
        print(f"❌ No credentials for user: {CURRENT_USER}")
        setup = input("Set up now? [y/N]: ").strip().lower()
        if setup == 'y':
            config.setup_user()
    else:
        print(f"✅ User authorized: {CURRENT_USER}")
        print(f"   Credentials: {config.creds_file}")

        # Test brain
        brain = get_brain()
        print(f"\n🧠 Brain loaded:")
        print(f"   Patterns: {len(brain.data.get('ticket_patterns', {}))}")
        print(f"   Templates: {len(brain.data.get('response_templates', {}))}")
        print(f"   WebIds: {len(brain.data.get('webid_shortcuts', {}))}")
