#!/usr/bin/env python3
"""
Error Detector Module - Flag anomalies in query results.
Detects data issues, suspicious patterns, and common problems.
"""

import re
from datetime import datetime, timedelta
from typing import Dict, List, Optional, Any, Tuple
from decimal import Decimal


class ErrorDetector:
    """Detect anomalies and errors in database query results."""

    # Known error status codes
    ERROR_STATUSES = {
        "promotion": {
            "2": "Rejected",
            "4": "Expired",
            "5": "Cancelled",
            "-1": "Error"
        },
        "transaction": {
            "2": "Rejected",
            "4": "Cancelled",
            "1009": "Expired/Rejected",
            "1016": "Admin Rejected"
        },
        "customer": {
            "1": "Suspended",
            "2": "Closed",
            "4": "Deleted"
        }
    }

    # Suspicious patterns
    SUSPICIOUS_PATTERNS = {
        "large_amount": 100000,  # Flag amounts over 100k
        "rapid_transactions": 10,  # More than 10 in 1 hour
        "multiple_rejections": 3,  # More than 3 rejections
        "balance_mismatch_threshold": 0.01,  # 1 cent tolerance
    }

    def __init__(self):
        self.alerts = []
        self.warnings = []
        self.info = []

    def reset(self):
        """Reset all findings."""
        self.alerts = []
        self.warnings = []
        self.info = []

    def analyze_query_result(self, query_type: str, data: List[Dict]) -> Dict:
        """
        Analyze query results for anomalies.

        Args:
            query_type: Type of query (customer, promotion, transaction, etc.)
            data: List of result rows as dicts

        Returns:
            Dict with alerts, warnings, info, and summary
        """
        self.reset()

        if not data:
            self.info.append("No data returned from query")
            return self._build_result()

        # Run type-specific analyzers
        analyzers = {
            "customer": self._analyze_customer,
            "promotion": self._analyze_promotion,
            "transaction": self._analyze_transaction,
            "balance": self._analyze_balance,
            "vip": self._analyze_vip,
            "betting": self._analyze_betting,
            "reject": self._analyze_rejection,
        }

        # Try to match query type
        for key, analyzer in analyzers.items():
            if key in query_type.lower():
                analyzer(data)
                break
        else:
            # Generic analysis
            self._analyze_generic(data)

        return self._build_result()

    def _build_result(self) -> Dict:
        """Build result dictionary."""
        return {
            "alerts": self.alerts,  # Critical issues
            "warnings": self.warnings,  # Potential problems
            "info": self.info,  # Informational notes
            "has_issues": len(self.alerts) > 0 or len(self.warnings) > 0,
            "severity": "critical" if self.alerts else ("warning" if self.warnings else "ok"),
            "summary": self._generate_summary()
        }

    def _generate_summary(self) -> str:
        """Generate human-readable summary."""
        if self.alerts:
            return f"CRITICAL: {len(self.alerts)} issue(s) found - {self.alerts[0]}"
        if self.warnings:
            return f"WARNING: {len(self.warnings)} potential issue(s) - {self.warnings[0]}"
        if self.info:
            return self.info[0]
        return "No issues detected"

    # ==================== ANALYZERS ====================

    def _analyze_customer(self, data: List[Dict]):
        """Analyze customer data."""
        if not data:
            return

        row = data[0]

        # Check customer status
        status = row.get("CustomerStatus", row.get("Status", 0))
        if status:
            status = int(status)
            if status & 1:
                self.alerts.append("Customer account is SUSPENDED (bit 1)")
            if status & 2:
                self.alerts.append("Customer account is CLOSED (bit 2)")
            if status & 4:
                self.alerts.append("Customer account is DELETED (bit 4)")
            if status & 524288:
                self.warnings.append("Customer excluded from VIP (bit 524288)")

        # Check VIP level
        vip_level = row.get("VipLevel", row.get("LevelId"))
        if vip_level == 1:
            self.info.append("Customer is VIP0 (lowest level)")

        # Check balance
        balance = row.get("Balance", row.get("MainWalletBalance"))
        if balance and float(balance) < 0:
            self.alerts.append(f"NEGATIVE BALANCE: {balance}")

    def _analyze_promotion(self, data: List[Dict]):
        """Analyze promotion request data."""
        rejection_count = 0
        for row in data:
            status = str(row.get("Status", row.get("RequestStatus", "")))

            # Check for rejections
            if status in ["2", "Rejected", "rejected"]:
                rejection_count += 1
                reason = row.get("RejectReason", row.get("Remark", "Unknown"))
                self.warnings.append(f"Rejected promotion: {reason[:50]}")

            # Check for pending status over 24h
            created = row.get("CreatedOn", row.get("RequestDate"))
            if created and status in ["1", "Pending", "pending"]:
                self._check_stale_pending(created, "promotion request")

        if rejection_count >= self.SUSPICIOUS_PATTERNS["multiple_rejections"]:
            self.alerts.append(f"Multiple rejections detected: {rejection_count} times")

    def _analyze_transaction(self, data: List[Dict]):
        """Analyze transaction data."""
        for row in data:
            # Check transaction status
            status = str(row.get("TransactionStatus", row.get("Status", "")))
            if status in ["2", "1009", "1016"]:
                self.warnings.append(f"Transaction rejected/cancelled (status: {status})")

            # Check for large amounts
            amount = row.get("Amount", 0)
            if amount and float(amount) > self.SUSPICIOUS_PATTERNS["large_amount"]:
                self.warnings.append(f"Large transaction: {amount}")

            # Balance mismatch check
            before = row.get("BeforeBalance")
            after = row.get("AfterBalance")
            trans_amount = row.get("Amount")
            if before and after and trans_amount:
                expected_after = float(before) + float(trans_amount)
                actual_after = float(after)
                if abs(expected_after - actual_after) > self.SUSPICIOUS_PATTERNS["balance_mismatch_threshold"]:
                    self.alerts.append(
                        f"Balance mismatch! Before:{before} + Amount:{trans_amount} "
                        f"!= After:{after} (diff: {abs(expected_after - actual_after):.2f})"
                    )

    def _analyze_balance(self, data: List[Dict]):
        """Analyze balance history."""
        if len(data) < 2:
            return

        # Check for gaps in balance
        prev_after = None
        for row in data:
            before = row.get("BeforeBalance")
            after = row.get("AfterBalance")

            if prev_after and before:
                if abs(float(prev_after) - float(before)) > 0.01:
                    self.warnings.append(
                        f"Balance gap detected: prev_after={prev_after}, next_before={before}"
                    )

            prev_after = after

        # Check for missing ActionType
        for row in data:
            action_type = row.get("ActionType")
            if not action_type:
                self.warnings.append("Missing ActionType in balance history")
                break

    def _analyze_vip(self, data: List[Dict]):
        """Analyze VIP data."""
        for row in data:
            # Check for downgrade
            from_level = row.get("FromLevelId", row.get("FromLevel"))
            to_level = row.get("ToLevelId", row.get("ToLevel"))
            if from_level and to_level:
                if int(to_level) < int(from_level):
                    self.info.append(f"VIP downgrade: Level {from_level} → {to_level}")
                elif int(to_level) > int(from_level):
                    self.info.append(f"VIP upgrade: Level {from_level} → {to_level}")

            # Check bonus status
            bonus_status = row.get("Status", row.get("BonusStatus"))
            if str(bonus_status) in ["2", "Rejected"]:
                self.warnings.append("VIP bonus was rejected")

    def _analyze_betting(self, data: List[Dict]):
        """Analyze betting data."""
        void_count = 0
        for row in data:
            status = str(row.get("Status", row.get("BetStatus", ""))).lower()

            if status == "void":
                void_count += 1
                self.info.append(f"Voided bet found: {row.get('BetId', row.get('Id', 'N/A'))}")

            # Check for resettlement
            if row.get("IsResettled") or row.get("ResettlementCount", 0) > 0:
                self.info.append("Bet was resettled")

            # Large stake check
            stake = row.get("Stake", row.get("Amount", 0))
            if stake and float(stake) > self.SUSPICIOUS_PATTERNS["large_amount"]:
                self.warnings.append(f"Large bet stake: {stake}")

        if void_count > 0:
            self.warnings.append(f"Total voided bets: {void_count}")

    def _analyze_rejection(self, data: List[Dict]):
        """Analyze rejection records."""
        reject_types = {}
        for row in data:
            reject_type = row.get("RejectSetting", row.get("RejectType", "Unknown"))
            reject_types[reject_type] = reject_types.get(reject_type, 0) + 1

        for rtype, count in reject_types.items():
            if count >= self.SUSPICIOUS_PATTERNS["multiple_rejections"]:
                self.alerts.append(f"Repeated rejection: {rtype} ({count} times)")
            else:
                self.info.append(f"Rejection reason: {rtype}")

    def _analyze_generic(self, data: List[Dict]):
        """Generic analysis for unknown query types."""
        self.info.append(f"Returned {len(data)} rows")

        # Check for any error-like fields
        for row in data:
            for key, value in row.items():
                key_lower = key.lower()
                value_str = str(value).lower() if value else ""

                if "error" in key_lower or "error" in value_str:
                    self.warnings.append(f"Error field found: {key}={value}")
                if "fail" in key_lower or "failed" in value_str:
                    self.warnings.append(f"Failure indicator: {key}={value}")
                if "reject" in key_lower or "rejected" in value_str:
                    self.warnings.append(f"Rejection indicator: {key}={value}")

    def _check_stale_pending(self, created_str: str, item_type: str):
        """Check if pending item is stale (over 24h)."""
        try:
            # Try to parse datetime
            if "T" in created_str:
                created = datetime.fromisoformat(created_str.replace("Z", ""))
            else:
                created = datetime.strptime(created_str[:19], "%Y-%m-%d %H:%M:%S")

            if datetime.now() - created > timedelta(hours=24):
                self.warnings.append(f"Stale pending {item_type}: created {created_str}")
        except (ValueError, TypeError):
            pass


class DataValidator:
    """Validate data integrity."""

    @staticmethod
    def check_required_fields(data: Dict, required: List[str]) -> List[str]:
        """
        Check if required fields are present and not null.

        Returns:
            List of missing field names
        """
        missing = []
        for field in required:
            if field not in data or data[field] is None:
                missing.append(field)
        return missing

    @staticmethod
    def check_date_range(start_date: str, end_date: str, max_days: int = 90) -> Optional[str]:
        """
        Validate date range.

        Returns:
            Error message if invalid, None if valid
        """
        try:
            start = datetime.strptime(start_date[:10], "%Y-%m-%d")
            end = datetime.strptime(end_date[:10], "%Y-%m-%d")

            if end < start:
                return "End date is before start date"
            if (end - start).days > max_days:
                return f"Date range exceeds {max_days} days"
            return None
        except (ValueError, TypeError) as e:
            return f"Invalid date format: {e}"

    @staticmethod
    def check_username_format(username: str) -> Optional[str]:
        """
        Validate username format.

        Returns:
            Error message if invalid, None if valid
        """
        if not username:
            return "Username is empty"
        if len(username) < 3:
            return "Username too short (min 3 chars)"
        if len(username) > 20:
            return "Username too long (max 20 chars)"
        if not re.match(r'^[A-Za-z0-9_]+$', username):
            return "Username contains invalid characters"
        return None


# Convenience functions
def analyze_results(query_type: str, data: List[Dict]) -> Dict:
    """Convenience function to analyze query results."""
    detector = ErrorDetector()
    return detector.analyze_query_result(query_type, data)


def validate_investigation_params(params: Dict) -> Tuple[bool, List[str]]:
    """
    Validate investigation parameters.

    Returns:
        Tuple of (is_valid, list of errors)
    """
    errors = []

    # Check username
    username_error = DataValidator.check_username_format(params.get("username", ""))
    if username_error:
        errors.append(username_error)

    # Check WebId
    webid = params.get("webid")
    if not webid:
        errors.append("WebId is required")
    elif not str(webid).isdigit():
        errors.append("WebId must be numeric")

    # Check dates if provided
    start = params.get("date_start")
    end = params.get("date_end")
    if start and end:
        date_error = DataValidator.check_date_range(start, end)
        if date_error:
            errors.append(date_error)

    return len(errors) == 0, errors


if __name__ == "__main__":
    # Test error detection
    detector = ErrorDetector()

    # Test customer with suspended status
    customer_data = [{"CustomerStatus": 3, "Balance": -50, "VipLevel": 1}]
    result = detector.analyze_query_result("customer", customer_data)
    print("Customer Analysis:")
    print(f"  Alerts: {result['alerts']}")
    print(f"  Warnings: {result['warnings']}")
    print(f"  Summary: {result['summary']}")

    # Test transaction with balance mismatch
    print("\nTransaction Analysis:")
    trans_data = [{"BeforeBalance": 100, "Amount": 50, "AfterBalance": 160}]
    result = detector.analyze_query_result("transaction", trans_data)
    print(f"  Alerts: {result['alerts']}")
    print(f"  Summary: {result['summary']}")

    # Test rejection analysis
    print("\nRejection Analysis:")
    reject_data = [
        {"RejectSetting": "SameFP"},
        {"RejectSetting": "SameFP"},
        {"RejectSetting": "SameFP"},
        {"RejectSetting": "TurnoverNotMet"},
    ]
    result = detector.analyze_query_result("reject", reject_data)
    print(f"  Alerts: {result['alerts']}")
    print(f"  Info: {result['info']}")
