#!/usr/bin/env python3
"""
Enhanced Investigation Script - Full 10/10 skill features.
- Auto-learn from investigations
- Similar ticket search
- Jira integration
- Error detection

Usage: python investigate.py -t promotion -w 20154 -u "player123"
       python investigate.py --ticket TCP-12345
"""

import sys
import json
import argparse
from datetime import datetime
from pathlib import Path
from typing import Dict, List, Optional, Any

# Add scripts dir to path
sys.path.insert(0, str(Path(__file__).parent))

from core import (
    get_user_config, get_brain, Artemis,
    get_customer_id, SKILL_DIR, SCREENSHOT_DIR
)
from brain import Brain, get_brain as get_learning_brain
from jira_integration import (
    TicketParser, SimilarTicketSearch, JiraCommentBuilder,
    parse_jira_ticket, build_similar_search_jql, format_to_cas
)
from error_detector import (
    ErrorDetector, DataValidator, analyze_results, validate_investigation_params
)

# Load queries
QUERIES_FILE = SKILL_DIR / "scripts/queries.json"


def load_queries():
    with open(QUERIES_FILE) as f:
        return json.load(f)


def fill_params(sql: str, params: dict) -> str:
    """Fill SQL parameters."""
    result = sql
    for key, val in params.items():
        result = result.replace(f"{{{key}}}", str(val))
    return result


class Investigation:
    """
    Enhanced investigation with all 10/10 features.
    """

    def __init__(self, headless: bool = False):
        self.config = get_user_config()
        self.brain = get_learning_brain()
        self.queries = load_queries()
        self.detector = ErrorDetector()
        self.headless = headless

        # Results tracking
        self.results = None
        self.errors_detected = []
        self.similar_tickets = []
        self.learned = False

    def check_credentials(self) -> bool:
        """Check if user has credentials."""
        return self.config.is_authorized()

    def validate_params(self, params: Dict) -> tuple:
        """Validate investigation parameters."""
        return validate_investigation_params(params)

    def get_similar_tickets_jql(self, ticket_type: str, webid: str = None) -> str:
        """Build JQL for similar ticket search."""
        return build_similar_search_jql(ticket_type, webid)

    def parse_ticket(self, summary: str, description: str) -> Dict:
        """Parse Jira ticket to extract parameters."""
        return parse_jira_ticket(summary, description)

    def investigate(self, ticket_type: str, webid: str, username: str,
                    customer_id: str = None, ticket_key: str = None,
                    similar_tickets: List[Dict] = None) -> Dict:
        """
        Run full investigation with all features.

        Args:
            ticket_type: Type of issue (promotion, payment, vip, betting, login)
            webid: WebId for the site
            username: Player username
            customer_id: Optional CustomerId (auto-fetched if not provided)
            ticket_key: Optional Jira ticket key
            similar_tickets: Optional list of similar tickets from Jira search

        Returns:
            Full results dict with data, screenshots, analysis, and To CAS
        """
        if not self.config.is_authorized():
            return {"error": "No credentials. Run credential setup first."}

        if ticket_type not in self.queries:
            return {
                "error": f"Unknown type: {ticket_type}",
                "available_types": list(self.queries.keys())
            }

        playbook = self.queries[ticket_type]
        self.similar_tickets = similar_tickets or []

        self.results = {
            "ticket_key": ticket_key,
            "type": ticket_type,
            "playbook": playbook["name"],
            "webid": webid,
            "username": username,
            "timestamp": datetime.now().isoformat(),
            "queries": [],
            "screenshots": [],
            "errors_detected": [],
            "similar_tickets": self.similar_tickets[:5],
            "summary": None,
            "to_cas": None,
            "learned": False
        }

        print(f"\n{'='*60}")
        print(f"  INVESTIGATION: {playbook['name']}")
        print(f"{'='*60}")
        print(f"  Ticket: {ticket_key or 'N/A'}")
        print(f"  WebId: {webid} | User: {username}")
        if self.similar_tickets:
            print(f"  Similar tickets found: {len(self.similar_tickets)}")
        print(f"{'='*60}\n")

        artemis = Artemis(self.config, headless=self.headless)

        try:
            artemis.connect()

            # Get CustomerId if not provided
            if not customer_id:
                print("  Getting CustomerId...")
                customer_id = get_customer_id(artemis, webid, username)
                if customer_id:
                    print(f"  CustomerId: {customer_id}")
                else:
                    print("  Customer not found, some queries may fail")
                    customer_id = "0"

            params = {
                "webId": webid,
                "username": username,
                "customerId": customer_id
            }
            self.results["customer_id"] = customer_id

            # Run queries with error detection
            for i, q in enumerate(playbook["queries"]):
                print(f"\n  [{i+1}/{len(playbook['queries'])}] {q['name']}")

                sql = fill_params(q["sql"], params)
                result = artemis.query(sql, q["database"], use_cache=False)

                print(f"      Rows: {result['count']}")

                # Error detection on results
                if result.get("rows"):
                    query_type = q.get("id", q["name"]).lower()
                    analysis = analyze_results(query_type, result["rows"])

                    if analysis["has_issues"]:
                        print(f"      Detected: {analysis['summary']}")
                        self.errors_detected.extend(analysis["alerts"])
                        self.errors_detected.extend(analysis["warnings"])

                # Screenshot
                filename = f"{ticket_key or ticket_type}_{q['id']}_{datetime.now().strftime('%H%M%S')}.png"
                screenshot = artemis.screenshot(filename)
                print(f"      Screenshot: {filename}")

                self.results["queries"].append({
                    "id": q["id"],
                    "name": q["name"],
                    "database": q["database"],
                    "count": result["count"],
                    "data": result["rows"][:10],
                    "screenshot": screenshot,
                    "analysis": analysis if result.get("rows") else None
                })
                self.results["screenshots"].append(screenshot)

            # Store detected errors
            self.results["errors_detected"] = list(set(self.errors_detected))

            # Generate summary with all context
            self.results["summary"] = self._generate_summary(playbook)
            self.results["to_cas"] = self._generate_to_cas()

        finally:
            artemis.close()

        return self.results

    def _generate_summary(self, playbook: Dict) -> Dict:
        """Generate summary from results with error detection insights."""
        findings = []
        root_cause = None
        conclusion = None

        for q in self.results["queries"]:
            if not q["data"]:
                continue

            row = q["data"][0]

            # Check common fields
            if "RejectSetting" in row and row["RejectSetting"]:
                reject = row["RejectSetting"]
                flags = playbook.get("flags", {})
                meaning = flags.get(reject, reject)
                findings.append(f"Rejected: {meaning}")

                # Set root cause
                if reject == "SameFP":
                    root_cause = "Fingerprint conflict"
                    conclusion = "Player's device was used by another account"
                elif reject == "SameIP":
                    root_cause = "IP address conflict"
                    conclusion = "Multiple accounts from same IP"
                elif reject == "TurnoverNotMet":
                    root_cause = "Turnover requirement not met"
                    conclusion = "Player needs more wagering before claiming"

            if "TransactionStatus" in row:
                status = str(row["TransactionStatus"])
                brain = get_brain()
                meaning = brain.get_status_meaning("transaction", status)
                findings.append(f"Transaction: {meaning}")

                if status == "1":
                    root_cause = "Transaction pending"
                    conclusion = "Waiting for processing"
                elif status == "2":
                    root_cause = "Transaction rejected"
                    conclusion = "Payment was declined"

            if "VipLevel" in row:
                findings.append(f"VIP Level: {row['VipLevel']}")

            if "Result" in row and str(row["Result"]) != "Success":
                findings.append(f"Login Result: {row['Result']}")
                if "Suspended" in str(row.get("Result", "")):
                    root_cause = "Account suspended"
                    conclusion = "Contact support for account review"

        # Add error detection findings
        if self.errors_detected:
            findings.extend([f"DETECTED: {e}" for e in self.errors_detected[:3]])
            if not root_cause:
                root_cause = self.errors_detected[0]

        # Add similar ticket insights
        if self.similar_tickets:
            for t in self.similar_tickets[:2]:
                if t.get("resolution"):
                    findings.append(f"Similar: {t['key']} - {t.get('resolution', '')[:50]}")

        # Defaults
        if not root_cause:
            root_cause = f"Investigated {self.results['type']} issue"
        if not conclusion:
            conclusion = "See attached screenshots for details"

        return {
            "findings": findings,
            "root_cause": root_cause,
            "conclusion": conclusion,
            "query_count": len(self.results["queries"]),
            "total_rows": sum(q["count"] for q in self.results["queries"]),
            "errors_found": len(self.errors_detected)
        }

    def _generate_to_cas(self) -> str:
        """Generate To CAS response (500 char max)."""
        summary = self.results["summary"]
        return format_to_cas(
            root_cause=summary["root_cause"],
            conclusion=summary["conclusion"],
            details="; ".join(summary["findings"][:3])
        )

    def learn(self, root_cause: str = None, solution: str = None) -> Dict:
        """
        Learn from this investigation for future reference.

        Args:
            root_cause: Override root cause (uses detected if not provided)
            solution: Override solution (uses conclusion if not provided)

        Returns:
            Learning pattern that was saved
        """
        if not self.results:
            return {"error": "No investigation results to learn from"}

        summary = self.results["summary"]

        pattern = self.brain.learn_from_investigation(
            ticket_key=self.results.get("ticket_key", f"manual_{datetime.now().strftime('%Y%m%d%H%M')}"),
            ticket_type=self.results["type"],
            root_cause=root_cause or summary["root_cause"],
            solution=solution or summary["conclusion"],
            query_results=self.results["queries"]
        )

        self.results["learned"] = True
        self.learned = True

        print(f"\n  Learned pattern from investigation")
        print(f"  Total patterns: {self.brain.get_stats()['total_patterns']}")

        return pattern

    def get_common_causes(self, ticket_type: str) -> List[Dict]:
        """Get common causes from past investigations."""
        return self.brain.get_common_causes(ticket_type)

    def get_similar_solutions(self, ticket_type: str, indicators: Dict) -> List[Dict]:
        """Get similar solutions from past investigations."""
        return self.brain.get_similar_solutions(ticket_type, indicators)

    def format_jira_comment(self) -> str:
        """Format results as Jira comment."""
        if not self.results:
            return "No investigation results"

        return JiraCommentBuilder.build_investigation_comment(
            ticket_key=self.results.get("ticket_key", "N/A"),
            username=self.results["username"],
            webid=self.results["webid"],
            findings={f.split(': ')[0]: f.split(": ")[-1]
                      for f in self.results["summary"]["findings"][:5]},
            root_cause=self.results["summary"]["root_cause"],
            conclusion=self.results["summary"]["conclusion"],
            queries_run=[q["name"] for q in self.results["queries"]],
            similar_tickets=self.similar_tickets[:3]
        )


def main():
    parser = argparse.ArgumentParser(description='Enhanced Artemis Investigation')
    parser.add_argument('-t', '--type', help='Type: promotion, payment, vip, betting, login')
    parser.add_argument('-w', '--webid', help='WebId')
    parser.add_argument('-u', '--username', help='Username')
    parser.add_argument('-c', '--customerid', help='CustomerId (auto-fetched)')
    parser.add_argument('-k', '--ticket', help='Jira ticket key (e.g., TCP-12345)')
    parser.add_argument('--headless', action='store_true', help='No browser window')
    parser.add_argument('--learn', action='store_true', help='Save learning from this investigation')
    parser.add_argument('--show-common', help='Show common causes for type')
    parser.add_argument('-o', '--output', help='Output JSON file')

    args = parser.parse_args()

    investigation = Investigation(headless=args.headless)

    # Check credentials
    if not investigation.check_credentials():
        print("\nNo credentials found!")
        print("Please run: python core.py")
        print("Or use AskUserQuestion in Claude to set up credentials")
        sys.exit(1)

    # Show common causes mode
    if args.show_common:
        causes = investigation.get_common_causes(args.show_common)
        print(f"\nCommon causes for {args.show_common}:")
        for c in causes:
            print(f"  [{c['count']}x] {c['cause']}: {c['solution'][:50]}...")
        sys.exit(0)

    # Validate required args
    if not args.type or not args.webid or not args.username:
        print("\nError: --type, --webid, and --username are required")
        print("       Or use --ticket to auto-parse from Jira")
        parser.print_help()
        sys.exit(1)

    # Run investigation
    results = investigation.investigate(
        ticket_type=args.type,
        webid=args.webid,
        username=args.username,
        customer_id=args.customerid,
        ticket_key=args.ticket
    )

    if "error" in results:
        print(f"\nError: {results['error']}")
        sys.exit(1)

    # Print summary
    print(f"\n{'='*60}")
    print("  SUMMARY")
    print(f"{'='*60}")
    for f in results["summary"]["findings"]:
        print(f"    {f}")

    if results["errors_detected"]:
        print(f"\n  DETECTED ISSUES:")
        for e in results["errors_detected"][:5]:
            print(f"    {e}")

    print(f"\n{'='*60}")
    print("  TO CAS:")
    print(f"{'='*60}")
    print(f"  {results['to_cas']}")
    print(f"  ({len(results['to_cas'])} chars)")
    print(f"{'='*60}")

    print(f"\n  Screenshots: {len(results['screenshots'])}")
    for s in results["screenshots"]:
        print(f"    {s}")

    # Learn if requested
    if args.learn:
        pattern = investigation.learn()
        print(f"\n  Saved learning pattern: {pattern.get('ticket_key')}")

    # Save JSON
    if args.output:
        with open(args.output, 'w') as f:
            json.dump(results, f, indent=2, default=str)
        print(f"\n  Saved: {args.output}")

    # Show brain stats
    stats = investigation.brain.get_stats()
    print(f"\n  Brain: {stats['total_patterns']} patterns learned")


if __name__ == '__main__':
    main()
