#!/usr/bin/env python3
"""
Jira Integration Module - Auto-fetch tickets, search similar, comment results.
Uses MCP tools via Claude, but also provides direct parsing utilities.
"""

import re
import json
from datetime import datetime, timedelta
from typing import Dict, List, Optional, Tuple, Any
from pathlib import Path

SKILL_DIR = Path.home() / ".claude/skills/artemis-debug-secure"

# WebId mappings
WEBID_MAP = {
    "saffaluck": "20154", "sfl": "20154",
    "nocmakatiinc": "20107", "nmi": "20107", "nocmakati": "20107",
    "bet25": "20120",
    "gbw": "20109", "gbw777": "20109",
    "lucky7": "20132",
    "jilibet": "20140",
    "betso88": "20145",
    "peso88": "20150",
    "pnxbet": "20155",
    "22bet": "20160",
    "1xbet": "20165",
}

# Ticket type detection keywords
TYPE_KEYWORDS = {
    "promotion": ["promotion", "bonus", "reject", "fp", "fingerprint", "lucky wheel",
                  "cannot apply", "not received", "促销", "奖金"],
    "payment": ["deposit", "withdrawal", "pending", "payment", "not credited",
                "transfer", "bank", "存款", "提款", "支付"],
    "vip": ["vip", "upgrade", "level", "benefits", "downgrade", "vip bonus",
            "升级", "降级", "会员"],
    "betting": ["bet", "settlement", "winning", "void", "resettlement", "odds",
                "stake", "payout", "投注", "结算"],
    "login": ["login", "locked", "2fa", "suspended", "password", "access",
              "cannot login", "登录", "锁定"]
}


class TicketParser:
    """Parse Jira ticket descriptions to extract investigation parameters."""

    # Username patterns
    USERNAME_PATTERNS = [
        r'(?:username|user|player|account|玩家|账号)[:\s]*["\']?([A-Za-z0-9_]+)["\']?',
        r'(?:member|用户)[:\s]*["\']?([A-Za-z0-9_]+)["\']?',
        r'\b([A-Za-z][A-Za-z0-9_]{4,15})\b'  # Fallback: word that looks like username
    ]

    # Date patterns
    DATE_PATTERNS = [
        r'(\d{4}[-/]\d{1,2}[-/]\d{1,2})',  # 2025-01-05 or 2025/01/05
        r'(\d{1,2}[-/]\d{1,2}[-/]\d{4})',  # 05-01-2025 or 05/01/2025
        r'(\d{4}年\d{1,2}月\d{1,2}日)',     # 2025年1月5日
    ]

    # Amount patterns
    AMOUNT_PATTERNS = [
        r'(?:amount|金额)[:\s]*[\$₱]?([0-9,]+\.?\d*)',
        r'[\$₱]([0-9,]+\.?\d*)',
    ]

    @classmethod
    def parse_ticket(cls, summary: str, description: str) -> Dict[str, Any]:
        """
        Parse ticket summary and description to extract parameters.

        Returns:
            Dict with: username, webid, ticket_type, date_range, amount, raw_text
        """
        text = f"{summary}\n{description}".lower()
        result = {
            "username": None,
            "webid": None,
            "ticket_type": None,
            "date_start": None,
            "date_end": None,
            "amount": None,
            "confidence": {},
            "raw_text": text
        }

        # Extract username
        for pattern in cls.USERNAME_PATTERNS:
            match = re.search(pattern, text, re.IGNORECASE)
            if match:
                username = match.group(1)
                # Validate: not a common word
                if username.lower() not in ['the', 'and', 'for', 'this', 'that', 'with']:
                    result["username"] = username
                    result["confidence"]["username"] = "high" if pattern != cls.USERNAME_PATTERNS[-1] else "low"
                    break

        # Extract WebId from site name
        for site, webid in WEBID_MAP.items():
            if site in text:
                result["webid"] = webid
                result["confidence"]["webid"] = "high"
                break

        # Detect ticket type
        type_scores = {}
        for ttype, keywords in TYPE_KEYWORDS.items():
            score = sum(1 for kw in keywords if kw in text)
            if score > 0:
                type_scores[ttype] = score

        if type_scores:
            result["ticket_type"] = max(type_scores, key=type_scores.get)
            result["confidence"]["ticket_type"] = "high" if type_scores[result["ticket_type"]] >= 2 else "medium"

        # Extract dates
        for pattern in cls.DATE_PATTERNS:
            matches = re.findall(pattern, text)
            if matches:
                result["date_start"] = matches[0]
                result["date_end"] = matches[-1] if len(matches) > 1 else matches[0]
                break

        # Extract amount
        for pattern in cls.AMOUNT_PATTERNS:
            match = re.search(pattern, text)
            if match:
                result["amount"] = match.group(1).replace(",", "")
                break

        return result


class SimilarTicketSearch:
    """Search for similar past tickets and extract solutions."""

    @staticmethod
    def build_jql(ticket_type: str, webid: str = None,
                  keywords: List[str] = None, status: str = "Done") -> str:
        """
        Build JQL query for similar ticket search.

        Args:
            ticket_type: Type of issue (promotion, payment, vip, etc.)
            webid: Optional WebId to filter by site
            keywords: Additional keywords to search
            status: Ticket status filter

        Returns:
            JQL query string
        """
        conditions = [f"project = TCP"]

        # Add type keywords to summary search
        type_keywords = TYPE_KEYWORDS.get(ticket_type, [])[:3]  # Top 3 keywords
        if type_keywords:
            keyword_condition = " OR ".join([f'summary ~ "{kw}"' for kw in type_keywords])
            conditions.append(f"({keyword_condition})")

        # Add status
        if status:
            conditions.append(f"status = {status}")

        # Add date range (last 30 days)
        conditions.append("created >= -30d")

        # Order by most recent
        jql = " AND ".join(conditions) + " ORDER BY created DESC"

        return jql

    @staticmethod
    def extract_solution_from_comments(comments: List[Dict]) -> Optional[Dict]:
        """
        Extract solution pattern from ticket comments.

        Args:
            comments: List of comment dicts with 'body' field

        Returns:
            Dict with root_cause, solution, by, date or None
        """
        solution_indicators = [
            "root cause", "because", "due to", "reason",
            "solution", "resolved", "fixed", "done",
            "to cas", "response to customer"
        ]

        for comment in reversed(comments):  # Start from newest
            body = comment.get("body", "").lower()

            # Check if comment contains solution indicators
            if any(indicator in body for indicator in solution_indicators):
                return {
                    "root_cause": body[:200],  # First 200 chars
                    "by": comment.get("author", {}).get("displayName", "Unknown"),
                    "date": comment.get("created", ""),
                    "full_text": comment.get("body", "")
                }

        return None

    @staticmethod
    def rank_similar_tickets(tickets: List[Dict], current_params: Dict) -> List[Dict]:
        """
        Rank similar tickets by relevance.

        Args:
            tickets: List of ticket dicts from Jira search
            current_params: Parsed params from current ticket

        Returns:
            Sorted list with relevance scores
        """
        ranked = []

        for ticket in tickets:
            score = 0
            summary = ticket.get("fields", {}).get("summary", "").lower()

            # Score based on keyword matches
            if current_params.get("ticket_type"):
                keywords = TYPE_KEYWORDS.get(current_params["ticket_type"], [])
                score += sum(2 for kw in keywords if kw in summary)

            # Score based on username match
            if current_params.get("username"):
                if current_params["username"].lower() in summary:
                    score += 5

            # Prefer resolved tickets
            status = ticket.get("fields", {}).get("status", {}).get("name", "")
            if status.lower() in ["done", "closed", "resolved"]:
                score += 3

            ranked.append({
                "key": ticket.get("key"),
                "summary": ticket.get("fields", {}).get("summary"),
                "status": status,
                "created": ticket.get("fields", {}).get("created"),
                "relevance_score": score
            })

        # Sort by score descending
        ranked.sort(key=lambda x: x["relevance_score"], reverse=True)
        return ranked[:10]  # Top 10


class JiraCommentBuilder:
    """Build formatted comments for Jira."""

    @staticmethod
    def build_investigation_comment(
        ticket_key: str,
        username: str,
        webid: str,
        findings: Dict,
        root_cause: str,
        conclusion: str,
        queries_run: List[str],
        similar_tickets: List[Dict] = None
    ) -> str:
        """
        Build investigation result comment for Jira.

        Returns:
            Formatted comment string
        """
        now = datetime.now().strftime("%Y-%m-%d %H:%M")

        comment = f"""
*DATABASE INVESTIGATION RESULTS*
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
*Ticket:* {ticket_key}
*Player:* {username}
*WebId:* {webid}
*Investigated:* {now}

*Queries Executed:*
"""
        for q in queries_run:
            comment += f"• {q}\n"

        comment += f"""
*Key Findings:*
"""
        for key, value in findings.items():
            comment += f"• *{key}:* {value}\n"

        if similar_tickets:
            comment += f"""
*Similar Past Tickets:*
"""
            for t in similar_tickets[:3]:
                comment += f"• {t['key']}: {t['summary'][:50]}... ({t['status']})\n"

        comment += f"""
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
*ROOT CAUSE:* {root_cause}

*CONCLUSION:* {conclusion}
"""
        return comment

    @staticmethod
    def build_to_cas_response(root_cause: str, conclusion: str, details: str = "") -> str:
        """
        Build To CAS response (500 char max).

        Returns:
            Formatted string under 500 characters
        """
        response = f"{root_cause}. {conclusion}."
        if details:
            response += f" {details}"

        # Truncate to 500 chars
        if len(response) > 500:
            response = response[:497] + "..."

        return response


# Export for use in investigate.py
def parse_jira_ticket(summary: str, description: str) -> Dict:
    """Convenience function to parse a Jira ticket."""
    return TicketParser.parse_ticket(summary, description)


def build_similar_search_jql(ticket_type: str, webid: str = None) -> str:
    """Convenience function to build JQL for similar tickets."""
    return SimilarTicketSearch.build_jql(ticket_type, webid)


def format_investigation_comment(**kwargs) -> str:
    """Convenience function to format investigation comment."""
    return JiraCommentBuilder.build_investigation_comment(**kwargs)


def format_to_cas(root_cause: str, conclusion: str, details: str = "") -> str:
    """Convenience function to format To CAS response."""
    return JiraCommentBuilder.build_to_cas_response(root_cause, conclusion, details)


if __name__ == "__main__":
    # Test parsing
    test_summary = "Player Thuliswa2 on Saffaluck cannot apply promotion"
    test_desc = "Username: Thuliswa2\nDate: 2025-01-05\nPromotion bonus not showing"

    result = parse_jira_ticket(test_summary, test_desc)
    print("Parsed ticket:")
    print(json.dumps(result, indent=2, default=str))

    # Test JQL builder
    jql = build_similar_search_jql("promotion", "20154")
    print(f"\nJQL: {jql}")

    # Test To CAS formatter
    to_cas = format_to_cas(
        "Fingerprint conflict detected",
        "Player's device was previously used by another account",
        "FP hash: abc123..."
    )
    print(f"\nTo CAS ({len(to_cas)} chars):")
    print(to_cas)
