"""
Message Bus - Async communication between agents.
"""

import asyncio
import uuid
from datetime import datetime
from enum import Enum
from dataclasses import dataclass, field
from typing import Dict, Any, Optional, Callable, List


class MessageType(Enum):
    """Types of messages agents can send."""
    TASK_REQUEST = "task_request"
    TASK_RESULT = "task_result"
    CONTEXT_UPDATE = "context_update"
    AGENT_STATUS = "agent_status"
    ERROR = "error"
    BROADCAST = "broadcast"


@dataclass
class Message:
    """Message structure for inter-agent communication."""
    type: MessageType
    sender: str
    payload: Dict[str, Any]
    recipient: str = "broadcast"  # Agent name or "broadcast"
    id: str = field(default_factory=lambda: str(uuid.uuid4())[:8])
    timestamp: datetime = field(default_factory=datetime.now)
    correlation_id: Optional[str] = None  # For request-response pairing

    def reply(self, payload: Dict[str, Any], msg_type: MessageType = MessageType.TASK_RESULT) -> 'Message':
        """Create a reply message."""
        return Message(
            type=msg_type,
            sender=self.recipient,
            recipient=self.sender,
            payload=payload,
            correlation_id=self.id
        )


class MessageBus:
    """
    Async message bus for agent communication.
    Supports pub/sub and direct messaging.
    """

    def __init__(self):
        self._queues: Dict[str, asyncio.Queue] = {}
        self._subscribers: Dict[str, List[Callable]] = {}
        self._lock = asyncio.Lock()
        self._running = False

    async def register(self, agent_name: str) -> asyncio.Queue:
        """Register an agent and get its message queue."""
        async with self._lock:
            if agent_name not in self._queues:
                self._queues[agent_name] = asyncio.Queue()
            return self._queues[agent_name]

    async def unregister(self, agent_name: str):
        """Unregister an agent."""
        async with self._lock:
            if agent_name in self._queues:
                del self._queues[agent_name]
            if agent_name in self._subscribers:
                del self._subscribers[agent_name]

    async def publish(self, message: Message):
        """
        Publish a message.
        - If recipient is "broadcast", sends to all agents
        - Otherwise sends to specific agent
        """
        if message.recipient == "broadcast":
            # Broadcast to all
            async with self._lock:
                for name, queue in self._queues.items():
                    if name != message.sender:
                        await queue.put(message)
        else:
            # Direct message
            async with self._lock:
                if message.recipient in self._queues:
                    await self._queues[message.recipient].put(message)

    async def send(self, sender: str, recipient: str, msg_type: MessageType, payload: Dict[str, Any]) -> Message:
        """Convenience method to create and send a message."""
        msg = Message(
            type=msg_type,
            sender=sender,
            recipient=recipient,
            payload=payload
        )
        await self.publish(msg)
        return msg

    async def request(self, sender: str, recipient: str, payload: Dict[str, Any],
                      timeout: float = 30.0) -> Optional[Message]:
        """
        Send a request and wait for response.
        Returns the response message or None on timeout.
        """
        # Create request message
        request = Message(
            type=MessageType.TASK_REQUEST,
            sender=sender,
            recipient=recipient,
            payload=payload
        )
        await self.publish(request)

        # Wait for response with matching correlation_id
        sender_queue = self._queues.get(sender)
        if not sender_queue:
            return None

        try:
            start = asyncio.get_event_loop().time()
            while True:
                remaining = timeout - (asyncio.get_event_loop().time() - start)
                if remaining <= 0:
                    return None

                try:
                    msg = await asyncio.wait_for(sender_queue.get(), timeout=remaining)
                    if msg.correlation_id == request.id:
                        return msg
                    else:
                        # Not our response, put it back
                        await sender_queue.put(msg)
                except asyncio.TimeoutError:
                    return None
        except Exception:
            return None

    async def receive(self, agent_name: str, timeout: Optional[float] = None) -> Optional[Message]:
        """Receive a message for an agent."""
        queue = self._queues.get(agent_name)
        if not queue:
            return None

        try:
            if timeout:
                return await asyncio.wait_for(queue.get(), timeout=timeout)
            else:
                return await queue.get()
        except asyncio.TimeoutError:
            return None

    def subscribe(self, agent_name: str, msg_type: MessageType, callback: Callable):
        """Subscribe to a message type with a callback."""
        key = f"{agent_name}:{msg_type.value}"
        if key not in self._subscribers:
            self._subscribers[key] = []
        self._subscribers[key].append(callback)

    async def start(self):
        """Start the message bus."""
        self._running = True

    async def stop(self):
        """Stop the message bus."""
        self._running = False
        # Clear all queues
        async with self._lock:
            for queue in self._queues.values():
                while not queue.empty():
                    try:
                        queue.get_nowait()
                    except asyncio.QueueEmpty:
                        break

    def get_stats(self) -> Dict[str, Any]:
        """Get bus statistics."""
        return {
            "registered_agents": list(self._queues.keys()),
            "queue_sizes": {name: q.qsize() for name, q in self._queues.items()},
            "subscribers": len(self._subscribers)
        }
