"""
Investigation Context - Shared state between agents.
"""

import asyncio
from datetime import datetime
from dataclasses import dataclass, field
from typing import Dict, List, Optional, Any


@dataclass
class QueryResult:
    """Result from a database query."""
    id: str
    name: str
    database: str
    sql: str
    count: int = 0
    rows: List[Dict] = field(default_factory=list)
    screenshot: Optional[str] = None
    error: Optional[str] = None
    duration_ms: int = 0


@dataclass
class SimilarTicket:
    """A similar past ticket."""
    key: str
    summary: str
    status: str
    resolution: Optional[str] = None
    relevance_score: int = 0


@dataclass
class AnalysisResult:
    """Result from analysis agent."""
    findings: List[str] = field(default_factory=list)
    errors: List[Dict] = field(default_factory=list)
    root_cause: str = "Unknown"
    conclusion: str = "Manual review required"
    confidence: str = "low"  # low, medium, high


@dataclass
class InvestigationContext:
    """
    Shared context for an investigation.
    Thread-safe access via asyncio.Lock.
    """
    # Request parameters
    ticket_key: Optional[str] = None
    ticket_type: str = ""
    webid: str = ""
    username: str = ""
    customer_id: Optional[str] = None

    # Jira data
    jira_summary: Optional[str] = None
    jira_description: Optional[str] = None

    # Intermediate results
    similar_tickets: List[SimilarTicket] = field(default_factory=list)
    query_results: List[QueryResult] = field(default_factory=list)
    screenshots: List[str] = field(default_factory=list)
    analysis: Optional[AnalysisResult] = None

    # Learning data
    common_causes: List[Dict] = field(default_factory=list)
    similar_solutions: List[Dict] = field(default_factory=list)

    # Final output
    summary: Optional[Dict] = None
    to_cas: Optional[str] = None
    jira_comment: Optional[str] = None

    # Metadata
    start_time: datetime = field(default_factory=datetime.now)
    end_time: Optional[datetime] = None
    errors: List[str] = field(default_factory=list)
    phase: str = "init"

    # Lock for thread-safe access
    _lock: asyncio.Lock = field(default_factory=asyncio.Lock, repr=False)

    async def update(self, **kwargs):
        """Thread-safe update of context fields."""
        async with self._lock:
            for key, value in kwargs.items():
                if hasattr(self, key):
                    setattr(self, key, value)

    async def add_query_result(self, result: QueryResult):
        """Add a query result."""
        async with self._lock:
            self.query_results.append(result)
            if result.screenshot:
                self.screenshots.append(result.screenshot)

    async def add_error(self, error: str):
        """Add an error."""
        async with self._lock:
            self.errors.append(error)

    async def set_phase(self, phase: str):
        """Set current phase."""
        async with self._lock:
            self.phase = phase

    async def set_similar_tickets(self, tickets: List):
        """Set similar tickets."""
        async with self._lock:
            self.similar_tickets = tickets

    async def set_analysis(self, analysis: AnalysisResult):
        """Set analysis result."""
        async with self._lock:
            self.analysis = analysis

    async def finalize(self):
        """Mark investigation as complete."""
        async with self._lock:
            self.end_time = datetime.now()
            self.phase = "complete"

    def duration_seconds(self) -> float:
        """Get investigation duration in seconds."""
        end = self.end_time or datetime.now()
        return (end - self.start_time).total_seconds()

    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary for serialization."""
        return {
            "ticket_key": self.ticket_key,
            "ticket_type": self.ticket_type,
            "webid": self.webid,
            "username": self.username,
            "customer_id": self.customer_id,
            "query_count": len(self.query_results),
            "screenshot_count": len(self.screenshots),
            "similar_tickets": len(self.similar_tickets),
            "errors": self.errors,
            "phase": self.phase,
            "duration_seconds": self.duration_seconds(),
            "to_cas": self.to_cas,
            "summary": self.summary
        }


class ContextManager:
    """
    Manages multiple investigation contexts.
    Allows running multiple investigations in parallel.
    """

    def __init__(self):
        self._contexts: Dict[str, InvestigationContext] = {}
        self._lock = asyncio.Lock()

    async def create(self, investigation_id: str, **kwargs) -> InvestigationContext:
        """Create a new investigation context."""
        async with self._lock:
            ctx = InvestigationContext(**kwargs)
            self._contexts[investigation_id] = ctx
            return ctx

    async def get(self, investigation_id: str) -> Optional[InvestigationContext]:
        """Get an investigation context."""
        return self._contexts.get(investigation_id)

    async def remove(self, investigation_id: str):
        """Remove an investigation context."""
        async with self._lock:
            if investigation_id in self._contexts:
                del self._contexts[investigation_id]

    async def list_active(self) -> List[str]:
        """List active investigation IDs."""
        return [
            id for id, ctx in self._contexts.items()
            if ctx.phase != "complete"
        ]
