"""
Connection Pool - Reusable browser sessions for Artemis.
"""

import asyncio
import time
from datetime import datetime
from typing import Optional, Dict, Any
from dataclasses import dataclass, field
from pathlib import Path

# Will use playwright async API
try:
    from playwright.async_api import async_playwright, Browser, Page, BrowserContext
    PLAYWRIGHT_AVAILABLE = True
except ImportError:
    PLAYWRIGHT_AVAILABLE = False


@dataclass
class BrowserConnection:
    """A single browser connection."""
    id: int
    browser: Any  # Browser
    context: Any  # BrowserContext
    page: Any  # Page
    created_at: datetime = field(default_factory=datetime.now)
    last_used: datetime = field(default_factory=datetime.now)
    query_count: int = 0
    logged_in: bool = False

    def is_expired(self, max_age_seconds: int) -> bool:
        """Check if connection is too old."""
        age = (datetime.now() - self.created_at).total_seconds()
        return age > max_age_seconds

    def is_stale(self, idle_seconds: int) -> bool:
        """Check if connection has been idle too long."""
        idle = (datetime.now() - self.last_used).total_seconds()
        return idle > idle_seconds

    async def refresh(self):
        """Refresh the connection (navigate to dashboard)."""
        try:
            await self.page.goto('https://artemis.568winex.com')
            await self.page.wait_for_load_state('networkidle')
            self.last_used = datetime.now()
        except Exception:
            self.logged_in = False

    async def close(self):
        """Close the connection."""
        try:
            await self.context.close()
            await self.browser.close()
        except Exception:
            pass


class ConnectionPool:
    """
    Pool of browser connections for Artemis.
    Reuses logged-in sessions for performance.
    """

    def __init__(self, size: int = 3, max_age_seconds: int = 300, idle_seconds: int = 120):
        """
        Initialize pool.

        Args:
            size: Maximum number of connections
            max_age_seconds: Max age before refresh (5 min default)
            idle_seconds: Max idle time before refresh (2 min default)
        """
        self.size = size
        self.max_age = max_age_seconds
        self.idle_timeout = idle_seconds
        self._pool: asyncio.Queue = asyncio.Queue(maxsize=size)
        self._all_connections: list = []
        self._lock = asyncio.Lock()
        self._initialized = False
        self._credentials: Dict[str, str] = {}
        self._playwright = None
        self._next_id = 0

    async def initialize(self, credentials: Dict[str, str], headless: bool = False):
        """
        Initialize pool with credentials.
        Pre-creates and logs in all connections.

        Args:
            credentials: Dict with 'url', 'username', 'password'
            headless: Run browsers in headless mode
        """
        if not PLAYWRIGHT_AVAILABLE:
            raise RuntimeError("Playwright not installed. Run: pip install playwright && playwright install")

        self._credentials = credentials
        self._playwright = await async_playwright().start()

        # Create connections with progress
        for i in range(self.size):
            print(f"  [Pool] Creating connection {i+1}/{self.size}...", flush=True)
            conn = await self._create_connection(headless)
            if conn:
                await self._pool.put(conn)
                self._all_connections.append(conn)
                print(f"  [Pool] Connection {i+1} ready ✓", flush=True)
            else:
                print(f"  [Pool] Connection {i+1} FAILED ✗", flush=True)

        self._initialized = True

    async def _create_connection(self, headless: bool = False) -> Optional[BrowserConnection]:
        """Create a new logged-in connection."""
        try:
            print(f"    → Launching browser...", flush=True)
            browser = await self._playwright.chromium.launch(headless=headless)
            context = await browser.new_context(viewport={'width': 1400, 'height': 900})
            page = await context.new_page()

            # Navigate to Artemis
            print(f"    → Navigating to Artemis...", flush=True)
            await page.goto(self._credentials.get('url', 'https://artemis.568winex.com'))
            await page.wait_for_load_state('networkidle')

            # Login
            print(f"    → Logging in as {self._credentials['username']}...", flush=True)
            await page.fill('input[type="text"]', self._credentials['username'])
            await page.fill('input[type="password"]', self._credentials['password'])

            # Try multiple selectors for login button
            login_selectors = [
                'button[type="submit"]',
                'button:has-text("Login")',
                'button:has-text("Sign in")',
                'input[type="submit"]',
                'button.btn-primary',
                'button'
            ]

            clicked = False
            for selector in login_selectors:
                try:
                    btn = page.locator(selector).first
                    if await btn.is_visible(timeout=2000):
                        await btn.click()
                        clicked = True
                        break
                except Exception:
                    continue

            if not clicked:
                # Fallback: press Enter
                await page.keyboard.press('Enter')

            await page.wait_for_load_state('networkidle')
            await page.wait_for_timeout(1000)

            self._next_id += 1
            conn = BrowserConnection(
                id=self._next_id,
                browser=browser,
                context=context,
                page=page,
                logged_in=True
            )
            print(f"    → Login successful!", flush=True)

            # Navigate to Query page if not already there
            print(f"    → Navigating to Query page...", flush=True)
            try:
                # Click Query menu item
                query_link = page.locator('a:text("Query"), a[href*="Query"]').first
                if await query_link.is_visible(timeout=3000):
                    await query_link.click()
                    await page.wait_for_load_state('networkidle')
                    print(f"    → On Query page ✓", flush=True)
            except Exception:
                # Try JavaScript approach
                await page.evaluate('''() => {
                    const links = document.querySelectorAll('a');
                    for (const a of links) {
                        if (a.textContent.trim() === 'Query') {
                            a.click();
                            return;
                        }
                    }
                }''')
                await page.wait_for_load_state('networkidle')

            return conn

        except Exception as e:
            print(f"    ✗ Failed: {e}", flush=True)
            return None

    async def acquire(self, timeout: float = 30.0) -> Optional[BrowserConnection]:
        """
        Acquire a connection from the pool.
        Blocks until one is available or timeout.

        Args:
            timeout: Max seconds to wait

        Returns:
            BrowserConnection or None on timeout
        """
        if not self._initialized:
            raise RuntimeError("Pool not initialized. Call initialize() first.")

        try:
            conn = await asyncio.wait_for(self._pool.get(), timeout=timeout)

            # Check if connection needs refresh
            if conn.is_expired(self.max_age) or conn.is_stale(self.idle_timeout):
                await conn.refresh()

            conn.last_used = datetime.now()
            conn.query_count += 1
            return conn

        except asyncio.TimeoutError:
            return None

    async def release(self, conn: BrowserConnection):
        """Release a connection back to the pool."""
        if conn and conn.logged_in:
            conn.last_used = datetime.now()
            await self._pool.put(conn)

    async def execute_query(self, conn: BrowserConnection, sql: str, database: str,
                             customer: str = "YY3_Ns_Prod") -> Dict[str, Any]:
        """
        Execute a query using a connection.

        Args:
            conn: Browser connection
            sql: SQL query
            database: Database name (e.g., Main, VIP, Promotion)
            customer: Customer/server name (default: YY3_Ns_Prod)

        Returns:
            Dict with 'count', 'rows', 'error'
        """
        try:
            page = conn.page
            print(f"    [Debug] execute_query: db={database}, customer={customer}", flush=True)

            # Find all select elements and their IDs for debugging
            selects_info = await page.evaluate('''() => {
                const selects = document.querySelectorAll('select');
                return Array.from(selects).map(s => ({
                    id: s.id,
                    name: s.name,
                    optionCount: s.options.length,
                    firstOptions: Array.from(s.options).slice(0, 3).map(o => o.text)
                }));
            }''')
            print(f"    [Debug] Found selects: {selects_info}", flush=True)

            # Select Customer (server) - try multiple selector patterns
            customer_selected = False
            customer_selectors = ['#CustomerList', '#Customer', 'select[name*="Customer"]', 'select:first-of-type']

            for selector in customer_selectors:
                try:
                    select = page.locator(selector).first
                    if await select.is_visible(timeout=1000):
                        options = await page.evaluate(f'''() => {{
                            const sel = document.querySelector('{selector}');
                            return sel ? Array.from(sel.options).map(o => ({{value: o.value, text: o.text}})) : [];
                        }}''')
                        print(f"    [Debug] {selector} options: {options[:5]}", flush=True)

                        # Find matching option
                        for opt in options:
                            if customer.lower() in opt['text'].lower() or customer.lower() in opt['value'].lower():
                                await page.select_option(selector, opt['value'])
                                await page.wait_for_timeout(1000)  # Wait for DB dropdown to populate
                                customer_selected = True
                                print(f"    [Debug] Selected customer: {opt['text']}", flush=True)
                                break
                        if customer_selected:
                            break
                except Exception as e:
                    continue

            if not customer_selected:
                print(f"    [Debug] WARNING: Could not select customer dropdown", flush=True)

            # Select Database - try multiple selector patterns
            db_selected = False
            db_selectors = ['#DatabaseList', '#Database', 'select[name*="Database"]', 'select[name*="DataBase"]']

            for selector in db_selectors:
                try:
                    select = page.locator(selector).first
                    if await select.is_visible(timeout=1000):
                        options = await page.evaluate(f'''() => {{
                            const sel = document.querySelector('{selector}');
                            return sel ? Array.from(sel.options).map(o => ({{value: o.value, text: o.text}})) : [];
                        }}''')
                        print(f"    [Debug] {selector} options: {options[:5]}", flush=True)

                        # Find matching option
                        for opt in options:
                            if database.lower() in opt['text'].lower() or database.lower() in opt['value'].lower():
                                await page.select_option(selector, opt['value'])
                                await page.wait_for_timeout(500)
                                db_selected = True
                                print(f"    [Debug] Selected database: {opt['text']}", flush=True)
                                break
                        if db_selected:
                            break
                except Exception as e:
                    continue

            if not db_selected:
                print(f"    [Debug] WARNING: Could not select database dropdown", flush=True)

            # Enter SQL in CodeMirror
            print(f"    [Debug] Entering SQL ({len(sql)} chars)...", flush=True)
            cm_result = await page.evaluate(f'''() => {{
                const cm = document.querySelector('.CodeMirror')?.CodeMirror;
                if (cm) {{
                    cm.setValue({repr(sql)});
                    return 'SQL entered, length: ' + cm.getValue().length;
                }}
                return 'CodeMirror not found';
            }}''')
            print(f"    [Debug] CodeMirror: {cm_result}", flush=True)

            # Take pre-submit screenshot
            pre_submit_path = str(Path.home() / '.playwright-mcp' / 'debug_pre_submit.png')
            await page.screenshot(path=pre_submit_path, full_page=True)
            print(f"    [Debug] Pre-submit screenshot: {pre_submit_path}", flush=True)

            # Click Submit button
            print(f"    [Debug] Looking for Submit button...", flush=True)

            # List ALL clickable elements for debugging (buttons, inputs, anchors with btn class)
            clickables_info = await page.evaluate('''() => {
                const elements = document.querySelectorAll('button, input[type="submit"], input[type="button"], a.btn, [role="button"]');
                return Array.from(elements).map(el => ({
                    tag: el.tagName,
                    text: el.textContent?.trim() || el.value || '',
                    type: el.type || '',
                    class: el.className,
                    id: el.id
                }));
            }''')
            print(f"    [Debug] Found clickables: {clickables_info}", flush=True)

            # Click Submit using comprehensive search
            submit_clicked = await page.evaluate('''() => {
                // 1. Try input[type="submit"] with value="Submit"
                const submitInputs = document.querySelectorAll('input[type="submit"]');
                for (const inp of submitInputs) {
                    if (inp.value === 'Submit') {
                        inp.click();
                        return 'clicked input[type=submit]:Submit';
                    }
                }

                // 2. Try button with exact "Submit" text
                const buttons = document.querySelectorAll('button');
                for (const btn of buttons) {
                    if (btn.textContent.trim() === 'Submit') {
                        btn.click();
                        return 'clicked button:Submit';
                    }
                }

                // 3. Try anchor tags styled as buttons
                const anchors = document.querySelectorAll('a.btn, a[role="button"]');
                for (const a of anchors) {
                    if (a.textContent.trim() === 'Submit') {
                        a.click();
                        return 'clicked anchor:Submit';
                    }
                }

                // 4. Try any element with btn class containing Submit text
                const btnElements = document.querySelectorAll('.btn');
                for (const el of btnElements) {
                    if (el.textContent.trim() === 'Submit') {
                        el.click();
                        return 'clicked .btn element:Submit - tag:' + el.tagName;
                    }
                }

                // 5. Try input[type="button"] with value="Submit"
                const buttonInputs = document.querySelectorAll('input[type="button"]');
                for (const inp of buttonInputs) {
                    if (inp.value === 'Submit') {
                        inp.click();
                        return 'clicked input[type=button]:Submit';
                    }
                }

                // 6. Fallback: Find element by ID patterns
                const byId = document.querySelector('#Submit, #submitBtn, #btnSubmit');
                if (byId) {
                    byId.click();
                    return 'clicked by ID: ' + byId.id;
                }

                return 'Submit not found after comprehensive search';
            }''')
            print(f"    [Debug] Submit click result: {submit_clicked}", flush=True)

            # If still not found, try Playwright native locators
            if 'not found' in submit_clicked:
                print(f"    [Debug] Trying Playwright locators...", flush=True)
                try:
                    # Try multiple Playwright locator patterns
                    locators = [
                        page.locator('input[value="Submit"]').first,
                        page.locator('button:text-is("Submit")').first,
                        page.locator('.btn:text-is("Submit")').first,
                        page.get_by_role('button', name='Submit'),
                    ]
                    for loc in locators:
                        try:
                            if await loc.is_visible(timeout=1000):
                                await loc.click()
                                print(f"    [Debug] Clicked via Playwright locator", flush=True)
                                submit_clicked = 'clicked via Playwright'
                                break
                        except Exception:
                            continue
                except Exception as e:
                    print(f"    [Debug] Playwright click failed: {e}", flush=True)

            # Wait for query to complete - check for loading indicator to disappear
            print(f"    [Debug] Waiting for query to complete...", flush=True)

            # Wait for any loading indicators to disappear
            try:
                # Wait for "Loading..." text to disappear
                await page.wait_for_function('''() => {
                    const loading = document.body.innerText.includes('Loading...');
                    return !loading;
                }''', timeout=15000)
            except Exception:
                pass  # Continue if timeout

            # Additional wait for network to settle
            await page.wait_for_load_state('networkidle')
            await page.wait_for_timeout(1500)  # Extra buffer for results to render

            # Wait for result table to appear (look for "Query Result" section)
            try:
                await page.wait_for_function('''() => {
                    // Check if results section appeared
                    const hasResults = document.body.innerText.includes('Query Result') ||
                                      document.querySelector('table tr td');
                    return hasResults;
                }''', timeout=10000)
            except Exception:
                print(f"    [Debug] Results section not found, continuing...", flush=True)

            # Scroll down to ensure results are visible
            await page.evaluate('window.scrollTo(0, document.body.scrollHeight)')
            await page.wait_for_timeout(500)  # Brief wait after scroll

            # Take full-page debug screenshot
            debug_path = str(Path.home() / '.playwright-mcp' / 'debug_query.png')
            await page.screenshot(path=debug_path, full_page=True)
            print(f"    [Debug] Full-page screenshot saved to: {debug_path}", flush=True)

            # Parse results - find the query results table (not form tables)
            rows = await page.evaluate('''() => {
                // Look specifically for the query results table
                // It should be near the "Query Result" header or have data columns like Id, CustomerId, etc.

                // First, try to find the results section by looking for "Query Result" text
                let resultSection = null;
                const allText = document.body.innerText;

                // Find all tables
                const allTables = document.querySelectorAll('table');
                let resultTable = null;

                for (const table of allTables) {
                    // Skip tables that are likely form elements
                    const tableText = table.innerText || '';

                    // Skip if this table contains form elements like "Save Query List" or "Customer List"
                    if (tableText.includes('Save Query List') ||
                        tableText.includes('Customer List') ||
                        tableText.includes('DataBase List') ||
                        tableText.includes('-- select')) {
                        continue;
                    }

                    // Skip tables inside select elements or with select children
                    if (table.querySelector('select') || table.closest('select')) {
                        continue;
                    }

                    // Check if this table has real data columns (like Id, CustomerId, CreatedOn)
                    const rows = table.querySelectorAll('tr');
                    if (rows.length < 2) continue;  // Need at least header + 1 data row

                    const firstRowCells = rows[0].querySelectorAll('th, td');
                    const cellTexts = Array.from(firstRowCells).map(c => c.textContent.trim().toLowerCase());

                    // Look for database-like column names
                    const dbColumns = ['id', 'customerid', 'username', 'createdon', 'modifiedon', 'webid', 'status'];
                    const hasDbColumns = cellTexts.some(t => dbColumns.some(col => t.includes(col)));

                    if (hasDbColumns) {
                        resultTable = table;
                        break;
                    }

                    // Also check if table has many columns (likely results, not form)
                    if (firstRowCells.length >= 4 && rows.length >= 2) {
                        // Make sure it's not a form layout table
                        const hasInputs = table.querySelectorAll('input, select, textarea').length > 0;
                        if (!hasInputs) {
                            resultTable = table;
                            break;
                        }
                    }
                }

                if (!resultTable) {
                    console.log('No result table found');
                    return [];
                }

                // Get headers from first row
                const headerRow = resultTable.querySelector('tr');
                const headers = Array.from(headerRow.querySelectorAll('th, td')).map(el => el.textContent.trim());
                console.log('Headers found:', headers);

                // Get data rows (skip header row)
                const rows = [];
                const allRows = resultTable.querySelectorAll('tr');
                for (let i = 1; i < allRows.length; i++) {
                    const tr = allRows[i];
                    const row = {};
                    const cells = tr.querySelectorAll('td');
                    cells.forEach((td, j) => {
                        const header = headers[j] || `col${j}`;
                        row[header] = td.textContent.trim();
                    });
                    if (Object.keys(row).length > 0 && Object.values(row).some(v => v)) {
                        rows.push(row);
                    }
                }

                console.log('Rows found:', rows.length);
                return rows;
            }''')

            return {
                'count': len(rows),
                'rows': rows,
                'error': None
            }

        except Exception as e:
            return {
                'count': 0,
                'rows': [],
                'error': str(e)
            }

    async def take_screenshot(self, conn: BrowserConnection, filename: str,
                               output_dir: str = None) -> str:
        """Take a screenshot."""
        if output_dir is None:
            output_dir = str(Path.home() / '.playwright-mcp')

        Path(output_dir).mkdir(parents=True, exist_ok=True)
        filepath = str(Path(output_dir) / filename)

        await conn.page.screenshot(path=filepath, full_page=False)
        return filepath

    async def shutdown(self):
        """Shutdown all connections and the pool."""
        async with self._lock:
            # Close all connections
            for conn in self._all_connections:
                await conn.close()

            self._all_connections.clear()
            self._initialized = False

            # Clear queue
            while not self._pool.empty():
                try:
                    self._pool.get_nowait()
                except asyncio.QueueEmpty:
                    break

            # Stop playwright
            if self._playwright:
                await self._playwright.stop()
                self._playwright = None

    def get_stats(self) -> Dict[str, Any]:
        """Get pool statistics."""
        return {
            "size": self.size,
            "available": self._pool.qsize(),
            "in_use": self.size - self._pool.qsize(),
            "total_connections": len(self._all_connections),
            "initialized": self._initialized
        }
