"""
Result Store - Aggregates results from all agents.
"""

import asyncio
from datetime import datetime
from typing import Dict, Any, List, Optional
from dataclasses import dataclass, field


@dataclass
class AgentResult:
    """Result from an agent task."""
    agent: str
    task: str
    success: bool
    data: Any = None
    error: Optional[str] = None
    timestamp: datetime = field(default_factory=datetime.now)
    duration_ms: int = 0


class ResultStore:
    """
    Thread-safe storage for investigation results.
    Aggregates results from all agents.
    """

    def __init__(self):
        self._results: Dict[str, List[AgentResult]] = {}
        self._data: Dict[str, Any] = {}
        self._lock = asyncio.Lock()

    async def store_result(self, investigation_id: str, result: AgentResult):
        """Store an agent result."""
        async with self._lock:
            if investigation_id not in self._results:
                self._results[investigation_id] = []
            self._results[investigation_id].append(result)

    async def store(self, investigation_id: str, key: str, value: Any):
        """Store arbitrary data."""
        async with self._lock:
            if investigation_id not in self._data:
                self._data[investigation_id] = {}
            self._data[investigation_id][key] = value

    async def get(self, investigation_id: str, key: str) -> Optional[Any]:
        """Get stored data."""
        data = self._data.get(investigation_id, {})
        return data.get(key)

    async def get_all(self, investigation_id: str) -> Dict[str, Any]:
        """Get all data for an investigation."""
        return self._data.get(investigation_id, {})

    async def get_results(self, investigation_id: str) -> List[AgentResult]:
        """Get all agent results for an investigation."""
        return self._results.get(investigation_id, [])

    async def get_results_by_agent(self, investigation_id: str, agent: str) -> List[AgentResult]:
        """Get results from a specific agent."""
        results = self._results.get(investigation_id, [])
        return [r for r in results if r.agent == agent]

    async def get_errors(self, investigation_id: str) -> List[str]:
        """Get all errors."""
        results = self._results.get(investigation_id, [])
        return [r.error for r in results if r.error]

    async def clear(self, investigation_id: str):
        """Clear results for an investigation."""
        async with self._lock:
            if investigation_id in self._results:
                del self._results[investigation_id]
            if investigation_id in self._data:
                del self._data[investigation_id]

    async def get_summary(self, investigation_id: str) -> Dict[str, Any]:
        """Get summary of results."""
        results = self._results.get(investigation_id, [])
        data = self._data.get(investigation_id, {})

        return {
            "total_tasks": len(results),
            "successful": sum(1 for r in results if r.success),
            "failed": sum(1 for r in results if not r.success),
            "agents": list(set(r.agent for r in results)),
            "data_keys": list(data.keys()),
            "errors": [r.error for r in results if r.error]
        }

    def get_stats(self) -> Dict[str, Any]:
        """Get store statistics."""
        return {
            "investigations": len(self._results),
            "total_results": sum(len(r) for r in self._results.values()),
            "data_entries": sum(len(d) for d in self._data.values())
        }
