#!/usr/bin/env python3
"""
Swarm Investigation - Multi-Agent parallel investigation system.

Usage:
    python swarm_investigate.py -t promotion -w 20154 -u "player"
    python swarm_investigate.py --ticket TCP-12345
    python swarm_investigate.py -t vip -w 20154 -u "player" --benchmark
"""

import argparse
import asyncio
import json
import os
import sys
from datetime import datetime
from pathlib import Path
from typing import Dict, Any, Optional

# Add parent to path for imports
sys.path.insert(0, str(Path(__file__).parent))

from swarm.bus import MessageBus
from swarm.store import ResultStore
from swarm.pool import ConnectionPool
from swarm.context import InvestigationContext
from agents.coordinator import CoordinatorAgent

SKILL_DIR = Path.home() / ".claude/skills/artemis-debug-secure"
CREDENTIALS_FILE = SKILL_DIR / ".credentials"
OUTPUT_DIR = Path.home() / ".playwright-mcp"


def log(msg: str, icon: str = "▶"):
    """Print timestamped log message."""
    ts = datetime.now().strftime("%H:%M:%S")
    print(f"[{ts}] {icon} {msg}", flush=True)


def load_credentials() -> Dict[str, str]:
    """Load Artemis credentials from file or environment."""
    # Try environment first
    if os.environ.get("ARTEMIS_USER") and os.environ.get("ARTEMIS_PASS"):
        return {
            "url": os.environ.get("ARTEMIS_URL", "https://artemis.568winex.com"),
            "username": os.environ["ARTEMIS_USER"],
            "password": os.environ["ARTEMIS_PASS"]
        }

    # Try credentials file
    if CREDENTIALS_FILE.exists():
        creds = {}
        with open(CREDENTIALS_FILE) as f:
            for line in f:
                line = line.strip()
                if "=" in line and not line.startswith("#"):
                    key, value = line.split("=", 1)
                    creds[key.strip()] = value.strip()

        if creds.get("ARTEMIS_USER") and creds.get("ARTEMIS_PASS"):
            return {
                "url": creds.get("ARTEMIS_URL", "https://artemis.568winex.com"),
                "username": creds["ARTEMIS_USER"],
                "password": creds["ARTEMIS_PASS"]
            }

    return {}


def print_banner():
    """Print swarm banner."""
    print("""
╔═══════════════════════════════════════════════════════════════╗
║           ARTEMIS SWARM INVESTIGATION SYSTEM                  ║
║              Multi-Agent Parallel Execution                   ║
╚═══════════════════════════════════════════════════════════════╝
""")


def print_phase(phase: str, status: str = ""):
    """Print phase indicator."""
    icons = {
        "init": "🔄",
        "data": "📊",
        "analysis": "🔍",
        "output": "📝",
        "complete": "✅",
        "error": "❌"
    }
    icon = icons.get(phase, "▶")
    print(f"\n{icon} Phase: {phase.upper()} {status}")


def print_results(results: Dict[str, Any]):
    """Print investigation results."""
    print("\n" + "═" * 60)
    print("INVESTIGATION RESULTS")
    print("═" * 60)

    if results.get("success"):
        print(f"\n✅ Status: SUCCESS")
        print(f"⏱  Duration: {results.get('duration_ms', 0)}ms")
        print(f"📊 Queries: {results.get('queries_executed', 0)}")
        print(f"📸 Screenshots: {len(results.get('screenshots', []))}")
        print(f"🎯 Confidence: {results.get('confidence', 'unknown')}")

        print(f"\n{'─' * 60}")
        print("ROOT CAUSE:")
        print(f"  {results.get('root_cause', 'Unknown')}")

        print(f"\n{'─' * 60}")
        print("CONCLUSION:")
        print(f"  {results.get('conclusion', 'N/A')}")

        if results.get("findings"):
            print(f"\n{'─' * 60}")
            print("KEY FINDINGS:")
            for i, f in enumerate(results["findings"][:5], 1):
                print(f"  {i}. {f}")

        if results.get("errors"):
            print(f"\n{'─' * 60}")
            print("ERRORS DETECTED:")
            for e in results["errors"][:3]:
                print(f"  • {e.get('cause', 'Unknown')}")

        if results.get("to_cas"):
            print(f"\n{'─' * 60}")
            print("TO CAS RESPONSE:")
            print(f"  {results['to_cas']}")

        if results.get("screenshots"):
            print(f"\n{'─' * 60}")
            print("SCREENSHOTS:")
            for s in results["screenshots"]:
                print(f"  📸 {s}")

    else:
        print(f"\n❌ Status: FAILED")
        print(f"Error: {results.get('error', 'Unknown error')}")
        print(f"Phase: {results.get('phase', 'unknown')}")

    print("\n" + "═" * 60)


async def run_benchmark(coordinator: CoordinatorAgent, context: InvestigationContext,
                        iterations: int = 3) -> Dict[str, Any]:
    """Run benchmark to measure performance."""
    print(f"\n🏁 Running benchmark ({iterations} iterations)...")

    times = []
    for i in range(iterations):
        start = datetime.now()
        await coordinator.execute("investigate", context=context)
        duration = (datetime.now() - start).total_seconds() * 1000
        times.append(duration)
        print(f"  Iteration {i+1}: {duration:.0f}ms")

    avg = sum(times) / len(times)
    min_time = min(times)
    max_time = max(times)

    print(f"\n📊 Benchmark Results:")
    print(f"  Average: {avg:.0f}ms")
    print(f"  Min: {min_time:.0f}ms")
    print(f"  Max: {max_time:.0f}ms")

    return {
        "iterations": iterations,
        "average_ms": avg,
        "min_ms": min_time,
        "max_ms": max_time,
        "times": times
    }


async def main():
    """Main entry point."""
    parser = argparse.ArgumentParser(
        description="Swarm Investigation - Multi-Agent parallel system"
    )

    parser.add_argument("-t", "--type", dest="ticket_type",
                        choices=["promotion", "vip", "payment", "betting", "login"],
                        help="Investigation type")
    parser.add_argument("-w", "--webid", help="WebId (e.g., 20154)")
    parser.add_argument("-u", "--username", help="Player username")
    parser.add_argument("--ticket", help="Jira ticket key (e.g., TCP-12345)")
    parser.add_argument("--customer-id", dest="customer_id", help="Customer ID (optional)")
    parser.add_argument("--headless", action="store_true", help="Run browsers in headless mode")
    parser.add_argument("--pool-size", type=int, default=1, help="Connection pool size (default: 1)")
    parser.add_argument("--benchmark", action="store_true", help="Run performance benchmark")
    parser.add_argument("--json", action="store_true", help="Output results as JSON")

    args = parser.parse_args()

    # Validate arguments
    if not args.ticket and (not args.ticket_type or not args.webid or not args.username):
        parser.error("Either --ticket OR (--type, --webid, --username) required")

    if not args.json:
        print_banner()

    # Load credentials
    credentials = load_credentials()
    if not credentials:
        log("No credentials found!", "❌")
        print("   Set ARTEMIS_USER and ARTEMIS_PASS environment variables")
        print("   Or create ~/.claude/skills/artemis-debug-secure/.credentials")
        sys.exit(1)

    # Create infrastructure
    bus = MessageBus()
    store = ResultStore()
    pool = ConnectionPool(size=args.pool_size)

    try:
        # Initialize pool
        if not args.json:
            log("Initializing browser...", "🔌")

        await pool.initialize(credentials, headless=args.headless)

        if not args.json:
            log(f"Browser ready (pool={args.pool_size})", "✅")

        # Create context
        context = InvestigationContext(
            ticket_key=args.ticket,
            ticket_type=args.ticket_type or "promotion",
            webid=args.webid or "",
            username=args.username or "",
            customer_id=args.customer_id
        )

        # Create coordinator
        coordinator = CoordinatorAgent(bus, store, pool)
        await coordinator.start()
        coordinator.set_context(context)

        if args.benchmark:
            # Run benchmark
            results = await run_benchmark(coordinator, context)
        else:
            # Run investigation
            if not args.json:
                log("Starting investigation...", "🚀")

            result = await coordinator.execute("investigate", context=context)
            results = result.data if result.success else {"success": False, "error": result.error}

        # Output results
        if args.json:
            print(json.dumps(results, indent=2, default=str))
        else:
            print_results(results)

        # Stop coordinator
        await coordinator.stop()

    except KeyboardInterrupt:
        log("Investigation cancelled", "⚠️")
    except Exception as e:
        log(f"Error: {e}", "❌")
        if not args.json:
            import traceback
            traceback.print_exc()
    finally:
        # Shutdown pool
        if not args.json:
            log("Shutting down...", "🔌")
        await pool.shutdown()


if __name__ == "__main__":
    asyncio.run(main())
