# Azure AD CLI Command Reference

This reference documents the `az ad` commands used for user and group management.

## Prerequisites

### Azure CLI Installation

```bash
# macOS
brew install azure-cli

# Linux (Ubuntu/Debian)
curl -sL https://aka.ms/InstallAzureCLIDeb | sudo bash

# Windows
winget install Microsoft.AzureCLI
```

### Authentication

```bash
# Interactive login
az login

# Login to specific tenant
az login --tenant TENANT_ID

# Check current session
az account show
```

## User Commands

### az ad user list

List users in the directory.

```bash
# Basic usage
az ad user list

# With output format
az ad user list -o table
az ad user list -o json

# With query (JMESPath)
az ad user list --query "[].{name:displayName, email:mail}"

# With OData filter
az ad user list --filter "startswith(displayName,'John')"
az ad user list --filter "userType eq 'Guest'"
az ad user list --filter "mail eq 'user@example.com'"
az ad user list --filter "accountEnabled eq true"
```

**Common Filters:**
| Filter | Description |
|--------|-------------|
| `startswith(displayName,'X')` | Name starts with X |
| `userType eq 'Guest'` | Guest users only |
| `userType eq 'Member'` | Member users only |
| `accountEnabled eq true` | Enabled accounts |
| `accountEnabled eq false` | Disabled accounts |

### az ad user show

Get details of a specific user.

```bash
# By email/UPN
az ad user show --id "user@example.com"

# By object ID
az ad user show --id "00000000-0000-0000-0000-000000000000"

# Specific properties
az ad user show --id "user@example.com" --query "{name:displayName, id:id}"
```

### az ad user get-member-groups

Get groups the user belongs to.

```bash
# Basic usage
az ad user get-member-groups --id "user@example.com"

# Security groups only
az ad user get-member-groups --id "user@example.com" --security-enabled-only true
```

## Group Commands

### az ad group list

List groups in the directory.

```bash
# Basic usage
az ad group list

# Security groups only
az ad group list --filter "securityEnabled eq true"

# Microsoft 365 groups
az ad group list --filter "groupTypes/any(c:c eq 'Unified')"

# Search by name
az ad group list --filter "startswith(displayName,'Engineering')"
```

### az ad group show

Get details of a specific group.

```bash
# By name
az ad group show --group "GroupName"

# By object ID
az ad group show --group "00000000-0000-0000-0000-000000000000"
```

### az ad group member list

List members of a group.

```bash
az ad group member list --group "GroupName"
az ad group member list --group "GroupName" -o table
```

### az ad group member add

Add a user to a group.

```bash
az ad group member add --group "GROUP_ID" --member-id "USER_ID"
```

**Note:** Both group and member-id must be object IDs, not display names.

### az ad group member remove

Remove a user from a group.

```bash
az ad group member remove --group "GROUP_ID" --member-id "USER_ID"
```

### az ad group member check

Check if a user is a member of a group.

```bash
az ad group member check --group "GROUP_ID" --member-id "USER_ID"
# Returns: {"value": true} or {"value": false}
```

## Graph API Commands

For operations not directly supported by `az ad`, use `az rest` with Microsoft Graph API.

### Invite Guest User

```bash
az rest --method POST \
  --url "https://graph.microsoft.com/v1.0/invitations" \
  --headers "Content-Type=application/json" \
  --body '{
    "invitedUserEmailAddress": "guest@external.com",
    "inviteRedirectUrl": "https://myapps.microsoft.com",
    "sendInvitationMessage": true
  }'
```

### Get Transitive Memberships

```bash
az rest --method GET \
  --url "https://graph.microsoft.com/v1.0/users/USER_ID/transitiveMemberOf" \
  -o json
```

### Get Directory Role Assignments

```bash
az rest --method GET \
  --url "https://graph.microsoft.com/v1.0/users/USER_ID/memberOf" \
  --query "value[?@odata.type=='#microsoft.graph.directoryRole']"
```

## Output Formats

| Format | Flag | Use Case |
|--------|------|----------|
| JSON | `-o json` | Scripting, full details |
| Table | `-o table` | Human readable |
| TSV | `-o tsv` | Single values for scripts |
| YAML | `-o yaml` | Configuration files |
| JSONC | `-o jsonc` | Colorized JSON (terminal) |

## JMESPath Queries

Common query patterns:

```bash
# Select specific fields
--query "[].{name:displayName, email:mail}"

# Filter results
--query "[?userType=='Guest']"

# Count results
--query "length(@)"

# First result only
--query "[0]"

# Combine filter and select
--query "[?userType=='Guest'].{name:displayName, email:mail}"
```

## Required Permissions

| Operation | Graph Permission | Azure AD Role |
|-----------|-----------------|---------------|
| List users | User.Read.All | Directory Readers |
| List groups | Group.Read.All | Directory Readers |
| Get user details | User.Read.All | Directory Readers |
| Get group members | GroupMember.Read.All | Directory Readers |
| Add to group | GroupMember.ReadWrite.All | Groups Administrator |
| Invite guests | User.Invite.All | Guest Inviter |
| Full user management | User.ReadWrite.All | User Administrator |
