# Azure AD Troubleshooting Guide

Common issues and solutions when managing Azure AD users and groups.

## Authentication Issues

### AADSTS50076 - MFA Required

**Error:**
```
AADSTS50076: Due to a configuration change made by your administrator,
or because you moved to a new location, you must use multi-factor authentication to access...
```

**Solution:**
```bash
# Re-authenticate with fresh credentials
az logout
az login --tenant YOUR_TENANT_ID
```

### AADSTS50079 - User Must Enroll MFA

**Error:**
```
AADSTS50079: Due to a configuration change made by your administrator,
you must enroll in multi-factor authentication.
```

**Solution:**
1. Go to https://aka.ms/mfasetup
2. Complete MFA enrollment
3. Re-run `az login`

### AADSTS700016 - Application Not Found

**Error:**
```
AADSTS700016: Application with identifier was not found in the directory
```

**Solution:**
```bash
# Use device code flow
az login --use-device-code
```

## Permission Issues

### Authorization_RequestDenied

**Error:**
```
Authorization_RequestDenied: Insufficient privileges to complete the operation.
```

**Causes & Solutions:**

| Operation | Required Role |
|-----------|--------------|
| List users | Directory Readers |
| List groups | Directory Readers |
| Add to group | Groups Administrator |
| Invite guests | Guest Inviter |
| Create users | User Administrator |

**Check your roles:**
```bash
# Get your role assignments
az rest --method GET \
  --url "https://graph.microsoft.com/v1.0/me/memberOf" \
  --query "value[?@odata.type=='#microsoft.graph.directoryRole'].displayName" \
  -o tsv
```

### Request_ResourceNotFound

**Error:**
```
Request_ResourceNotFound: Resource 'user@example.com' does not exist
```

**Solutions:**
1. Verify the user/group exists:
   ```bash
   az ad user list --filter "mail eq 'user@example.com'"
   ```
2. Check for typos in the email/UPN
3. Use object ID instead of email

### Request_BadRequest - Invalid Filter

**Error:**
```
Request_BadRequest: Invalid filter clause
```

**Common Causes:**

1. **Unsupported property:**
   ```bash
   # Wrong - email is not filterable
   az ad user list --filter "email eq 'x'"

   # Correct - use mail
   az ad user list --filter "mail eq 'x'"
   ```

2. **Missing quotes:**
   ```bash
   # Wrong
   az ad user list --filter "displayName eq John"

   # Correct
   az ad user list --filter "displayName eq 'John'"
   ```

3. **Case sensitivity:**
   ```bash
   # OData functions are case-sensitive
   # Wrong
   az ad user list --filter "StartsWith(displayName,'J')"

   # Correct
   az ad user list --filter "startswith(displayName,'J')"
   ```

## Group Operations Issues

### Cannot Add User to Group

**Error:**
```
One or more added object references already exist
```

**Solution:**
The user is already a member. Check first:
```bash
GROUP_ID=$(az ad group show --group "GroupName" --query "id" -o tsv)
USER_ID=$(az ad user show --id "user@example.com" --query "id" -o tsv)
az ad group member check --group "$GROUP_ID" --member-id "$USER_ID"
```

### Group Not Found

**Error:**
```
Resource 'GroupName' does not exist
```

**Solutions:**
1. List groups to find exact name:
   ```bash
   az ad group list --query "[].displayName" -o tsv | grep -i "partial"
   ```
2. Use object ID instead of display name

### Cannot Remove Last Owner

**Error:**
```
Cannot remove the last owner of a group
```

**Solution:**
Add another owner before removing:
```bash
az ad group owner add --group "GROUP_ID" --owner-object-id "NEW_OWNER_ID"
```

## Guest Invitation Issues

### Invitation Not Sent

**Error:**
Invitation created but email not received.

**Solutions:**
1. Check spam/junk folder
2. Verify email address is correct
3. Check if `sendInvitationMessage` is `true`:
   ```bash
   az rest --method POST \
     --url "https://graph.microsoft.com/v1.0/invitations" \
     --body '{"invitedUserEmailAddress": "guest@ext.com", "inviteRedirectUrl": "https://myapps.microsoft.com", "sendInvitationMessage": true}'
   ```

### Guest Already Exists

**Error:**
```
Invitation failed because the invitee's email address belongs to an existing user
```

**Solution:**
The user already has an account. Find them:
```bash
az ad user list --filter "mail eq 'guest@external.com'" -o table
```

### External Collaboration Blocked

**Error:**
```
External collaboration is blocked for your organization
```

**Solution:**
Contact your Azure AD administrator to enable B2B collaboration in:
- Azure Portal > Azure Active Directory > External Identities > External collaboration settings

## Performance Issues

### Slow Queries on Large Tenants

**Solutions:**

1. Use filters to reduce results:
   ```bash
   az ad user list --filter "startswith(displayName,'A')"
   ```

2. Select only needed fields:
   ```bash
   az ad user list --query "[].{name:displayName, id:id}" -o table
   ```

3. Use `$top` for pagination (via Graph API):
   ```bash
   az rest --method GET \
     --url "https://graph.microsoft.com/v1.0/users?\$top=100"
   ```

### Timeout Errors

**Solution:**
```bash
# Increase timeout
az config set core.http_timeout=300
```

## Debugging

### Enable Debug Output

```bash
az ad user list --debug
```

### Check API Version

```bash
az version
```

### View Raw API Response

```bash
az ad user show --id "user@example.com" -o json --debug 2>&1 | grep "Response content"
```

## Getting Help

### Azure CLI Documentation
```bash
az ad user --help
az ad group --help
az ad user list --help
```

### Microsoft Graph API Reference
- Users: https://learn.microsoft.com/graph/api/resources/user
- Groups: https://learn.microsoft.com/graph/api/resources/group
- Invitations: https://learn.microsoft.com/graph/api/resources/invitation
