#!/usr/bin/env -S uv run --script
# /// script
# requires-python = ">=3.11"
# dependencies = [
#     "httpx",
#     "click",
#     "rich",
#     "azure-identity",
# ]
# ///
"""
Azure AD Guest User Invitation

Invite external guest users to an Azure AD tenant using Microsoft Graph API.

Usage:
    uv run invite_guest.py invite --email EMAIL [--redirect-url URL] [--send-email] [--message MSG]
    uv run invite_guest.py check

Authentication:
    Uses DefaultAzureCredential which automatically uses:
    1. Environment variables (AZURE_CLIENT_ID, AZURE_CLIENT_SECRET, AZURE_TENANT_ID)
    2. Azure CLI login (`az login`)
    3. Managed Identity (in Azure environments)

Required Permissions:
    User.Invite.All (delegated or application)
    User must have: Guest Inviter, Directory Writer, or User Administrator role
"""

import json
import re
import sys

import click
import httpx
from azure.identity import DefaultAzureCredential
from rich.console import Console

console = Console(stderr=True)

GRAPH_API_BASE = "https://graph.microsoft.com/v1.0"
GRAPH_SCOPE = "https://graph.microsoft.com/.default"
DEFAULT_REDIRECT_URL = "https://myapps.microsoft.com"


def is_valid_email(email: str) -> bool:
    """Validate email format."""
    pattern = r"^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$"
    return bool(re.match(pattern, email))


def get_access_token() -> str:
    """Get access token using DefaultAzureCredential."""
    credential = DefaultAzureCredential()
    token = credential.get_token(GRAPH_SCOPE)
    return token.token


def check_user_exists(token: str, email: str) -> dict | None:
    """Check if a user with this email already exists in the tenant."""
    url = f"{GRAPH_API_BASE}/users"
    headers = {
        "Authorization": f"Bearer {token}",
        "Content-Type": "application/json",
    }
    params = {
        "$filter": f"mail eq '{email}' or otherMails/any(x:x eq '{email}')",
        "$select": "id,displayName,mail,userType",
    }

    with httpx.Client() as client:
        response = client.get(url, headers=headers, params=params)
        if response.status_code == 200:
            data = response.json()
            users = data.get("value", [])
            if users:
                return users[0]
    return None


def invite_guest_user(
    token: str,
    email: str,
    redirect_url: str = DEFAULT_REDIRECT_URL,
    send_invitation: bool = True,
    custom_message: str | None = None,
) -> dict:
    """Invite a guest user to the tenant."""
    url = f"{GRAPH_API_BASE}/invitations"

    headers = {
        "Authorization": f"Bearer {token}",
        "Content-Type": "application/json",
    }

    body = {
        "invitedUserEmailAddress": email,
        "inviteRedirectUrl": redirect_url,
        "sendInvitationMessage": send_invitation,
    }

    if custom_message:
        body["invitedUserMessageInfo"] = {
            "customizedMessageBody": custom_message,
        }

    with httpx.Client() as client:
        response = client.post(url, headers=headers, json=body)

        if response.status_code == 201:
            data = response.json()
            return {
                "success": True,
                "invitation_id": data.get("id"),
                "email": data.get("invitedUserEmailAddress"),
                "status": data.get("status"),
                "redeem_url": data.get("inviteRedeemUrl"),
                "user_id": data.get("invitedUser", {}).get("id"),
            }
        elif response.status_code == 400:
            error_data = response.json()
            error_msg = error_data.get("error", {}).get("message", "Bad request")
            return {"success": False, "error": "bad_request", "message": error_msg}
        elif response.status_code == 403:
            return {
                "success": False,
                "error": "permission_denied",
                "message": "Insufficient permissions. Requires User.Invite.All and Guest Inviter role.",
            }
        else:
            response.raise_for_status()
            return {"success": False, "error": "unknown", "message": "Unexpected response"}


@click.group()
def cli():
    """Azure AD Guest Invitation - Invite external users to your tenant."""
    pass


@cli.command()
@click.option("--email", required=True, help="Email address of the user to invite")
@click.option(
    "--redirect-url",
    default=DEFAULT_REDIRECT_URL,
    help=f"URL to redirect user after accepting (default: {DEFAULT_REDIRECT_URL})",
)
@click.option("--send-email/--no-send-email", default=True, help="Send invitation email (default: yes)")
@click.option("--message", default=None, help="Custom message to include in invitation email")
@click.option("--json", "output_json", is_flag=True, help="Output as JSON")
def invite(email: str, redirect_url: str, send_email: bool, message: str | None, output_json: bool):
    """Invite a guest user to the Azure AD tenant."""
    # Validate email format
    if not is_valid_email(email):
        result = {"success": False, "error": "invalid_email", "message": f"Invalid email format: {email}"}
        if output_json:
            print(json.dumps(result))
        else:
            console.print(f"[red]✗[/red] Invalid email format: {email}")
        sys.exit(1)

    if not output_json:
        console.print(f"\n[bold blue]Inviting Guest User[/bold blue]")
        console.print(f"Email: {email}")
        console.print(f"Redirect URL: {redirect_url}")
        console.print(f"Send email: {'Yes' if send_email else 'No'}")
        if message:
            console.print(f"Custom message: {message[:50]}...")
        console.print("")

    try:
        # Get access token
        if not output_json:
            with console.status("[bold green]Authenticating..."):
                token = get_access_token()
            console.print("[green]✓[/green] Authenticated successfully")
        else:
            token = get_access_token()

        # Check if user already exists
        if not output_json:
            with console.status("[bold green]Checking if user exists..."):
                existing_user = check_user_exists(token, email)
        else:
            existing_user = check_user_exists(token, email)

        if existing_user:
            result = {
                "success": False,
                "error": "user_exists",
                "message": f"User {email} already exists in tenant",
                "existing_user": {
                    "id": existing_user.get("id"),
                    "displayName": existing_user.get("displayName"),
                    "userType": existing_user.get("userType"),
                },
            }
            if output_json:
                print(json.dumps(result))
            else:
                console.print(f"[yellow]⚠[/yellow] User already exists: {existing_user.get('displayName')} ({existing_user.get('userType')})")
            return

        # Send invitation
        if not output_json:
            with console.status("[bold green]Sending invitation..."):
                result = invite_guest_user(token, email, redirect_url, send_email, message)
        else:
            result = invite_guest_user(token, email, redirect_url, send_email, message)

        if output_json:
            print(json.dumps(result))
        else:
            if result["success"]:
                console.print(f"[green]✓[/green] Invitation sent successfully!")
                console.print(f"  Status: {result.get('status')}")
                console.print(f"  User ID: {result.get('user_id')}")
                if not send_email:
                    console.print(f"  Redeem URL: {result.get('redeem_url')}")
            else:
                console.print(f"[red]✗[/red] {result.get('message')}")
                sys.exit(1)

    except Exception as e:
        result = {"success": False, "error": "auth_failed", "message": str(e)}
        if output_json:
            print(json.dumps(result))
        else:
            console.print(f"[red]✗[/red] Authentication failed: {e}")
            console.print("\n[dim]Ensure you are logged in with 'az login' and have User.Invite.All permission[/dim]")
        sys.exit(1)


@cli.command()
@click.option("--json", "output_json", is_flag=True, help="Output as JSON")
def check(output_json: bool):
    """Check authentication and permissions for guest invitations."""
    if not output_json:
        console.print("\n[bold blue]Checking Guest Invitation Permissions[/bold blue]\n")

    try:
        # Test authentication
        if not output_json:
            with console.status("[bold green]Testing authentication..."):
                token = get_access_token()
            console.print("[green]✓[/green] DefaultAzureCredential authentication successful")
        else:
            token = get_access_token()

        # Test Graph API access by getting current user
        url = f"{GRAPH_API_BASE}/me"
        headers = {"Authorization": f"Bearer {token}"}

        with httpx.Client() as client:
            response = client.get(url, headers=headers)

        if response.status_code == 200:
            user_data = response.json()
            result = {
                "success": True,
                "authenticated": True,
                "user": {
                    "displayName": user_data.get("displayName"),
                    "mail": user_data.get("mail") or user_data.get("userPrincipalName"),
                },
            }
            if output_json:
                print(json.dumps(result))
            else:
                console.print("[green]✓[/green] Graph API access verified")
                console.print(f"  Logged in as: {user_data.get('displayName')}")
                console.print(f"  Email: {user_data.get('mail') or user_data.get('userPrincipalName')}")
                console.print("\n[dim]Note: Actual invitation requires User.Invite.All permission and Guest Inviter role[/dim]")
        else:
            result = {
                "success": False,
                "authenticated": True,
                "graph_access": False,
                "error": f"Graph API returned {response.status_code}",
            }
            if output_json:
                print(json.dumps(result))
            else:
                console.print(f"[yellow]⚠[/yellow] Graph API returned: {response.status_code}")
            sys.exit(1)

    except Exception as e:
        result = {"success": False, "authenticated": False, "error": str(e)}
        if output_json:
            print(json.dumps(result))
        else:
            console.print(f"[red]✗[/red] Authentication failed: {e}")
            console.print("\n[dim]Run 'az login' to authenticate with Azure CLI[/dim]")
        sys.exit(1)


if __name__ == "__main__":
    cli()
