---
name: azure-osdu
description: Queries OSDU users and entitlements via the OSDU Entitlements API. Use when listing OSDU users, checking roles (Viewer, Editor, Admin, Ops), resolving GUIDs to names, or troubleshooting OSDU access.
---

# OSDU Entitlements Operations

Reference documentation for querying OSDU users and entitlements via the OSDU Entitlements API.

## Prerequisites

Verify environment variables are set before any operation:

```bash
echo "AI_OSDU_HOST: ${AI_OSDU_HOST:-NOT SET}"
echo "AI_OSDU_DATA_PARTITION: ${AI_OSDU_DATA_PARTITION:-NOT SET}"
echo "AI_OSDU_CLIENT: ${AI_OSDU_CLIENT:-NOT SET}"
echo "AI_OSDU_SECRET: ${AI_OSDU_SECRET:+SET}"
echo "AI_OSDU_TENANT_ID: ${AI_OSDU_TENANT_ID:-NOT SET}"
```

Test configuration and authentication:

```bash
uv run .claude/skills/azure-osdu/scripts/osdu_manage.py check
```

## Quick Reference

### List Users and Roles

```bash
# List all OSDU users with their roles
uv run .claude/skills/azure-osdu/scripts/osdu_users.py

# JSON output for processing
uv run .claude/skills/azure-osdu/scripts/osdu_users.py --json

# Filter to specific role
uv run .claude/skills/azure-osdu/scripts/osdu_users.py --role Admin

# Skip GUID resolution (faster)
uv run .claude/skills/azure-osdu/scripts/osdu_users.py --no-resolve
```

### Manage User Roles

```bash
# Add user as Viewer
uv run .claude/skills/azure-osdu/scripts/osdu_manage.py add \
  --user "user@example.com" --role Viewer

# Add user as Editor with OWNER rights
uv run .claude/skills/azure-osdu/scripts/osdu_manage.py add \
  --user "user@example.com" --role Editor --as-owner

# Remove from specific role
uv run .claude/skills/azure-osdu/scripts/osdu_manage.py remove \
  --user "user@example.com" --role Viewer

# Remove from all roles
uv run .claude/skills/azure-osdu/scripts/osdu_manage.py remove \
  --user "user@example.com" --all-roles
```

### Owner Management

Users in OSDU groups have one of two membership types:
- **OWNER**: Can add/remove members from the group (management capability)
- **MEMBER**: Has access granted by the group (no management capability)

```bash
# Change user from MEMBER to OWNER (promote)
uv run .claude/skills/azure-osdu/scripts/osdu_manage.py update \
  --user "user@example.com" --role Editor --to OWNER

# Change user from OWNER to MEMBER (demote)
uv run .claude/skills/azure-osdu/scripts/osdu_manage.py update \
  --user "user@example.com" --role Editor --to MEMBER

# Remove an OWNER (blocked by default if last owner)
uv run .claude/skills/azure-osdu/scripts/osdu_manage.py remove \
  --user "owner@example.com" --role Admin

# Force removal of last OWNER (use with caution!)
uv run .claude/skills/azure-osdu/scripts/osdu_manage.py remove \
  --user "owner@example.com" --role Admin --force
```

**Last-Owner Protection**: By default, the script prevents removing or demoting the last OWNER of a group to avoid orphaning it. Use `--force` to override this protection when necessary.

## GUID Resolution

OSDU stores users as GUIDs. The `osdu_users.py` script automatically resolves GUIDs to human-readable names via Azure AD.

### How It Works

1. **Check cache** - Session-scoped cache prevents repeated lookups
2. **Try user** - Look up in Azure AD users
3. **Try service principal** - Check Azure AD service principals (prefixed with `[SP]`)
4. **Try application** - Check Azure AD applications (prefixed with `[App]`)

### Requirements

- Azure CLI authenticated: `az login`
- Permission to read Azure AD objects

### Performance

**GUID Resolution Performance:**
- With `--no-resolve`: ~2 seconds (fast)
- With resolution: ~20-30 seconds (parallel resolution of ~150 GUIDs)

The script uses **parallel GUID resolution** with 10 concurrent workers to minimize lookup time.

Use `--no-resolve` flag when:
- Processing large user lists
- Cross-referencing with Azure AD (you already have names from tenant)
- Running in `/audit` workflows (names come from tenant lookup)
- Azure CLI not available
- Speed is more important than readability

```bash
# Fast mode without GUID resolution
uv run .claude/skills/azure-osdu/scripts/osdu_users.py --no-resolve

# For audit workflows, always use --no-resolve with --json
uv run .claude/skills/azure-osdu/scripts/osdu_users.py --json --no-resolve
```

## Role Groups

| Role | Group Pattern | Access Level |
|------|---------------|--------------|
| Viewer | `users.datalake.viewers@{partition}.{domain}` | Read-only |
| Editor | `users.datalake.editors@{partition}.{domain}` | Read-write |
| Admin | `users.datalake.admins@{partition}.{domain}` | Administrative |
| Ops | `users.datalake.ops@{partition}.{domain}` | Operations |
| Root | `users.data.root@{partition}.{domain}` | Superuser |

## Output Formats

| Format | Flag | Use Case |
|--------|------|----------|
| Table | (default) | Human readable with colors |
| JSON | `--json` | Automation and processing |

### JSON Output Structure

```json
{
  "success": true,
  "host": "osdu.energy.azure.com",
  "partition": "opendes",
  "total_users": 5,
  "users": [
    {
      "email": "user@example.com",
      "display_name": "John Doe",
      "is_root": false,
      "roles": {"Viewer": "MEMBER", "Editor": "OWNER"}
    }
  ]
}
```

## Environment Variables

| Variable | Required | Description |
|----------|----------|-------------|
| `AI_OSDU_HOST` | Yes | OSDU instance hostname |
| `AI_OSDU_DATA_PARTITION` | Yes | Data partition ID |
| `AI_OSDU_CLIENT` | Yes | App registration client ID |
| `AI_OSDU_SECRET` | Yes | App registration secret |
| `AI_OSDU_TENANT_ID` | Yes | Azure AD tenant ID |
| `AI_OSDU_DOMAIN` | No | Entitlements domain (default: `dataservices.energy`) |

## Error Handling

| Error | Cause | Solution |
|-------|-------|----------|
| 401 Unauthorized | Invalid/expired token | Check AI_OSDU_SECRET, re-authenticate |
| 403 Forbidden | Missing API permissions | Add Entitlements API permissions in OSDU |
| 404 Not Found | Group/user not found | Verify group name format and partition ID |
| 409 Conflict | User already in group | User is already a member (not an error) |
| Last owner | Removing/demoting last OWNER | Add another OWNER first, or use `--force` |
| Missing env vars | Not configured | Set required AI_OSDU_* environment variables |
| GUID not resolved | Azure AD issue | Check `az login`, try `--no-resolve` |

## Best Practices

1. **Verify environment** - Run `check` command before operations
2. **Start with Viewer** - Assign least privilege, escalate as needed
3. **Use email addresses** - Easier to track than GUIDs
4. **Use --no-resolve for bulk** - Faster for large user lists
5. **Check Azure CLI auth** - GUID resolution requires `az login`

## Reference Files

- [reference/entitlements.md](reference/entitlements.md) - Full API reference
- [reference/troubleshooting.md](reference/troubleshooting.md) - Detailed error solutions
- [scripts/osdu_users.py](scripts/osdu_users.py) - User listing script
- [scripts/osdu_manage.py](scripts/osdu_manage.py) - User management script
