# OSDU Entitlements Reference

OSDU uses the OSDU Entitlements service for access control.

## GUID Resolution

OSDU stores user identities as GUIDs (UUIDs). To convert these to human-readable names, the `osdu_users.py` script queries Azure AD.

### Resolution Flow

```
GUID detected → Cache check → User lookup → SP lookup → App lookup → Cache result
```

### Identity Types

| Prefix | Type | Example |
|--------|------|---------|
| (none) | User | `John Doe` |
| `[SP]` | Service Principal | `[SP] My Managed Identity` |
| `[App]` | Application | `[App] My App Registration` |

### Requirements

- Azure CLI authenticated: `az login`
- Permissions: Read access to Azure AD users/apps

### Cross-Skill Integration

GUID resolution uses the same Azure CLI auth as the azure-ad skill. If you can query Azure AD users directly, GUID resolution will work.

```bash
# Test Azure AD access
az ad user show --id "YOUR_GUID" --query "displayName" -o tsv
```

## Entitlement Groups

### Standard Role Groups

| Group Pattern | Purpose |
|--------------|---------|
| `users.datalake.viewers@{partition}.{domain}` | Read-only access to data |
| `users.datalake.editors@{partition}.{domain}` | Read-write access to data |
| `users.datalake.admins@{partition}.{domain}` | Administrative access |
| `users.datalake.ops@{partition}.{domain}` | Operations and maintenance |
| `users.data.root@{partition}.{domain}` | Superuser access (use sparingly) |

### Group Email Format

```
{group-type}@{data-partition}.{domain}

Example:
users.datalake.editors@opendes.dataservices.energy
```

## API Endpoints

Base URL: `https://{host}/api/entitlements/v2`

### Authentication

```bash
# Get OAuth token (v1 endpoint)
POST https://login.microsoftonline.com/{tenant}/oauth2/token
Content-Type: application/x-www-form-urlencoded

grant_type=client_credentials
client_id={client_id}
client_secret={client_secret}
resource={client_id}
```

### List Group Members

```bash
GET /groups/{group-email}/members
Headers:
  Authorization: Bearer {token}
  data-partition-id: {partition}
  Accept: application/json

Response:
{
  "members": [
    {"email": "user@example.com", "role": "MEMBER"},
    {"email": "admin@example.com", "role": "OWNER"}
  ]
}
```

### Add Group Member

```bash
POST /groups/{group-email}/members
Headers:
  Authorization: Bearer {token}
  data-partition-id: {partition}
  Content-Type: application/json

Body:
{
  "email": "user@example.com",
  "role": "MEMBER"  // or "OWNER"
}

Response: 200 OK
{
  "email": "user@example.com",
  "role": "MEMBER"
}
```

### Remove Group Member

```bash
DELETE /groups/{group-email}/members/{member-email}
Headers:
  Authorization: Bearer {token}
  data-partition-id: {partition}

Response: 204 No Content
```

### List All Groups

```bash
GET /groups
Headers:
  Authorization: Bearer {token}
  data-partition-id: {partition}

Response:
{
  "groups": [
    {"email": "users.datalake.viewers@opendes.dataservices.energy", "name": "viewers"},
    ...
  ]
}
```

## Member Roles

| Role | Permissions |
|------|------------|
| `MEMBER` | Access granted by the group |
| `OWNER` | Can add/remove members from the group |

## Owner Management

### Understanding Ownership

Each member in an OSDU entitlement group has one of two membership types:

- **OWNER**: Can manage the group (add/remove other members)
- **MEMBER**: Has access granted by the group but cannot manage membership

A group should always have at least one OWNER to allow membership management. Removing all OWNERs "orphans" the group, making it impossible to manage without admin intervention.

### Last-Owner Protection

The `osdu_manage.py` script implements client-side protection against orphaning groups:

1. Before removing a user, it checks if they are an OWNER
2. If OWNER, it counts how many OWNERs exist in the group
3. If only one OWNER remains, removal is blocked by default
4. Use `--force` to override this protection when intentional

**Note**: The OSDU Entitlements API does not enforce this protection at the server level. The protection is implemented in the CLI script for safety.

### Changing Membership Type

To change a user's membership type without removing them:

```bash
# Using the update command
uv run osdu_manage.py update --user "user@example.com" --role Viewer --to OWNER

# This is equivalent to:
# 1. DELETE /groups/{group}/members/{user}
# 2. POST /groups/{group}/members with {"email": user, "role": "OWNER"}
```

The `update` command performs an atomic remove+add operation with the same last-owner protection as the `remove` command.

### Recommended Workflow

When transferring ownership:

1. **Add new owner first**: `add --user newowner@example.com --role Admin --as-owner`
2. **Then demote or remove old owner**: `update --user oldowner@example.com --role Admin --to MEMBER`

This ensures the group is never orphaned.

## Common Patterns

### Grant Standard Access

```bash
# Typical new user setup
POST /groups/users.datalake.viewers@{partition}.{domain}/members
{"email": "user@example.com", "role": "MEMBER"}
```

### Grant Elevated Access

```bash
# Editor access (can modify data)
POST /groups/users.datalake.editors@{partition}.{domain}/members
{"email": "user@example.com", "role": "MEMBER"}
```

### Grant Admin Access

```bash
# Admin access (use carefully)
POST /groups/users.datalake.admins@{partition}.{domain}/members
{"email": "admin@example.com", "role": "MEMBER"}
```

### Make Group Owner

```bash
# Can manage other members
POST /groups/users.datalake.viewers@{partition}.{domain}/members
{"email": "manager@example.com", "role": "OWNER"}
```

## HTTP Status Codes

| Code | Meaning |
|------|---------|
| 200 | Success (with response body) |
| 204 | Success (no response body) |
| 400 | Bad request (invalid email format) |
| 401 | Unauthorized (invalid/expired token) |
| 403 | Forbidden (insufficient permissions) |
| 404 | Not found (group or member doesn't exist) |
| 409 | Conflict (member already exists) |

## Required App Permissions

The app registration needs these OSDU API permissions:

| Permission | Required For |
|-----------|-------------|
| `Entitlements.Read` | List groups and members |
| `Entitlements.Write` | Add/remove members |
| `Entitlements.Admin` | Create/delete groups |

## Best Practices

1. **Least Privilege**: Start with Viewer, escalate as needed
2. **Use Email Addresses**: Easier to track than GUIDs
3. **Avoid Root**: Use Admin instead when possible
4. **Document Changes**: Keep audit trail of access changes
5. **Review Regularly**: Periodic access reviews
6. **Use OWNER Sparingly**: Only for those managing access

## Troubleshooting

### Token Issues

```bash
# Test token acquisition
curl -s -X POST \
  "https://login.microsoftonline.com/${AI_OSDU_TENANT_ID}/oauth2/token" \
  -d "grant_type=client_credentials&client_id=${AI_OSDU_CLIENT}&client_secret=${AI_OSDU_SECRET}&resource=${AI_OSDU_CLIENT}" \
  | jq '.access_token // .error_description'
```

### Group Not Found

```bash
# List all groups to find correct name
curl -s -X GET \
  "https://${HOST}/api/entitlements/v2/groups" \
  -H "Authorization: Bearer ${TOKEN}" \
  -H "data-partition-id: ${PARTITION}" \
  | jq '.groups[].email'
```

### Permission Denied

Check that the app registration has the required API permissions in OSDU configuration.

### Cannot Remove Last Owner

**Error**: `Cannot remove last OWNER from {role}. Use --force to override.`

**Cause**: Attempting to remove or demote the only OWNER of a group.

**Solutions**:
1. **Add another owner first**:
   ```bash
   uv run osdu_manage.py add --user "newowner@example.com" --role Admin --as-owner
   ```
2. **Then remove the original owner**:
   ```bash
   uv run osdu_manage.py remove --user "oldowner@example.com" --role Admin
   ```
3. **Or force removal** (use with caution - orphans the group):
   ```bash
   uv run osdu_manage.py remove --user "owner@example.com" --role Admin --force
   ```
